<?php
/**
 * Setup handler class for admin page and AJAX processing
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Fresh_Install_Assistant_Setup_Handler {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Register AJAX handlers
        add_action('wp_ajax_fresh_install_assistant_run_setup', array($this, 'ajax_run_setup'));
        add_action('wp_ajax_fresh_install_assistant_deactivate', array($this, 'ajax_deactivate_plugin'));
    }
    
    /**
     * Render the setup page
     */
    public function render_setup_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'fresh-install-assistant'));
        }
        
        // Include the template
        include FRESH_INSTALL_ASSISTANT_PLUGIN_DIR . 'templates/setup-page.php';
    }
    
    /**
     * AJAX handler for running setup
     */
    public function ajax_run_setup() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'fresh_install_assistant_nonce')) {
            wp_die(esc_html__('Security check failed', 'fresh-install-assistant'));
        }
        
        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions', 'fresh-install-assistant'));
        }
        
        // Get selected options with proper sanitization
        $cleanup_tasks = isset($_POST['cleanup_tasks']) ? array_map('sanitize_text_field', wp_unslash($_POST['cleanup_tasks'])) : array();
        $setup_tasks = isset($_POST['setup_tasks']) ? array_map('sanitize_text_field', wp_unslash($_POST['setup_tasks'])) : array();
        $theme_task = isset($_POST['theme_task']) ? sanitize_text_field(wp_unslash($_POST['theme_task'])) : '';

        // Get and sanitize site info
        $site_title = isset($_POST['site_title']) ? sanitize_text_field(wp_unslash($_POST['site_title'])) : '';
        $site_tagline = isset($_POST['site_tagline']) ? sanitize_text_field(wp_unslash($_POST['site_tagline'])) : '';

        // Get and sanitize selected pages
        $pages_to_create = isset($_POST['pages_to_create']) ? array_map('sanitize_text_field', wp_unslash($_POST['pages_to_create'])) : array();

        $results = array(
            'cleanup' => array(),
            'setup' => array(),
            'theme' => array(),
            'success' => true
        );

        // Prepare sanitized data to pass to task classes
        $task_data = array(
            'site_title' => $site_title,
            'site_tagline' => $site_tagline,
            'pages_to_create' => $pages_to_create
        );
        
        // Process cleanup tasks
        if (!empty($cleanup_tasks)) {
            $cleanup = new Fresh_Install_Assistant_Cleanup_Tasks();
            
            foreach ($cleanup_tasks as $task) {
                // Validate task name to prevent arbitrary method calls
                if (!preg_match('/^[a-z_]+$/', $task)) {
                    continue;
                }
                
                $method = 'task_' . $task;
                if (method_exists($cleanup, $method)) {
                    try {
                        $result = $cleanup->$method();
                        $results['cleanup'][$task] = $result;
                        if ($result['status'] === 'failed') {
                            $results['success'] = false;
                        }
                    } catch (Exception $e) {
                        $results['cleanup'][$task] = array(
                            'status' => 'failed',
                            /* translators: %s: error message */
                            'message' => sprintf(__('Error: %s', 'fresh-install-assistant'), $e->getMessage())
                        );
                        $results['success'] = false;
                    }
                }
            }
        }
        
        // Process setup tasks
        if (!empty($setup_tasks)) {
            $setup = new Fresh_Install_Assistant_Setup_Tasks($task_data);
            
            foreach ($setup_tasks as $task) {
                // Validate task name to prevent arbitrary method calls
                if (!preg_match('/^[a-z_]+$/', $task)) {
                    continue;
                }
                
                $method = 'task_' . $task;
                if (method_exists($setup, $method)) {
                    try {
                        $result = $setup->$method();
                        $results['setup'][$task] = $result;
                        if ($result['status'] === 'failed') {
                            $results['success'] = false;
                        }
                    } catch (Exception $e) {
                        $results['setup'][$task] = array(
                            'status' => 'failed',
                            /* translators: %s: error message */
                            'message' => sprintf(__('Error: %s', 'fresh-install-assistant'), $e->getMessage())
                        );
                        $results['success'] = false;
                    }
                }
            }
        }
        
        // Process theme installation
        if (!empty($theme_task) && $theme_task !== 'none') {
            // Validate theme task name
            if (preg_match('/^[a-z-]+$/', $theme_task)) {
                try {
                    $theme_installer = new Fresh_Install_Assistant_Theme_Tasks();
                    $result = $theme_installer->install_theme($theme_task);
                    $results['theme'][$theme_task] = $result;
                    if ($result['status'] === 'failed') {
                        $results['success'] = false;
                    }
                } catch (Exception $e) {
                    $results['theme'][$theme_task] = array(
                        'status' => 'failed',
                        /* translators: %s: error message */
                        'message' => sprintf(__('Error: %s', 'fresh-install-assistant'), $e->getMessage())
                    );
                    $results['success'] = false;
                }
            }
        }
        
        // Return results
        wp_send_json_success($results);
    }
    
    /**
     * AJAX handler for plugin deactivation
     */
    public function ajax_deactivate_plugin() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'fresh_install_assistant_nonce')) {
            wp_die(esc_html__('Security check failed', 'fresh-install-assistant'));
        }
        
        // Check capabilities
        if (!current_user_can('activate_plugins')) {
            wp_die(esc_html__('Insufficient permissions', 'fresh-install-assistant'));
        }
        
        // Deactivate the plugin
        deactivate_plugins(plugin_basename(FRESH_INSTALL_ASSISTANT_PLUGIN_FILE));
        
        wp_send_json_success(array(
            'redirect_url' => admin_url()
        ));
    }
}