/**
 * Fresh Install Assistant (FIA) Admin JavaScript
 * Version: 1.0.2
 */

jQuery(document).ready(function($) {
    'use strict';
    
    // Handle select all/deselect all functionality
    $(document).on('change', '.select-all', function() {
        var section = $(this).data('section');
        var isChecked = $(this).is(':checked');
        
        $('input[name="' + section + '_tasks[]"]').prop('checked', isChecked);
    });
    
    // Update select all checkbox when individual checkboxes change
    $('input[name$="_tasks[]"]').on('change', function() {
        var section = $(this).attr('name').replace('_tasks[]', '');
        var allChecked = $('input[name="' + section + '_tasks[]"]:not(:checked)').length === 0;
        $('.select-all[data-section="' + section + '"]').prop('checked', allChecked);
    });
    
    // Toggle site info fields based on checkbox
    $('input[value="set_site_info"]').on('change', function() {
        if ($(this).is(':checked')) {
            $('.site-info-fields').slideDown(200);
        } else {
            $('.site-info-fields').slideUp(200);
        }
    });
    
    // Toggle pages selection based on checkbox - uncheck all pages instead of hiding
    $('input[value="create_pages"]').on('change', function() {
        if ($(this).is(':checked')) {
            // Check all page checkboxes when 'Create starter pages' is checked
            $('.page-checkbox').prop('checked', true);
        } else {
            // Uncheck all page checkboxes when 'Create starter pages' is unchecked
            $('.page-checkbox').prop('checked', false);
            // Also uncheck 'Set Blog page as posts page' since there's no blog page
            $('input[value="set_blog_page"]').prop('checked', false);
        }
    });
    
    // Handle blog page checkbox interaction with set_blog_page task
    $('input[value="blog"].page-checkbox').on('change', function() {
        if (!$(this).is(':checked')) {
            // If blog page is unchecked, also uncheck 'set blog page as posts page' task
            $('input[value="set_blog_page"]').prop('checked', false);
        }
    });
    
    // Handle set_blog_page task checkbox interaction
    $('input[value="set_blog_page"]').on('change', function() {
        if ($(this).is(':checked')) {
            // If 'Set Blog page as posts page' is checked, ensure blog page is also checked
            $('input[value="blog"].page-checkbox').prop('checked', true);
            // Also ensure 'Create starter pages' is checked
            $('input[value="create_pages"]').prop('checked', true);
            // Update the indeterminate state
            updateCreatePagesCheckbox();
        }
    });
    
    // Function to update 'Create starter pages' checkbox state
    function updateCreatePagesCheckbox() {
        var anyChecked = $('.page-checkbox:checked').length > 0;

        // Update the main checkbox state - checked if ANY pages are selected
        if (anyChecked) {
            $('input[value="create_pages"]').prop('checked', true);
        } else {
            $('input[value="create_pages"]').prop('checked', false);
        }
    }
    
    // Update 'Create starter pages' checkbox based on individual page selections
    $('.page-checkbox').on('change', function() {
        updateCreatePagesCheckbox();
    });
    
    // Handle form submission
    $('#fresh-install-assistant-form').on('submit', function(e) {
        e.preventDefault();

        // Disable submit button
        $('#fresh-install-assistant-run').prop('disabled', true);

        // Gather form data directly as object
        var data = {
            action: 'fresh_install_assistant_run_setup',
            nonce: freshInstallAssistant.nonce,
            cleanup_tasks: [],
            setup_tasks: [],
            theme_task: ''
        };

        // Get cleanup tasks
        $('input[name="cleanup_tasks[]"]:checked').each(function() {
            data.cleanup_tasks.push($(this).val());
        });

        // Get setup tasks
        $('input[name="setup_tasks[]"]:checked').each(function() {
            data.setup_tasks.push($(this).val());
        });

        // Get theme task
        data.theme_task = $('input[name="theme_task"]:checked').val() || 'none';

        // Get site info
        data.site_title = $('#site_title').val();
        data.site_tagline = $('#site_tagline').val();

        // Get selected pages
        data.pages_to_create = [];
        $('input[name="pages_to_create[]"]:checked').each(function() {
            data.pages_to_create.push($(this).val());
        });

        // Hide form and show progress, then run AJAX
        $('#fresh-install-assistant-setup-form').slideUp(300, function() {
            // Remove force-hidden class before showing progress
            $('#fresh-install-assistant-progress').removeClass('force-hidden').slideDown(300, function() {
                // Run AJAX request after animations complete
                $.ajax({
            url: freshInstallAssistant.ajaxUrl,
            type: 'POST',
            data: data,
            success: function(response) {
                // Stop any animations and hide progress section immediately
                $('#fresh-install-assistant-progress').stop(true, true).hide().css('display', 'none').addClass('force-hidden');
                $('#progress-text').text(''); // Clear the progress text

                try {
                    if (response.success) {
                        displayResults(response.data);

                        // Show results
                        $('#fresh-install-assistant-results').slideDown(300);

                        if (response.data.success) {
                            // Show completion message after results
                            setTimeout(function() {
                                showCompletionMessage();
                            }, 500);
                        }
                    } else {
                        var errorMsg = response.data && response.data.message ? response.data.message : 'Setup failed. Please try again.';
                        displayError(errorMsg);
                        $('#fresh-install-assistant-results').slideDown(300);
                    }
                } catch(e) {
                    console.error('Error processing response:', e);
                    displayError('Invalid response format. Please try again.');
                    $('#fresh-install-assistant-results').slideDown(300);
                }
            },
            error: function(xhr, status, error) {
                // Stop animations and hide progress on error too
                $('#fresh-install-assistant-progress').stop(true, true).hide().css('display', 'none').addClass('force-hidden');

                console.error('AJAX error:', status, error);
                displayError('An error occurred: ' + status + '. Please try again.');
                $('#fresh-install-assistant-results').slideDown(300);
            },
            complete: function() {
                // Ensure progress is hidden (fallback with animation stop)
                $('#fresh-install-assistant-progress').stop(true, true).hide().css('display', 'none').addClass('force-hidden');
            }
                });
            });
        });
    });
    
    // Display results
    function displayResults(data) {
        var html = '';
        
        // Cleanup results
        if (data.cleanup && Object.keys(data.cleanup).length > 0) {
            html += '<div class="fresh-install-assistant-result-section">';
            html += '<h3>' + 'Essential Cleanup Results' + '</h3>';
            
            $.each(data.cleanup, function(task, result) {
                html += formatResultItem(task, result);
            });
            
            html += '</div>';
        }
        
        // Setup results
        if (data.setup && Object.keys(data.setup).length > 0) {
            html += '<div class="fresh-install-assistant-result-section">';
            html += '<h3>' + 'Initial Site Setup Results' + '</h3>';
            
            $.each(data.setup, function(task, result) {
                html += formatResultItem(task, result);
            });
            
            html += '</div>';
        }
        
        // Theme results
        if (data.theme && Object.keys(data.theme).length > 0) {
            html += '<div class="fresh-install-assistant-result-section">';
            html += '<h3>' + 'Theme Installation Results' + '</h3>';
            
            $.each(data.theme, function(task, result) {
                html += formatResultItem(task, result);
            });
            
            html += '</div>';
        }
        
        // Handle any other unexpected sections (for debugging)
        $.each(data, function(section, tasks) {
            if (section !== 'cleanup' && section !== 'setup' && section !== 'theme' && section !== 'success') {
                console.warn('Unexpected section in results:', section);
                // Try to display it properly anyway
                if (typeof tasks === 'object' && tasks !== null) {
                    html += '<div class="fresh-install-assistant-result-section">';
                    html += '<h3>' + getTaskDisplayName(section) + ' Results</h3>';
                    
                    $.each(tasks, function(task, result) {
                        html += formatResultItem(task, result);
                    });
                    
                    html += '</div>';
                }
            }
        });
        
        $('#fresh-install-assistant-results-content').html(html);
    }
    
    // Format individual result item
    function formatResultItem(task, result) {
        var statusIcon = '';
        var taskName = getTaskDisplayName(task);
        
        switch(result.status) {
            case 'completed':
                statusIcon = '✅';
                break;
            case 'skipped':
                statusIcon = '⏭️';
                break;
            case 'failed':
                statusIcon = '❌';
                break;
        }
        
        return '<div class="fresh-install-assistant-result-item ' + result.status + '">' +
               '<span class="status-icon">' + statusIcon + '</span>' +
               '<span class="task-name">' + taskName + '</span>' +
               '<span class="task-message">' + $('<div>').text(result.message).html() + '</span>' +
               '</div>';
    }
    
    // Get display name for task
    function getTaskDisplayName(task) {
        var names = {
            'remove_hello_world': 'Remove Hello World Post',
            'remove_sample_page': 'Remove Sample Page',
            'rename_uncategorized': 'Rename Uncategorized Category',
            'set_permalinks': 'Set Permalink Structure',
            'remove_hello_dolly': 'Remove Hello Dolly Plugin',
            'remove_akismet': 'Remove Akismet Plugin',
            'disable_comments': 'Disable Comments',
            'hide_dashboard_widgets': 'Hide Dashboard Widgets',
            'set_site_info': 'Set Site Title and Tagline',
            'create_pages': 'Create Pages',
            'set_front_page': 'Set Front Page',
            'set_blog_page': 'Set Blog Page as Posts Page',
            'create_menu': 'Create Navigation Menu',
            'astra': 'Astra Theme',
            'generatepress': 'GeneratePress Theme',
            'oceanwp': 'OceanWP Theme',
            'blocksy': 'Blocksy Theme',
            'neve': 'Neve Theme',
            'hello-elementor': 'Hello Elementor Theme',
            'kadence': 'Kadence Theme',
            'blog': 'Blog Page'
        };
        
        return names[task] || task;
    }
    
    // Display error message (with proper escaping)
    function displayError(message) {
        var escapedMessage = $('<div>').text(message).html();
        $('#fresh-install-assistant-results-content').html(
            '<div class="notice notice-error"><p>' + escapedMessage + '</p></div>'
        );
    }
    
    // Show completion message and deactivate
    function showCompletionMessage() {
        $('#fresh-install-assistant-complete').slideDown(300);
        
        // Deactivate the plugin silently
        $.ajax({
            url: freshInstallAssistant.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fresh_install_assistant_deactivate',
                nonce: freshInstallAssistant.nonce
            },
            success: function(response) {
                // Plugin deactivated, update the message (with proper escaping)
                var message = freshInstallAssistant.strings.deactivated || 'The plugin has been deactivated. You can now safely delete it from the Plugins page.';
                var escapedMessage = $('<div>').text(message).html();
                $('#deactivation-message').html(escapedMessage);
            },
            error: function() {
                // If deactivation fails, still show a helpful message
                $('#deactivation-message').html('Setup completed successfully! You can manually deactivate and delete this plugin from the Plugins page.');
            }
        });
    }
    
});