<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly 
/**
 * Freightmate_Rest_Api Class
 *
 * A class to handle REST API requests related to consignments.
 */
class Freightmate_Rest_Api
{

    /**
     * @var mixed $freightmate_connect An instance of the object used to make API requests.
     */
    public $freightmate_connect;

    /**
     * Constructor
     *
     * Initializes the Freightmate_Rest_Api object with the provided $freightmate_connect.
     *
     * @param mixed $freightmate_connect An object used to make API requests.
     */
    public function __construct($freightmate_connect)
    {
        $this->freightmate_connect = $freightmate_connect;
    }

    /**
     * Get Consignment
     *
     * Retrieves consignment data from the API.
     *
     * @param string $endpoint The endpoint to which the request will be made.
     * @param array $args Optional. Additional arguments to be appended to the endpoint URL. Default is an array with 'state' set to 'OPEN'.
     * @return array The retrieved consignment data.
     */
    public static function get_consignment($freightmate_connect, $args = ['state' => 'OPEN', 'sort' => 'DESC'])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/consignment';

        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);

        // Make the API request
        $consignment = $freightmate_connect->get($endpoint);
        $consignment = json_decode($consignment, true);
        // Check if consignment data is retrieved successfully
        if ($consignment && isset($consignment['data']['results']) && !empty($consignment['data']['results'])) {
            return $consignment; // Return consignment data
        } else {
            // Log the error if consignment data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($consignment));
            return $consignment; // Return empty consignment data
        }
    }
    public static function get_items($freightmate_connect, $args = ['statusId' => true])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/item';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $items = $freightmate_connect->get($endpoint);
        $items = json_decode($items, true);
        // Check if items data is retrieved successfully
        if ($items && isset($items['data']['results']) && !empty($items['data']['results'])) {
            return $items; // Return items data
        } else {
            // Log the error if items data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($items));
            return $items; // Return empty items data
        }
    }
    public static function get_itemsById($freightmate_connect, $args = ['itemTypeId' => ''])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/item/byId';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $items = $freightmate_connect->get($endpoint);
        $items = json_decode($items, true);
        // Check if items data is retrieved successfully
        if ($items && isset($items['data']['results']) && !empty($items['data']['results'])) {
            return $items; // Return items data
        } else {
            // Log the error if items data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($items));
            return $items; // Return empty items data
        }
    }

    public static function get_client_address($freightmate_connect, $is_formatted = false)
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/address/default';
        // Make the API request
        $client_address = $freightmate_connect->get($endpoint);
        $client_address = json_decode($client_address, true);
        // Check if consignment data is retrieved successfully
        if ($client_address && isset($client_address['data']) && !empty($client_address['data'])) {
            if ($is_formatted == true) {
                $clientAddress = $client_address['data'];
                $company = isset($clientAddress['company']) ? $clientAddress['company'] : '';
                $addressLine1 = isset($clientAddress['addressLine1']) ? $clientAddress['addressLine1'] : '';
                $addressLine2 = isset($clientAddress['addressLine2']) ? $clientAddress['addressLine2'] : '';
                $suburb = isset($clientAddress['suburb']) ? $clientAddress['suburb']['name'] : '';
                $postcode = isset($clientAddress['suburb']['postcode']) ? $clientAddress['suburb']['postcode'] : '';
                $clientId = isset($clientAddress['id']) ? $clientAddress['id'] : '';
                $address = [$addressLine1, $addressLine2, $suburb, $postcode];
                return ['company' => $company, 'clientId' => $clientId,  'address' => implode(',', $address)];
            }
            return $client_address; // Return client_address data
        } else {
            // Log the error if client_address data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($client_address));
            return $client_address; // Return empty client_address data
        }
    }
    public static function get_carriers($freightmate_connect, $args = ['page' => 0, 'size' => 10])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/carrier/services';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $carriers = $freightmate_connect->get($endpoint);
        $carriers = json_decode($carriers, true);
        // Check if carriers data is retrieved successfully
        if ($carriers && isset($carriers['data']['results']) && !empty($carriers['data']['results'])) {
            return $carriers; // Return carriers data
        } else {
            // Log the error if carriers data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($carriers));
            return $carriers; // Return empty carriers data
        }
    }
    public static function search_address($freightmate_connect, $args = ['filter' => '', 'size' => 10])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/address';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $address = $freightmate_connect->get($endpoint);
        $address = json_decode($address, true);
        // Check if address data is retrieved successfully
        if ($address && isset($address['data']['results']) && !empty($address['data']['results'])) {
            return $address; // Return address data
        } else {
            // Log the error if address data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($address));
            return $address; // Return empty address data
        }
    }
    public static function search_suburb($freightmate_connect, $args = ['filter' => '', 'size' => 10])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/address/suburb';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $suburb = $freightmate_connect->get($endpoint);
        $suburb = json_decode($suburb, true);
        // Check if suburb data is retrieved successfully
        if ($suburb && isset($suburb['data']['results']) && !empty($suburb['data']['results'])) {
            return $suburb; // Return suburb data
        } else {
            // Log the error if suburb data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($suburb));
            return $suburb; // Return empty suburb data
        }
    }
    public static function get_offers($freightmate_connect, $body)
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/consignment/offers';
        // Make the API request
        $offers = $freightmate_connect->post($endpoint, $body);
        $offers = json_decode($offers, true);
        // Check if offers data is retrieved successfully
        if ($offers && isset($offers['data']['results']) && !empty($offers['data']['results'])) {
            return $offers; // Return offers data
        } else {
            // Log the error if offers data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($offers));
            return $offers; // Return empty offers data
        }
    }
    public static function create_consignment($freightmate_connect, $body)
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/consignment';
        // Make the API request
        $consignment = $freightmate_connect->post($endpoint, $body);
        $consignment = json_decode($consignment, true);

        // Check if consignment data is retrieved successfully
        if ($consignment && isset($consignment['data']['results']) && !empty($consignment['data']['results'])) {
            return $consignment; // Return consignment data
        } else {
            // Log the error if consignment data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($consignment));
        }
        // Return the last response after max retries
        return $consignment; // Return empty consignment data or the last failed response
    }
    public static function delete_consignment($freightmate_connect, $args = ['connoteNumbers' => ''])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/consignment';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $consignment = $freightmate_connect->delete($endpoint);
        $consignment = json_decode($consignment, true);
        // Check if consignment data is retrieved successfully
        // Log the error if consignment data retrieval fails
        Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($consignment));
        return $consignment; // Return empty consignment data
    }
    public static function track_consignment($freightmate_connect, $args = ['connote_number' => ''])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/tracking';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $trackConsignment = $freightmate_connect->get($endpoint);
        $trackConsignment = json_decode($trackConsignment, true);
        // Check if trackConsignment data is retrieved successfully
        // Log the error if trackConsignment data retrieval fails
        Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($trackConsignment));
        return $trackConsignment; // Return empty trackConsignment data
    }
    public static function manifest_list($freightmate_connect, $args = ['page' => 0, 'size' => 10])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/manifest';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $manifestList = $freightmate_connect->get($endpoint);
        $manifestList = json_decode($manifestList, true);
        // Check if manifestList data is retrieved successfully
        // Log the error if manifestList data retrieval fails
        Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($manifestList));
        return $manifestList; // Return empty manifestList data
    }
    public static function create_manifest($freightmate_connect, $body)
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/manifest';
        // Make the API request
        $manifest = $freightmate_connect->post($endpoint, $body);
        $manifest = json_decode($manifest, true);
        // Check if manifest data is retrieved successfully
        if ($manifest && isset($manifest['data']['results']) && !empty($manifest['data']['results'])) {
            return $manifest; // Return manifest data
        } else {
            // Log the error if manifest data retrieval fails
            Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($manifest));
        }
        // Return the last response after max retries
        return $manifest; // Return empty consignment data or the last failed response
    }
    public static function print_manifest($freightmate_connect, $args = ['manifestId' => ''])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/paperwork/manifest';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $printManifest = $freightmate_connect->get($endpoint);
        $printManifest = json_decode($printManifest, true);
        // Check if printManifest data is retrieved successfully
        // Log the error if printManifest data retrieval fails
        Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($printManifest));
        return $printManifest; // Return empty printManifest data
    }
    public static function print_manifest_label($freightmate_connect, $args = ['manifestId' => ''])
    {
        // Set the default endpoint
        $endpoint = 'b2b/v1/paperwork/label';
        // Append any additional arguments to the endpoint
        $endpoint = add_query_arg($args, $endpoint);
        // Make the API request
        $printManifestLabel = $freightmate_connect->get($endpoint);
        $printManifestLabel = json_decode($printManifestLabel, true);
        // Check if printManifestLabel data is retrieved successfully
        // Log the error if printManifestLabel data retrieval fails
        Freightmate_Logs_Table_Manager::insert_log(wp_json_encode($printManifestLabel));
        return $printManifestLabel; // Return empty printManifestLabel data
    }
}
