<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly 
if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/screen.php');
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}
class Freightmate_Order_List_Table extends WP_List_Table
{

    public function __construct()
    {
        parent::__construct(array(
            'singular' => 'order',
            'plural'   => 'orders',
            'ajax'     => false
        ));
    }

    public function no_items()
    {
        esc_html_e('No orders found.', 'freightmate-for-woocommerce');
    }

    public function column_default($item, $column_name)
    {
        return $item[$column_name];
    }

    public function column_order_id($item)
    {
        $actions = array(
            'view' => sprintf(
                '<a href="?page=%s&action=%s&order_id=%s">%s</a>', 
                isset($_REQUEST['page']) ? esc_attr(sanitize_text_field(wp_unslash($_REQUEST['page']))) : '', 
                'view', 
                isset($item['order_id']) ? esc_attr($item['order_id']) : '', 
                __('View', 'freightmate-for-woocommerce')
            ),
        );
        
        return sprintf('%1$s %2$s', esc_html($item['order_id'] ?? ''), $this->row_actions($actions));        
    }

    public function prepare_items()
    {
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();

        $this->_column_headers = array($columns, $hidden, $sortable);

        $data = $this->list_orders_data(); // Function to get order data
        //usort($data, array(&$this, 'usort_reorder'));

        $per_page = 10;
        $current_page = $this->get_pagenum();
        $total_items = count($data);

        $data = array_slice($data, (($current_page - 1) * $per_page), $per_page);

        $this->items = $data;

        $this->set_pagination_args(array(
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ));
    }

    public function get_columns()
    {
        $columns = array(
            'order'       => __('Order', 'freightmate-for-woocommerce'),
            'customer_name'  => __('Customer Name', 'freightmate-for-woocommerce'),
            'status'  => __('Status', 'freightmate-for-woocommerce'),
            'date'  => __('Date', 'freightmate-for-woocommerce'),
            'total'  => __('Total', 'freightmate-for-woocommerce'),
            'action'  => __('Action', 'freightmate-for-woocommerce'),
            // Add more columns as needed
        );
        return $columns;
    }

    public function get_sortable_columns()
    {
        $sortable_columns = array(
            'order'      => array('order', false),
            'customer_name' => array('customer_name', false),
            'status' => array('status', false),
            'date' => array('date', false),
            'total' => array('total', false),
            'action' => array('action', true),
            // Add more sortable columns as needed
        );
        return [];
        return $sortable_columns;
    }

    private function list_orders_data()
    {
        $paged = (get_query_var('paged')) ? get_query_var('paged') : 1;
        $per_page = 10;
        $search = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
        $order_ids = freightmateCommonMethods::getEligibleOrderIdsConsignment();
        $args = array(
            'limit'     => $per_page,
            'paged'     => $paged,
            's'         => $search,
            'exclude' => $order_ids,
            'status' => 'processing',
            'orderby' => 'ID', 
            'order' => 'DESC'
        );
        $orderData = [];
        $orders = wc_get_orders($args);
        foreach ($orders as $order) {
            $orderInfo = $this->get_order_column_data_by_order($order);
            $orderData[] = array(
                'order'       => $orderInfo['order'],
                'customer_name'  => $orderInfo['buyer'],
                'status' => $orderInfo['status'],
                'date' => $orderInfo['date'],
                'total' => $orderInfo['total'],
                'action' => $orderInfo['consignment_link']
            );
        }
        return $orderData;
    }

    function usort_reorder($a, $b)
    {
        $orderby = (isset($_REQUEST['orderby']) && !empty($_REQUEST['orderby'])) ? esc_attr(sanitize_text_field(wp_unslash($_REQUEST['orderby']))) : 'order';
        $order = (isset($_REQUEST['order']) && !empty($_REQUEST['order'])) ? esc_attr(sanitize_text_field(wp_unslash($_REQUEST['order']))) : 'asc';
        $result = strcmp($a[$orderby], $b[$orderby]);
        return ($order === 'asc') ? $result : -$result;
    }
    function get_order_column_data_by_order($order)
    {
        $buyer = '';
        if ($order->get_billing_first_name() || $order->get_billing_last_name()) {
            // translators: %1$s is the first name, %2$s is the last name
            $buyer = trim(sprintf(_x('%1$s %2$s', 'full name', 'freightmate-for-woocommerce'), $order->get_billing_first_name(), $order->get_billing_last_name()));
        } elseif ($order->get_billing_company()) {
            $buyer = trim($order->get_billing_company());
        } elseif ($order->get_customer_id()) {
            $user  = get_user_by('id', $order->get_customer_id());
            $buyer = ucwords($user->display_name);
        }
        $data_order_id =  '<strong>#' . esc_attr($order->get_order_number()) . ' ' . esc_html($buyer) . '</strong>';
        $status_name = wc_get_order_status_name($order->get_status());
        if ($order->get_payment_method_title()) {
            /* translators: %s: method */
            $total =  '<span class="tips" data-tip="' . esc_attr(sprintf(__('via %s', 'freightmate-for-woocommerce'), $order->get_payment_method_title())) . '">' . wp_kses_post($order->get_formatted_order_total()) . '</span>';
        } else {
            $total =  wp_kses_post($order->get_formatted_order_total());
        }
        $order_timestamp = $order->get_date_created() ? $order->get_date_created()->getTimestamp() : '';
        $show_date = '';
        if ($order_timestamp > strtotime('-1 day', time()) && $order_timestamp <= time()) {

            
            $show_date = sprintf(
                // translators: %s is the ago (e.g. "5 minutes ago"), %s is the human-readable time difference
                _x('%s ago', '%s = human-readable time difference', 'freightmate-for-woocommerce'),
                human_time_diff($order->get_date_created()->getTimestamp(), time())
            );
        } else {
            $show_date = $order->get_date_created()->date_i18n(apply_filters('woocommerce_admin_order_date_format', __('M j, Y', 'freightmate-for-woocommerce'))); // phpcs:ignore WooCommerce.Commenting.CommentHooks.MissingHookComment
        }
        // Construct the final data
        $final_date = sprintf(
            '<time datetime="%1$s" title="%2$s">%3$s</time>',
            esc_attr($order->get_date_created()->date('c')),
            esc_html($order->get_date_created()->date_i18n(get_option('date_format') . ' ' . get_option('time_format'))),
            esc_html($show_date)
        );
        $create_consignment = '<div class="create-order-consignment-wrap"><a href="' . esc_url( add_query_arg( ['order_id' => $order->get_id(), 'action' => 'create-order-consignment'], $this->get_order_edit_link( $order )) ) . '" class="create-order-consignment">Create Consignment</a></div>';
        return [
            'order' => $data_order_id,
            'date' => $final_date,
            'total' => $total,
            'status' => $status_name,
            'buyer' => $buyer,
            'consignment_link' => $create_consignment
        ];
    }
}
