<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly 
class Freightmate_Admin_Menu
{
    public function __construct()
    {
        add_action('admin_menu', array($this, 'freightmate_add_menu_cb'));
        add_action('admin_enqueue_scripts', array($this, 'freightmate_enqueue_styles'));
    }
    public function freightmate_enqueue_styles()
    {
        $order_redirect_link = admin_url('admin.php?page=freightmate-orders');
        $consignment_redirect_link = admin_url('admin.php?page=freightmate-consignment');
        $manifested_redirect_link = admin_url('admin.php?page=freightmate-manifested-consignment');
        wp_enqueue_style('freightmate-admin-styles', FREIGHTMATE_APP_PLUGIN_URL . 'css/freightmate-admin-styles.css', '', time());
        wp_enqueue_script('freightmate-admin-js', FREIGHTMATE_APP_PLUGIN_URL . '/js/freightmate-admin-script.js', [], time(), true);
        wp_enqueue_script('freightmate-admin-header-js', FREIGHTMATE_APP_PLUGIN_URL . '/js/freightmate-admin-header-script.js', [], time(), false);
        $items = freightmateCommonMethods::getItems();
        wp_localize_script('freightmate-admin-js', 'freightmate', [
            'ajaxURL' => admin_url('admin-ajax.php'),
            'images' => FREIGHTMATE_APP_PLUGIN_URL . 'images/',
            'timezone' => freightmateCommonMethods::freightmateTimeZone(),
            'redirect' => [
                'orders' => $order_redirect_link,
                'consignment' => $consignment_redirect_link,
                'manifested' => $manifested_redirect_link
            ],
            'items' => $items,
            'nonce' => wp_create_nonce('wp_searchKeyword_ajax_action')
        ]);
    }
    public function freightmate_add_menu_cb()
    {
        $is_freightmate_active = freightmateCommonMethods::is_freightmate_active();
        $dashboard_heading = $is_freightmate_active ? 'Dashboard' : 'Connect';
        add_menu_page(
            'Freightmate',
            'Freightmate',
            'manage_options',
            'freightmate-dashboard',
            array($this, 'dashboard_page'),
            FREIGHTMATE_APP_PLUGIN_URL . '/images/Freightmate_logo.svg',
            80
        );

        $submenus = array(
            array(
                'title' => $dashboard_heading,
                'slug' => 'freightmate-dashboard',
                'callback' => array($this, 'dashboard_page')
            ),
            array(
                'title' => 'Orders',
                'slug' => 'freightmate-orders',
                'callback' => array($this, 'orders_page')
            ),
            array(
                'title' => 'Consignments',
                'slug' => 'freightmate-consignment',
                'callback' => array($this, 'consignment_page')
            ),
            array(
                'title' => 'Manifests',
                'slug' => 'freightmate-manifested-consignment',
                'callback' => array($this, 'manifested_consignment_page')
            ),
            array(
                'title' => 'Track and trace',
                'slug' => 'freightmate-track-and-trace',
                'callback' => array($this, 'track_and_trace_page')
            ),
            array(
                'title' => 'Settings',
                'slug' => 'freightmate-settings',
                'callback' => array($this, 'shipping_settings_page')
            ),
        );

        foreach ($submenus as $submenu) {
            add_submenu_page(
                'freightmate-dashboard',
                $submenu['title'],
                $submenu['title'],
                'manage_options',
                $submenu['slug'],
                $submenu['callback']
            );
        }
    }

    public function dashboard_page()
    {
        // Dashboard page content
        $response = $this->freightmate_handle_form_submission('dashboard');
        $response['heading'] = "Freightmate dashboard";
        $is_freightmate_active = freightmateCommonMethods::is_freightmate_active();
        $dashboard_template = FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-dashboard.php';
        if ($is_freightmate_active == false) {
            $response['heading'] = "Connect your Freightmate account";
            $dashboard_template = FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-connect.php';
        }
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-pages-header.php';
        require_once $dashboard_template;
    }

    /**
     * Handles the Orders Page.
     *
     * This function manages the display and functionality of the Orders page.
     * If the action is to create an order consignment, it gathers necessary data and displays the consignment creation page.
     * Otherwise, it displays the general Orders page.
     */
    public function orders_page()
    {
        $order_id = ( isset($_REQUEST['order_id']) && (sanitize_text_field(wp_unslash($_REQUEST['order_id'])))  )  ? (sanitize_text_field(wp_unslash($_REQUEST['order_id']))) : '';
        $response['heading'] = "Orders";
        $freightmate_file = FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-orders.php';
        if ( ( isset($_REQUEST['action']) && (sanitize_text_field(wp_unslash($_REQUEST['action']) )) ) && (sanitize_text_field(wp_unslash($_REQUEST['action']) )) == 'create-order-consignment') {
            // Initialize Freightmate API connection
            $freightmate_connection = new Freightmate_API_Connection(FREIGHTMATE_API_URL);
            $freightmate_api = new Freightmate_Rest_Api($freightmate_connection);

            // Retrieve necessary data for consignment creation
            $client_address = $freightmate_api->get_client_address($freightmate_connection, true);
            $items = freightmateCommonMethods::getItems();
            $carriers = $freightmate_api->get_carriers($freightmate_connection);
            // Initialize order service
            $orderRepository = new WooCommerceOrderRepository();
            $orderService = new FreightmateOrderService($orderRepository);

            // Prepare response data
            $response['items'] = $items;
            $response['carriers'] = $carriers;
            $response['client_address'] = $client_address;
            $response['shipping'] = $orderService->getOrderShippingData($order_id);
            $response['heading'] = "New Consignment for Order ID: #" . $order_id;

            // Set consignment creation page file
            $freightmate_file = FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-create-order-consignment.php';
        }
        // Include necessary files for the page
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-pages-header.php';
        require_once $freightmate_file;
    }


    public function consignment_page()
    {
        $order_id = (isset($_REQUEST['order_id']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['order_id']))) ) ? wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['order_id']))) : '';
        $consignment_id = ( isset($_REQUEST['consignment_id']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['consignment_id']))) ) ? wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['consignment_id']))) : '';

        if (isset($_REQUEST['submit']) && sanitize_text_field(wp_unslash($_REQUEST['submit'])) == 'Create manifest') {
            $response = $this->freightmate_handle_form_submission('create-manifest');
        }
        $freightmate_file =  FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-consignments.php';
        if (isset($_REQUEST['action']) && sanitize_text_field(wp_unslash($_REQUEST['action'])) == 'delete-order-consignment') {
            $response = $this->freightmate_handle_form_submission('delete-consignment');
        }
        $response['heading'] = "Consignments";
        if (isset($_REQUEST['action']) && sanitize_text_field(wp_unslash($_REQUEST['action'])) == 'create-order-manifest') {
            $response['heading'] = "New Manifest for Order ID: #" . $order_id . ' and Consignment : ' . $consignment_id;
            // Set consignment creation page file
            $freightmate_file = FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-create-manifest.php';
        }
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-pages-header.php';
        require_once $freightmate_file;
    }

    public function manifested_consignment_page()
    {
        $response['heading'] = "Manifested Consignment";
        // Include necessary files for the page
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-pages-header.php';
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-manifest.php';
    }

    public function track_and_trace_page()
    {
        // if (isset($_REQUEST['submit']) && $_REQUEST['submit']) {
        //     $response = $this->freightmate_handle_form_submission('track-consignment');
        // }
        $response['heading'] = "Track and trace";
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-pages-header.php';
        // Track and trace page content
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-track-consignment.php';
    }

    public function shipping_settings_page()
    {
         // Check if the action type is set, sanitize, unslash, and verify the nonce
        if (isset($_REQUEST['freightmate_action_type']) && ($action_type = sanitize_text_field(wp_unslash($_REQUEST['freightmate_action_type'])))) {
            
            // Verify nonce for 'save-shipping' action
            if ($action_type == 'save-shipping' && isset($_POST['freightmate_shipping_setting_nonce'])) {
                // Unslash and validate the nonce before verification
                $freightmate_shipping_nonce = sanitize_text_field(wp_unslash($_POST['freightmate_shipping_setting_nonce']));
                if (wp_verify_nonce($freightmate_shipping_nonce, 'freightmate_shipping_setting_action')) {
                    $response = $this->freightmate_handle_form_submission('shipping');
                } else {
                    $response = ['status' => 0, 'message' => 'Nonce verification failed for saving shipping settings.'];
                }
            }

            // Verify nonce for 'delete-account' action
            if ($action_type == 'delete-account' && isset($_POST['freightmate_uninstall_nonce'])) {
                // Unslash and validate the nonce before verification
                $freightmate_uninstall_nonce = sanitize_text_field(wp_unslash($_POST['freightmate_uninstall_nonce']));
                if (wp_verify_nonce($freightmate_uninstall_nonce, 'freightmate_uninstall_action')) {
                    $response = $this->freightmate_handle_form_submission('delete-account');
                } else {
                    $response = ['status' => 0, 'message' => 'Nonce verification failed for deleting account.'];
                }
            }
        }

        // Set the heading for the page
        if (!isset($response['heading'])) {
            $response['heading'] = "Shipping settings";
        }
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-pages-header.php';
        // Shipping settings page content
        require_once FREIGHTMATE_APP_PLUGIN_DIR . 'pages/freightmate-shipping-settings.php';
    }
    private function freightmate_handle_form_submission($form_type)
    {
        $response = ['status' => true, 'message' => ''];
        if (isset($_REQUEST['freightmate_action']) && $_REQUEST['freightmate_action'] == 'submit') {
            // Handle form submission logic specific to the form type
            switch ($form_type) {
                case 'dashboard':
                    // Handle form submission logic for dashboard page
                    if (!isset($_REQUEST['freightmate_creds_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['freightmate_creds_nonce'])), 'freightmate_creds_action')) {
                        $response['message'] = "Sorry, your nonce did not verify for dashboard.";
                        $response['status'] = false;
                    } else {
                        $email = sanitize_text_field(wp_unslash($_POST['freightmate-api-key'] ?? ''));
                        $password = sanitize_text_field(wp_unslash($_POST['freightmate-password'] ?? ''));
                        $environment = sanitize_text_field(wp_unslash($_POST['freightmate-environment'] ?? ''));
                        $stage_api_URL = sanitize_text_field(wp_unslash($_POST['freightmate-api-url-stage'] ?? ''));
                        $live_api_URL = sanitize_text_field(wp_unslash($_POST['freightmate-api-url-live'] ?? ''));
                        if (empty($email)) {
                            $response['message'] = "Please enter API Key!";
                            $response['status'] = false;
                        } elseif (empty($password)) {
                            $response['message'] = "Please enter password";
                            $response['status'] = false;
                        } elseif (empty($environment)) {
                            $response['message'] = "Please environment type";
                            $response['status'] = false;
                        } elseif (empty($stage_api_URL) && $environment == 'STAGE') {
                            $response['message'] = "Please enter Stage API URL";
                            $response['status'] = false;
                        } elseif (empty($live_api_URL) && $environment == 'LIVE') {
                            $response['message'] = "Please enter Live API URL";
                            $response['status'] = false;
                        } else {
                        }
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'client_api_key', $email);
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'client_password', base64_encode($password));
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'environment_type', $environment);
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'live_api_url', $live_api_URL);
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'stage_api_url', $stage_api_URL);
                        if ($response['status'] == true) {
                            wp_redirect(admin_url('admin.php?page=freightmate-dashboard'));
                        }
                    }
                    break;
                case 'shipping':
                    // Handle form submission logic for dashboard page
                    if (!isset($_REQUEST['freightmate_shipping_setting_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['freightmate_shipping_setting_nonce'])), 'freightmate_shipping_setting_action')) {
                        $response['message'] = "Sorry, your nonce did not verify for dashboard.";
                        $response['status'] = false;
                    } else {
                        $freightmate_shipping = sanitize_text_field(wp_unslash($_POST['freightmate_shipping'] ?? ''));
                        $freightmate_shipping_charges = sanitize_text_field(wp_unslash($_POST['freightmate_shipping_charges'] ?? ''));
                        $woo_defaultshipping = isset($_POST['woo_defaultshipping']) ? 1 : 0;
                        if (empty($freightmate_shipping)) {
                            $response['message'] = "Please select option for shipping.";
                            $response['status'] = false;
                        } elseif (!empty($freightmate_shipping) && in_array($freightmate_shipping, ['freightmate_flat', 'freightmate_percentage']) && empty($freightmate_shipping_charges)) {
                            $response['message'] = "Please enter number in shipping charges input.";
                            $response['status'] = false;
                        } else {
                            update_option(FREIGHTMATE_OPTION_PREFIX . 'freightmate_shipping', $freightmate_shipping);
                            update_option(FREIGHTMATE_OPTION_PREFIX . 'freightmate_shipping_charges', $freightmate_shipping_charges);
                            update_option(FREIGHTMATE_OPTION_PREFIX . 'woo_defaultshipping', $woo_defaultshipping);
                            $response['message'] = "Your setting has been save successfully!";
                        }
                    }
                    break;
                case 'delete-consignment':
                    // Handle form submission logic for dashboard page
                    if (isset($_REQUEST['action']) && $_REQUEST['action'] != 'delete-order-consignment') {
                        $response['message'] = "Sorry, your nonce did not verify for Delete consignment.";
                        $response['status'] = false;
                    } else if (!isset($_REQUEST['connoteNumber']) || empty($_REQUEST['connoteNumber'])) {
                        $response['message'] = "Invalid connoteNumber.";
                        $response['status'] = false;
                    } else {
                        // Initialize Freightmate API connection
                        $freightmate_connection = new Freightmate_API_Connection(FREIGHTMATE_API_URL);
                        $freightmate_api = new Freightmate_Rest_Api($freightmate_connection);

                        // Sanitize connoteNumber before using it
                        $connoteNumber = isset($_REQUEST['connoteNumber']) ? sanitize_text_field(wp_unslash($_REQUEST['connoteNumber'])) : '';

                        // Check if connoteNumber is provided
                        if (!empty($connoteNumber)) {
                            $delete_consignment = $freightmate_api->delete_consignment($freightmate_connection, ['connoteNumbers' => $connoteNumber]);

                            $response['message'] = "Your consignment number : " . esc_html($connoteNumber) . " has been deleted successfully!";
                            $statusCode = isset($delete_consignment['status_code']) ? $delete_consignment['status_code'] : 0;
                            $response['is_deleted'] = ($statusCode == 200) ? 1 : 0;

                            if ($delete_consignment && $statusCode == 200) {
                                if (isset($_REQUEST['order_id']) && sanitize_text_field(wp_unslash($_REQUEST['order_id'])) ) {
                                    // Sanitize order_id before using it
                                    $order_id = sanitize_text_field(wp_unslash($_REQUEST['order_id']));
                                    update_post_meta($order_id, FREIGHTMATE_META_PREFIX . 'is_consignment_created', false);
                                }
                            }
                        } else {
                            $response['message'] = "Invalid consignment number.";
                            $response['status'] = false;
                        }
                    }
                    break;
                case 'track-consignment':
                    // Handle form submission logic for dashboard page
                    if (!isset($_REQUEST['track_consignment_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['track_consignment_nonce'])), 'track_consignment_action')) {
                        $response['message'] = "Sorry, your nonce did not verify for track consignment.";
                        $response['status'] = false;
                    } else {
                        // Sanitize consignment_number before using it
                        $consignment_number = isset($_POST['consignment_number']) ? sanitize_text_field(wp_unslash($_POST['consignment_number'])) : '';

                        // Check if consignment_number is provided
                        if (!empty($consignment_number)) {
                            $response = freightmateCommonMethods::track_consignment_template($consignment_number);
                        } else {
                            $response['message'] = "Consignment number cannot be empty.";
                            $response['status'] = false;
                        }
                    }
                    break;
                case 'create-manifest':
                    // Handle form submission logic for dashboard page
                    if (
                        !isset($_REQUEST['freightmate_create_manifest_nonce']) ||
                        !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['freightmate_create_manifest_nonce'])), 'freightmate_create_manifest_action')
                    ) {
                        $response['message'] = "Sorry, your nonce did not verify for track consignment.";
                        $response['status'] = false;
                    } else if (!isset($_REQUEST['connoteNumber']) || empty($_REQUEST['connoteNumber'])) {
                        $response['message'] = "Invalid connoteNumber.";
                        $response['status'] = false;
                    } else if (!isset($_REQUEST['readyTime']) || empty($_REQUEST['readyTime'])) {
                        $response['message'] = "Invalid readyTime.";
                        $response['status'] = false;
                    } else if (!isset($_REQUEST['closingTime']) || empty($_REQUEST['closingTime'])) {
                        $response['message'] = "Invalid closingTime.";
                        $response['status'] = false;
                    } else {
                        $consignment_number = isset($_REQUEST['connoteNumber']) ? sanitize_text_field(wp_unslash($_REQUEST['connoteNumber'])) : '';
                        $readyTimeInput = isset($_REQUEST['readyTime']) ? sanitize_text_field(wp_unslash($_REQUEST['readyTime'])) : '';
                        $closingTimeInput = isset($_REQUEST['closingTime']) ? sanitize_text_field(wp_unslash($_REQUEST['closingTime'])) : '';
                        $order_id = isset($_REQUEST['order_id']) ? sanitize_text_field(wp_unslash($_REQUEST['order_id'])) : '';
                        $cut_off = isset($_REQUEST['cut_off']) ? sanitize_text_field(wp_unslash($_REQUEST['cut_off'])) : '';

                        // Get the current date
                        $currentDate = gmdate('Y-m-d', time());
                        // Consignment info
                        $consignmentRes = get_post_meta($order_id, FREIGHTMATE_META_PREFIX . 'consignment_result', true);
                        $consignmentInfo = $consignmentRes['data'] ?? [];
                        $dispatchedAt = $consignmentInfo['dispatchedAt'] ?? '';
                        $dispatchedAtdt = freightmateCommonMethods::convert_iso8601_to_wp_timezone($dispatchedAt);
                        $dispatchedDate = $dispatchedAtdt['date'];
                        $PickupStart = $consignmentInfo['preferredPickupStart'] ?? '';
                        $PickupEnd = $consignmentInfo['preferredPickupEnd'] ?? '';
                        $cutOffDateTime = freightmateCommonMethods::convert_iso8601_to_wp_timezone($cut_off);
                        $cutOffDate = $cutOffDateTime['date'];
                        $cutOffTime = $cutOffDateTime['time'];
                        $manifestDate = $dispatchedDate;
                        if ($manifestDate < $currentDate) {
                            $manifestDate = $currentDate;
                        }
                        if ($readyTimeInput > $PickupStart) {
                            // Assuming you want to add +1 day to $manifestDate if $readyTimeInput is less than $PickupStart
                            $manifestDate = gmdate('Y-m-d', strtotime('+1 day', strtotime($manifestDate)));
                        } elseif ($manifestDate < $cutOffDate) {
                            $manifestDate = $cutOffDate;
                        } elseif ($manifestDate >= $currentDate && $readyTimeInput > $cutOffTime) {
                            // Assuming you want to add +1 day to $manifestDate if $readyTimeInput is less than $cutOffTime
                            $manifestDate = gmdate('Y-m-d', strtotime('+1 day', strtotime($cutOffDate)));
                        }
                        $readyTime = freightmateCommonMethods::freightmateFormatDateTimeInUTC($manifestDate, $readyTimeInput);
                        $closingTime = freightmateCommonMethods::freightmateFormatDateTimeInUTC($manifestDate, $closingTimeInput);
                        $body = [
                            "readyTime" => $readyTime,
                            "closingTime" => $closingTime,
                            "consignmentIds" => [$consignment_number]
                        ];
                        // Initialize Freightmate API connection
                        $freightmate_connection = new Freightmate_API_Connection(FREIGHTMATE_API_URL);
                        $freightmate_api = new Freightmate_Rest_Api($freightmate_connection);
                        $manifestResult = $freightmate_api->create_manifest($freightmate_connection, wp_json_encode($body));
                        $manifestNumber = isset($manifestResult['data']) && isset($manifestResult['data']['manifestNumber']) ?  $manifestResult['data']['manifestNumber'] : '';
                        $manifestId = isset($manifestResult['data']) && isset($manifestResult['data']['id']) ?  $manifestResult['data']['id'] : '';
                        if ($manifestResult['status_code'] == 200 && $manifestNumber) {
                            if ($order_id) {
                                update_post_meta($order_id, FREIGHTMATE_META_PREFIX . 'manifest_result', $manifestResult);
                                update_post_meta($order_id, FREIGHTMATE_META_PREFIX . 'manifest_number', $manifestNumber);
                                update_post_meta($order_id, FREIGHTMATE_META_PREFIX . 'manifest_number_id', $manifestId);
                                update_post_meta($order_id, FREIGHTMATE_META_PREFIX . 'is_manifest_created', true);
                                update_post_meta($order_id, FREIGHTMATE_META_PREFIX . 'manifest_created_at', gmdate('Y-m-d H:i:s'));
                                update_post_meta($order_id, FREIGHTMATE_META_PREFIX . 'create_manifest_api_response', wp_json_encode($manifestResult));
                            }
                            $response['status'] = true;
                            $response['message'] = "Your manifest has been created successfully!";
                            wp_redirect(admin_url('admin.php?page=freightmate-consignment'));
                        } else {
                            $errorStatus = $manifestResult['data']['status'] ?? '';
                            $validationErrors = $manifestResult['data']['validationErrors'] ?? '';
                            $error_message = isset($validationErrors[0]['message']) ? $validationErrors[0]['message'] : '';
                            $response['message'] = '<strong>' . $errorStatus . '</strong>: ' . $error_message . '';
                            $response['status'] = false;
                        }
                    }
                    break;
                case 'delete-account';
                    if (
                        !isset($_REQUEST['freightmate_uninstall_nonce']) ||
                        !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['freightmate_uninstall_nonce'])), 'freightmate_uninstall_action')
                    ) {
                        $response['message'] = "Sorry, your nonce did not verify for Uninstall Account.";
                        $response['status'] = false;
                    } else {
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'client_api_key', '');
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'client_password', '');
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'environment_type', '');
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'live_api_url', '');
                        update_option(FREIGHTMATE_OPTION_PREFIX . 'stage_api_url', '');
                        wp_redirect(admin_url('admin.php?page=freightmate-dashboard'));
                    }
            }
        }
        return $response;
    }
}

new Freightmate_Admin_Menu();
