<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly 
/**
 * Freightmate_API_Connection Class
 *
 * A class to handle connections to the Freightmate API.
 */
class Freightmate_API_Connection
{

    /**
     * @var string $api_url The base URL of the Freightmate API.
     */
    private $api_url;

    /**
     * Constructor
     *
     * Initializes the Freightmate_API_Connection object with the provided $api_url.
     *
     * @param string $api_url The base URL of the Freightmate API.
     */
    public function __construct( $api_url ) {
        // Sanitize the URL input and ensure it ends with a slash
        $this->api_url = trailingslashit( esc_url_raw( $api_url ) );
    }    

    /**
     * Get Auth
     *
     * Retrieves the API authentication credentials.
     *
     * @return string The authentication credentials.
     */
    private function get_auth()
    {
        return FREIGHTMATE_API_KEY . ':' . FREIGHTMATE_API_SECRET;
    }

    /**
     * Get
     *
     * Makes a GET request to the Freightmate API.
     *
     * @param string $endpoint The API endpoint to which the request will be made.
     * @return string JSON-encoded response containing either success data or error message.
     */
    public function get($endpoint)
    {
        try {
            $url = $this->api_url . $endpoint;

            // Make GET request using WordPress HTTP API
            $response = wp_remote_get($url, [
                'headers' => [
                    'Authorization' => 'Basic ' . base64_encode($this->get_auth())
                ],
                'timeout'     => 120
            ]);
            $status_code = wp_remote_retrieve_response_code($response);

            // Check for WP_Error
            if (is_wp_error($response)) {
                $error_message = $response->get_error_message();
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => $error_message
                ];
                return wp_json_encode($error_response);
            }

            // Check for different status codes
            if ($status_code === 401) {
                // Authentication error
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => 'Authentication failed. Please check your credentials.'
                ];
                return wp_json_encode($error_response);
            } elseif ($status_code === 503) {
                // Service unavailable error
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => '[FREIGHTMATE] - API Service unavailable. Please try again later.'
                ];
                return wp_json_encode($error_response);
            }

            // Parse response body
            $data = json_decode(wp_remote_retrieve_body($response), true);
            $success_response = [
                'success' => true,
                'status_code' => $status_code,
                'data' => $data
            ];
            return wp_json_encode($success_response);
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            $error_response = [
                'success' => false,
                'status_code' => $status_code,
                'error' => $error_message
            ];
            return wp_json_encode($error_response);
        }
    }
    /**
     * Post
     *
     * Makes a POST request to the Freightmate API.
     *
     * @param string $endpoint The API endpoint to which the request will be made.
     * @return string JSON-encoded response containing either success data or error message.
     */

    public function post($endpoint, $body)
    {
        try {
            $url = $this->api_url . $endpoint;

            // Make GET request using WordPress HTTP API
            $response = wp_remote_post($url, [
                'body'        => $body,
                'headers' => [
                    'Content-Type'  => 'application/json; charset=utf-8',
                    'Authorization' => 'Basic ' . base64_encode($this->get_auth())
                ],
                'timeout'     => 120
            ]);
            $status_code = wp_remote_retrieve_response_code($response);

            // Check for WP_Error
            if (is_wp_error($response)) {
                $error_message = $response->get_error_message();
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => $error_message
                ];
                return wp_json_encode($error_response);
            }

            // Check for different status codes
            if ($status_code === 401) {
                // Authentication error
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => 'Authentication failed. Please check your credentials.'
                ];
                return wp_json_encode($error_response);
            } elseif ($status_code === 503) {
                // Service unavailable error
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => '[FREIGHTMATE] - API Service unavailable. Please try again later.'
                ];
                return wp_json_encode($error_response);
            }

            // Parse response body
            $data = json_decode(wp_remote_retrieve_body($response), true);
            $success_response = [
                'success' => true,
                'status_code' => $status_code,
                'data' => $data
            ];
            return wp_json_encode($success_response);
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            $error_response = [
                'success' => false,
                'status_code' => $status_code,
                'error' => $error_message
            ];
            return wp_json_encode($error_response);
        }
    }
    /**
     * Delete
     *
     * Makes a DELETE request to the Freightmate API.
     *
     * @param string $endpoint The API endpoint to which the request will be made.
     * @return string JSON-encoded response containing either success data or error message.
     */
    public function delete($endpoint)
    {
        try {
            $url = $this->api_url . $endpoint;

            // Make DELETE request using WordPress HTTP API
            $response = wp_remote_request($url, [
                'method'      => 'DELETE',
                'headers'     => [
                    'Authorization' => 'Basic ' . base64_encode($this->get_auth())
                ]
            ]);
            $status_code = wp_remote_retrieve_response_code($response);

            // Check for WP_Error
            if (is_wp_error($response)) {
                $error_message = $response->get_error_message();
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => $error_message
                ];
                return wp_json_encode($error_response);
            }

            // Check for different status codes
            if ($status_code === 401) {
                // Authentication error
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => 'Authentication failed. Please check your credentials.'
                ];
                return wp_json_encode($error_response);
            } elseif ($status_code === 503) {
                // Service unavailable error
                $error_response = [
                    'success' => false,
                    'status_code' => $status_code,
                    'error' => '[FREIGHTMATE] - API Service unavailable. Please try again later.'
                ];
                return wp_json_encode($error_response);
            }

            // Parse response body
            $data = json_decode(wp_remote_retrieve_body($response), true);
            $success_response = [
                'success' => true,
                'status_code' => $status_code,
                'data' => $data
            ];
            return wp_json_encode($success_response);
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            $error_response = [
                'success' => false,
                'status_code' => $status_code,
                'error' => $error_message
            ];
            return wp_json_encode($error_response);
        }
    }
}
