<?php
/**
 * Frontend functionality for FreedomReader
 *
 * @package FreedomReader
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Frontend Class
 *
 * @package FreedomReader
 */
class FREEDO_Frontend {

	/**
	 * Single instance of the class
	 *
	 * @var FREEDO_Frontend|null
	 */
	private static $instance = null;

	/**
	 * Database instance
	 *
	 * @var FREEDO_Database
	 */
	private $db;

	/**
	 * User manager instance
	 *
	 * @var FREEDO_User_Manager
	 */
	private $user_manager;

	/**
	 * Get single instance
	 *
	 * @return FREEDO_Frontend Single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->db           = FREEDO_Database::get_instance();
		$this->user_manager = FREEDO_User_Manager::get_instance();
		$this->init_hooks();
	}

	/**
	 * Initialize frontend hooks
	 *
	 * @package FreedomReader
	 */
	private function init_hooks() {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'wp_footer', array( $this, 'add_paypal_sdk' ) );
		add_action( 'wp_ajax_freedomreader_cancel_subscription', array( $this, 'cancel_subscription' ) );
		add_action( 'wp_head', array( $this, 'add_structured_data' ) );
		add_filter( 'body_class', array( $this, 'add_body_classes' ) );
	}


	/**
	 * Enqueue frontend scripts and styles
	 */
	public function enqueue_scripts() {
		// Only enqueue on pages that might have locked content.
		if ( ! is_singular() && ! is_page() ) {
			return;
		}

		wp_enqueue_script(
			'freedomreader-frontend',
			FREEDOMREADER_PLUGIN_URL . 'assets/js/frontend.js',
			array( 'jquery' ),
			FREEDOMREADER_VERSION,
			true
		);

		wp_enqueue_style(
			'freedomreader-frontend',
			FREEDOMREADER_PLUGIN_URL . 'assets/css/frontend.css',
			array(),
			FREEDOMREADER_VERSION
		);

		// Enqueue subscription-cancelled styles if on that page.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking page parameter for style loading
		if ( is_page() && isset( $_GET['freedomreader_page'] ) && 'subscription-cancelled' === sanitize_text_field( wp_unslash( $_GET['freedomreader_page'] ) ) ) {
			wp_enqueue_style(
				'freedomreader-subscription-cancelled',
				FREEDOMREADER_PLUGIN_URL . 'assets/css/subscription-cancelled.css',
				array(),
				FREEDOMREADER_VERSION
			);
		}

		// Localize script with settings.
		global $post;
		wp_localize_script(
			'freedomreader-frontend',
			'freedomreader',
			array(
				'ajax_url'             => esc_url( admin_url( 'admin-ajax.php' ) ),
				'nonce'                => wp_create_nonce( 'freedomreader_nonce' ),
				'paypal_client_id'     => get_option( 'freedomreader_paypal_client_id' ),
				'paypal_mode'          => get_option( 'freedomreader_paypal_mode', 'sandbox' ),
				'currency'             => get_option( 'freedomreader_currency', 'USD' ),
				'is_user_logged_in'    => is_user_logged_in(),
				'post_id'              => is_single() && $post ? $post->ID : 0,
				'login_url'            => wp_login_url( get_permalink() ),
				'register_url'         => get_option( 'users_can_register' ) ? wp_registration_url() : wp_login_url( get_permalink() ) . '?action=register',
				'registration_enabled' => get_option( 'users_can_register' ),
				'messages'             => array(
					'processing'     => esc_html__( 'Processing payment...', 'freedomreader' ),
					'success'        => esc_html__( 'Payment successful! Refreshing page...', 'freedomreader' ),
					'error'          => esc_html__( 'Payment failed. Please try again.', 'freedomreader' ),
					'login_required' => esc_html__( 'Please log in to purchase content.', 'freedomreader' ),
					'auth_required'  => esc_html__( 'Authentication Required', 'freedomreader' ),
					'confirm_cancel' => esc_html__( 'Are you sure you want to cancel your subscription?', 'freedomreader' ),
				),
			)
		);
	}


	/**
	 * Add PayPal SDK to the page
	 */
	public function add_paypal_sdk() {
		$client_id = get_option( 'freedomreader_paypal_client_id' );
		$currency  = get_option( 'freedomreader_currency', 'USD' );

		if ( empty( $client_id ) ) {
			return;
		}

		// Only load on pages that might need PayPal.
		if ( ! is_singular() && ! is_page() ) {
			return;
		}

		wp_enqueue_script(
			'paypal-sdk',
			'https://www.paypal.com/sdk/js?client-id=' . esc_attr( $client_id ) . '&currency=' . esc_attr( $currency ) . '&intent=capture',
			array(),
			'1.0.0',
			true
		);
	}




	/**
	 * Cancel subscription via AJAX
	 */
	public function cancel_subscription() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
			return;
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Please log in.', 'freedomreader' ) ) );
			return;
		}

		$subscription_id = intval( wp_unslash( $_POST['subscription_id'] ?? 0 ) );
		$user_id         = get_current_user_id();

		// Verify user owns this subscription.
		$subscription = $this->db->get_user_subscription( $user_id );
		if ( ! $subscription || $subscription->id !== $subscription_id ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid subscription.', 'freedomreader' ) ) );
			return;
		}

		// Cancel with PayPal.
		$paypal = FREEDO_PayPal::get_instance();
		$result = $paypal->cancel_subscription( $subscription->paypal_subscription_id );

		if ( $result['success'] ) {
			// Update local database.
			global $wpdb;
			$table = $wpdb->prefix . 'freedo_subscriptions';

			// Clear cache before update.
			$cache_key = 'freedomreader_subscription_' . $subscription_id;
			wp_cache_delete( $cache_key, 'freedomreader' );

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Subscription status update with cache invalidation
			$result = $wpdb->update(
				$table,
				array(
					'status'     => 'cancelled',
					'updated_at' => current_time( 'mysql' ),
				),
				array( 'id' => $subscription_id ),
				array( '%s', '%s' ),
				array( '%d' )
			);

			// Clear related caches.
			wp_cache_delete( 'freedomreader_user_subscriptions_' . get_current_user_id(), 'freedomreader' );

			wp_send_json_success(
				array(
					'message' => esc_html__( 'Subscription cancelled successfully.', 'freedomreader' ),
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message' => $result['message'] ? $result['message'] : esc_html__( 'Failed to cancel subscription.', 'freedomreader' ),
				)
			);
		}
	}




	/**
	 * Add structured data for locked content
	 */
	public function add_structured_data() {
		if ( ! is_singular() ) {
			return;
		}

		global $post;
		$lock_settings = $this->db->get_content_lock( $post->ID );

		if ( ! $lock_settings || ! $lock_settings->price ) {
			return;
		}

		$currency = get_option( 'freedomreader_currency', 'USD' );

		$structured_data = array(
			'@context'            => 'https://schema.org',
			'@type'               => 'Article',
			'headline'            => get_the_title(),
			'author'              => array(
				'@type' => 'Person',
				'name'  => get_the_author(),
			),
			'datePublished'       => get_the_date( 'c' ),
			'dateModified'        => get_the_modified_date( 'c' ),
			'isAccessibleForFree' => false,
			'hasPart'             => array(
				'@type'               => 'WebPageElement',
				'isAccessibleForFree' => false,
				'cssSelector'         => '.fr-locked-content',
			),
			'offers'              => array(
				'@type'         => 'Offer',
				'price'         => $lock_settings->price,
				'priceCurrency' => $currency,
				'availability'  => 'https://schema.org/InStock',
			),
		);

		echo '<script type="application/ld+json">' . wp_json_encode( $structured_data ) . '</script>';
	}


	/**
	 * Add body classes for locked content
	 *
	 * @param array $classes Existing body classes.
	 * @return array Modified body classes.
	 */
	public function add_body_classes( $classes ) {
		if ( is_singular() ) {
			global $post;
			$lock_settings = $this->db->get_content_lock( $post->ID );

			if ( $lock_settings ) {
				$classes[] = 'fr-has-locked-content';
				$classes[] = 'fr-lock-type-' . $lock_settings->lock_type;

				$user_id = get_current_user_id();
				if ( $this->db->user_has_access( $user_id, $post->ID ) ) {
					$classes[] = 'fr-user-has-access';
				} else {
					$classes[] = 'fr-user-no-access';
				}
			}
		}

		// Add user access level class.
		if ( is_user_logged_in() ) {
			$user_id      = get_current_user_id();
			$access_level = $this->user_manager->get_user_access_level( $user_id );
			$classes[]    = 'fr-access-' . $access_level;
		} else {
			$classes[] = 'fr-access-guest';
		}

		return $classes;
	}

	/**
	 * Get user's purchase history for a specific post
	 *
	 * @param int $user_id User ID.
	 * @param int $post_id Post ID.
	 * @return array Purchase history.
	 */
	public function get_user_post_purchases( $user_id, $post_id ) {
		$cache_key = 'freedomreader_user_post_purchases_' . $user_id . '_' . $post_id;
		$purchases = wp_cache_get( $cache_key );

		if ( false === $purchases ) {
			global $wpdb;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for user purchase history with caching
			$purchases = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM {$wpdb->prefix}freedo_purchases WHERE user_id = %d AND post_id = %d ORDER BY created_at DESC",
					$user_id,
					$post_id
				)
			);

			wp_cache_set( $cache_key, $purchases, '', 300 ); // Cache for 5 minutes.
		}

		return $purchases;
	}

	/**
	 * Check if content should show teaser
	 *
	 * @param int $post_id Post ID.
	 * @return bool Whether to show teaser.
	 */
	public function should_show_teaser( $post_id ) {
		$lock_settings = $this->db->get_content_lock( $post_id );

		if ( ! $lock_settings ) {
			return false;
		}

		return 'partial' === $lock_settings->lock_type;
	}

	/**
	 * Get teaser content
	 *
	 * @param string $content Post content.
	 * @param int    $post_id Post ID.
	 * @return string Teaser content.
	 */
	public function get_teaser_content( $content, $post_id ) {
		$lock_settings = $this->db->get_content_lock( $post_id );

		if ( ! $lock_settings || 'partial' !== $lock_settings->lock_type ) {
			return '';
		}

		$teaser_length = $lock_settings->teaser_length ? $lock_settings->teaser_length : 150;
		return wp_trim_words( $content, $teaser_length, '...' );
	}

	/**
	 * Generate purchase button HTML
	 *
	 * @param int    $post_id Post ID.
	 * @param float  $price   Price override.
	 * @param string $text    Button text override.
	 * @return string Button HTML.
	 */
	public function generate_purchase_button( $post_id, $price = null, $text = null ) {
		$lock_settings = $this->db->get_content_lock( $post_id );

		if ( ! $lock_settings ) {
			return '';
		}

		$price       = $price ? $price : $lock_settings->price;
		$button_text = $text ? $text : get_option( 'freedomreader_unlock_button_text', esc_html__( 'Unlock Content', 'freedomreader' ) );

		if ( $price ) {
			/* translators: %s: price amount */
			$button_text = sprintf( esc_html__( 'Unlock for $%s', 'freedomreader' ), $price );
		}

		// Check if user already has access.
		$user_id = get_current_user_id();
		if ( $this->db->user_has_access( $user_id, $post_id ) ) {
			return '<p class="fr-already-purchased">' . esc_html__( 'You already have access to this content.', 'freedomreader' ) . '</p>';
		}

		$html = '<div class="fr-purchase-button-container">';

		if ( ! is_user_logged_in() ) {
			$html .= '<p class="fr-login-required">' . esc_html__( 'Please log in to purchase content.', 'freedomreader' ) . '</p>';
			$html .= '<a href="' . wp_login_url( get_permalink( $post_id ) ) . '" class="fr-login-btn">' . esc_html__( 'Login', 'freedomreader' ) . '</a>';
		} else {
			$html .= '<button class="fr-purchase-btn" data-post-id="' . $post_id . '" data-price="' . $price . '">';
			$html .= esc_html( $button_text );
			$html .= '</button>';

			// Add subscription options if available.
			$subscription_plans = get_option( 'freedomreader_subscription_plans', array() );
			if ( ! empty( $subscription_plans ) ) {
				$html .= '<div class="fr-subscription-options">';
				$html .= '<p>' . esc_html__( 'Or subscribe for unlimited access:', 'freedomreader' ) . '</p>';

				foreach ( $subscription_plans as $plan_id => $plan ) {
					$html .= '<button class="fr-subscribe-btn" data-plan-id="' . $plan_id . '">';
					/* translators: 1: plan name, 2: price, 3: billing interval */
					$html .= sprintf( esc_html__( '%1$s - $%2$s/%3$s', 'freedomreader' ), $plan['name'], $plan['price'], $plan['interval'] );
					$html .= '</button>';
				}

				$html .= '</div>';
			}
		}

		$html .= '</div>';

		return $html;
	}

	/**
	 * Get subscription status message
	 *
	 * @param int $user_id User ID.
	 * @return string Status message.
	 */
	public function get_subscription_status_message( $user_id ) {
		$subscription = $this->db->get_user_subscription( $user_id );

		if ( ! $subscription ) {
			return '';
		}

		$message = '';

		switch ( $subscription->status ) {
			case 'trial':
				if ( $subscription->trial_ends_at ) {
					$days_left = ceil( ( strtotime( $subscription->trial_ends_at ) - time() ) / ( 60 * 60 * 24 ) );
					/* translators: %d: number of days left in trial */
					$message = sprintf( esc_html__( 'You have %d days left in your free trial.', 'freedomreader' ), $days_left );
				}
				break;
			case 'active':
				$message = esc_html__( 'You have an active subscription with unlimited access.', 'freedomreader' );
				break;
			case 'cancelled':
				$message = esc_html__( 'Your subscription has been cancelled.', 'freedomreader' );
				break;
			case 'expired':
				$message = esc_html__( 'Your subscription has expired.', 'freedomreader' );
				break;
		}

		return $message;
	}
}
