<?php
/**
 * FreedomReader Functions
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Initialize the plugin.
 *
 * @return FreedomReader
 * @package FreedomReader
 */
function freedomreader_init() {
	return FreedomReader::get_instance();
}

/**
 * Get FreedomReader instance.
 *
 * @return FreedomReader
 */
function freedomreader() {
	return FreedomReader::get_instance();
}

/**
 * Create order via AJAX.
 */
function freedomreader_create_order() {
	status_header( 200 );
	header( 'Content-Type: application/json; charset=utf-8' );

	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
		echo wp_json_encode(
			array(
				'success' => false,
				'message' => 'Security verification failed.',
			)
		);
		wp_die();
	}

	$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
	$type    = isset( $_POST['type'] ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : '';

	if ( ! $post_id || ! in_array( $type, array( 'single', 'subscription' ), true ) ) {
		echo wp_json_encode(
			array(
				'success' => false,
				'message' => 'Invalid request parameters.',
			)
		);
		wp_die();
	}

	$paypal = FREEDO_PayPal::get_instance();

	if ( 'single' === $type ) {
		// Get the price from post meta or settings.
		$price = get_post_meta( $post_id, '_freedomreader_price', true );
		if ( ! $price ) {
			$price = get_option( 'freedomreader_default_price', 5.00 );
		}

		$currency   = get_option( 'freedomreader_currency', 'USD' );
		$post_title = get_the_title( $post_id );

		$order_data = array(
			'amount'      => floatval( $price ),
			'currency'    => $currency,
			/* translators: %s: Post title */
			'description' => sprintf( esc_html__( 'Access to: %s', 'freedomreader' ), $post_title ),
			'return_url'  => get_permalink( $post_id ),
			'cancel_url'  => get_permalink( $post_id ),
		);

		$result = $paypal->create_order_api( $order_data );
	} else {
		$result = $paypal->create_subscription();
	}

	echo wp_json_encode( $result );
	wp_die();
}

/**
 * Capture order via AJAX.
 */
function freedomreader_capture_order() {
	status_header( 200 );
	header( 'Content-Type: application/json; charset=utf-8' );

	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
		echo wp_json_encode(
			array(
				'success' => false,
				'message' => 'Security verification failed.',
			)
		);
		wp_die();
	}

	$order_id = isset( $_POST['order_id'] ) ? sanitize_text_field( wp_unslash( $_POST['order_id'] ) ) : '';
	$post_id  = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;

	if ( ! $order_id || ! $post_id ) {
		echo wp_json_encode(
			array(
				'success' => false,
				'message' => 'Invalid request parameters.',
			)
		);
		wp_die();
	}

	$paypal = FREEDO_PayPal::get_instance();
	$result = $paypal->capture_payment( $order_id );

	echo wp_json_encode( $result );
	wp_die();
}

/**
 * Process payment via AJAX.
 *
 * @package FreedomReader
 */
function freedomreader_process_payment() {
	// Verify nonce for security.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
		wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
	}

	// Check if user is logged in.
	if ( ! is_user_logged_in() ) {
		wp_send_json_error( array( 'message' => esc_html__( 'Please log in to purchase content.', 'freedomreader' ) ) );
	}

	$post_id         = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
	$payment_type    = isset( $_POST['payment_type'] ) ? sanitize_text_field( wp_unslash( $_POST['payment_type'] ) ) : '';
	$paypal_order_id = isset( $_POST['paypal_order_id'] ) ? sanitize_text_field( wp_unslash( $_POST['paypal_order_id'] ) ) : '';

	if ( ! $post_id || ! $paypal_order_id ) {
		wp_send_json_error( array( 'message' => esc_html__( 'Invalid request parameters.', 'freedomreader' ) ) );
	}

	// Process with PayPal.
	$paypal = FREEDO_PayPal::get_instance();
	$result = $paypal->capture_payment( $paypal_order_id );

	if ( $result['success'] ) {

		$purchase_data = array(
			'user_id'               => get_current_user_id(),
			'post_id'               => $post_id,
			'purchase_type'         => $payment_type,
			'amount'                => $result['amount'],
			'currency'              => $result['currency'],
			'paypal_transaction_id' => $result['transaction_id'],
			'status'                => 'completed',
		);

		$db = FREEDO_Database::get_instance();
		$db->record_purchase( $purchase_data );

		// Log the transaction.
		$db->log_payment(
			array(
				'user_id'               => get_current_user_id(),
				'transaction_type'      => 'purchase',
				'paypal_transaction_id' => $result['transaction_id'],
				'amount'                => $result['amount'],
				'status'                => 'completed',
				'response_data'         => $result,
			)
		);

		do_action( 'freedomreader_after_payment_success', $purchase_data, $result );

		wp_send_json_success(
			array(
				'message'  => esc_html__( 'Payment successful! You now have access to this content.', 'freedomreader' ),
				'redirect' => get_permalink( $post_id ),
			)
		);
	} else {
		wp_send_json_error( array( 'message' => $result['message'] ) );
	}
}


add_action( 'wp_ajax_freedomreader_create_order', 'freedomreader_create_order' );
add_action( 'wp_ajax_nopriv_freedomreader_create_order', 'freedomreader_create_order' );
add_action( 'wp_ajax_freedomreader_capture_order', 'freedomreader_capture_order' );
add_action( 'wp_ajax_nopriv_freedomreader_capture_order', 'freedomreader_capture_order' );
add_action( 'wp_ajax_freedomreader_process_payment', 'freedomreader_process_payment' );
add_action( 'wp_ajax_nopriv_freedomreader_process_payment', 'freedomreader_process_payment' );
