<?php
/**
 * Main FreedomReader Class
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main FreedomReader Class
 *
 * Handles plugin initialization, activation, deactivation and core functionality.
 *
 * @package FreedomReader
 * @since 1.0.0
 */
class FreedomReader {

	/**
	 * Single instance of the class
	 *
	 * @var FreedomReader|null
	 */
	private static $instance = null;

	/**
	 * Get single instance
	 *
	 * @return FreedomReader Single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->define_constants();
		$this->includes();
		$this->init_components();
	}

	/**
	 * Define plugin constants
	 *
	 * @package FreedomReader
	 */
	private function define_constants() {
		if ( ! defined( 'FREEDOMREADER_VERSION' ) ) {
			define( 'FREEDOMREADER_VERSION', '1.0.1' );
		}

		if ( ! defined( 'FREEDOMREADER_PLUGIN_PATH' ) ) {
			define( 'FREEDOMREADER_PLUGIN_PATH', plugin_dir_path( __DIR__ ) );
		}

		if ( ! defined( 'FREEDOMREADER_PLUGIN_URL' ) ) {
			define( 'FREEDOMREADER_PLUGIN_URL', plugin_dir_url( __DIR__ ) );
		}

		if ( ! defined( 'FREEDOMREADER_PLUGIN_BASENAME' ) ) {
			define( 'FREEDOMREADER_PLUGIN_BASENAME', plugin_basename( __DIR__ . '/freedomreader.php' ) );
		}
	}

	/**
	 * Include required files
	 *
	 * @package FreedomReader
	 */
	private function includes() {
		// Core classes.
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-database.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-paypal.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-content-locker.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-advanced-locks.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-user-manager.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-shortcodes.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-content-enhancements.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-analytics.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-reports.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-backup-system.php';
		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/class-freedo-performance-monitor.php';

		require_once FREEDOMREADER_PLUGIN_PATH . 'includes/functions.php';

		if ( is_admin() ) {
			require_once FREEDOMREADER_PLUGIN_PATH . 'admin/class-freedo-admin.php';
			require_once FREEDOMREADER_PLUGIN_PATH . 'admin/class-freedo-admin-reports.php';
		}

		if ( ! is_admin() ) {
			require_once FREEDOMREADER_PLUGIN_PATH . 'public/class-freedo-frontend.php';
		}
	}

	/**
	 * Initialize components
	 *
	 * @package FreedomReader
	 */
	private function init_components() {
		// Initialize admin or frontend immediately.
		if ( is_admin() ) {
			FREEDO_Admin::get_instance();
		} else {
			FREEDO_Frontend::get_instance();
		}

		// Initialize backup system.
		FREEDO_Backup_System::get_instance();

		// Initialize performance monitor.
		FREEDO_Performance_Monitor::get_instance();
	}

	/**
	 * Plugin activation hook
	 *
	 * @package FreedomReader
	 */
	public static function activate() {
		// Create database tables.
		FREEDO_Database::create_tables();

		// Set default options.
		$default_options = array(
			'freedomreader_paypal_mode'              => 'sandbox',
			'freedomreader_currency'                 => 'USD',
			'freedomreader_teaser_message'           => esc_html__( 'This content is locked. Purchase to unlock.', 'freedomreader' ),
			'freedomreader_unlock_button_text'       => esc_html__( 'Unlock Content', 'freedomreader' ),
			'freedomreader_subscription_button_text' => esc_html__( 'Subscribe', 'freedomreader' ),
		);

		foreach ( $default_options as $option => $value ) {
			if ( false === get_option( $option ) ) {
				add_option( $option, $value );
			}
		}

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation hook
	 *
	 * @package FreedomReader
	 */
	public static function deactivate() {
		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Check if table exists
	 *
	 * @param string $table Table name.
	 * @return bool Whether table exists.
	 * @package FreedomReader
	 */
	public static function table_exists( $table ) {
		global $wpdb;

		$cache_key    = 'freedomreader_table_exists_' . md5( $table );
		$table_exists = wp_cache_get( $cache_key );

		if ( false === $table_exists ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table existence check with caching
			$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );
			wp_cache_set( $cache_key, $table_exists, '', 3600 ); // Cache for 1 hour.
		}

		return ! empty( $table_exists );
	}
}
