<?php
/**
 * User Management for FreedomReader
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * User Manager Class
 *
 * @package FreedomReader
 */
class FREEDO_User_Manager {

	/**
	 * Single instance of the class
	 *
	 * @var FREEDO_User_Manager|null
	 */
	private static $instance = null;

	/**
	 * Database instance
	 *
	 * @var FREEDO_Database
	 */
	private $db;

	/**
	 * Get single instance
	 *
	 * @return FREEDO_User_Manager Single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->db = FREEDO_Database::get_instance();
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		add_action( 'init', array( $this, 'add_rewrite_rules' ) );
		add_action( 'template_redirect', array( $this, 'handle_user_pages' ) );
		add_action( 'wp_ajax_freedomreader_cancel_subscription', array( $this, 'cancel_subscription' ) );
		add_action( 'wp_ajax_freedomreader_download_invoice', array( $this, 'download_invoice' ) );
		add_filter( 'query_vars', array( $this, 'add_query_vars' ) );
	}

	/**
	 * Add rewrite rules for user dashboard
	 */
	public function add_rewrite_rules() {
		add_rewrite_rule(
			'^freedomreader/dashboard/?$',
			'index.php?freedomreader_page=dashboard',
			'top'
		);

		add_rewrite_rule(
			'^freedomreader/purchases/?$',
			'index.php?freedomreader_page=purchases',
			'top'
		);

		add_rewrite_rule(
			'^freedomreader/subscription/?$',
			'index.php?freedomreader_page=subscription',
			'top'
		);

		add_rewrite_rule(
			'^freedomreader/subscription-success/?$',
			'index.php?freedomreader_page=subscription_success',
			'top'
		);

		add_rewrite_rule(
			'^freedomreader/subscription-cancelled/?$',
			'index.php?freedomreader_page=subscription_cancelled',
			'top'
		);
	}

	/**
	 * Add query vars
	 *
	 * @param array $vars Query variables.
	 * @return array Modified query variables.
	 */
	public function add_query_vars( $vars ) {
		$vars[] = 'freedomreader_page';
		return $vars;
	}

	/**
	 * Handle user dashboard pages
	 */
	public function handle_user_pages() {
		$page = get_query_var( 'freedomreader_page' );

		if ( ! $page ) {
			return;
		}

		// Require login for most pages.
		if ( ! is_user_logged_in() && ! in_array( $page, array( 'subscription_success', 'subscription_cancelled' ), true ) ) {
			// phpcs:ignore WordPress.Security.SafeRedirect.wp_redirect_wp_redirect -- Redirecting to login page
			wp_redirect( wp_login_url( home_url( '/freedomreader/' . $page . '/' ) ) );
			exit;
		}

		switch ( $page ) {
			case 'dashboard':
				$this->render_dashboard();
				break;
			case 'purchases':
				$this->render_purchases();
				break;
			case 'subscription':
				$this->render_subscription();
				break;
			case 'subscription_success':
				$this->render_subscription_success();
				break;
			case 'subscription_cancelled':
				$this->render_subscription_cancelled();
				break;
		}
	}

	/**
	 * Render user dashboard
	 */
	private function render_dashboard() {
		$user_id      = get_current_user_id();
		$purchases    = $this->db->get_user_purchases( $user_id );
		$subscription = $this->db->get_user_subscription( $user_id );

		$this->render_page(
			'dashboard',
			array(
				'purchases'    => $purchases,
				'subscription' => $subscription,
				'user'         => wp_get_current_user(),
			)
		);
	}

	/**
	 * Render purchases page
	 */
	private function render_purchases() {
		$user_id   = get_current_user_id();
		$purchases = $this->db->get_user_purchases( $user_id );

		$this->render_page(
			'purchases',
			array(
				'purchases' => $purchases,
			)
		);
	}

	/**
	 * Render subscription page
	 */
	private function render_subscription() {
		$user_id            = get_current_user_id();
		$subscription       = $this->db->get_user_subscription( $user_id );
		$subscription_plans = get_option( 'freedomreader_subscription_plans', array() );

		$this->render_page(
			'subscription',
			array(
				'subscription' => $subscription,
				'plans'        => $subscription_plans,
			)
		);
	}

	/**
	 * Render subscription success page
	 */
	private function render_subscription_success() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- PayPal callback parameter
		$subscription_id = isset( $_GET['subscription_id'] ) ? sanitize_text_field( wp_unslash( $_GET['subscription_id'] ) ) : '';

		if ( $subscription_id && is_user_logged_in() ) {
			// Record subscription in database.
			$this->record_subscription_from_paypal( $subscription_id );
		}

		$this->render_page(
			'subscription_success',
			array(
				'subscription_id' => $subscription_id,
			)
		);
	}

	/**
	 * Render subscription cancelled page
	 */
	private function render_subscription_cancelled() {
		$this->render_page( 'subscription_cancelled' );
	}

	/**
	 * Render page template
	 *
	 * @param string $template Template name.
	 * @param array  $data     Template data.
	 */
	private function render_page( $template, $data = array() ) {
		// Extract data for template.
		// phpcs:ignore WordPress.PHP.DontExtract.extract_extract -- Template variables needed.
		extract( $data );

		// Start output buffering.
		ob_start();

		// Include template.
		$template_file = FREEDOMREADER_PLUGIN_PATH . 'templates/user/' . $template . '.php';

		if ( file_exists( $template_file ) ) {
			include $template_file;
		} else {
			echo '<div class="fr-error">Template not found: ' . esc_html( $template ) . '</div>';
		}

		$content = ob_get_clean();

		// Use WordPress template.
		get_header();
		echo '<div class="freedomreader-user-page">' . wp_kses_post( $content ) . '</div>';
		get_footer();
		exit;
	}

	/**
	 * Cancel subscription
	 */
	public function cancel_subscription() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed', 'freedomreader' ) );
		}

		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Please log in', 'freedomreader' ) ) );
		}

		$user_id      = get_current_user_id();
		$subscription = $this->db->get_user_subscription( $user_id );

		if ( ! $subscription ) {
			wp_send_json_error( array( 'message' => esc_html__( 'No active subscription found', 'freedomreader' ) ) );
		}

		$paypal = FREEDO_PayPal::get_instance();
		$result = $this->cancel_paypal_subscription( $subscription->paypal_subscription_id );

		if ( $result ) {

			global $wpdb;
			$table = $wpdb->prefix . 'freedo_subscriptions';

			$wpdb->upgmdate(
				$table,
				array( 'status' => 'cancelled' ),
				array( 'id' => $subscription->id )
			);

			do_action( 'freedomreader_on_subscription_cancelled', $subscription->paypal_subscription_id, $subscription );

			wp_send_json_success(
				array(
					'message' => esc_html__( 'Subscription cancelled successfully', 'freedomreader' ),
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'Failed to cancel subscription', 'freedomreader' ),
				)
			);
		}
	}

	/**
	 * Cancel PayPal subscription
	 *
	 * @param string $subscription_id PayPal subscription ID.
	 * @return bool Whether cancellation was successful.
	 */
	private function cancel_paypal_subscription( $subscription_id ) {
		$paypal       = FREEDO_PayPal::get_instance();
		$access_token = $paypal->get_access_token();

		if ( ! $access_token ) {
			return false;
		}

		$mode     = get_option( 'freedomreader_paypal_mode', 'sandbox' );
		$base_url = ( 'live' === $mode ) ? 'https://api.paypal.com' : 'https://api.sandbox.paypal.com';

		$url = $base_url . '/v1/billing/subscriptions/' . $subscription_id . '/cancel';

		$cancel_data = array(
			'reason' => esc_html__( 'Cancelled by user', 'freedomreader' ),
		);

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => wp_json_encode( $cancel_data ),
		);

		$response = wp_remote_request( $url, $args );

		return ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 204;
	}

	/**
	 * Download invoice
	 *
	 * @package FreedomReader
	 */
	public function download_invoice() {

		if ( ! isset( $_GET['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed', 'freedomreader' ) );
		}

		if ( ! is_user_logged_in() ) {
			wp_die( esc_html__( 'Please log in', 'freedomreader' ) );
		}

		$purchase_id = isset( $_GET['purchase_id'] ) ? intval( wp_unslash( $_GET['purchase_id'] ) ) : 0;
		$user_id     = get_current_user_id();

		// Get purchase details.
		global $wpdb;
		$table = $wpdb->prefix . 'freedo_purchases';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Purchase details lookup
		$purchase = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table}` WHERE id = %d AND user_id = %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$purchase_id,
				$user_id
			)
		);

		if ( ! $purchase ) {
			wp_die( esc_html__( 'Purchase not found', 'freedomreader' ) );
		}

		$this->generate_invoice_pdf( $purchase );
	}

	/**
	 * Generate invoice PDF
	 *
	 * @param object $purchase Purchase object.
	 */
	private function generate_invoice_pdf( $purchase ) {
		// Simple HTML invoice (you can enhance this with a PDF library).
		$user       = get_userdata( $purchase->user_id );
		$post_title = $purchase->post_id ? get_the_title( $purchase->post_id ) : esc_html__( 'Subscription', 'freedomreader' );

		$html = '
        <!DOCTYPE html>
        <html>
        <head>
            <title>Invoice #' . $purchase->id . '</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 40px; }
                .header { border-bottom: 2px solid #333; padding-bottom: 20px; margin-bottom: 30px; }
                .invoice-details { margin-bottom: 30px; }
                .total { font-weight: bold; font-size: 18px; }
            </style>
        </head>
        <body>
            <div class="header">
                <h1>' . esc_html( get_bloginfo( 'name' ) ) . '</h1>
                <h2>Invoice #' . $purchase->id . '</h2>
            </div>
            
            <div class="invoice-details">
                <p><strong>Date:</strong> ' . gmdate( 'F j, Y', strtotime( $purchase->created_at ) ) . '</p>
                <p><strong>Customer:</strong> ' . $user->display_name . '</p>
                <p><strong>Email:</strong> ' . $user->user_email . '</p>
                <p><strong>Transaction ID:</strong> ' . $purchase->paypal_transaction_id . '</p>
            </div>
            
            <table border="1" cellpadding="10" cellspacing="0" width="100%">
                <tr>
                    <th>Description</th>
                    <th>Amount</th>
                </tr>
                <tr>
                    <td>' . $post_title . '</td>
                    <td>$' . number_format( $purchase->amount, 2 ) . ' ' . $purchase->currency . '</td>
                </tr>
            </table>
            
            <div class="total" style="margin-top: 20px; text-align: right;">
                Total: $' . number_format( $purchase->amount, 2 ) . ' ' . $purchase->currency . '
            </div>
        </body>
        </html>';

		// Set headers for download.
		header( 'Content-Type: text/html' );
		header( 'Content-Disposition: attachment; filename="invoice-' . $purchase->id . '.html"' );

		echo esc_html( $html );
		exit;
	}

	/**
	 * Record subscription from PayPal success
	 *
	 * @param string $subscription_id PayPal subscription ID.
	 */
	private function record_subscription_from_paypal( $subscription_id ) {
		$user_id = get_current_user_id();

		// Check if already recorded.
		global $wpdb;
		$table = $wpdb->prefix . 'freedo_subscriptions';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Subscription existence check
		$exists = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table}` WHERE paypal_subscription_id = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$subscription_id
			)
		);

		if ( $exists ) {
			return;
		}

		// Get trial days.
		$trial_days    = get_option( 'freedomreader_trial_days', 7 );
		$trial_ends_at = gmdate( 'Y-m-d H:i:s', strtotime( '+' . $trial_days . ' days' ) );

		// Record subscription.
		$this->db->record_subscription(
			array(
				'user_id'                => $user_id,
				'plan_id'                => 'monthly', // Default, should be passed from PayPal.
				'paypal_subscription_id' => $subscription_id,
				'status'                 => 'trial',
				'trial_ends_at'          => $trial_ends_at,
			)
		);
	}

	/**
	 * Get user access level
	 *
	 * @param int $user_id User ID.
	 * @return string User access level.
	 */
	public function get_user_access_level( $user_id ) {
		$subscription = $this->db->get_user_subscription( $user_id );

		if ( $subscription ) {
			if ( 'active' === $subscription->status ) {
				return 'premium';
			} elseif ( 'trial' === $subscription->status && $subscription->trial_ends_at > current_time( 'mysql' ) ) {
				return 'trial';
			}
		}

		return 'basic';
	}

	/**
	 * Check if user has premium access
	 *
	 * @param int $user_id User ID.
	 * @return bool Whether user has premium access.
	 */
	public function user_has_premium_access( $user_id ) {
		$access_level = $this->get_user_access_level( $user_id );
		return in_array( $access_level, array( 'premium', 'trial' ), true );
	}
}
