<?php
/**
 * Shortcodes for FreedomReader
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Shortcodes Class
 *
 * @package FreedomReader
 */
class FREEDO_Shortcodes {

	/**
	 * Single instance of the class
	 *
	 * @var FREEDO_Shortcodes|null
	 */
	private static $instance = null;

	/**
	 * Database instance
	 *
	 * @var FREEDO_Database
	 */
	private $db;

	/**
	 * User manager instance
	 *
	 * @var FREEDO_User_Manager
	 */
	private $user_manager;

	/**
	 * Get single instance
	 *
	 * @return FREEDO_Shortcodes Single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->db           = FREEDO_Database::get_instance();
		$this->user_manager = FREEDO_User_Manager::get_instance();
		$this->init_shortcodes();
	}

	/**
	 * Initialize shortcodes
	 */
	private function init_shortcodes() {
		add_shortcode( 'freedomreader_premium_content', array( $this, 'premium_content_shortcode' ) );
		add_shortcode( 'freedomreader_subscription_plans', array( $this, 'subscription_plans_shortcode' ) );
		add_shortcode( 'freedomreader_purchase_button', array( $this, 'purchase_button_shortcode' ) );
		add_shortcode( 'freedomreader_user_dashboard', array( $this, 'user_dashboard_shortcode' ) );
		add_shortcode( 'freedomreader_login_form', array( $this, 'login_form_shortcode' ) );
		add_shortcode( 'freedomreader_access_level', array( $this, 'access_level_shortcode' ) );
	}

	/**
	 * Premium content shortcode
	 * Usage: [freedomreader_premium_content]Content for premium users only[/freedomreader_premium_content]
	 *
	 * @param array  $atts    Shortcode attributes.
	 * @param string $content Shortcode content.
	 * @return string Processed content.
	 */
	public function premium_content_shortcode( $atts, $content = '' ) {
		$atts = shortcode_atts(
			array(
				'subscription_required' => 'true',
				'post_id'               => '',
				'price'                 => '',
				'message'               => '',
			),
			$atts
		);

		$user_id = get_current_user_id();

		// Check if user has premium access.
		if ( 'true' === $atts['subscription_required'] ) {
			if ( ! $this->user_manager->user_has_premium_access( $user_id ) ) {
				return $this->render_premium_required_message( $atts );
			}
		} else {
			// Check for specific post access.
			$post_id = $atts['post_id'] ? $atts['post_id'] : get_the_ID();
			if ( ! $this->db->user_has_access( $user_id, $post_id ) ) {
				return $this->render_purchase_required_message( $atts, $post_id );
			}
		}

		return do_shortcode( $content );
	}

	/**
	 * Subscription plans shortcode
	 * Usage: [freedomreader_subscription_plans]
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string Subscription plans HTML.
	 */
	public function subscription_plans_shortcode( $atts ) {
		$atts = shortcode_atts(
			array(
				'show_trial' => 'true',
				'layout'     => 'grid',
			),
			$atts
		);

		$subscription_plans = get_option( 'freedomreader_subscription_plans', array() );
		$trial_days         = get_option( 'freedomreader_trial_days', 7 );

		if ( empty( $subscription_plans ) ) {
			return '<p>' . esc_html__( 'No subscription plans available.', 'freedomreader' ) . '</p>';
		}

		$output = '<div class="fr-subscription-plans fr-layout-' . esc_attr( $atts['layout'] ) . '">';

		foreach ( $subscription_plans as $plan_id => $plan ) {
			$output .= '<div class="fr-plan fr-plan-' . esc_attr( $plan_id ) . '">';
			$output .= '<h3 class="fr-plan-name">' . esc_html( $plan['name'] ) . '</h3>';
			$output .= '<div class="fr-plan-price">$' . esc_html( $plan['price'] ) . '/' . esc_html( $plan['interval'] ) . '</div>';

			if ( 'true' === $atts['show_trial'] && $trial_days > 0 ) {
				/* translators: %d: number of trial days */
				$output .= '<div class="fr-plan-trial">' . sprintf( esc_html__( '%d days free trial', 'freedomreader' ), $trial_days ) . '</div>';
			}

			$output .= '<button class="fr-subscribe-btn" data-plan-id="' . esc_attr( $plan_id ) . '">';
			$output .= esc_html__( 'Subscribe Now', 'freedomreader' );
			$output .= '</button>';
			$output .= '</div>';
		}

		$output .= '</div>';

		return $output;
	}

	/**
	 * Purchase button shortcode
	 * Usage: [freedomreader_purchase_button post_id="123" price="9.99"]
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string Purchase button HTML.
	 */
	public function purchase_button_shortcode( $atts ) {
		$atts = shortcode_atts(
			array(
				'post_id' => get_the_ID(),
				'price'   => '',
				'text'    => '',
				'class'   => 'fr-purchase-btn',
			),
			$atts
		);

		$post_id = intval( $atts['post_id'] );
		$price   = floatval( $atts['price'] );
		/* translators: %s: price amount */
		$text = $atts['text'] ? $atts['text'] : sprintf( esc_html__( 'Purchase for $%s', 'freedomreader' ), $price );

		// Check if user already has access.
		$user_id = get_current_user_id();
		if ( $this->db->user_has_access( $user_id, $post_id ) ) {
			return '<p class="fr-already-purchased">' . esc_html__( 'You already have access to this content.', 'freedomreader' ) . '</p>';
		}

		return sprintf(
			'<button class="%s" data-post-id="%d" data-price="%s">%s</button>',
			esc_attr( $atts['class'] ),
			$post_id,
			$price,
			esc_html( $text )
		);
	}

	/**
	 * User dashboard shortcode
	 * Usage: [freedomreader_user_dashboard]
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string Dashboard HTML.
	 */
	public function user_dashboard_shortcode( $atts ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Parameter required for WordPress shortcode interface
		if ( ! is_user_logged_in() ) {
			return '<p>' . esc_html__( 'Please log in to view your dashboard.', 'freedomreader' ) . ' <a href="' . wp_login_url() . '">' . esc_html__( 'Login', 'freedomreader' ) . '</a></p>';
		}

		$user_id      = get_current_user_id();
		$user         = wp_get_current_user();
		$purchases    = $this->db->get_user_purchases( $user_id );
		$subscription = $this->db->get_user_subscription( $user_id );
		$access_level = $this->user_manager->get_user_access_level( $user_id );

		$output = '<div class="fr-user-dashboard">';

		// User info.
		$output .= '<div class="fr-user-info">';
		/* translators: %s: user display name */
		$output .= '<h3>' . sprintf( esc_html__( 'Welcome, %s', 'freedomreader' ), $user->display_name ) . '</h3>';
		/* translators: %s: user access level */
		$output .= '<p>' . sprintf( esc_html__( 'Access Level: %s', 'freedomreader' ), ucfirst( $access_level ) ) . '</p>';
		$output .= '</div>';

		// Subscription info.
		if ( $subscription ) {
			$output .= '<div class="fr-subscription-info">';
			$output .= '<h4>' . esc_html__( 'Your Subscription', 'freedomreader' ) . '</h4>';
			/* translators: %s: subscription status */
			$output .= '<p>' . sprintf( esc_html__( 'Status: %s', 'freedomreader' ), ucfirst( $subscription->status ) ) . '</p>';

			if ( 'trial' === $subscription->status && $subscription->trial_ends_at ) {
				/* translators: %s: trial end date */
				$output .= '<p>' . sprintf( esc_html__( 'Trial ends: %s', 'freedomreader' ), gmdate( 'F j, Y', strtotime( $subscription->trial_ends_at ) ) ) . '</p>';
			}

			if ( $subscription->next_billing_date ) {
				/* translators: %s: next billing date */
				$output .= '<p>' . sprintf( esc_html__( 'Next billing: %s', 'freedomreader' ), gmdate( 'F j, Y', strtotime( $subscription->next_billing_date ) ) ) . '</p>';
			}

			if ( 'active' === $subscription->status ) {
				$output .= '<button class="fr-cancel-subscription" data-subscription-id="' . $subscription->id . '">' . esc_html__( 'Cancel Subscription', 'freedomreader' ) . '</button>';
			}

			$output .= '</div>';
		}

		// Recent purchases.
		if ( ! empty( $purchases ) ) {
			$output .= '<div class="fr-recent-purchases">';
			$output .= '<h4>' . esc_html__( 'Recent Purchases', 'freedomreader' ) . '</h4>';
			$output .= '<ul>';

			foreach ( array_slice( $purchases, 0, 5 ) as $purchase ) {
				$post_title = $purchase->post_id ? get_the_title( $purchase->post_id ) : esc_html__( 'Subscription', 'freedomreader' );
				$output    .= '<li>';
				$output    .= '<span class="fr-purchase-item">' . esc_html( $post_title ) . '</span>';
				$output    .= '<span class="fr-purchase-amount">$' . esc_html( $purchase->amount ) . '</span>';
				$output    .= '<span class="fr-purchase-date">' . gmdate( 'M j, Y', strtotime( $purchase->created_at ) ) . '</span>';
				$output    .= '</li>';
			}

			$output .= '</ul>';
			$output .= '<a href="' . home_url( '/freedomreader/purchases/' ) . '">' . esc_html__( 'View All Purchases', 'freedomreader' ) . '</a>';
			$output .= '</div>';
		}

		$output .= '</div>';

		return $output;
	}

	/**
	 * Login form shortcode
	 * Usage: [freedomreader_login_form]
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string Login form HTML.
	 */
	public function login_form_shortcode( $atts ) {
		if ( is_user_logged_in() ) {
			return '<p>' . esc_html__( 'You are already logged in.', 'freedomreader' ) . ' <a href="' . wp_logout_url() . '">' . esc_html__( 'Logout', 'freedomreader' ) . '</a></p>';
		}

		$atts = shortcode_atts(
			array(
				'redirect'       => '',
				'label_username' => esc_html__( 'Username or Email', 'freedomreader' ),
				'label_password' => esc_html__( 'Password', 'freedomreader' ),
				'label_remember' => esc_html__( 'Remember Me', 'freedomreader' ),
				'label_log_in'   => esc_html__( 'Log In', 'freedomreader' ),
			),
			$atts
		);

		$redirect_url = $atts['redirect'] ? $atts['redirect'] : get_permalink();

		$form_args = array(
			'echo'           => false,
			'redirect'       => $redirect_url,
			'label_username' => $atts['label_username'],
			'label_password' => $atts['label_password'],
			'label_remember' => $atts['label_remember'],
			'label_log_in'   => $atts['label_log_in'],
		);

		return '<div class="fr-login-form">' . wp_login_form( $form_args ) . '</div>';
	}

	/**
	 * Access level shortcode - shows content based on user access level
	 * Usage: [freedomreader_access_level level="premium"]Content for premium users[/freedomreader_access_level]
	 *
	 * @param array  $atts    Shortcode attributes.
	 * @param string $content Shortcode content.
	 * @return string Processed content.
	 */
	public function access_level_shortcode( $atts, $content = '' ) {
		$atts = shortcode_atts(
			array(
				'level'        => 'premium',
				'show_message' => 'true',
			),
			$atts
		);

		$user_id           = get_current_user_id();
		$user_access_level = $this->user_manager->get_user_access_level( $user_id );

		$access_hierarchy = array(
			'basic'   => 1,
			'trial'   => 2,
			'premium' => 3,
		);

		$required_level = $access_hierarchy[ $atts['level'] ] ?? 3;
		$current_level  = $access_hierarchy[ $user_access_level ] ?? 1;

		if ( $current_level >= $required_level ) {
			return do_shortcode( $content );
		}

		if ( 'true' === $atts['show_message'] ) {
			return '<div class="fr-access-required">' .
					/* translators: %s: required access level */
					sprintf( esc_html__( 'This content requires %s access.', 'freedomreader' ), $atts['level'] ) .
					'</div>';
		}

		return '';
	}

	/**
	 * Render premium required message
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string Premium required message HTML.
	 */
	private function render_premium_required_message( $atts ) {
		$message = $atts['message'] ? $atts['message'] : esc_html__( 'This content is available to premium subscribers only.', 'freedomreader' );

		$output  = '<div class="fr-premium-required">';
		$output .= '<p>' . esc_html( $message ) . '</p>';
		$output .= do_shortcode( '[freedomreader_subscription_plans]' );
		$output .= '</div>';

		return $output;
	}

	/**
	 * Render purchase required message
	 *
	 * @param array $atts    Shortcode attributes.
	 * @param int   $post_id Post ID.
	 * @return string Purchase required message HTML.
	 */
	private function render_purchase_required_message( $atts, $post_id ) {
		$price   = $atts['price'];
		$message = $atts['message'] ? $atts['message'] : esc_html__( 'Purchase this content to continue reading.', 'freedomreader' );

		$output  = '<div class="fr-purchase-required">';
		$output .= '<p>' . esc_html( $message ) . '</p>';

		if ( $price ) {
			$output .= do_shortcode( '[freedomreader_purchase_button post_id="' . $post_id . '" price="' . $price . '"]' );
		}

		$output .= '</div>';

		return $output;
	}
}
