<?php
/**
 * FreedomReader Reports Class
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Reports Class
 *
 * @package FreedomReader
 */
class FREEDO_Reports {

	/**
	 * Single instance of the class
	 *
	 * @var FREEDO_Reports|null
	 */
	private static $instance = null;

	/**
	 * Get single instance
	 *
	 * @return FREEDO_Reports Single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Constructor.
	}

	/**
	 * Get all reports data
	 *
	 * @return array Reports data.
	 */
	public function get_reports_data() {
		return array(
			'revenue_data'           => $this->get_revenue_data(),
			'subscription_analytics' => $this->get_subscription_analytics(),
			'purchase_analytics'     => $this->get_purchase_analytics(),
			'content_analytics'      => $this->get_content_analytics(),
			'recent_activity'        => array(
				'purchases'     => $this->get_recent_purchases( 10 ),
				'subscriptions' => $this->get_recent_subscriptions( 10 ),
			),
		);
	}

	/**
	 * Get revenue data for charts
	 *
	 * @return array Revenue data with labels and amounts.
	 */
	public function get_revenue_data() {
		$cache_key = 'freedomreader_revenue_data_12_months';
		$results   = wp_cache_get( $cache_key );

		if ( false === $results ) {
			global $wpdb;

			// Get revenue for last 12 months.
			$table_name = $wpdb->prefix . 'freedo_purchases';
			$query      = sprintf(
				"SELECT
					DATE_FORMAT(created_at, '%%Y-%%m') as month,
					SUM(amount) as revenue,
					COUNT(*) as transactions
				FROM `%s`
				WHERE status = %%s
				AND created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
				GROUP BY DATE_FORMAT(created_at, '%%Y-%%m')
				ORDER BY month ASC",
				$table_name
			);

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for reports with caching
			$results = $wpdb->get_results(
				$wpdb->prepare( $query, 'completed' ) // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			);

			wp_cache_set( $cache_key, $results, '', 600 );
		}

		$labels  = array();
		$amounts = array();

		if ( ! empty( $results ) ) {
			foreach ( $results as $row ) {
				$labels[]  = gmdate( 'M Y', strtotime( $row->month . '-01' ) );
				$amounts[] = floatval( $row->revenue );
			}
		} else {
			// If no data, show last 6 months with zero values.
			for ( $i = 5; $i >= 0; $i-- ) {
				$labels[]  = gmdate( 'M Y', strtotime( "-{$i} months" ) );
				$amounts[] = 0;
			}
		}

		return array(
			'labels'             => $labels,
			'amounts'            => $amounts,
			'total_revenue'      => array_sum( $amounts ),
			'total_transactions' => count( $results ),
		);
	}

	/**
	 * Get subscription analytics
	 *
	 * @return array Subscription analytics data.
	 */
	public function get_subscription_analytics() {
		$cache_key = 'freedomreader_subscription_analytics_main';
		$analytics = wp_cache_get( $cache_key );

		if ( false === $analytics ) {
			global $wpdb;

			// Get subscription stats.
			$subscriptions_table = $wpdb->prefix . 'freedo_subscriptions';
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for subscription analytics with caching
			$active_subscriptions = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM `{$subscriptions_table}` WHERE status = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
					'active'
				)
			);

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for subscription analytics with caching
			$cancelled_subscriptions = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM `{$subscriptions_table}` WHERE status = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
					'cancelled'
				)
			);

			$total_subscriptions = $active_subscriptions + $cancelled_subscriptions;

			// Calculate MRR (Monthly Recurring Revenue).
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for MRR calculation with caching
			$mrr = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT SUM(amount) FROM `{$subscriptions_table}` WHERE status = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
					'active'
				)
			);

			// Calculate churn rate.
			$churn_rate = $total_subscriptions > 0 ? ( $cancelled_subscriptions / $total_subscriptions ) * 100 : 0;

			$analytics = array(
				'active_subscriptions'    => intval( $active_subscriptions ),
				'cancelled_subscriptions' => intval( $cancelled_subscriptions ),
				'total_subscriptions'     => intval( $total_subscriptions ),
				'mrr'                     => floatval( $mrr ),
				'churn_rate'              => floatval( $churn_rate ),
				'trial_conversion_rate'   => 0, // Placeholder for now.
			);

			wp_cache_set( $cache_key, $analytics, '', 600 );
		}

		return $analytics;
	}

	/**
	 * Get purchase analytics
	 *
	 * @return array Purchase analytics data.
	 */
	public function get_purchase_analytics() {
		$cache_key = 'freedomreader_purchase_analytics';
		$analytics = wp_cache_get( $cache_key );

		if ( false === $analytics ) {
			global $wpdb;

			// Get purchase stats.
			$purchases_table = $wpdb->prefix . 'freedo_purchases';
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table query for purchase analytics with caching
			$total_purchases = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$purchases_table}` WHERE status = %s", 'completed' ) );

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table query for revenue analytics with caching
			$total_revenue = $wpdb->get_var( $wpdb->prepare( "SELECT SUM(amount) FROM `{$purchases_table}` WHERE status = %s", 'completed' ) );

			$average_purchase = $total_purchases > 0 ? $total_revenue / $total_purchases : 0;

			// Get top selling content.
			$posts_table = $wpdb->posts;
			$query       = sprintf(
				'SELECT
					p.post_id,
					posts.post_title,
					COUNT(*) as purchase_count,
					SUM(p.amount) as total_revenue
				FROM `%s` p
				LEFT JOIN `%s` posts ON p.post_id = posts.ID
				WHERE p.status = %%s
				GROUP BY p.post_id
				ORDER BY purchase_count DESC
				LIMIT 10',
				$purchases_table,
				$posts_table
			);

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for top content analytics with caching
			$top_content = $wpdb->get_results(
				$wpdb->prepare( $query, 'completed' ) // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			);

			$analytics = array(
				'total_purchases'  => intval( $total_purchases ),
				'total_revenue'    => floatval( $total_revenue ),
				'average_purchase' => floatval( $average_purchase ),
				'top_content'      => $top_content,
			);

			wp_cache_set( $cache_key, $analytics, '', 600 );
		}

		return $analytics;
	}

	/**
	 * Get content analytics
	 *
	 * @return array Content analytics data.
	 */
	public function get_content_analytics() {
		$cache_key = 'freedomreader_content_analytics';
		$analytics = wp_cache_get( $cache_key );

		if ( false === $analytics ) {
			global $wpdb;

			// Get locked content stats.
			$locks_table     = $wpdb->prefix . 'freedo_content_locks';
			$purchases_table = $wpdb->prefix . 'freedo_purchases';
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table query for content analytics with caching
			$total_locked_content = $wpdb->get_var( "SELECT COUNT(*) FROM `{$locks_table}`" );

			// Get conversion rates.
			$query = sprintf(
				'SELECT
					cl.post_id,
					cl.price,
					COUNT(p.id) as purchases,
					(COUNT(p.id) * 100.0 / GREATEST(1, cl.views)) as conversion_rate
				FROM `%s` cl
				LEFT JOIN `%s` p ON cl.post_id = p.post_id AND p.status = %%s
				GROUP BY cl.post_id
				ORDER BY conversion_rate DESC
				LIMIT 10',
				$locks_table,
				$purchases_table
			);

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for content performance analytics with caching
			$content_stats = $wpdb->get_results(
				$wpdb->prepare( $query, 'completed' ) // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			);

			$analytics = array(
				'total_locked_content' => intval( $total_locked_content ),
				'content_performance'  => $content_stats,
			);

			wp_cache_set( $cache_key, $analytics, '', 600 );
		}

		return $analytics;
	}

	/**
	 * Get recent purchases
	 *
	 * @param int $limit Number of purchases to retrieve.
	 * @return array Recent purchases data.
	 */
	public function get_recent_purchases( $limit = 20 ) {
		$cache_key = 'freedomreader_recent_purchases_' . $limit;
		$results   = wp_cache_get( $cache_key );

		if ( false === $results ) {
			global $wpdb;

			$purchases_table = $wpdb->prefix . 'freedo_purchases';
			$users_table     = $wpdb->users;
			$posts_table     = $wpdb->posts;
			$query           = sprintf(
				"SELECT
					p.*,
					u.display_name as user_name,
					posts.post_title
				FROM `%s` p
				LEFT JOIN `%s` u ON p.user_id = u.ID
				LEFT JOIN `%s` posts ON p.post_id = posts.ID
				WHERE p.status = 'completed'
				ORDER BY p.created_at DESC
				LIMIT %%d",
				$purchases_table,
				$users_table,
				$posts_table
			);

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for recent purchases with caching
			$results = $wpdb->get_results(
				$wpdb->prepare( $query, $limit ) // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			);

			wp_cache_set( $cache_key, $results, '', 300 );
		}

		return $results;
	}

	/**
	 * Get recent subscriptions
	 *
	 * @param int $limit Number of subscriptions to retrieve.
	 * @return array Recent subscriptions data.
	 */
	public function get_recent_subscriptions( $limit = 20 ) {
		$cache_key = 'freedomreader_recent_subscriptions_' . $limit;
		$results   = wp_cache_get( $cache_key );

		if ( false === $results ) {
			global $wpdb;

			$subscriptions_table = $wpdb->prefix . 'freedo_subscriptions';
			$users_table         = $wpdb->users;
			$query               = sprintf(
				'SELECT
	                s.*,
	                u.display_name as user_name
	            FROM `%s` s
	            LEFT JOIN `%s` u ON s.user_id = u.ID
	            ORDER BY s.created_at DESC
	            LIMIT %%d',
				$subscriptions_table,
				$users_table
			);

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for recent subscriptions with caching
			$results = $wpdb->get_results(
				$wpdb->prepare( $query, $limit ) // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			);

			wp_cache_set( $cache_key, $results, '', 300 );
		}

		return $results;
	}
}
