<?php
/**
 * FreedomReader Performance Monitor
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Performance Monitor Class
 */
class FREEDO_Performance_Monitor {

	/**
	 * Instance of this class
	 *
	 * @var FREEDO_Performance_Monitor
	 */
	private static $instance = null;

	/**
	 * Performance metrics
	 *
	 * @var array
	 */
	private $metrics = array();

	/**
	 * Start times for tracking
	 *
	 * @var array
	 */
	private $start_times = array();

	/**
	 * Get instance
	 *
	 * @return FREEDO_Performance_Monitor
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		if ( get_option( 'freedomreader_performance_monitoring', 1 ) ) {
			$this->init();
		}
	}

	/**
	 * Initialize performance monitoring
	 */
	private function init() {
		// Hook into WordPress actions for monitoring.
		add_action( 'init', array( $this, 'start_page_load_tracking' ) );
		add_action( 'wp_footer', array( $this, 'end_page_load_tracking' ) );
		add_action( 'admin_footer', array( $this, 'end_page_load_tracking' ) );

		// Monitor database queries.
		add_filter( 'query', array( $this, 'monitor_database_query' ) );

		// Schedule performance report.
		if ( ! wp_next_scheduled( 'freedomreader_performance_report' ) ) {
			wp_schedule_event( time(), 'daily', 'freedomreader_performance_report' );
		}
		add_action( 'freedomreader_performance_report', array( $this, 'generate_daily_report' ) );
	}

	/**
	 * Start tracking an operation
	 *
	 * @param string $operation Operation name.
	 */
	public function start_tracking( $operation ) {
		$this->start_times[ $operation ] = microtime( true );
	}

	/**
	 * End tracking an operation
	 *
	 * @param string $operation Operation name.
	 * @param array  $context   Additional context data.
	 */
	public function end_tracking( $operation, $context = array() ) {
		if ( ! isset( $this->start_times[ $operation ] ) ) {
			return;
		}

		$duration = microtime( true ) - $this->start_times[ $operation ];

		$this->record_metric( $operation, $duration, $context );

		unset( $this->start_times[ $operation ] );
	}

	/**
	 * Record a performance metric
	 *
	 * @param string $operation Operation name.
	 * @param float  $duration  Duration in seconds.
	 * @param array  $context   Additional context data.
	 */
	public function record_metric( $operation, $duration, $context = array() ) {
		$metric = array(
			'operation' => $operation,
			'duration'  => $duration,
			'timestamp' => time(),
			'memory'    => memory_get_usage( true ),
			'context'   => $context,
		);

		// Store in transient for current session.
		$session_metrics   = get_transient( 'freedomreader_performance_session' );
		$session_metrics   = $session_metrics ? $session_metrics : array();
		$session_metrics[] = $metric;
		set_transient( 'freedomreader_performance_session', $session_metrics, HOUR_IN_SECONDS );

		// Store in database for historical data.
		$this->store_metric_in_db( $metric );

		// Check for performance alerts.
		$this->check_performance_alerts( $metric );
	}

	/**
	 * Start page load tracking
	 */
	public function start_page_load_tracking() {
		if ( is_admin() ) {
			$this->start_tracking( 'admin_page_load' );
		} else {
			$this->start_tracking( 'frontend_page_load' );
		}
	}

	/**
	 * End page load tracking
	 */
	public function end_page_load_tracking() {
		global $wpdb;

		$context = array(
			'queries'    => $wpdb->num_queries,
			'page'       => is_admin() ? 'admin' : 'frontend',
			'url'        => isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '',
			'user_agent' => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',
		);

		if ( is_admin() ) {
			$this->end_tracking( 'admin_page_load', $context );
		} else {
			$this->end_tracking( 'frontend_page_load', $context );
		}
	}

	/**
	 * Monitor database queries
	 *
	 * @param string $query SQL query.
	 * @return string
	 */
	public function monitor_database_query( $query ) {
		// Only monitor FreedomReader queries.
		if ( strpos( $query, 'freedo_' ) !== false ) {
			$start_time = microtime( true );

			add_filter(
				'query',
				function ( $q ) use ( $query, $start_time ) {
					if ( $q === $query ) {
						$duration = microtime( true ) - $start_time;
						$this->record_metric( 'database_query', $duration, array( 'query' => substr( $query, 0, 100 ) ) );
					}
					return $q;
				},
				10,
				1
			);
		}

		return $query;
	}

	/**
	 * Store metric in database
	 *
	 * @param array $metric Metric data.
	 */
	private function store_metric_in_db( $metric ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'freedo_performance_metrics';

		$this->maybe_create_metrics_table();

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Performance monitoring
		$wpdb->insert(
			$table_name,
			array(
				'operation'  => $metric['operation'],
				'duration'   => $metric['duration'],
				'memory'     => $metric['memory'],
				'context'    => wp_json_encode( $metric['context'] ),
				'created_at' => gmdate( 'Y-m-d H:i:s', $metric['timestamp'] ),
			),
			array( '%s', '%f', '%d', '%s', '%s' )
		);
	}

	/**
	 * Maybe create performance metrics table
	 */
	private function maybe_create_metrics_table() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'freedo_performance_metrics';

		// Check if table exists.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table existence check, table names cannot be prepared
		$table_exists = $wpdb->get_var( "SHOW TABLES LIKE '{$table_name}'" );

		if ( ! $table_exists ) {
			$charset_collate = $wpdb->get_charset_collate();

			$sql = "CREATE TABLE {$table_name} (
				id bigint(20) NOT NULL AUTO_INCREMENT,
				operation varchar(100) NOT NULL,
				duration float NOT NULL,
				memory bigint(20) NOT NULL,
				context text,
				created_at datetime NOT NULL,
				PRIMARY KEY (id),
				KEY operation (operation),
				KEY created_at (created_at)
			) {$charset_collate};";

			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
			dbDelta( $sql );
		}
	}

	/**
	 * Check for performance alerts
	 *
	 * @param array $metric Metric data.
	 */
	private function check_performance_alerts( $metric ) {
		$alert_thresholds = array(
			'frontend_page_load' => 3.0, // 3 seconds.
			'admin_page_load'    => 5.0, // 5 seconds.
			'database_query'     => 1.0, // 1 second.
		);

		$threshold = $alert_thresholds[ $metric['operation'] ] ?? null;

		if ( $threshold && $metric['duration'] > $threshold ) {
			$this->send_performance_alert( $metric, $threshold );
		}
	}

	/**
	 * Send performance alert
	 *
	 * @param array $metric    Metric data.
	 * @param float $threshold Alert threshold.
	 */
	private function send_performance_alert( $metric, $threshold ) {
		if ( ! get_option( 'freedomreader_email_performance_alerts', 0 ) ) {
			return;
		}

		// Throttle alerts (max 1 per hour per operation type).
		$alert_key = 'freedomreader_perf_alert_' . $metric['operation'];
		if ( get_transient( $alert_key ) ) {
			return;
		}
		set_transient( $alert_key, true, HOUR_IN_SECONDS );

		$admin_email = get_option( 'admin_email' );
		$site_name   = get_bloginfo( 'name' );
		$subject     = sprintf( '[%s] FreedomReader Performance Alert', $site_name );

		$message = sprintf(
			"Performance alert for FreedomReader:\n\n" .
			"Operation: %s\n" .
			"Duration: %.2f seconds (threshold: %.2f seconds)\n" .
			"Memory Usage: %s\n" .
			"Time: %s\n" .
			"Context: %s\n\n" .
			'Please check your site performance.',
			$metric['operation'],
			$metric['duration'],
			$threshold,
			size_format( $metric['memory'] ),
			gmdate( 'Y-m-d H:i:s', $metric['timestamp'] ),
			wp_json_encode( $metric['context'] )
		);

		wp_mail( $admin_email, $subject, $message );
	}

	/**
	 * Get performance statistics
	 *
	 * @param string $period Time period (hour, day, week, month).
	 * @return array
	 */
	public function get_performance_stats( $period = 'day' ) {
		global $wpdb;

		// Calculate time range.
		$intervals = array(
			'hour'  => HOUR_IN_SECONDS,
			'day'   => DAY_IN_SECONDS,
			'week'  => WEEK_IN_SECONDS,
			'month' => MONTH_IN_SECONDS,
		);

		$interval = $intervals[ $period ] ?? DAY_IN_SECONDS;
		$since    = gmdate( 'Y-m-d H:i:s', time() - $interval );

		// Build the query with proper table name (no placeholders needed for table names).
		$query = "SELECT operation, AVG(duration) as avg_duration, MAX(duration) as max_duration,
				 MIN(duration) as min_duration, COUNT(*) as count, AVG(memory) as avg_memory
				 FROM `{$wpdb->prefix}freedo_performance_metrics`
				 WHERE created_at >= %s
				 GROUP BY operation";

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Performance statistics query with caching
		$cache_key = 'freedomreader_performance_metrics_' . md5( $since );
		$metrics   = wp_cache_get( $cache_key, 'freedomreader' );

		if ( false === $metrics ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared -- Performance metrics query with proper prepared statement and table name interpolation
			$metrics = $wpdb->get_results( $wpdb->prepare( $query, $since ), ARRAY_A );
			wp_cache_set( $cache_key, $metrics, 'freedomreader', 5 * MINUTE_IN_SECONDS );
		}

		return $metrics ? $metrics : array();
	}

	/**
	 * Generate daily performance report
	 */
	public function generate_daily_report() {
		$stats = $this->get_performance_stats( 'day' );

		if ( empty( $stats ) ) {
			return;
		}

		// Store report in options for dashboard display.
		update_option(
			'freedomreader_daily_performance_report',
			array(
				'date'  => current_time( 'Y-m-d' ),
				'stats' => $stats,
			)
		);

		// Clean old metrics (keep only last 30 days).
		$this->cleanup_old_metrics();
	}

	/**
	 * Cleanup old performance metrics
	 */
	private function cleanup_old_metrics() {
		global $wpdb;

		$cutoff = gmdate( 'Y-m-d H:i:s', time() - ( 30 * DAY_IN_SECONDS ) );

		// Use proper prepared statement with table name validation.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup old metrics with proper prepared statement
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM `{$wpdb->prefix}freedo_performance_metrics` WHERE created_at < %s",
				$cutoff
			)
		);

		// Clear related caches after cleanup.
		wp_cache_flush_group( 'freedomreader' );
	}

	/**
	 * Get current session metrics
	 *
	 * @return array
	 */
	public function get_session_metrics() {
		$metrics = get_transient( 'freedomreader_performance_session' );
		return $metrics ? $metrics : array();
	}

	/**
	 * Clear session metrics
	 */
	public function clear_session_metrics() {
		delete_transient( 'freedomreader_performance_session' );
	}
}
