<?php
/**
 * PayPal Integration Class
 *
 * @package FreedomReader
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * PayPal Integration Class
 *
 * @package FreedomReader
 */
class FREEDO_PayPal {

	/**
	 * Single instance of the class
	 *
	 * @var FREEDO_PayPal|null
	 */
	private static $instance = null;

	/**
	 * PayPal client ID
	 *
	 * @var string
	 */
	private $client_id;

	/**
	 * PayPal client secret
	 *
	 * @var string
	 */
	private $client_secret;

	/**
	 * PayPal mode (sandbox or live)
	 *
	 * @var string
	 */
	private $mode;

	/**
	 * PayPal base URL
	 *
	 * @var string
	 */
	private $base_url;

	/**
	 * Get single instance
	 *
	 * @return FREEDO_PayPal Single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->mode          = get_option( 'freedomreader_paypal_mode', 'sandbox' );
		$this->client_id     = get_option( 'freedomreader_paypal_client_id' );
		$this->client_secret = get_option( 'freedomreader_paypal_client_secret' );

		$this->base_url = ( 'live' === $this->mode )
			? 'https://api.paypal.com'
			: 'https://api.sandbox.paypal.com';

		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		add_action( 'wp_ajax_freedomreader_create_paypal_order', array( $this, 'create_order' ) );
		add_action( 'wp_ajax_nopriv_freedomreader_create_paypal_order', array( $this, 'create_order' ) );
		add_action( 'wp_ajax_freedomreader_create_subscription', array( $this, 'create_subscription' ) );
		add_action( 'wp_ajax_nopriv_freedomreader_create_subscription', array( $this, 'create_subscription' ) );
		add_action( 'wp_ajax_freedomreader_cancel_subscription', array( $this, 'cancel_subscription' ) );
		add_action( 'wp_ajax_nopriv_freedomreader_cancel_subscription', array( $this, 'cancel_subscription' ) );
		add_action( 'init', array( $this, 'handle_webhook' ) );
	}


	/**
	 * Get PayPal access token
	 *
	 * @return string|false Access token or false on failure.
	 */
	public function get_access_token() {
		$cache_key    = 'freedomreader_paypal_token_' . $this->mode;
		$cached_token = get_transient( $cache_key );

		if ( $cached_token ) {
			return $cached_token;
		}

		// Check if credentials are set.
		if ( empty( $this->client_id ) || empty( $this->client_secret ) ) {
			// PayPal credentials not configured - silently fail.
			return false;
		}

		$url = $this->base_url . '/v1/oauth2/token';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Accept'          => 'application/json',
				'Accept-Language' => 'en_US',
				// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Required for PayPal API authentication
				'Authorization'   => 'Basic ' . base64_encode( $this->client_id . ':' . $this->client_secret ),
			),
			'body'    => 'grant_type=client_credentials',
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			// PayPal token request failed - silently fail.
			return false;
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		// Process PayPal token response.

		if ( isset( $body['access_token'] ) ) {
			// Cache token for 50 minutes (expires in 1 hour).
			set_transient( $cache_key, $body['access_token'], 3000 );
			return $body['access_token'];
		}

		// PayPal token not found in response.
		return false;
	}

	/**
	 * Create PayPal order for single purchase
	 */
	public function create_order() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed', 'freedomreader' ) );
		}

		$post_id  = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		$price    = isset( $_POST['price'] ) ? floatval( $_POST['price'] ) : 0;
		$currency = get_option( 'freedomreader_currency', 'USD' );

		$access_token = $this->get_access_token();

		if ( ! $access_token ) {
			wp_send_json_error( array( 'message' => esc_html__( 'PayPal authentication failed', 'freedomreader' ) ) );
		}

		$post_title = get_the_title( $post_id );

		$order_data = array(
			'intent'              => 'CAPTURE',
			'purchase_units'      => array(
				array(
					'reference_id' => 'post_' . $post_id,
					'amount'       => array(
						'currency_code' => $currency,
						'value'         => number_format( $price, 2, '.', '' ),
					),
					/* translators: %s: post title */
					'description'  => sprintf( esc_html__( 'Access to: %s', 'freedomreader' ), $post_title ),
				),
			),
			'application_context' => array(
				'return_url' => get_permalink( $post_id ),
				'cancel_url' => get_permalink( $post_id ),
			),
		);

		$url = $this->base_url . '/v2/checkout/orders';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => wp_json_encode( $order_data ),
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to create PayPal order', 'freedomreader' ) ) );
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		// Process PayPal order creation response.

		if ( isset( $body['id'] ) ) {
			wp_send_json_success(
				array(
					'order_id'     => $body['id'],
					'approval_url' => $this->get_approval_url( $body['links'] ),
				)
			);
		} else {
			$error_message = esc_html__( 'Failed to create PayPal order', 'freedomreader' );
			if ( isset( $body['message'] ) ) {
				$error_message = $body['message'];
			} elseif ( isset( $body['details'] ) && is_array( $body['details'] ) && ! empty( $body['details'] ) ) {
				$error_message = $body['details'][0]['description'] ?? $error_message;
			}

			// PayPal order creation failed.
			wp_send_json_error( array( 'message' => $error_message ) );
		}
	}

	/**
	 * Create PayPal order (method version)
	 *
	 * @param array $order_data Order data.
	 * @return array Order creation result.
	 */
	public function create_order_api( $order_data ) {
		$access_token = $this->get_access_token();

		if ( ! $access_token ) {
			return array(
				'success' => false,
				'message' => esc_html__( 'PayPal authentication failed', 'freedomreader' ),
			);
		}

		$currency    = $order_data['currency'] ?? 'USD';
		$amount      = $order_data['amount'];
		$description = $order_data['description'] ?? esc_html__( 'Premium Content Purchase', 'freedomreader' );

		$paypal_order_data = array(
			'intent'              => 'CAPTURE',
			'purchase_units'      => array(
				array(
					'amount'      => array(
						'currency_code' => $currency,
						'value'         => number_format( $amount, 2, '.', '' ),
					),
					'description' => $description,
				),
			),
			'application_context' => array(
				'return_url'  => $order_data['return_url'] ?? home_url(),
				'cancel_url'  => $order_data['cancel_url'] ?? home_url(),
				'brand_name'  => esc_html( get_bloginfo( 'name' ) ),
				'user_action' => 'PAY_NOW',
			),
		);

		$url = $this->base_url . '/v2/checkout/orders';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => wp_json_encode( $paypal_order_data ),
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => esc_html__( 'PayPal connection failed', 'freedomreader' ),
			);
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( isset( $body['id'] ) ) {
			return array(
				'success'      => true,
				'order_id'     => $body['id'],
				'approval_url' => $this->get_approval_url( $body['links'] ?? array() ),
			);
		} else {
			return array(
				'success' => false,
				'message' => isset( $body['message'] ) ? $body['message'] : esc_html__( 'Failed to create PayPal order', 'freedomreader' ),
			);
		}
	}

	/**
	 * Capture payment after approval
	 *
	 * @param string $order_id PayPal order ID.
	 * @return array Capture result.
	 */
	public function capture_payment( $order_id ) {
		$access_token = $this->get_access_token();

		if ( ! $access_token ) {
			return array(
				'success' => false,
				'message' => esc_html__( 'PayPal authentication failed', 'freedomreader' ),
			);
		}

		$url = $this->base_url . '/v2/checkout/orders/' . $order_id . '/capture';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => esc_html__( 'Failed to capture payment', 'freedomreader' ),
			);
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		// Process PayPal capture response.

		if ( isset( $body['status'] ) && 'COMPLETED' === $body['status'] ) {
			$capture = $body['purchase_units'][0]['payments']['captures'][0];

			return array(
				'success'        => true,
				'transaction_id' => $capture['id'],
				'amount'         => $capture['amount']['value'],
				'currency'       => $capture['amount']['currency_code'],
			);
		}

		// Log error details.
		$error_message = esc_html__( 'Payment capture failed', 'freedomreader' );
		if ( isset( $body['message'] ) ) {
			$error_message = $body['message'];
		} elseif ( isset( $body['details'] ) && is_array( $body['details'] ) && ! empty( $body['details'] ) ) {
			$error_message = $body['details'][0]['description'] ?? $error_message;
		}

		// PayPal capture failed.

		return array(
			'success' => false,
			'message' => $error_message,
		);
	}

	/**
	 * Create subscription
	 */
	public function create_subscription() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed', 'freedomreader' ) );
		}

		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Please log in to subscribe', 'freedomreader' ) ) );
		}

		$plan_id            = isset( $_POST['plan_id'] ) ? sanitize_text_field( wp_unslash( $_POST['plan_id'] ) ) : '';
		$subscription_plans = get_option( 'freedomreader_subscription_plans', array() );

		if ( ! isset( $subscription_plans[ $plan_id ] ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid subscription plan', 'freedomreader' ) ) );
		}

		$plan         = $subscription_plans[ $plan_id ];
		$access_token = $this->get_access_token();

		if ( ! $access_token ) {
			wp_send_json_error( array( 'message' => esc_html__( 'PayPal authentication failed', 'freedomreader' ) ) );
		}

		// First create a product.
		$product_id = $this->create_product( $plan['name'] );

		if ( ! $product_id ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to create subscription product', 'freedomreader' ) ) );
		}

		// Create billing plan.
		$billing_plan_id = $this->create_billing_plan( $product_id, $plan );

		if ( ! $billing_plan_id ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to create billing plan', 'freedomreader' ) ) );
		}

		// Create subscription.
		$subscription_data = array(
			'plan_id'             => $billing_plan_id,
			'subscriber'          => array(
				'name'          => array(
					'given_name' => wp_get_current_user()->first_name ? wp_get_current_user()->first_name : 'Customer',
					'surname'    => wp_get_current_user()->last_name ? wp_get_current_user()->last_name : 'User',
				),
				'email_address' => wp_get_current_user()->user_email,
			),
			'application_context' => array(
				'brand_name'          => esc_html( get_bloginfo( 'name' ) ),
				'locale'              => 'en-US',
				'shipping_preference' => 'NO_SHIPPING',
				'user_action'         => 'SUBSCRIBE_NOW',
				'payment_method'      => array(
					'payer_selected'  => 'PAYPAL',
					'payee_preferred' => 'IMMEDIATE_PAYMENT_REQUIRED',
				),
				'return_url'          => home_url( '/freedomreader/subscription-success/' ),
				'cancel_url'          => home_url( '/freedomreader/subscription-cancelled/' ),
			),
		);

		$url = $this->base_url . '/v1/billing/subscriptions';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => wp_json_encode( $subscription_data ),
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to create subscription', 'freedomreader' ) ) );
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( isset( $body['id'] ) ) {
			wp_send_json_success(
				array(
					'subscription_id' => $body['id'],
					'approval_url'    => $this->get_approval_url( $body['links'] ),
				)
			);
		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to create subscription', 'freedomreader' ) ) );
		}
	}

	/**
	 * Create PayPal product
	 *
	 * @param string $name Product name.
	 * @return string|false Product ID or false on failure.
	 */
	private function create_product( $name ) {
		$access_token = $this->get_access_token();

		$product_data = array(
			'name'        => $name,
			/* translators: 1: subscription plan name, 2: site name */
			'description' => sprintf(
				/* translators: 1: subscription plan name, 2: site name */
				esc_html__( '%1$s subscription for %2$s', 'freedomreader' ),
				$name,
				esc_html( get_bloginfo( 'name' ) )
			),
			'type'        => 'SERVICE',
			'category'    => 'SOFTWARE',
		);

		$url = $this->base_url . '/v1/catalogs/products';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => wp_json_encode( $product_data ),
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return false;
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		return isset( $body['id'] ) ? $body['id'] : false;
	}

	/**
	 * Create billing plan
	 *
	 * @param string $product_id PayPal product ID.
	 * @param array  $plan       Plan data.
	 * @return string|false Plan ID or false on failure.
	 */
	private function create_billing_plan( $product_id, $plan ) {
		$access_token = $this->get_access_token();
		$currency     = get_option( 'freedomreader_currency', 'USD' );

		$billing_plan_data = array(
			'product_id'          => $product_id,
			'name'                => $plan['name'],
			'description'         => $plan['name'] . ' subscription',
			'billing_cycles'      => array(
				array(
					'frequency'      => array(
						'interval_unit'  => strtoupper( $plan['interval'] ),
						'interval_count' => 1,
					),
					'tenure_type'    => 'REGULAR',
					'sequence'       => 1,
					'total_cycles'   => 0,
					'pricing_scheme' => array(
						'fixed_price' => array(
							'value'         => number_format( $plan['price'], 2, '.', '' ),
							'currency_code' => $currency,
						),
					),
				),
			),
			'payment_preferences' => array(
				'auto_bill_outstanding'     => true,
				'setup_fee_failure_action'  => 'CONTINUE',
				'payment_failure_threshold' => 3,
			),
		);

		$url = $this->base_url . '/v1/billing/plans';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => wp_json_encode( $billing_plan_data ),
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return false;
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		return isset( $body['id'] ) ? $body['id'] : false;
	}

	/**
	 * Get approval URL from PayPal response links
	 *
	 * @param array $links PayPal response links.
	 * @return string|null Approval URL or null.
	 */
	private function get_approval_url( $links ) {
		foreach ( $links as $link ) {
			if ( 'approve' === $link['rel'] ) {
				return $link['href'];
			}
		}
		return false;
	}

	/**
	 * Sanitize webhook data recursively
	 *
	 * @param array $data Webhook data to sanitize.
	 * @return array Sanitized webhook data.
	 */
	private function sanitize_webhook_data( $data ) {
		if ( ! is_array( $data ) ) {
			return sanitize_text_field( $data );
		}

		$sanitized = array();
		foreach ( $data as $key => $value ) {
			$sanitized_key = sanitize_key( $key );
			if ( is_array( $value ) ) {
				$sanitized[ $sanitized_key ] = $this->sanitize_webhook_data( $value );
			} elseif ( is_string( $value ) ) {
				$sanitized[ $sanitized_key ] = sanitize_text_field( $value );
			} elseif ( is_numeric( $value ) ) {
				$sanitized[ $sanitized_key ] = is_float( $value ) ? floatval( $value ) : intval( $value );
			} elseif ( is_bool( $value ) ) {
				$sanitized[ $sanitized_key ] = (bool) $value;
			} else {
				// For other types, convert to string and sanitize.
				$sanitized[ $sanitized_key ] = sanitize_text_field( (string) $value );
			}
		}
		return $sanitized;
	}

	/**
	 * Handle PayPal webhooks
	 */
	public function handle_webhook() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Webhook verification handled by PayPal signature validation
		if ( ! isset( $_GET['freedomreader_webhook'] ) || sanitize_text_field( wp_unslash( $_GET['freedomreader_webhook'] ) ) !== 'paypal' ) {
			return;
		}

		$input = file_get_contents( 'php://input' );
		if ( empty( $input ) ) {
			http_response_code( 400 );
			exit;
		}

		$webhook_data = json_decode( $input, true );

		if ( ! $webhook_data || ! is_array( $webhook_data ) ) {
			http_response_code( 400 );
			exit;
		}

		// Validate required webhook fields.
		if ( ! isset( $webhook_data['event_type'] ) || empty( $webhook_data['event_type'] ) ) {
			http_response_code( 400 );
			exit;
		}

		// Sanitize event type.
		$event_type = sanitize_text_field( $webhook_data['event_type'] );

		// Validate event type against allowed values.
		$allowed_events = array(
			'BILLING.SUBSCRIPTION.ACTIVATED',
			'BILLING.SUBSCRIPTION.CANCELLED',
			'PAYMENT.SALE.COMPLETED',
		);

		if ( ! in_array( $event_type, $allowed_events, true ) ) {
			http_response_code( 400 );
			exit;
		}

		// Sanitize webhook data recursively.
		$sanitized_webhook_data = $this->sanitize_webhook_data( $webhook_data );

		// Log webhook.
		FREEDO_Database::get_instance()->log_payment(
			array(
				'transaction_type' => 'webhook',
				'status'           => $event_type,
				'response_data'    => $sanitized_webhook_data,
			)
		);

		// Handle different webhook events.
		switch ( $event_type ) {
			case 'BILLING.SUBSCRIPTION.ACTIVATED':
				$this->handle_subscription_activated( $sanitized_webhook_data );
				break;
			case 'BILLING.SUBSCRIPTION.CANCELLED':
				$this->handle_subscription_cancelled( $sanitized_webhook_data );
				break;
			case 'PAYMENT.SALE.COMPLETED':
				$this->handle_payment_completed( $sanitized_webhook_data );
				break;
		}

		http_response_code( 200 );
		exit;
	}

	/**
	 * Handle subscription activated webhook
	 *
	 * @param array $webhook_data Webhook data.
	 */
	private function handle_subscription_activated( $webhook_data ) {
		$subscription_id = $webhook_data['resource']['id'];

		// Update subscription status in database.
		global $wpdb;
		$table = $wpdb->prefix . 'freedo_subscriptions';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update for webhook processing
		$wpdb->update(
			$table,
			array( 'status' => 'active' ),
			array( 'paypal_subscription_id' => $subscription_id )
		);
	}

	/**
	 * Handle subscription cancelled webhook
	 *
	 * @param array $webhook_data Webhook data.
	 */
	private function handle_subscription_cancelled( $webhook_data ) {
		$subscription_id = $webhook_data['resource']['id'];

		// Update subscription status in database.
		global $wpdb;
		$table = $wpdb->prefix . 'freedo_subscriptions';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update for webhook processing
		$wpdb->update(
			$table,
			array( 'status' => 'cancelled' ),
			array( 'paypal_subscription_id' => $subscription_id )
		);

		// Fire action hook for developers.
		do_action( 'freedomreader_on_subscription_cancelled', $subscription_id, $webhook_data );
	}

	/**
	 * Handle payment completed webhook
	 *
	 * @param array $webhook_data Webhook data.
	 */
	private function handle_payment_completed( $webhook_data ) {
		// Handle recurring payment completion.
		// Update next billing date, etc.
		// Handle PayPal payment completion webhook.
	}

	/**
	 * Cancel subscription via AJAX
	 *
	 * @package FreedomReader
	 */
	public function cancel_subscription() {
		// Verify nonce for security.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Please log in to cancel subscription.', 'freedomreader' ) ) );
		}

		$subscription_id = isset( $_POST['subscription_id'] ) ? sanitize_text_field( wp_unslash( $_POST['subscription_id'] ) ) : '';

		if ( empty( $subscription_id ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid subscription ID', 'freedomreader' ) ) );
		}

		// Verify user owns this subscription.
		global $wpdb;
		$table = $wpdb->prefix . 'freedo_purchases';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Subscription ownership verification
		$subscription = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}freedo_purchases WHERE paypal_subscription_id = %s AND user_id = %d AND purchase_type = 'subscription'",
				$subscription_id,
				get_current_user_id()
			)
		);

		if ( ! $subscription ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Subscription not found or access denied', 'freedomreader' ) ) );
		}

		if ( 'cancelled' === $subscription->status ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Subscription already cancelled', 'freedomreader' ) ) );
		}

		// Cancel subscription via PayPal API.
		$access_token = $this->get_access_token();

		if ( ! $access_token ) {
			wp_send_json_error( array( 'message' => esc_html__( 'PayPal authentication failed', 'freedomreader' ) ) );
		}

		$url = $this->base_url . '/v1/billing/subscriptions/' . $subscription_id . '/cancel';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => wp_json_encode(
				array(
					'reason' => esc_html__( 'User requested cancellation', 'freedomreader' ),
				)
			),
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to cancel subscription', 'freedomreader' ) ) );
		}

		$response_code = wp_remote_retrieve_response_code( $response );

		if ( 204 === $response_code ) {
			// Update subscription status in database.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Status update after cancellation
			$wpdb->update(
				$table,
				array(
					'status'     => 'cancelled',
					'updated_at' => current_time( 'mysql' ),
				),
				array( 'id' => $subscription->id )
			);

			// Clear relevant caches.
			wp_cache_delete( 'freedomreader_dashboard_stats' );
			wp_cache_delete( 'freedomreader_user_subscriptions_' . get_current_user_id() );

			wp_send_json_success( array( 'message' => esc_html__( 'Subscription cancelled successfully', 'freedomreader' ) ) );
		} else {
			$body          = json_decode( wp_remote_retrieve_body( $response ), true );
			$error_message = esc_html__( 'Failed to cancel subscription', 'freedomreader' );

			if ( isset( $body['message'] ) ) {
				$error_message = $body['message'];
			} elseif ( isset( $body['details'] ) && is_array( $body['details'] ) && ! empty( $body['details'] ) ) {
				$error_message = $body['details'][0]['description'] ?? $error_message;
			}

			wp_send_json_error( array( 'message' => $error_message ) );
		}
	}

	/**
	 * Refund payment via PayPal API
	 *
	 * @param string $transaction_id PayPal transaction ID.
	 * @return array Refund result.
	 */
	public function refund_payment( $transaction_id ) {
		$access_token = $this->get_access_token();

		if ( ! $access_token ) {
			return array(
				'success' => false,
				'message' => esc_html__( 'PayPal authentication failed', 'freedomreader' ),
			);
		}

		$url = $this->base_url . '/v2/payments/captures/' . $transaction_id . '/refund';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Content-Type'  => 'application/json',
				'Authorization' => 'Bearer ' . $access_token,
			),
			'body'    => wp_json_encode( array() ), // Empty body for full refund.
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => esc_html__( 'Failed to process refund', 'freedomreader' ),
			);
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$body          = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 201 === $response_code && isset( $body['id'] ) ) {
			return array(
				'success'   => true,
				'refund_id' => $body['id'],
				'message'   => esc_html__( 'Refund processed successfully', 'freedomreader' ),
			);
		} else {
			$error_message = esc_html__( 'Failed to process refund', 'freedomreader' );

			if ( isset( $body['message'] ) ) {
				$error_message = $body['message'];
			} elseif ( isset( $body['details'] ) && is_array( $body['details'] ) && ! empty( $body['details'] ) ) {
				$error_message = $body['details'][0]['description'] ?? $error_message;
			}

			return array(
				'success' => false,
				'message' => $error_message,
			);
		}
	}
}
