<?php
/**
 * Database operations for FreedomReader
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Database operations class for FreedomReader plugin.
 *
 * @package FreedomReader
 */
class FREEDO_Database {

	/**
	 * Singleton instance.
	 *
	 * @var FREEDO_Database|null
	 */
	private static $instance = null;

	/**
	 * Get singleton instance.
	 *
	 * @return FREEDO_Database
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		// Constructor.
	}

	/**
	 * Create plugin tables
	 *
	 * @package FreedomReader
	 */
	public static function create_tables() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		// Purchases table.
		$purchases_table = $wpdb->prefix . 'freedo_purchases';
		$purchases_sql   = "CREATE TABLE $purchases_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) NOT NULL,
            post_id bigint(20) DEFAULT NULL,
            purchase_type enum('single','subscription') NOT NULL,
            amount decimal(10,2) NOT NULL,
            currency varchar(3) NOT NULL DEFAULT 'USD',
            paypal_transaction_id varchar(255) DEFAULT NULL,
            paypal_subscription_id varchar(255) DEFAULT NULL,
            status enum('pending','completed','cancelled','refunded') NOT NULL DEFAULT 'pending',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            expires_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY post_id (post_id),
            KEY status (status)
        ) $charset_collate;";

		// Subscriptions table.
		$subscriptions_table = $wpdb->prefix . 'freedo_subscriptions';
		$subscriptions_sql   = "CREATE TABLE $subscriptions_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) NOT NULL,
            plan_id varchar(50) NOT NULL,
            paypal_subscription_id varchar(255) NOT NULL,
            status enum('active','cancelled','expired','trial') NOT NULL DEFAULT 'trial',
            amount decimal(10,2) DEFAULT NULL,
            currency varchar(3) DEFAULT 'USD',
            trial_ends_at datetime DEFAULT NULL,
            next_billing_date datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY status (status),
            UNIQUE KEY paypal_subscription_id (paypal_subscription_id)
        ) $charset_collate;";

		// Content locks table.
		$content_locks_table = $wpdb->prefix . 'freedo_content_locks';
		$content_locks_sql   = "CREATE TABLE $content_locks_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            lock_type enum('full','partial','subscription_only') NOT NULL DEFAULT 'full',
            price decimal(10,2) DEFAULT NULL,
            teaser_length int(11) DEFAULT 150,
            subscription_required tinyint(1) DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY post_id (post_id)
        ) $charset_collate;";

		// Payment logs table.
		$payment_logs_table = $wpdb->prefix . 'freedo_payment_logs';
		$payment_logs_sql   = "CREATE TABLE $payment_logs_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) DEFAULT NULL,
            transaction_type enum('purchase','subscription','refund','webhook') NOT NULL,
            paypal_transaction_id varchar(255) DEFAULT NULL,
            amount decimal(10,2) DEFAULT NULL,
            currency varchar(3) DEFAULT 'USD',
            status varchar(50) NOT NULL,
            response_data text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY transaction_type (transaction_type),
            KEY created_at (created_at)
        ) $charset_collate;";

		// Global lock rules table.
		$lock_rules_table = $wpdb->prefix . 'freedo_lock_rules';
		$lock_rules_sql   = "CREATE TABLE $lock_rules_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            rule_name varchar(255) NOT NULL,
            rule_type varchar(50) NOT NULL,
            rule_value text NOT NULL,
            lock_conditions text NOT NULL,
            priority int(11) DEFAULT 10,
            is_active tinyint(1) DEFAULT 1,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY rule_type (rule_type),
            KEY is_active (is_active),
            KEY priority (priority)
        ) $charset_collate;";

		// Geolocation cache table.
		$geo_cache_table = $wpdb->prefix . 'freedo_geo_cache';
		$geo_cache_sql   = "CREATE TABLE $geo_cache_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            ip_address varchar(45) NOT NULL,
            country_code varchar(2) NOT NULL,
            country_name varchar(100) NOT NULL,
            cached_at datetime DEFAULT CURRENT_TIMESTAMP,
            expires_at datetime NOT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY ip_address (ip_address),
            KEY expires_at (expires_at)
        ) $charset_collate;";

		// Email unlock requests table.
		$email_unlocks_table = $wpdb->prefix . 'freedo_email_unlocks';
		$email_unlocks_sql   = "CREATE TABLE $email_unlocks_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            email varchar(255) NOT NULL,
            post_id bigint(20) unsigned NOT NULL,
            unlock_token varchar(64) NOT NULL,
            ip_address varchar(45) NOT NULL,
            user_agent text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            expires_at datetime NOT NULL,
            used_at datetime NULL,
            PRIMARY KEY (id),
            UNIQUE KEY unlock_token (unlock_token),
            KEY email (email),
            KEY post_id (post_id),
            KEY expires_at (expires_at)
        ) $charset_collate;";

		// Analytics table.
		$analytics_table = $wpdb->prefix . 'freedo_analytics';
		$analytics_sql   = "CREATE TABLE $analytics_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            post_id bigint(20) unsigned NOT NULL,
            user_id bigint(20) unsigned NULL,
            ip_address varchar(45) NOT NULL,
            user_agent text,
            referrer text,
            device_type varchar(20) NOT NULL,
            reading_time int(11) DEFAULT 0,
            scroll_depth int(3) DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY post_id (post_id),
            KEY user_id (user_id),
            KEY device_type (device_type),
            KEY created_at (created_at)
        ) $charset_collate;";

		// Author questions table.
		$author_questions_table = $wpdb->prefix . 'freedo_author_questions';
		$author_questions_sql   = "CREATE TABLE $author_questions_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            post_id bigint(20) unsigned NOT NULL,
            author_id bigint(20) unsigned NOT NULL,
            questioner_name varchar(255) NOT NULL,
            questioner_email varchar(255) NOT NULL,
            question text NOT NULL,
            answer text NULL,
            status varchar(20) DEFAULT 'pending',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            answered_at datetime NULL,
            PRIMARY KEY (id),
            KEY post_id (post_id),
            KEY author_id (author_id),
            KEY status (status)
        ) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		dbDelta( $purchases_sql );
		dbDelta( $subscriptions_sql );
		dbDelta( $content_locks_sql );
		dbDelta( $payment_logs_sql );
		dbDelta( $lock_rules_sql );
		dbDelta( $geo_cache_sql );
		dbDelta( $email_unlocks_sql );
		dbDelta( $analytics_sql );
		dbDelta( $author_questions_sql );

		// Update database version.
		update_option( 'freedomreader_db_version', FREEDOMREADER_VERSION );
	}

	/**
	 * Get user purchases.
	 *
	 * @param int $user_id User ID.
	 * @return array|false User purchases or false on failure.
	 * @package FreedomReader
	 */
	public function get_user_purchases( $user_id ) {
		$cache_key = 'freedomreader_user_purchases_' . $user_id;
		$purchases = wp_cache_get( $cache_key );

		if ( false === $purchases ) {
			global $wpdb;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- User purchases lookup with caching
			$purchases = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM `{$wpdb->prefix}freedo_purchases` WHERE user_id = %d ORDER BY created_at DESC",
					$user_id
				)
			);

			wp_cache_set( $cache_key, $purchases, '', 300 );
		}

		return $purchases;
	}

	/**
	 * Get user subscription.
	 *
	 * @param int $user_id User ID.
	 * @return object|false User subscription or false on failure.
	 * @package FreedomReader
	 */
	public function get_user_subscription( $user_id ) {
		$cache_key    = 'freedomreader_user_subscription_' . $user_id;
		$subscription = wp_cache_get( $cache_key );

		if ( false === $subscription ) {
			global $wpdb;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- User subscription lookup with caching
			$subscription = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM `{$wpdb->prefix}freedo_subscriptions` WHERE user_id = %d AND status IN ('active', 'trial') ORDER BY created_at DESC LIMIT 1",
					$user_id
				)
			);

			wp_cache_set( $cache_key, $subscription, '', 300 );
		}

		return $subscription;
	}

	/**
	 * Check if user has access to content.
	 *
	 * @param int $user_id User ID.
	 * @param int $post_id Post ID.
	 * @return bool True if user has access, false otherwise.
	 * @package FreedomReader
	 */
	public function user_has_access( $user_id, $post_id ) {
		$cache_key  = 'freedomreader_user_access_' . $user_id . '_' . $post_id;
		$has_access = wp_cache_get( $cache_key );

		if ( false === $has_access ) {
			global $wpdb;

			// Check for single purchase.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Access verification query with caching
			$single_purchase = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM `{$wpdb->prefix}freedo_purchases`
	             WHERE user_id = %d AND post_id = %d AND status = 'completed'
	             AND (expires_at IS NULL OR expires_at > NOW())",
					$user_id,
					$post_id
				)
			);

			if ( 0 < $single_purchase ) {
				$has_access = true;
			} else {
				// Check for active subscription.
				$subscription = $this->get_user_subscription( $user_id );
				if ( $subscription && ( 'active' === $subscription->status ||
					( 'trial' === $subscription->status && $subscription->trial_ends_at > current_time( 'mysql' ) ) ) ) {
					$has_access = true;
				} else {
					$has_access = false;
				}
			}

			wp_cache_set( $cache_key, $has_access, '', 180 );
		}

		return $has_access;
	}

	/**
	 * Record purchase.
	 *
	 * @param array $data Purchase data.
	 * @return int|false Insert ID on success, false on failure.
	 * @package FreedomReader
	 */
	public function record_purchase( $data ) {
		global $wpdb;

		$defaults = array(
			'user_id'                => 0,
			'post_id'                => null,
			'purchase_type'          => 'single',
			'amount'                 => 0,
			'currency'               => 'USD',
			'paypal_transaction_id'  => null,
			'paypal_subscription_id' => null,
			'status'                 => 'pending',
			'created_at'             => current_time( 'mysql' ),
			'expires_at'             => null,
		);

		$data = wp_parse_args( $data, $defaults );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Purchase record insertion
		$result = $wpdb->insert( $wpdb->prefix . 'freedo_purchases', $data );

		// Clear related caches when purchase is recorded.
		if ( false !== $result && isset( $data['user_id'] ) ) {
			wp_cache_delete( 'freedomreader_user_purchases_' . $data['user_id'] );
			wp_cache_delete( 'freedomreader_dashboard_stats' );
			wp_cache_delete( 'freedomreader_purchase_analytics' );
			wp_cache_delete( 'freedomreader_content_analytics' );
			wp_cache_delete( 'freedomreader_admin_recent_activity' );
			wp_cache_delete( 'freedomreader_admin_top_selling_content' );
			wp_cache_delete( 'freedomreader_recent_purchases_20' );
			wp_cache_delete( 'freedomreader_purchases_total_count' );
			if ( isset( $data['post_id'] ) ) {
				wp_cache_delete( 'freedomreader_user_access_' . $data['user_id'] . '_' . $data['post_id'] );
				wp_cache_delete( 'freedomreader_user_post_purchases_' . $data['user_id'] . '_' . $data['post_id'] );
			}
		}

		return $result;
	}

	/**
	 * Record subscription.
	 *
	 * @param array $data Subscription data.
	 * @return int|false Insert ID on success, false on failure.
	 * @package FreedomReader
	 */
	public function record_subscription( $data ) {
		global $wpdb;

		$defaults = array(
			'user_id'                => 0,
			'plan_id'                => '',
			'paypal_subscription_id' => '',
			'status'                 => 'trial',
			'trial_ends_at'          => null,
			'next_billing_date'      => null,
			'created_at'             => current_time( 'mysql' ),
		);

		$data = wp_parse_args( $data, $defaults );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Subscription record insertion
		$result = $wpdb->insert( $wpdb->prefix . 'freedo_subscriptions', $data );

		// Clear related caches when subscription is recorded.
		if ( false !== $result && isset( $data['user_id'] ) ) {
			wp_cache_delete( 'freedomreader_user_subscription_' . $data['user_id'] );
			wp_cache_delete( 'freedomreader_dashboard_stats' );
			wp_cache_delete( 'freedomreader_admin_subscription_analytics' );
			wp_cache_delete( 'freedomreader_admin_recent_activity' );
			wp_cache_delete( 'freedomreader_recent_subscriptions_20' );
			wp_cache_delete( 'freedomreader_subscriptions_total_count' );
			// Clear user access cache for all posts since subscription affects global access.
			// Use pattern-based cache clearing for user access.
			$this->clear_user_access_cache( $data['user_id'] );
		}

		return $result;
	}

	/**
	 * Log payment activity.
	 *
	 * @param array $data Payment data.
	 * @return int|false Insert ID on success, false on failure.
	 * @package FreedomReader
	 */
	public function log_payment( $data ) {
		global $wpdb;

		$defaults = array(
			'user_id'               => null,
			'transaction_type'      => 'purchase',
			'paypal_transaction_id' => null,
			'amount'                => null,
			'currency'              => 'USD',
			'status'                => '',
			'response_data'         => '',
			'created_at'            => current_time( 'mysql' ),
		);

		$data = wp_parse_args( $data, $defaults );

		if ( is_array( $data['response_data'] ) || is_object( $data['response_data'] ) ) {
			$data['response_data'] = wp_json_encode( $data['response_data'] );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Payment log insertion
		$result = $wpdb->insert( $wpdb->prefix . 'freedo_payment_logs', $data );

		// Clear related caches when payment is logged.
		if ( false !== $result ) {
			wp_cache_delete( 'freedomreader_dashboard_stats' );
			// Clear reports caches as payment logs affect analytics.
			wp_cache_delete( 'freedomreader_purchase_analytics' );
			wp_cache_delete( 'freedomreader_admin_recent_activity' );
		}

		return $result;
	}

	/**
	 * Get content lock settings.
	 *
	 * @param int $post_id Post ID.
	 * @return object|false Lock settings or false on failure.
	 * @package FreedomReader
	 */
	public function get_content_lock( $post_id ) {
		$cache_key     = 'freedomreader_content_lock_' . $post_id;
		$lock_settings = wp_cache_get( $cache_key );

		if ( false === $lock_settings ) {
			global $wpdb;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Content lock settings lookup with caching
			$lock_settings = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM `{$wpdb->prefix}freedo_content_locks` WHERE post_id = %d",
					$post_id
				)
			);

			wp_cache_set( $cache_key, $lock_settings, '', 600 ); // Cache for 10 minutes.
		}

		return $lock_settings;
	}

	/**
	 * Save content lock settings.
	 *
	 * @param int   $post_id Post ID.
	 * @param array $data Lock settings data.
	 * @return int|false Number of rows affected or false on failure.
	 * @package FreedomReader
	 */
	public function save_content_lock( $post_id, $data ) {
		global $wpdb;

		$defaults = array(
			'lock_type'             => 'full',
			'price'                 => null,
			'teaser_length'         => 150,
			'subscription_required' => 0,
		);

		$data            = wp_parse_args( $data, $defaults );
		$data['post_id'] = $post_id;

		// Check if record exists.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Content lock existence check
		$exists = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$wpdb->prefix}freedo_content_locks` WHERE post_id = %d",
				$post_id
			)
		);

		if ( $exists ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Content lock update
			$result = $wpdb->update( $wpdb->prefix . 'freedo_content_locks', $data, array( 'post_id' => $post_id ) );
		} else {
			$data['created_at'] = current_time( 'mysql' );
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Content lock insertion
			$result = $wpdb->insert( $wpdb->prefix . 'freedo_content_locks', $data );
		}

		// Clear related caches when content lock is saved.
		if ( false !== $result ) {
			wp_cache_delete( 'freedomreader_content_lock_' . $post_id );
			wp_cache_delete( 'freedomreader_dashboard_stats' );
		}

		return $result;
	}

	/**
	 * Clear user access cache for all posts.
	 *
	 * @param int $user_id User ID.
	 * @package FreedomReader
	 */
	private function clear_user_access_cache( $user_id ) {
		// Since WordPress doesn't support wildcard cache deletion,
		// we'll use a more targeted approach by clearing common cache keys
		// This is a simplified version - in production you might want to track cache keys.

		// Clear access cache for recent posts (common scenario).
		$recent_posts = get_posts(
			array(
				'numberposts' => 50,
				'post_status' => 'publish',
				'fields'      => 'ids',
			)
		);

		foreach ( $recent_posts as $post_id ) {
			wp_cache_delete( 'freedomreader_user_access_' . $user_id . '_' . $post_id );
		}
	}
}
