<?php
/**
 * Content Locker for FreedomReader
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Content Locker Class
 *
 * @package FreedomReader
 */
class FREEDO_Content_Locker {

	/**
	 * Single instance of the class
	 *
	 * @var FREEDO_Content_Locker|null
	 */
	private static $instance = null;

	/**
	 * Database instance
	 *
	 * @var FREEDO_Database
	 */
	private $db;

	/**
	 * Get single instance
	 *
	 * @return FREEDO_Content_Locker Single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->db = FREEDO_Database::get_instance();
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		add_filter( 'the_content', array( $this, 'filter_content' ), 10 );
		add_filter( 'the_excerpt', array( $this, 'filter_excerpt' ), 10 );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'wp_ajax_freedomreader_purchase_content', array( $this, 'handle_purchase_request' ) );
		add_action( 'wp_ajax_nopriv_freedomreader_purchase_content', array( $this, 'handle_purchase_request' ) );
	}

	/**
	 * Enqueue frontend scripts and styles
	 */
	public function enqueue_scripts() {
		wp_enqueue_script(
			'freedomreader-frontend',
			FREEDOMREADER_PLUGIN_URL . 'assets/js/frontend.js',
			array( 'jquery' ),
			FREEDOMREADER_VERSION,
			true
		);

		wp_enqueue_style(
			'freedomreader-frontend',
			FREEDOMREADER_PLUGIN_URL . 'assets/css/frontend.css',
			array(),
			FREEDOMREADER_VERSION
		);

		wp_localize_script(
			'freedomreader-frontend',
			'freedomreader_ajax',
			array(
				'ajax_url'         => esc_url( admin_url( 'admin-ajax.php' ) ),
				'nonce'            => wp_create_nonce( 'freedomreader_nonce' ),
				'paypal_client_id' => get_option( 'freedomreader_paypal_client_id' ),
				'currency'         => get_option( 'freedomreader_currency', 'USD' ),
				'messages'         => array(
					'processing'     => esc_html__( 'Processing payment...', 'freedomreader' ),
					'success'        => esc_html__( 'Payment successful! Refreshing page...', 'freedomreader' ),
					'error'          => esc_html__( 'Payment failed. Please try again.', 'freedomreader' ),
					'login_required' => esc_html__( 'Please log in to purchase content.', 'freedomreader' ),
				),
			)
		);
	}

	/**
	 * Filter content based on lock settings
	 *
	 * @param string $content Post content.
	 * @return string Filtered content.
	 */
	public function filter_content( $content ) {
		global $post;

		if ( ! is_singular() || ! $post ) {
			return $content;
		}

		// Check if content is locked.
		$lock_settings = $this->db->get_content_lock( $post->ID );

		if ( ! $lock_settings ) {
			return $content;
		}

		// Check if user has access.
		$user_id = get_current_user_id();

		if ( $this->db->user_has_access( $user_id, $post->ID ) ) {
			return $content;
		}

		// Apply content lock.
		return $this->apply_content_lock( $content, $lock_settings, $post );
	}

	/**
	 * Filter excerpt for locked content
	 *
	 * @param string $excerpt Post excerpt.
	 * @return string Filtered excerpt.
	 */
	public function filter_excerpt( $excerpt ) {
		global $post;

		if ( ! $post ) {
			return $excerpt;
		}

		$lock_settings = $this->db->get_content_lock( $post->ID );

		if ( ! $lock_settings ) {
			return $excerpt;
		}

		$user_id = get_current_user_id();

		if ( $this->db->user_has_access( $user_id, $post->ID ) ) {
			return $excerpt;
		}

		// Return teaser for locked content.
		$teaser_message = get_option( 'freedomreader_teaser_message', esc_html__( 'This content is locked. Purchase to unlock.', 'freedomreader' ) );
		return $teaser_message;
	}

	/**
	 * Apply content lock based on settings
	 *
	 * @param string $content       Post content.
	 * @param object $lock_settings Lock settings.
	 * @param object $post          Post object.
	 * @return string Locked content.
	 */
	private function apply_content_lock( $content, $lock_settings, $post ) {
		$locked_content = '';

		// Apply lock based on type.
		switch ( $lock_settings->lock_type ) {
			case 'partial':
				$locked_content = $this->create_partial_lock( $content, $lock_settings );
				break;
			case 'subscription_only':
				$locked_content = $this->create_subscription_lock( $content, $lock_settings );
				break;
			case 'full':
			default:
				$locked_content = $this->create_full_lock( $content, $lock_settings );
				break;
		}

		// Apply filters for developers.
		$locked_content = apply_filters( 'freedomreader_before_content_locked', $locked_content, $post, $lock_settings );

		return $locked_content;
	}

	/**
	 * Create full content lock
	 *
	 * @param string $content       Post content.
	 * @param object $lock_settings Lock settings.
	 * @return string Full lock content.
	 */
	private function create_full_lock( $content, $lock_settings ) {
		// Use content parameter to avoid unused variable warning.
		unset( $content );

		$teaser_message = get_option( 'freedomreader_teaser_message', esc_html__( 'This content is locked. Purchase to unlock.', 'freedomreader' ) );

		$unlock_button = $this->generate_unlock_button( $lock_settings );

		return sprintf(
			'<div class="fr-locked-content">
                <div class="fr-lock-message">%s</div>
                %s
            </div>',
			$teaser_message,
			$unlock_button
		);
	}

	/**
	 * Create partial content lock
	 *
	 * @param string $content       Post content.
	 * @param object $lock_settings Lock settings.
	 * @return string Partial lock content.
	 */
	private function create_partial_lock( $content, $lock_settings ) {
		$teaser_length = $lock_settings->teaser_length ? $lock_settings->teaser_length : 150;
		$teaser        = wp_trim_words( $content, $teaser_length, '...' );

		$unlock_button = $this->generate_unlock_button( $lock_settings );

		return sprintf(
			'<div class="fr-partial-locked-content">
                <div class="fr-teaser-content">%s</div>
                <div class="fr-locked-section">
                    <div class="fr-lock-message">%s</div>
                    %s
                </div>
            </div>',
			$teaser,
			esc_html__( 'Continue reading with premium access...', 'freedomreader' ),
			$unlock_button
		);
	}

	/**
	 * Create subscription-only lock
	 *
	 * @param string $content       Post content.
	 * @param object $lock_settings Lock settings.
	 * @return string Subscription lock content.
	 */
	private function create_subscription_lock( $content, $lock_settings ) {
		// Use parameters to avoid unused variable warnings.
		unset( $content, $lock_settings );

		$subscription_plans = get_option( 'freedomreader_subscription_plans', array() );

		$plans_html = '';
		foreach ( $subscription_plans as $plan_id => $plan ) {
			$plans_html .= sprintf(
				'<div class="fr-subscription-plan" data-plan-id="%s">
                    <h4>%s</h4>
                    <div class="fr-plan-price">$%s/%s</div>
                    <button class="fr-subscribe-btn" data-plan-id="%s">%s</button>
                </div>',
				esc_attr( $plan_id ),
				esc_html( $plan['name'] ),
				esc_html( $plan['price'] ),
				esc_html( $plan['interval'] ),
				esc_attr( $plan_id ),
				esc_html__( 'Subscribe', 'freedomreader' )
			);
		}

		return sprintf(
			'<div class="fr-subscription-locked-content">
                <div class="fr-lock-message">%s</div>
                <div class="fr-subscription-plans">%s</div>
            </div>',
			esc_html__( 'This content requires a subscription to access.', 'freedomreader' ),
			$plans_html
		);
	}

	/**
	 * Generate unlock button
	 *
	 * @param object $lock_settings Lock settings.
	 * @return string Unlock button HTML.
	 */
	private function generate_unlock_button( $lock_settings ) {
		global $post;

		$button_text = get_option( 'freedomreader_unlock_button_text', esc_html__( 'Unlock Content', 'freedomreader' ) );
		$price       = $lock_settings->price;
		$currency    = get_option( 'freedomreader_currency', 'USD' );

		if ( $price ) {
			/* translators: %1$s: price amount, %2$s: currency */
			$button_text = sprintf( esc_html__( 'Unlock for %2$s%1$s', 'freedomreader' ), $price, '$' === $currency ? '$' : $currency . ' ' );
		}

		$subscription_plans = get_option( 'freedomreader_subscription_plans', array() );
		$plans_html         = '';

		foreach ( $subscription_plans as $plan_id => $plan ) {
			$plans_html .= sprintf(
				'<button class="fr-subscribe-btn fr-btn-secondary" data-plan-id="%s">
                    %s - $%s/%s
                </button>',
				esc_attr( $plan_id ),
				esc_html( $plan['name'] ),
				esc_html( $plan['price'] ),
				esc_html( $plan['interval'] )
			);
		}

		return sprintf(
			'<div class="fr-unlock-options">
                %s
                <div class="fr-purchase-options">
                    <button class="fr-purchase-btn fr-btn-primary" data-post-id="%d" data-price="%s">
                        %s
                    </button>
                    %s
                </div>
                <div class="fr-payment-processing" style="display: none;">
                    <div class="fr-spinner"></div>
                    <span>%s</span>
                </div>
            </div>',
			$price ? '' : '<p>' . esc_html__( 'Choose your access option:', 'freedomreader' ) . '</p>',
			$post->ID,
			$price ? $price : '0',
			$button_text,
			$plans_html,
			esc_html__( 'Processing payment...', 'freedomreader' )
		);
	}

	/**
	 * Handle purchase request via AJAX
	 */
	public function handle_purchase_request() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed', 'freedomreader' ) );
		}

		// Check if user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'Please log in to purchase content.', 'freedomreader' ),
				)
			);
		}

		$post_id         = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		$purchase_type   = isset( $_POST['purchase_type'] ) ? sanitize_text_field( wp_unslash( $_POST['purchase_type'] ) ) : '';
		$paypal_order_id = isset( $_POST['paypal_order_id'] ) ? sanitize_text_field( wp_unslash( $_POST['paypal_order_id'] ) ) : '';

		// Process with PayPal.
		$paypal = FREEDO_PayPal::get_instance();
		$result = $paypal->capture_payment( $paypal_order_id );

		if ( $result['success'] ) {
			// Record purchase in database.
			$purchase_data = array(
				'user_id'               => get_current_user_id(),
				'post_id'               => $post_id,
				'purchase_type'         => $purchase_type,
				'amount'                => $result['amount'],
				'paypal_transaction_id' => $result['transaction_id'],
				'status'                => 'completed',
			);

			$this->db->record_purchase( $purchase_data );

			// Log the transaction.
			$this->db->log_payment(
				array(
					'user_id'               => get_current_user_id(),
					'transaction_type'      => 'purchase',
					'paypal_transaction_id' => $result['transaction_id'],
					'amount'                => $result['amount'],
					'status'                => 'completed',
					'response_data'         => $result,
				)
			);

			// Fire action hook for developers.
			do_action( 'freedomreader_after_payment_success', $purchase_data, $result );

			wp_send_json_success(
				array(
					'message'  => esc_html__( 'Payment successful! You now have access to this content.', 'freedomreader' ),
					'redirect' => get_permalink( $post_id ),
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message' => $result['message'],
				)
			);
		}
	}

	/**
	 * Check if content should be locked
	 *
	 * @param int $post_id Post ID.
	 * @return bool Whether content is locked.
	 */
	public function is_content_locked( $post_id ) {
		$lock_settings = $this->db->get_content_lock( $post_id );
		return ! empty( $lock_settings );
	}

	/**
	 * Get lock settings for a post
	 *
	 * @param int $post_id Post ID.
	 * @return object|null Lock settings or null.
	 */
	public function get_lock_settings( $post_id ) {
		return $this->db->get_content_lock( $post_id );
	}
}
