<?php
/**
 * Content Enhancement Features
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Content enhancements functionality for FreedomReader
 *
 * @package FreedomReader
 */
class FREEDO_Content_Enhancements {

	/**
	 * Singleton instance
	 *
	 * @var FREEDO_Content_Enhancements|null
	 */
	private static $instance = null;

	/**
	 * Database instance
	 *
	 * @var FREEDO_Database
	 */
	private $db;

	/**
	 * Get singleton instance
	 *
	 * @return FREEDO_Content_Enhancements
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->db = FREEDO_Database::get_instance();
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {

		add_filter( 'the_content', array( $this, 'enhance_content' ), 20 );
		add_action( 'wp_head', array( $this, 'add_reading_progress_styles' ) );
		add_action( 'wp_footer', array( $this, 'add_reading_progress_script' ) );

		add_action( 'wp_ajax_freedomreader_request_email_unlock', array( $this, 'request_email_unlock' ) );
		add_action( 'wp_ajax_nopriv_freedomreader_request_email_unlock', array( $this, 'request_email_unlock' ) );
		add_action( 'wp_ajax_freedomreader_ask_author', array( $this, 'ask_author' ) );
		add_action( 'wp_ajax_nopriv_freedomreader_ask_author', array( $this, 'ask_author' ) );
		add_action( 'wp_ajax_freedomreader_track_reading', array( $this, 'track_reading' ) );
		add_action( 'wp_ajax_nopriv_freedomreader_track_reading', array( $this, 'track_reading' ) );

		add_action( 'add_meta_boxes', array( $this, 'add_content_meta_boxes' ) );
		add_action( 'save_post', array( $this, 'save_content_meta' ) );

		add_action( 'init', array( $this, 'handle_email_unlock' ) );
	}

	/**
	 * Enhance content with reading features
	 *
	 * @param string $content The content to enhance.
	 * @return string Enhanced content.
	 * @package FreedomReader
	 */
	public function enhance_content( $content ) {
		if ( ! is_single() || ! in_the_loop() || ! is_main_query() ) {
			return $content;
		}

		global $post;
		$enhancements = '';

		// Add reading time.
		if ( get_option( 'freedomreader_show_reading_time', true ) ) {
			$reading_time  = $this->get_reading_time( $post->ID );
			$enhancements .= '<div class="fr-reading-time">' .
				/* translators: %d: reading time in minutes */
				sprintf( esc_html__( '%d min read', 'freedomreader' ), $reading_time ) .
				'</div>';
		}

		// Add progress bar.
		if ( get_option( 'freedomreader_show_progress_bar', true ) ) {
			$enhancements .= '<div id="fr-reading-progress"></div>';
		}

		$content = $this->process_inline_notes( $content );

		if ( get_option( 'freedomreader_auto_internal_links', false ) ) {
			$content = $this->add_auto_internal_links( $content );
		}

		// Add ask author button (only for users with access to premium content).
		if ( get_option( 'freedomreader_show_ask_author', true ) ) {
			$enhancements .= $this->get_ask_author_button( $post );
		}

		return $enhancements . $content;
	}

	/**
	 * Get reading time for a post
	 *
	 * @param int $post_id The post ID.
	 * @return string Reading time HTML.
	 */
	private function get_reading_time( $post_id ) {
		// Check if manual reading time is set.
		$manual_time  = get_post_meta( $post_id, '_freedomreader_reading_time_manual', true );
		$reading_mode = get_option( 'freedomreader_reading_time_mode', 'auto' );
		$post_mode    = get_post_meta( $post_id, '_freedomreader_reading_time_mode', true );

		// Post-specific mode overrides global setting.
		if ( $post_mode ) {
			$reading_mode = $post_mode;
		}

		if ( 'manual' === $reading_mode && $manual_time ) {
			return intval( $manual_time );
		}

		if ( 'hybrid' === $reading_mode && $manual_time ) {
			return intval( $manual_time );
		}

		// Calculate automatically.
		$cached_time = get_post_meta( $post_id, '_freedomreader_reading_time_auto', true );
		if ( $cached_time ) {
			return intval( $cached_time );
		}

		$post          = get_post( $post_id );
		$word_count    = str_word_count( wp_strip_all_tags( $post->post_content ) );
		$reading_speed = get_option( 'freedomreader_reading_speed', 200 ); // words per minute.
		$reading_time  = max( 1, ceil( $word_count / $reading_speed ) );

		// Cache the calculated time.
		update_post_meta( $post_id, '_freedomreader_reading_time_auto', $reading_time );

		return $reading_time;
	}

	/**
	 * Process inline notes in content
	 *
	 * @param string $content Post content.
	 * @return string Processed content.
	 */
	private function process_inline_notes( $content ) {
		// Look for [note]content[/note] shortcodes.
		$pattern      = '/\[note\](.*?)\[\/note\]/s';
		$note_counter = 0;

		return preg_replace_callback(
			$pattern,
			function ( $matches ) use ( &$note_counter ) {
				$note_counter++;
				$note_id = 'fr-note-' . $note_counter;

				return sprintf(
					'<span class="fr-inline-note" data-note-id="%s">
                    <sup class="fr-note-trigger">%d</sup>
                    <span class="fr-note-content" id="%s">%s</span>
                </span>',
					$note_id,
					$note_counter,
					$note_id,
					$matches[1]
				);
			},
			$content
		);
	}

	/**
	 * Add automatic internal links
	 *
	 * @param string $content Post content.
	 * @return string Content with auto links.
	 */
	private function add_auto_internal_links( $content ) {
		$keywords = get_option( 'freedomreader_auto_link_keywords', array() );

		if ( empty( $keywords ) ) {
			return $content;
		}

		foreach ( $keywords as $keyword => $post_id ) {
			$post_url = get_permalink( $post_id );
			if ( $post_url ) {
				$pattern     = '/\b' . preg_quote( $keyword, '/' ) . '\b/i';
				$replacement = '<a href="' . esc_url( $post_url ) . '" class="fr-auto-link">' . $keyword . '</a>';
				$content     = preg_replace( $pattern, $replacement, $content, 1 ); // Only replace first occurrence.
			}
		}

		return $content;
	}

	/**
	 * Get ask author button
	 *
	 * @param WP_Post $post Post object.
	 * @return string Ask author button HTML.
	 */
	private function get_ask_author_button( $post ) {
		// Check if content is locked.
		$lock_settings = $this->db->get_content_lock( $post->ID );

		// If content is not locked, show the button to everyone.
		if ( ! $lock_settings ) {
			return sprintf(
				'<div class="fr-ask-author-section">
	                <button class="fr-ask-author-btn" data-post-id="%d" data-author-id="%d">
	                    %s
	                </button>
	            </div>',
				$post->ID,
				$post->post_author,
				esc_html__( 'Ask the Author', 'freedomreader' )
			);
		}

		// If content is locked, only show button to users with access.
		$user_id = get_current_user_id();
		if ( ! $this->db->user_has_access( $user_id, $post->ID ) ) {
			return ''; // Don't show button to users without access.
		}

		return sprintf(
			'<div class="fr-ask-author-section">
                <button class="fr-ask-author-btn" data-post-id="%d" data-author-id="%d">
                    %s
                </button>
            </div>',
			$post->ID,
			$post->post_author,
			esc_html__( 'Ask the Author', 'freedomreader' )
		);
	}

	/**
	 * Add reading progress styles
	 *
	 * @package FreedomReader
	 */
	public function add_reading_progress_styles() {
		if ( ! is_single() || ! get_option( 'freedomreader_show_progress_bar', true ) ) {
			return;
		}

		$css = '
		#fr-reading-progress {
			position: fixed;
			top: 0;
			left: 0;
			width: 0%;
			height: 3px;
			background: linear-gradient(90deg, #007cba, #00a0d2);
			z-index: 9999;
			transition: width 0.1s ease;
		}

		.fr-reading-time {
			font-size: 14px;
			color: #666;
			margin-bottom: 15px;
			padding: 8px 12px;
			background: #f8f9fa;
			border-radius: 4px;
			display: inline-block;
		}

		.fr-inline-note {
			position: relative;
		}

		.fr-note-trigger {
			color: #007cba;
			cursor: pointer;
			font-weight: bold;
			margin-left: 2px;
		}

		.fr-note-content {
			display: none;
			position: absolute;
			bottom: 100%;
			left: 50%;
			transform: translateX(-50%);
			background: #333;
			color: #fff;
			padding: 8px 12px;
			border-radius: 4px;
			font-size: 12px;
			white-space: nowrap;
			z-index: 1000;
			max-width: 300px;
			white-space: normal;
		}

		.fr-note-content:before {
			content: "";
			position: absolute;
			top: 100%;
			left: 50%;
			transform: translateX(-50%);
			border: 5px solid transparent;
			border-top-color: #333;
		}

		.fr-ask-author-section {
			margin: 30px 0;
			text-align: center;
			padding: 20px;
			background: #f8f9fa;
			border-radius: 6px;
		}

		.fr-ask-author-btn {
			background: #007cba;
			color: #fff;
			border: none;
			padding: 12px 24px;
			border-radius: 4px;
			cursor: pointer;
			font-size: 14px;
			font-weight: 500;
		}

		.fr-ask-author-btn:hover {
			background: #005a87;
		}

		.fr-auto-link {
			color: #007cba;
			text-decoration: underline;
		}

		.fr-email-unlock-form {
			background: #fff3cd;
			border: 1px solid #ffeaa7;
			border-radius: 6px;
			padding: 20px;
			margin: 20px 0;
			text-align: center;
		}

		.fr-email-unlock-form h3 {
			margin: 0 0 15px 0;
			color: #856404;
		}

		.fr-email-unlock-form input[type="email"] {
			padding: 10px;
			border: 1px solid #ddd;
			border-radius: 4px;
			width: 250px;
			margin-right: 10px;
		}

		.fr-email-unlock-form button {
			background: #28a745;
			color: #fff;
			border: none;
			padding: 10px 20px;
			border-radius: 4px;
			cursor: pointer;
		}';

		wp_add_inline_style( 'freedomreader-frontend', $css );
	}

	/**
	 * Add reading progress script
	 *
	 * @package FreedomReader
	 */
	public function add_reading_progress_script() {
		if ( ! is_single() ) {
			return;
		}

		$js = '
		jQuery(document).ready(function($) {
			// Reading progress bar.
			if ($("#fr-reading-progress").length) {
				$(window).scroll(function() {
					var scrollTop = $(window).scrollTop();
					var docHeight = $(document).height();
					var winHeight = $(window).height();
					var scrollPercent = (scrollTop) / (docHeight - winHeight);
					var scrollPercentRounded = Math.round(scrollPercent * 100);
					$("#fr-reading-progress").css("width", scrollPercentRounded + "%");
				});
			}

			// Inline notes.
			$(".fr-note-trigger").hover(
				function() {
					var noteId = $(this).parent().data("note-id");
					$("#" + noteId).show();
				},
				function() {
					var noteId = $(this).parent().data("note-id");
					$("#" + noteId).hide();
				}
			);

			// Ask author button.
			$(".fr-ask-author-btn").click(function() {
				var postId = $(this).data("post-id");
				var authorId = $(this).data("author-id");

				var modal = $("<div class=\"fr-modal-overlay\">" +
					"<div class=\"fr-modal\">" +
						"<div class=\"fr-modal-header\">" +
							"<h3>Ask the Author</h3>" +
							"<button class=\"fr-modal-close\">&times;</button>" +
						"</div>" +
						"<form class=\"fr-ask-author-form\">" +
							"<input type=\"hidden\" name=\"post_id\" value=\"" + postId + "\">" +
							"<input type=\"hidden\" name=\"author_id\" value=\"" + authorId + "\">" +
							"<p><label>Your Name:</label><input type=\"text\" name=\"questioner_name\" required></p>" +
							"<p><label>Your Email:</label><input type=\"email\" name=\"questioner_email\" required></p>" +
							"<p><label>Your Question:</label><textarea name=\"question\" rows=\"4\" required></textarea></p>" +
							"<p><button type=\"submit\">Send Question</button></p>" +
						"</form>" +
					"</div>" +
				"</div>");

				$("body").append(modal);
				modal.show();
			});

			// Modal close.
			$(document).on("click", ".fr-modal-close, .fr-modal-overlay", function(e) {
				if (e.target === this) {
					$(".fr-modal-overlay").remove();
				}
			});

			// Ask author form submission.
			$(document).on("submit", ".fr-ask-author-form", function(e) {
				e.preventDefault();

				var formData = $(this).serialize();
				formData += "&action=freedomreader_ask_author&nonce=" + freedomreader.nonce;

				$.ajax({
					url: freedomreader.ajax_url,
					type: "POST",
					data: formData,
					success: function(response) {
						if (response.success) {
							$(".fr-modal-overlay").html("<div class=\"fr-modal\"><div class=\"fr-modal-header\"><h3>Success</h3><button class=\"fr-modal-close\">&times;</button></div><div style=\"padding:20px;\">Your question has been sent to the author!</div></div>");
						} else {
							$(".fr-modal-overlay").html("<div class=\"fr-modal\"><div class=\"fr-modal-header\"><h3>Error</h3><button class=\"fr-modal-close\">&times;</button></div><div style=\"padding:20px;\">Error: " + response.data.message + "</div></div>");
						}
					}
				});
			});

			// Email unlock form.
			$(document).on("submit", ".fr-email-unlock-form", function(e) {
				e.preventDefault();

				var email = $(this).find("input[type=\"email\"]").val();
				var postId = $(this).data("post-id");

				$.ajax({
					url: freedomreader.ajax_url,
					type: "POST",
					data: {
						action: "freedomreader_request_email_unlock",
						nonce: freedomreader.nonce,
						email: email,
						post_id: postId
					},
					success: function(response) {
						if (response.success) {
							$(".fr-email-unlock-form").html("<p style=\"color:green;\">Check your email for the unlock link!</p>");
						} else {
							$(".fr-email-unlock-form").append("<p style=\"color:red;\">Error: " + response.data.message + "</p>");
						}
					}
				});
			});

			// Track reading analytics.
			var startTime = Date.now();
			var maxScroll = 0;

			$(window).scroll(function() {
				var scrollPercent = Math.round(($(window).scrollTop() / ($(document).height() - $(window).height())) * 100);
				maxScroll = Math.max(maxScroll, scrollPercent);
			});

			$(window).on("beforeunload", function() {
				var readingTime = Math.round((Date.now() - startTime) / 1000);

				$.ajax({
					url: freedomreader.ajax_url,
					type: "POST",
					data: {
						action: "freedomreader_track_reading",
						nonce: freedomreader.nonce,
						post_id: freedomreader.post_id,
						reading_time: readingTime,
						scroll_depth: maxScroll
					},
					async: false
				});
			});
		});';

		wp_add_inline_script( 'freedomreader-frontend', $js );

		$modal_css = '
		.fr-modal-overlay {
			position: fixed;
			top: 0;
			left: 0;
			width: 100%;
			height: 100%;
			background: rgba(0,0,0,0.5);
			z-index: 999999;
			display: flex;
			align-items: center;
			justify-content: center;
		}

		.fr-modal {
			background: #fff;
			border-radius: 8px;
			padding: 0;
			max-width: 500px;
			width: 90%;
			max-height: 90vh;
			overflow-y: auto;
		}

		.fr-modal-header {
			padding: 20px;
			border-bottom: 1px solid #ddd;
			display: flex;
			justify-content: space-between;
			align-items: center;
		}

		.fr-modal-close {
			background: none;
			border: none;
			font-size: 24px;
			cursor: pointer;
		}

		.fr-ask-author-form {
			padding: 20px;
		}

		.fr-ask-author-form label {
			display: block;
			margin-bottom: 5px;
			font-weight: 500;
		}

		.fr-ask-author-form input,
		.fr-ask-author-form textarea {
			width: 100%;
			padding: 8px;
			border: 1px solid #ddd;
			border-radius: 4px;
			margin-bottom: 15px;
		}

		.fr-ask-author-form button {
			background: #007cba;
			color: #fff;
			border: none;
			padding: 10px 20px;
			border-radius: 4px;
			cursor: pointer;
		}';

		wp_add_inline_style( 'freedomreader-frontend', $modal_css );
	}

	/**
	 * Request email unlock
	 *
	 * @package FreedomReader
	 */
	public function request_email_unlock() {
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		$email   = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;

		if ( ! $email || ! $post_id ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid email or post ID', 'freedomreader' ) ) );
		}

		// Generate unlock token.
		$token      = wp_generate_password( 32, false );
		$ip_address = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
		$user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';

		// Save to database.
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Email unlock token storage
		$result = $wpdb->insert(
			$wpdb->prefix . 'freedo_email_unlocks',
			array(
				'email'        => $email,
				'post_id'      => $post_id,
				'unlock_token' => $token,
				'ip_address'   => $ip_address,
				'user_agent'   => $user_agent,
				'expires_at'   => gmdate( 'Y-m-d H:i:s', time() + 86400 ), // 24 hours.
			)
		);

		if ( $result ) {
			// Send email.
			$post_title = get_the_title( $post_id );
			$unlock_url = add_query_arg( 'unlock_token', $token, get_permalink( $post_id ) );

			/* translators: %s: post title */
			$subject = sprintf( esc_html__( 'Unlock: %s', 'freedomreader' ), $post_title );
			/* translators: 1: post title, 2: unlock URL, 3: site name */
			$message = sprintf(
				esc_html__( "Hi there!\n\nYou requested access to: %1\$s\n\nClick here to unlock: %2\$s\n\nThis link expires in 24 hours.\n\nBest regards,\n%3\$s", 'freedomreader' ),
				$post_title,
				$unlock_url,
				esc_html( get_bloginfo( 'name' ) )
			);

			wp_mail( $email, $subject, $message );

			wp_send_json_success( array( 'message' => esc_html__( 'Check your email for the unlock link!', 'freedomreader' ) ) );
		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to process request', 'freedomreader' ) ) );
		}
	}

	/**
	 * Ask author question
	 *
	 * @package FreedomReader
	 */
	public function ask_author() {
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		$post_id          = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		$author_id        = isset( $_POST['author_id'] ) ? intval( $_POST['author_id'] ) : 0;
		$questioner_name  = isset( $_POST['questioner_name'] ) ? sanitize_text_field( wp_unslash( $_POST['questioner_name'] ) ) : '';
		$questioner_email = isset( $_POST['questioner_email'] ) ? sanitize_email( wp_unslash( $_POST['questioner_email'] ) ) : '';
		$question         = isset( $_POST['question'] ) ? sanitize_textarea_field( wp_unslash( $_POST['question'] ) ) : '';

		if ( ! $post_id || ! $author_id || ! $questioner_name || ! $questioner_email || ! $question ) {
			wp_send_json_error( array( 'message' => esc_html__( 'All fields are required', 'freedomreader' ) ) );
		}

		// Check if content is locked and user has access.
		$lock_settings = $this->db->get_content_lock( $post_id );
		if ( $lock_settings ) {
			$user_id = get_current_user_id();
			if ( ! $this->db->user_has_access( $user_id, $post_id ) ) {
				wp_send_json_error( array( 'message' => esc_html__( 'You need to purchase access to this content before asking questions.', 'freedomreader' ) ) );
			}
		}

		// Save to database.
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Author question storage
		$result = $wpdb->insert(
			$wpdb->prefix . 'freedo_author_questions',
			array(
				'post_id'          => $post_id,
				'author_id'        => $author_id,
				'questioner_name'  => $questioner_name,
				'questioner_email' => $questioner_email,
				'question'         => $question,
				'status'           => 'pending',
			)
		);

		if ( $result ) {
			// Notify author.
			$author     = get_userdata( $author_id );
			$post_title = get_the_title( $post_id );

			/* translators: %s: post title */
			$subject = sprintf( esc_html__( 'New question about: %s', 'freedomreader' ), $post_title );
			/* translators: 1: author name, 2: post title, 3: questioner name, 4: questioner email, 5: question text, 6: site name */
			$message = sprintf(
				esc_html__( "Hi %1\$s,\n\nYou have a new question about your post: %2\$s\n\nFrom: %3\$s (%4\$s)\nQuestion: %5\$s\n\nYou can respond in your WordPress admin area.\n\nBest regards,\n%6\$s", 'freedomreader' ),
				$author->display_name,
				$post_title,
				$questioner_name,
				$questioner_email,
				$question,
				esc_html( get_bloginfo( 'name' ) )
			);

			wp_mail( $author->user_email, $subject, $message );

			wp_send_json_success( array( 'message' => esc_html__( 'Your question has been sent!', 'freedomreader' ) ) );
		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to send question', 'freedomreader' ) ) );
		}
	}

	/**
	 * Track reading analytics
	 *
	 * @package FreedomReader
	 */
	public function track_reading() {
		// Verify nonce for security.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		$post_id      = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		$reading_time = isset( $_POST['reading_time'] ) ? intval( $_POST['reading_time'] ) : 0;
		$scroll_depth = isset( $_POST['scroll_depth'] ) ? intval( $_POST['scroll_depth'] ) : 0;

		if ( ! $post_id ) {
			return;
		}

		global $wpdb;

		$user_id    = is_user_logged_in() ? get_current_user_id() : null;
		$ip_address = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
		$user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
		$referrer   = isset( $_SERVER['HTTP_REFERER'] ) ? esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) ) : '';

		// Detect device type.
		$device_type = 'desktop';
		if ( wp_is_mobile() ) {
			$device_type = 'mobile';
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data storage
		$wpdb->insert(
			$wpdb->prefix . 'freedo_analytics',
			array(
				'post_id'      => $post_id,
				'user_id'      => $user_id,
				'ip_address'   => $ip_address,
				'user_agent'   => $user_agent,
				'referrer'     => $referrer,
				'device_type'  => $device_type,
				'reading_time' => $reading_time,
				'scroll_depth' => $scroll_depth,
			)
		);
	}

	/**
	 * Handle email unlock from URL
	 *
	 * @package FreedomReader
	 */
	public function handle_email_unlock() {
		if ( isset( $_GET['unlock_token'] ) && is_single() ) {
			// Verify nonce for security.
			if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'email_unlock_' . get_the_ID() ) ) {
				return;
			}

			$token = sanitize_text_field( wp_unslash( $_GET['unlock_token'] ) );
			global $post;

			if ( $post && $this->verify_email_unlock_token( $token, $post->ID ) ) {
				// Token verified and marked as used in verify_email_unlock_token method.
				add_action(
					'wp_head',
					function () {
						$js = 'jQuery(document).ready(function($) {
							$("body").prepend("<div style=\"background:#d4edda;color:#155724;padding:15px;margin:10px 0;border:1px solid #c3e6cb;border-radius:4px;text-align:center;\">Content unlocked! You now have access to this article.</div>");
							setTimeout(function() { $("body > div:first-child").fadeOut(); }, 5000);
						});';
						wp_add_inline_script( 'freedomreader-frontend', $js );
					}
				);
			}
		}
	}

	/**
	 * Verify email unlock token (moved from advanced locks)
	 *
	 * @param string $token   Unlock token.
	 * @param int    $post_id Post ID.
	 * @return bool Whether token is valid.
	 */
	private function verify_email_unlock_token( $token, $post_id ) {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Email unlock token verification
		$unlock_request = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$wpdb->prefix}freedo_email_unlocks` WHERE unlock_token = %s AND post_id = %d AND expires_at > NOW() AND used_at IS NULL",
				$token,
				$post_id
			)
		);

		if ( $unlock_request ) {
			// Mark as used.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Email unlock token usage tracking
			$wpdb->update(
				$wpdb->prefix . 'freedo_email_unlocks',
				array( 'used_at' => current_time( 'mysql' ) ),
				array( 'id' => $unlock_request->id )
			);

			// Set session to remember unlock.
			if ( ! session_id() ) {
				session_start();
			}
			$_SESSION['freedomreader_email_unlocked'][ $post_id ] = true;

			return true;
		}

		return false;
	}

	/**
	 * Add content meta boxes
	 *
	 * @package FreedomReader
	 */
	public function add_content_meta_boxes() {
		add_meta_box(
			'fr-content-enhancements',
			esc_html__( 'Content Enhancements', 'freedomreader' ),
			array( $this, 'content_enhancements_metabox' ),
			array( 'post', 'page' ),
			'side',
			'default'
		);
	}

	/**
	 * Content enhancements metabox
	 *
	 * @param WP_Post $post Post object.
	 */
	public function content_enhancements_metabox( $post ) {
		wp_nonce_field( 'freedomreader_content_meta', 'freedomreader_content_meta_nonce' );

		$reading_time_mode   = get_post_meta( $post->ID, '_freedomreader_reading_time_mode', true );
		$manual_reading_time = get_post_meta( $post->ID, '_freedomreader_reading_time_manual', true );
		$auto_reading_time   = get_post_meta( $post->ID, '_freedomreader_reading_time_auto', true );

		?>
		<div class="fr-content-meta">
			<p>
				<label for="freedomreader_reading_time_mode"><?php esc_html_e( 'Reading Time Mode:', 'freedomreader' ); ?></label>
				<select id="freedomreader_reading_time_mode" name="freedomreader_reading_time_mode">
					<option value=""><?php esc_html_e( 'Use Global Setting', 'freedomreader' ); ?></option>
					<option value="auto" <?php selected( $reading_time_mode, 'auto' ); ?>><?php esc_html_e( 'Auto Calculate', 'freedomreader' ); ?></option>
					<option value="manual" <?php selected( $reading_time_mode, 'manual' ); ?>><?php esc_html_e( 'Manual', 'freedomreader' ); ?></option>
					<option value="hybrid" <?php selected( $reading_time_mode, 'hybrid' ); ?>><?php esc_html_e( 'Hybrid', 'freedomreader' ); ?></option>
				</select>
			</p>

			<p class="fr-manual-time" style="<?php echo esc_attr( ( 'manual' === $reading_time_mode || 'hybrid' === $reading_time_mode ) ? '' : 'display:none;' ); ?>">
				<label for="freedomreader_manual_reading_time"><?php esc_html_e( 'Manual Reading Time (minutes):', 'freedomreader' ); ?></label>
				<input type="number" id="freedomreader_manual_reading_time" name="freedomreader_manual_reading_time" value="<?php echo esc_attr( $manual_reading_time ); ?>" min="1">
			</p>

			<?php if ( $auto_reading_time ) : ?>
			<p class="description">
				<?php
				/* translators: %d: auto-calculated reading time in minutes */
				printf( esc_html__( 'Auto-calculated: %d minutes', 'freedomreader' ), esc_html( $auto_reading_time ) );
				?>
			</p>
			<?php endif; ?>

			<p class="description">
				<?php esc_html_e( 'Use [note]Your note here[/note] to add inline expandable notes.', 'freedomreader' ); ?>
			</p>
		</div>

		<?php
		$js = '
		jQuery(document).ready(function($) {
			$("#freedomreader_reading_time_mode").change(function() {
				var mode = $(this).val();
				if (mode === "manual" || mode === "hybrid") {
					$(".fr-manual-time").show();
				} else {
					$(".fr-manual-time").hide();
				}
			});
		});';
		wp_add_inline_script( 'freedomreader-admin', $js );
	}

	/**
	 * Save content meta
	 *
	 * @param int $post_id Post ID.
	 */
	public function save_content_meta( $post_id ) {
		if ( ! isset( $_POST['freedomreader_content_meta_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['freedomreader_content_meta_nonce'] ) ), 'freedomreader_content_meta' ) ) {
			return;
		}

		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Save reading time settings.
		if ( isset( $_POST['freedomreader_reading_time_mode'] ) ) {
			update_post_meta( $post_id, '_freedomreader_reading_time_mode', sanitize_text_field( wp_unslash( $_POST['freedomreader_reading_time_mode'] ) ) );
		}

		if ( isset( $_POST['freedomreader_manual_reading_time'] ) ) {
			update_post_meta( $post_id, '_freedomreader_reading_time_manual', intval( wp_unslash( $_POST['freedomreader_manual_reading_time'] ) ) );
		}

		// Clear auto-calculated cache when content changes.
		delete_post_meta( $post_id, '_freedomreader_reading_time_auto' );
	}
}
