<?php
/**
 * FreedomReader Backup System
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Backup System Class
 */
class FREEDO_Backup_System {

	/**
	 * Instance of this class
	 *
	 * @var FREEDO_Backup_System
	 */
	private static $instance = null;

	/**
	 * Backup directory path
	 *
	 * @var string
	 */
	private $backup_dir;

	/**
	 * Get instance
	 *
	 * @return FREEDO_Backup_System
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->backup_dir = wp_upload_dir()['basedir'] . '/freedomreader-backups';
		$this->init();
	}

	/**
	 * Initialize backup system
	 */
	private function init() {
		// Create backup directory if it doesn't exist.
		if ( ! file_exists( $this->backup_dir ) ) {
			wp_mkdir_p( $this->backup_dir );

			// Add .htaccess for security.
			$htaccess_content = "Order deny,allow\nDeny from all\n";
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Required for backup security
			file_put_contents( $this->backup_dir . '/.htaccess', $htaccess_content );
		}

		// Schedule automatic backups.
		if ( get_option( 'freedomreader_auto_backup', 0 ) ) {
			$this->schedule_automatic_backups();
		}

		// Hook for cleanup old backups.
		add_action( 'freedomreader_cleanup_old_backups', array( $this, 'cleanup_old_backups' ) );
	}

	/**
	 * Schedule automatic backups
	 */
	public function schedule_automatic_backups() {
		$frequency = get_option( 'freedomreader_backup_frequency', 'weekly' );

		// Clear existing schedule.
		wp_clear_scheduled_hook( 'freedomreader_auto_backup' );

		// Schedule new backup.
		if ( ! wp_next_scheduled( 'freedomreader_auto_backup' ) ) {
			wp_schedule_event( time(), $frequency, 'freedomreader_auto_backup' );
		}

		// Hook the backup function.
		add_action( 'freedomreader_auto_backup', array( $this, 'create_automatic_backup' ) );
	}

	/**
	 * Create automatic backup
	 */
	public function create_automatic_backup() {
		$backup_file = $this->create_backup( 'automatic' );

		if ( $backup_file ) {
			// Send notification if enabled.
			if ( get_option( 'freedomreader_email_performance_alerts', 0 ) ) {
				$this->send_backup_notification( $backup_file );
			}

			// Log the backup.
			$this->log_backup_activity( 'Automatic backup created: ' . basename( $backup_file ) );
		}
	}

	/**
	 * Create backup
	 *
	 * @param string $type Backup type (manual, automatic, migration).
	 * @return string|false Backup file path or false on failure.
	 */
	public function create_backup( $type = 'manual' ) {
		global $wpdb;

		try {
			// Prepare backup data.
			$backup_data = array(
				'metadata' => array(
					'version'    => FREEDOMREADER_VERSION,
					'created_at' => current_time( 'mysql' ),
					'type'       => $type,
					'site_url'   => get_site_url(),
					'wp_version' => get_bloginfo( 'version' ),
				),
				'settings' => $this->get_plugin_settings(),
				'data'     => $this->get_plugin_data(),
			);

			// Create backup filename.
			$timestamp = gmdate( 'Y-m-d_H-i-s' );
			$filename  = "freedomreader-backup-{$type}-{$timestamp}.json";
			$filepath  = $this->backup_dir . '/' . $filename;

			// Write backup file.
			$json_data = wp_json_encode( $backup_data, JSON_PRETTY_PRINT );
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Required for backup functionality
			$result = file_put_contents( $filepath, $json_data );

			if ( false === $result ) {
				return false;
			}

			// Log backup creation.
			$this->log_backup_activity( "Backup created: {$filename}" );

			return $filepath;

		} catch ( Exception $e ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Required for backup error logging
			error_log( 'FreedomReader Backup Error: ' . $e->getMessage() );
			return false;
		}
	}

	/**
	 * Get plugin settings for backup
	 *
	 * @return array
	 */
	private function get_plugin_settings() {
		$settings     = array();
		$option_names = array(
			'freedomreader_paypal_mode',
			'freedomreader_currency',
			'freedomreader_trial_days',
			'freedomreader_unlock_button_text',
			'freedomreader_teaser_message',
			'freedomreader_keep_data_on_uninstall',
			'freedomreader_auto_cleanup',
			'freedomreader_export_format',
			'freedomreader_backup_frequency',
			'freedomreader_auto_backup',
			'freedomreader_backup_retention',
			'freedomreader_performance_monitoring',
			'freedomreader_user_activity_logs',
			'freedomreader_email_new_purchases',
			'freedomreader_email_new_subscriptions',
			'freedomreader_email_failed_payments',
			'freedomreader_email_performance_alerts',
		);

		foreach ( $option_names as $option ) {
			$settings[ $option ] = get_option( $option );
		}

		return $settings;
	}

	/**
	 * Get plugin data for backup
	 *
	 * @return array
	 */
	private function get_plugin_data() {
		global $wpdb;

		$data = array();

		// Get purchases.
		$purchases_table = $wpdb->prefix . 'freedo_purchases';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Backup functionality, table names cannot be prepared
		$data['purchases'] = $wpdb->get_results( "SELECT * FROM `{$purchases_table}` ORDER BY created_at DESC", ARRAY_A );

		// Get subscriptions.
		$subscriptions_table = $wpdb->prefix . 'freedo_subscriptions';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Backup functionality, table names cannot be prepared
		$data['subscriptions'] = $wpdb->get_results( "SELECT * FROM `{$subscriptions_table}` ORDER BY created_at DESC", ARRAY_A );

		// Get content locks.
		$locks_table = $wpdb->prefix . 'freedo_content_locks';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Backup functionality, table names cannot be prepared
		$data['content_locks'] = $wpdb->get_results( "SELECT * FROM `{$locks_table}` ORDER BY created_at DESC", ARRAY_A );

		// Get payment logs (last 1000 entries).
		$logs_table = $wpdb->prefix . 'freedo_payment_logs';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Backup functionality, table names cannot be prepared
		$data['payment_logs'] = $wpdb->get_results( "SELECT * FROM `{$logs_table}` ORDER BY created_at DESC LIMIT 1000", ARRAY_A );

		return $data;
	}

	/**
	 * Restore from backup
	 *
	 * @param string $backup_file Backup file path.
	 * @param bool   $overwrite   Whether to overwrite existing data.
	 * @return bool Success status.
	 */
	public function restore_backup( $backup_file, $overwrite = false ) {
		if ( ! file_exists( $backup_file ) ) {
			return false;
		}

		global $wpdb;

		try {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Required for backup restoration
			$backup_content = file_get_contents( $backup_file );
			$backup_data    = json_decode( $backup_content, true );

			if ( ! $backup_data || ! isset( $backup_data['data'] ) ) {
				return false;
			}

			// Start transaction.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for backup restoration transaction
			$wpdb->query( 'START TRANSACTION' );

			// Restore settings.
			if ( isset( $backup_data['settings'] ) ) {
				foreach ( $backup_data['settings'] as $option => $value ) {
					update_option( $option, $value );
				}
			}

			// Restore data.
			if ( $overwrite ) {
				// Clear existing data if overwrite is enabled.
				$this->clear_plugin_data();
			}

			$this->restore_plugin_data( $backup_data['data'] );

			// Commit transaction.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for backup restoration transaction
			$wpdb->query( 'COMMIT' );

			// Log restoration.
			$this->log_backup_activity( 'Backup restored: ' . basename( $backup_file ) );

			return true;

		} catch ( Exception $e ) {
			// Rollback on error.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for backup restoration transaction rollback
			$wpdb->query( 'ROLLBACK' );
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Required for backup error logging
			error_log( 'FreedomReader Restore Error: ' . $e->getMessage() );
			return false;
		}
	}

	/**
	 * Clear plugin data
	 */
	private function clear_plugin_data() {
		global $wpdb;

		$tables = array(
			$wpdb->prefix . 'freedo_purchases',
			$wpdb->prefix . 'freedo_subscriptions',
			$wpdb->prefix . 'freedo_content_locks',
		);

		foreach ( $tables as $table ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Backup restoration, table names cannot be prepared
			$wpdb->query( "TRUNCATE TABLE `{$table}`" );
		}
	}

	/**
	 * Restore plugin data
	 *
	 * @param array $data Plugin data to restore.
	 */
	private function restore_plugin_data( $data ) {
		global $wpdb;

		// Restore purchases.
		if ( ! empty( $data['purchases'] ) ) {
			$purchases_table = $wpdb->prefix . 'freedo_purchases';
			foreach ( $data['purchases'] as $purchase ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Backup restoration
				$wpdb->insert( $purchases_table, $purchase );
			}
		}

		// Restore subscriptions.
		if ( ! empty( $data['subscriptions'] ) ) {
			$subscriptions_table = $wpdb->prefix . 'freedo_subscriptions';
			foreach ( $data['subscriptions'] as $subscription ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Backup restoration
				$wpdb->insert( $subscriptions_table, $subscription );
			}
		}

		// Restore content locks.
		if ( ! empty( $data['content_locks'] ) ) {
			$locks_table = $wpdb->prefix . 'freedo_content_locks';
			foreach ( $data['content_locks'] as $lock ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Backup restoration
				$wpdb->insert( $locks_table, $lock );
			}
		}
	}

	/**
	 * Cleanup old backups
	 */
	public function cleanup_old_backups() {
		$retention_days = (int) get_option( 'freedomreader_backup_retention', 30 );
		$cutoff_time    = time() - ( $retention_days * DAY_IN_SECONDS );

		$files = glob( $this->backup_dir . '/freedomreader-backup-*.json' );

		foreach ( $files as $file ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_filemtime -- Required for backup file management
			if ( filemtime( $file ) < $cutoff_time ) {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Required for backup cleanup
				unlink( $file );
				$this->log_backup_activity( 'Old backup deleted: ' . basename( $file ) );
			}
		}
	}

	/**
	 * Get backup files list
	 *
	 * @return array
	 */
	public function get_backup_files() {
		$files   = glob( $this->backup_dir . '/freedomreader-backup-*.json' );
		$backups = array();

		foreach ( $files as $file ) {
			$backups[] = array(
				'filename' => basename( $file ),
				'filepath' => $file,
				'size'     => filesize( $file ),
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_filemtime -- Required for backup file info
				'created'  => filemtime( $file ),
				'type'     => $this->get_backup_type_from_filename( basename( $file ) ),
			);
		}

		// Sort by creation time (newest first).
		usort(
			$backups,
			function ( $a, $b ) {
				return $b['created'] - $a['created'];
			}
		);

		return $backups;
	}

	/**
	 * Get backup type from filename
	 *
	 * @param string $filename Backup filename.
	 * @return string
	 */
	private function get_backup_type_from_filename( $filename ) {
		if ( strpos( $filename, '-automatic-' ) !== false ) {
			return 'automatic';
		} elseif ( strpos( $filename, '-migration-' ) !== false ) {
			return 'migration';
		}
		return 'manual';
	}

	/**
	 * Send backup notification
	 *
	 * @param string $backup_file Backup file path.
	 */
	private function send_backup_notification( $backup_file ) {
		$admin_email = get_option( 'admin_email' );
		$site_name   = get_bloginfo( 'name' );
		$subject     = sprintf( '[%s] FreedomReader Backup Created', $site_name );

		$message = sprintf(
			"A new FreedomReader backup has been created:\n\n" .
			"Backup File: %s\n" .
			"Created: %s\n" .
			"Site: %s\n\n" .
			'This is an automated notification.',
			basename( $backup_file ),
			current_time( 'Y-m-d H:i:s' ),
			get_site_url()
		);

		wp_mail( $admin_email, $subject, $message );
	}

	/**
	 * Log backup activity
	 *
	 * @param string $message Log message.
	 */
	private function log_backup_activity( $message ) {
		if ( get_option( 'freedomreader_user_activity_logs', 1 ) ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Required for backup activity logging
			error_log( 'FreedomReader Backup: ' . $message );
		}
	}
}
