<?php
/**
 * Analytics System
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Analytics functionality for FreedomReader
 *
 * @package FreedomReader
 */
class FREEDO_Analytics {

	/**
	 * Singleton instance
	 *
	 * @var FREEDO_Analytics|null
	 */
	private static $instance = null;

	/**
	 * Get singleton instance
	 *
	 * @return FREEDO_Analytics
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'wp_dashboard_setup', array( $this, 'add_dashboard_widget' ) );
	}

	/**
	 * Add dashboard widget
	 *
	 * @package FreedomReader
	 */
	public function add_dashboard_widget() {
		wp_add_dashboard_widget(
			'freedo_analytics_widget',
			esc_html__( 'FreedomReader Analytics', 'freedomreader' ),
			array( $this, 'dashboard_widget_content' )
		);
	}

	/**
	 * Dashboard widget content
	 *
	 * @package FreedomReader
	 */
	public function dashboard_widget_content() {
		$stats = $this->get_dashboard_stats();
		?>
		<div class="fr-analytics-widget">
			<div class="fr-stats-grid">
				<div class="fr-stat-item">
					<div class="fr-stat-number"><?php echo number_format( $stats['total_views'] ); ?></div>
					<div class="fr-stat-label"><?php esc_html_e( 'Total Views', 'freedomreader' ); ?></div>
				</div>
				
				<div class="fr-stat-item">
					<div class="fr-stat-number"><?php echo number_format( $stats['unique_visitors'] ); ?></div>
					<div class="fr-stat-label"><?php esc_html_e( 'Unique Visitors', 'freedomreader' ); ?></div>
				</div>
				
				<div class="fr-stat-item">
					<div class="fr-stat-number"><?php echo esc_html( $stats['avg_reading_time'] ); ?>m</div>
					<div class="fr-stat-label"><?php esc_html_e( 'Avg Reading Time', 'freedomreader' ); ?></div>
				</div>
				
				<div class="fr-stat-item">
					<div class="fr-stat-number"><?php echo esc_html( $stats['avg_scroll_depth'] ); ?>%</div>
					<div class="fr-stat-label"><?php esc_html_e( 'Avg Scroll Depth', 'freedomreader' ); ?></div>
				</div>
			</div>
			
			<div class="fr-top-posts">
				<h4><?php esc_html_e( 'Top Posts (Last 7 Days)', 'freedomreader' ); ?></h4>
				<?php if ( ! empty( $stats['top_posts'] ) ) : ?>
					<ul>
						<?php foreach ( $stats['top_posts'] as $post ) : ?>
							<li>
								<a href="<?php echo esc_url( get_edit_post_link( $post->post_id ) ); ?>">
									<?php echo esc_html( get_the_title( $post->post_id ) ); ?>
								</a>
								<span class="fr-post-views"><?php echo number_format( $post->views ); ?> views</span>
							</li>
						<?php endforeach; ?>
					</ul>
				<?php else : ?>
					<p><?php esc_html_e( 'No data available yet.', 'freedomreader' ); ?></p>
				<?php endif; ?>
			</div>
			
			<div class="fr-device-breakdown">
				<h4><?php esc_html_e( 'Device Breakdown', 'freedomreader' ); ?></h4>
				<div class="fr-device-stats">
					<div class="fr-device-item">
						<span class="fr-device-label"><?php esc_html_e( 'Desktop', 'freedomreader' ); ?></span>
						<span class="fr-device-percent"><?php echo esc_html( $stats['device_breakdown']['desktop'] ); ?>%</span>
					</div>
					<div class="fr-device-item">
						<span class="fr-device-label"><?php esc_html_e( 'Mobile', 'freedomreader' ); ?></span>
						<span class="fr-device-percent"><?php echo esc_html( $stats['device_breakdown']['mobile'] ); ?>%</span>
					</div>
					<div class="fr-device-item">
						<span class="fr-device-label"><?php esc_html_e( 'Tablet', 'freedomreader' ); ?></span>
						<span class="fr-device-percent"><?php echo esc_html( $stats['device_breakdown']['tablet'] ); ?>%</span>
					</div>
				</div>
			</div>
			
			<div class="fr-top-referrers">
				<h4><?php esc_html_e( 'Top Referrers', 'freedomreader' ); ?></h4>
				<?php if ( ! empty( $stats['top_referrers'] ) ) : ?>
					<ul>
						<?php foreach ( $stats['top_referrers'] as $referrer ) : ?>
							<li>
								<span class="fr-referrer-domain"><?php echo esc_html( $referrer->domain ); ?></span>
								<span class="fr-referrer-count"><?php echo number_format( $referrer->visits ); ?></span>
							</li>
						<?php endforeach; ?>
					</ul>
				<?php else : ?>
					<p><?php esc_html_e( 'No referrer data available.', 'freedomreader' ); ?></p>
				<?php endif; ?>
			</div>
		</div>
		
		<?php
		// Enqueue analytics widget styles.
		$analytics_css = '
		.fr-analytics-widget {
			font-size: 13px;
		}

		.fr-stats-grid {
			display: grid;
			grid-template-columns: repeat(2, 1fr);
			gap: 15px;
			margin-bottom: 20px;
		}

		.fr-stat-item {
			text-align: center;
			padding: 10px;
			background: #f8f9fa;
			border-radius: 4px;
		}

		.fr-stat-number {
			font-size: 24px;
			font-weight: bold;
			color: #007cba;
			line-height: 1;
		}

		.fr-stat-label {
			font-size: 11px;
			color: #666;
			margin-top: 5px;
		}

		.fr-top-posts h4,
		.fr-device-breakdown h4,
		.fr-top-referrers h4 {
			margin: 15px 0 10px 0;
			font-size: 14px;
			color: #333;
		}

		.fr-top-posts ul,
		.fr-top-referrers ul {
			margin: 0;
			padding: 0;
			list-style: none;
		}

		.fr-top-posts li,
		.fr-top-referrers li {
			display: flex;
			justify-content: space-between;
			align-items: center;
			padding: 5px 0;
			border-bottom: 1px solid #eee;
		}

		.fr-top-posts li:last-child,
		.fr-top-referrers li:last-child {
			border-bottom: none;
		}

		.fr-post-views,
		.fr-referrer-count {
			font-size: 11px;
			color: #666;
			font-weight: 500;
		}

		.fr-device-stats {
			display: flex;
			flex-direction: column;
			gap: 5px;
		}

		.fr-device-item {
			display: flex;
			justify-content: space-between;
			align-items: center;
			padding: 5px 0;
		}

		.fr-device-label {
			font-size: 12px;
		}

		.fr-device-percent {
			font-size: 12px;
			font-weight: 500;
			color: #007cba;
		}

		.fr-referrer-domain {
			font-size: 12px;
			max-width: 150px;
			overflow: hidden;
			text-overflow: ellipsis;
			white-space: nowrap;
		}';
		wp_add_inline_style( 'freedomreader-admin', $analytics_css );
		?>
		<?php
	}

	/**
	 * Get dashboard statistics
	 *
	 * @package FreedomReader
	 */
	private function get_dashboard_stats() {
		global $wpdb;

		// Total views (last 30 days).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$total_views = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_analytics WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
				30
			)
		);

		// Unique visitors (last 30 days).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$unique_visitors = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT ip_address) FROM {$wpdb->prefix}freedo_analytics WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)",
				30
			)
		);

		// Average reading time (last 30 days).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$avg_reading_time = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT AVG(reading_time) FROM {$wpdb->prefix}freedo_analytics WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY) AND reading_time > %d",
				30,
				0
			)
		);

		// Average scroll depth (last 30 days).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$avg_scroll_depth = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT AVG(scroll_depth) FROM {$wpdb->prefix}freedo_analytics WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY) AND scroll_depth > %d",
				30,
				0
			)
		);

		// Top posts (last 7 days).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$top_posts = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT post_id, COUNT(*) as views
				FROM {$wpdb->prefix}freedo_analytics
				WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
				GROUP BY post_id
				ORDER BY views DESC
				LIMIT %d",
				7,
				5
			)
		);

		// Device breakdown (last 30 days).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$device_stats = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT device_type, COUNT(*) as count
				FROM {$wpdb->prefix}freedo_analytics
				WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
				GROUP BY device_type",
				30
			)
		);

		$device_breakdown   = array(
			'desktop' => 0,
			'mobile'  => 0,
			'tablet'  => 0,
		);
		$total_device_views = 0;

		foreach ( $device_stats as $stat ) {
			$device_breakdown[ $stat->device_type ] = intval( $stat->count );
			$total_device_views                    += intval( $stat->count );
		}

		// Convert to percentages.
		if ( $total_device_views > 0 ) {
			foreach ( $device_breakdown as $device => $count ) {
				$device_breakdown[ $device ] = round( ( $count / $total_device_views ) * 100 );
			}
		}

		// Top referrers (last 30 days).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$referrer_data = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT referrer, COUNT(*) as visits
				FROM {$wpdb->prefix}freedo_analytics
				WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
				AND referrer != %s
				AND referrer IS NOT NULL
				GROUP BY referrer
				ORDER BY visits DESC
				LIMIT %d",
				30,
				'',
				5
			)
		);

		$top_referrers = array();
		foreach ( $referrer_data as $referrer ) {
			$domain = wp_parse_url( $referrer->referrer, PHP_URL_HOST );
			if ( $domain ) {
				$top_referrers[] = (object) array(
					'domain' => $domain,
					'visits' => $referrer->visits,
				);
			}
		}

		return array(
			'total_views'      => intval( $total_views ),
			'unique_visitors'  => intval( $unique_visitors ),
			'avg_reading_time' => round( floatval( $avg_reading_time ) ),
			'avg_scroll_depth' => round( floatval( $avg_scroll_depth ) ),
			'top_posts'        => $top_posts,
			'device_breakdown' => $device_breakdown,
			'top_referrers'    => $top_referrers,
		);
	}

	/**
	 * Get detailed analytics for admin page
	 *
	 * @param int $days Number of days to analyze.
	 * @package FreedomReader
	 */
	public function get_detailed_analytics( $days = 30 ) {
		global $wpdb;

		// Daily views for chart.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$daily_views = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT DATE(created_at) as date, COUNT(*) as views
				FROM {$wpdb->prefix}freedo_analytics
				WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
				GROUP BY DATE(created_at)
				ORDER BY date ASC",
				$days
			)
		);

		// Most popular content.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Analytics data query
		$popular_content = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT post_id, COUNT(*) as views, AVG(reading_time) as avg_time, AVG(scroll_depth) as avg_scroll
				FROM {$wpdb->prefix}freedo_analytics
				WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
				GROUP BY post_id
				ORDER BY views DESC
				LIMIT 10",
				$days
			)
		);

		return array(
			'daily_views'     => $daily_views,
			'popular_content' => $popular_content,
		);
	}
}
