<?php
/**
 * Advanced Content Locking System
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Advanced Content Locking System
 *
 * @package FreedomReader
 */
class FREEDO_Advanced_Locks {

	/**
	 * Single instance of the class
	 *
	 * @var FREEDO_Advanced_Locks|null
	 */
	private static $instance = null;

	/**
	 * Database instance
	 *
	 * @var FREEDO_Database
	 */
	private $db;

	/**
	 * Geo location cache
	 *
	 * @var array
	 */
	private $geo_cache = array();

	/**
	 * Get single instance
	 *
	 * @return FREEDO_Advanced_Locks Single instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->db = FREEDO_Database::get_instance();
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		add_filter( 'freedomreader_is_content_locked', array( $this, 'check_advanced_locks' ), 10, 2 );
		add_action( 'wp_ajax_freedomreader_save_lock_rule', array( $this, 'save_lock_rule' ) );
		add_action( 'wp_ajax_freedomreader_delete_lock_rule', array( $this, 'delete_lock_rule' ) );
		add_action( 'wp_ajax_freedomreader_toggle_lock_rule', array( $this, 'toggle_lock_rule' ) );

		// Clean up expired geo cache daily.
		add_action( 'wp_scheduled_delete', array( $this, 'cleanup_geo_cache' ) );
	}

	/**
	 * Check if content is locked by advanced rules
	 *
	 * @param bool $is_locked Current lock status.
	 * @param int  $post_id   Post ID.
	 * @return bool Whether content is locked.
	 */
	public function check_advanced_locks( $is_locked, $post_id ) {
		// Skip if already locked by basic rules.
		if ( $is_locked ) {
			return $is_locked;
		}

		if ( current_user_can( 'manage_options' ) ) {
			return false;
		}

		$post = get_post( $post_id );
		if ( ! $post ) {
			return false;
		}

		// Get all active lock rules ordered by priority.
		$rules = $this->get_active_lock_rules();

		foreach ( $rules as $rule ) {
			if ( $this->evaluate_lock_rule( $rule, $post ) ) {
				return $this->check_rule_conditions( $rule );
			}
		}

		return false;
	}

	/**
	 * Get all active lock rules
	 *
	 * @package FreedomReader
	 */
	private function get_active_lock_rules() {
		global $wpdb;

		$cache_key = 'freedo_active_lock_rules';
		$rules     = wp_cache_get( $cache_key );

		if ( false === $rules ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for lock rules
			$rules = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM `{$wpdb->prefix}freedo_lock_rules` WHERE is_active = %d ORDER BY priority ASC, id ASC",
					1
				)
			);
			wp_cache_set( $cache_key, $rules, '', 300 );

		}

		return $rules;
	}

	/**
	 * Evaluate if a lock rule applies to a post
	 *
	 * @param object $rule Lock rule object.
	 * @param object $post Post object.
	 * @return bool Whether rule applies.
	 */
	private function evaluate_lock_rule( $rule, $post ) {
		switch ( $rule->rule_type ) {
			case 'category':
				return $this->check_category_rule( $rule->rule_value, $post );

			case 'tag':
				return $this->check_tag_rule( $rule->rule_value, $post );

			case 'author':
				return $this->check_author_rule( $rule->rule_value, $post );

			case 'country':
				return $this->check_country_rule( $rule->rule_value );

			case 'time_window':
				return $this->check_time_window_rule( $rule->rule_value );

			case 'user_role':
				return $this->check_user_role_rule( $rule->rule_value );

			case 'device_type':
				return $this->check_device_type_rule( $rule->rule_value );

			case 'email_verification':
				return $this->check_email_verification_rule( $rule->rule_value, $post );

			case 'purchase_history':
				return $this->check_purchase_history_rule( $rule->rule_value );

			default:
				return false;
		}
	}

	/**
	 * Check category-based lock rule
	 *
	 * @param string $rule_value Rule value (JSON encoded categories).
	 * @param object $post       Post object.
	 * @return bool Whether rule applies.
	 */
	private function check_category_rule( $rule_value, $post ) {
		$categories = json_decode( $rule_value, true );
		if ( ! is_array( $categories ) ) {
			return false;
		}

		$post_categories = wp_get_post_categories( $post->ID );
		return ! empty( array_intersect( $categories, $post_categories ) );
	}

	/**
	 * Check tag-based lock rule
	 *
	 * @param string $rule_value Rule value (JSON encoded tags).
	 * @param object $post       Post object.
	 * @return bool Whether rule applies.
	 */
	private function check_tag_rule( $rule_value, $post ) {
		$tags = json_decode( $rule_value, true );
		if ( ! is_array( $tags ) ) {
			return false;
		}

		$post_tags = wp_get_post_tags( $post->ID, array( 'fields' => 'ids' ) );
		return ! empty( array_intersect( $tags, $post_tags ) );
	}

	/**
	 * Check author-based lock rule
	 *
	 * @param string $rule_value Rule value (JSON encoded author IDs).
	 * @param object $post       Post object.
	 * @return bool Whether rule applies.
	 */
	private function check_author_rule( $rule_value, $post ) {
		$authors = json_decode( $rule_value, true );
		if ( ! is_array( $authors ) ) {
			return false;
		}

		return in_array( $post->post_author, $authors, true );
	}

	/**
	 * Check country-based lock rule
	 *
	 * @param string $rule_value Rule value.
	 * @return bool Whether rule passes.
	 */
	private function check_country_rule( $rule_value ) {
		$rule_data = json_decode( $rule_value, true );
		if ( ! is_array( $rule_data ) ) {
			return false;
		}

		$user_country = $this->get_user_country();
		if ( ! $user_country ) {
			return false;
		}

		$allowed_countries = $rule_data['allowed'] ?? array();
		$blocked_countries = $rule_data['blocked'] ?? array();

		// If blocked countries are specified and user is in one.
		if ( ! empty( $blocked_countries ) && in_array( $user_country, $blocked_countries, true ) ) {
			return true;
		}

		// If allowed countries are specified and user is not in one.
		if ( ! empty( $allowed_countries ) && ! in_array( $user_country, $allowed_countries, true ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Check time window lock rule
	 *
	 * @param string $rule_value Rule value (JSON encoded time window).
	 * @return bool Whether rule applies.
	 */
	private function check_time_window_rule( $rule_value ) {
		$rule_data = json_decode( $rule_value, true );
		if ( ! is_array( $rule_data ) ) {
			return false;
		}

		$current_time = time();

		if ( isset( $rule_data['start_date'] ) && isset( $rule_data['end_date'] ) ) {
			$start_time = strtotime( $rule_data['start_date'] );
			$end_time   = strtotime( $rule_data['end_date'] . ' 23:59:59' );

			if ( $current_time < $start_time || $current_time > $end_time ) {
				return true;
			}
		}

		if ( isset( $rule_data['start_time'] ) && isset( $rule_data['end_time'] ) ) {
			$current_hour_min = gmdate( 'H:i', $current_time );

			if ( $current_hour_min < $rule_data['start_time'] || $current_hour_min > $rule_data['end_time'] ) {
				return true;
			}
		}

		if ( isset( $rule_data['allowed_days'] ) ) {
			$current_day = gmdate( 'w', $current_time ); // 0 = Sunday.
			if ( ! in_array( $current_day, $rule_data['allowed_days'], true ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check user role rule
	 *
	 * @param string $rule_value Rule value (JSON encoded user roles).
	 * @return bool Whether rule applies.
	 */
	private function check_user_role_rule( $rule_value ) {
		if ( ! is_user_logged_in() ) {
			return true;
		}

		$required_roles = json_decode( $rule_value, true );
		if ( ! is_array( $required_roles ) ) {
			return false;
		}

		$user       = wp_get_current_user();
		$user_roles = $user->roles;

		return empty( array_intersect( $required_roles, $user_roles ) );
	}

	/**
	 * Check rule conditions (subscription, payment, etc.)
	 *
	 * @param object $rule Lock rule object.
	 * @return bool Whether conditions are met.
	 */
	private function check_rule_conditions( $rule ) {
		$conditions = json_decode( $rule->lock_conditions, true );
		if ( ! is_array( $conditions ) ) {
			return true;
		}

		// Check if user has active subscription.
		if ( isset( $conditions['requires_subscription'] ) && $conditions['requires_subscription'] ) {
			if ( ! $this->user_has_active_subscription() ) {
				return true;
			}
		}

		// Check if user has specific role.
		if ( isset( $conditions['required_roles'] ) && ! empty( $conditions['required_roles'] ) ) {
			if ( ! is_user_logged_in() ) {
				return true;
			}

			$user = wp_get_current_user();
			if ( empty( array_intersect( $conditions['required_roles'], $user->roles ) ) ) {
				return true;
			}
		}

		if ( isset( $conditions['requires_login'] ) && $conditions['requires_login'] ) {
			if ( ! is_user_logged_in() ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Get user's country from IP address
	 *
	 * @package FreedomReader
	 */
	private function get_user_country() {
		$ip = $this->get_user_ip();
		if ( ! $ip || '127.0.0.1' === $ip ) {
			return false;
		}

		// Check cache first.
		if ( isset( $this->geo_cache[ $ip ] ) ) {
			return $this->geo_cache[ $ip ];
		}

		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Geo cache lookup
		$cached = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT country_code FROM `{$wpdb->prefix}freedo_geo_cache` WHERE ip_address = %s AND expires_at > NOW()",
				$ip
			)
		);

		if ( $cached ) {
			$this->geo_cache[ $ip ] = $cached->country_code;
			return $cached->country_code;
		}

		// Fetch from API.
		$country = $this->fetch_country_from_api( $ip );
		if ( $country ) {
			// Cache in database for 24 hours.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Geo cache storage
			$wpdb->replace(
				$wpdb->prefix . 'freedo_geo_cache',
				array(
					'ip_address'   => $ip,
					'country_code' => $country['code'],
					'country_name' => $country['name'],
					'expires_at'   => gmdate( 'Y-m-d H:i:s', time() + 86400 ),
				)
			);

			$this->geo_cache[ $ip ] = $country['code'];
			return $country['code'];
		}

		return false;
	}

	/**
	 * Get user's IP address
	 *
	 * @package FreedomReader
	 */
	private function get_user_ip() {
		$ip_keys = array( 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'REMOTE_ADDR' );

		foreach ( $ip_keys as $key ) {
			if ( ! empty( $_SERVER[ $key ] ) ) {
				$ip = sanitize_text_field( wp_unslash( $_SERVER[ $key ] ) );
				// Handle comma-separated IPs.
				if ( strpos( $ip, ',' ) !== false ) {
					$ip = trim( explode( ',', $ip )[0] );
				}
				if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE ) ) {
					return $ip;
				}
			}
		}

		return isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
	}

	/**
	 * Fetch country from IP API
	 *
	 * @param string $ip IP address.
	 * @return string|false Country code or false on failure.
	 */
	private function fetch_country_from_api( $ip ) {
		$api_url = "http://ip-api.com/json/{$ip}?fields=status,country,countryCode";

		$response = wp_remote_get(
			$api_url,
			array(
				'timeout'    => 5,
				'user-agent' => 'FreedomReader WordPress Plugin',
			)
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( isset( $data['status'] ) && 'success' === $data['status'] ) {
			return array(
				'code' => $data['countryCode'],
				'name' => $data['country'],
			);
		}

		return false;
	}

	/**
	 * Check if user has active subscription
	 *
	 * @package FreedomReader
	 */
	private function user_has_active_subscription() {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		return $this->db->get_user_subscription( get_current_user_id() ) !== null;
	}

	/**
	 * Check device type rule
	 *
	 * @param string $rule_value Rule value (JSON encoded device types).
	 * @return bool Whether rule applies.
	 */
	private function check_device_type_rule( $rule_value ) {
		$rule_data = json_decode( $rule_value, true );
		if ( ! is_array( $rule_data ) ) {
			return false;
		}

		$user_device    = $this->get_user_device_type();
		$locked_devices = $rule_data['locked_devices'] ?? array();

		return in_array( $user_device, $locked_devices, true );
	}

	/**
	 * Check email verification rule
	 *
	 * @param string $rule_value Rule value (unused for now).
	 * @param object $post       Post object.
	 * @return bool Whether rule applies.
	 */
	private function check_email_verification_rule( $rule_value, $post ) {
		// Check if user has already unlocked this content via email.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Email unlock token verification
		if ( isset( $_GET['unlock_token'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Email unlock token verification
			return ! $this->verify_email_unlock_token( sanitize_text_field( wp_unslash( $_GET['unlock_token'] ) ), $post->ID );
		}

		// If no token, content is locked.
		// Note: $rule_value parameter is available for future use.
		return true;
	}

	/**
	 * Check purchase history rule
	 *
	 * @param string $rule_value Rule value (JSON encoded purchase criteria).
	 * @return bool Whether rule applies.
	 */
	private function check_purchase_history_rule( $rule_value ) {
		if ( ! is_user_logged_in() ) {
			return true;
		}

		$rule_data = json_decode( $rule_value, true );
		if ( ! is_array( $rule_data ) ) {
			return false;
		}

		$required_products = $rule_data['required_products'] ?? array();
		$user_id           = get_current_user_id();

		foreach ( $required_products as $product_id ) {
			if ( $this->user_has_purchased_product( $user_id, $product_id ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Get user's device type
	 *
	 * @package FreedomReader
	 */
	private function get_user_device_type() {
		$user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';

		// Mobile detection.
		if ( preg_match( '/Mobile|Android|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i', $user_agent ) ) {
			if ( preg_match( '/iPad/i', $user_agent ) ) {
				return 'tablet';
			}
			return 'mobile';
		}

		// Tablet detection (additional patterns).
		if ( preg_match( '/Tablet|PlayBook|Kindle|Silk/i', $user_agent ) ) {
			return 'tablet';
		}

		return 'desktop';
	}

	/**
	 * Verify email unlock token
	 *
	 * @param string $token   Unlock token.
	 * @param int    $post_id Post ID.
	 * @return bool Whether token is valid.
	 */
	private function verify_email_unlock_token( $token, $post_id ) {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Email unlock token verification
		$unlock_request = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$wpdb->prefix}freedo_email_unlocks` WHERE unlock_token = %s AND post_id = %d AND expires_at > NOW() AND used_at IS NULL",
				$token,
				$post_id
			)
		);

		if ( $unlock_request ) {
			// Mark as used.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Email unlock token usage tracking
			$wpdb->update(
				$wpdb->prefix . 'freedo_email_unlocks',
				array( 'used_at' => current_time( 'mysql' ) ),
				array( 'id' => $unlock_request->id )
			);

			if ( ! session_id() ) {
				session_start();
			}
			$_SESSION['freedo_email_unlocked'][ $post_id ] = true;

			return true;
		}

		return false;
	}

	/**
	 * Check if user has purchased a specific product
	 *
	 * @param int $user_id    User ID.
	 * @param int $product_id Product ID.
	 * @return bool Whether user has purchased the product.
	 */
	private function user_has_purchased_product( $user_id, $product_id ) {
		global $wpdb;

		// Check FreedomReader purchases.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Purchase verification query
		$freedo_purchase = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$wpdb->prefix}freedo_purchases` WHERE user_id = %d AND post_id = %d AND status = %s",
				$user_id,
				$product_id,
				'completed'
			)
		);

		if ( $freedo_purchase > 0 ) {
			return true;
		}

		// Check WooCommerce orders if available.
		if ( class_exists( 'WooCommerce' ) && function_exists( 'wc_get_orders' ) ) {
			try {
				$orders = wc_get_orders(
					array(
						'customer_id' => $user_id,
						'status'      => array( 'completed', 'processing' ),
						'limit'       => -1,
					)
				);

				foreach ( $orders as $order ) {
					foreach ( $order->get_items() as $item ) {
						if ( $item->get_product_id() === $product_id || $item->get_variation_id() === $product_id ) {
							return true;
						}
					}
				}
			} catch ( Exception $e ) {
				// Continue without WooCommerce check if there's an error.
				// Silently fail to avoid breaking the lock check process.
				unset( $e ); // Prevent unused variable warning.
			}
		}

		return false;
	}

	/**
	 * Clean up expired geo cache entries
	 *
	 * @package FreedomReader
	 */
	public function cleanup_geo_cache() {
		global $wpdb;
		$current_time = current_time( 'mysql' );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup expired cache entries with prepared statement
		$wpdb->query( $wpdb->prepare( "DELETE FROM `{$wpdb->prefix}freedo_geo_cache` WHERE expires_at < %s", $current_time ) );

		// Also clean up expired email unlock tokens.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cleanup expired tokens with prepared statement
		$wpdb->query( $wpdb->prepare( "DELETE FROM `{$wpdb->prefix}freedo_email_unlocks` WHERE expires_at < %s", $current_time ) );
	}
}
