/**
 * Lock Rules Admin JavaScript
 *
 * @package FreedomReader
 */

(function ($) {
	'use strict';

	var LockRules = {

		init: function () {
			this.bindEvents();
		},

		bindEvents: function () {
			// Add new rule.
			$( '#fr-add-rule-btn' ).on( 'click', this.showAddRuleModal );

			// Edit rule.
			$( '.fr-edit-rule' ).on( 'click', this.showEditRuleModal );

			// Delete rule.
			$( '.fr-delete-rule' ).on( 'click', this.deleteRule );

			// Toggle rule active status.
			$( '.fr-rule-toggle' ).on( 'change', this.toggleRule );

			// Modal controls.
			$( '.fr-modal-close, #fr-cancel-rule' ).on( 'click', this.hideModal );

			// Rule type change.
			$( '#fr-rule-type' ).on( 'change', this.showRuleConfig );

			// Form submission.
			$( '#fr-rule-form' ).on( 'submit', this.saveRule );

			// Close modal on outside click.
			$( '#fr-rule-modal' ).on(
				'click',
				function (e) {
					if (e.target === this) {
						LockRules.hideModal();
					}
				}
			);
		},

		showAddRuleModal: function () {
			$( '#fr-modal-title' ).text( 'Add Lock Rule' );
			$( '#fr-rule-form' )[0].reset();
			$( '#fr-rule-id' ).val( '' );
			$( '.fr-rule-config' ).hide();
			$( '#fr-rule-modal' ).show();
		},

		showEditRuleModal: function () {
			var ruleId = $( this ).data( 'rule-id' );

			// Load rule data via AJAX.
			$.ajax(
				{
					url: freedomreader_admin.ajax_url,
					type: 'POST',
					data: {
						action: 'freedomreader_get_lock_rule',
						nonce: freedomreader_admin.nonce,
						rule_id: ruleId
					},
					success: function (response) {
						if (response.success) {
							LockRules.populateRuleForm( response.data );
							$( '#fr-modal-title' ).text( 'Edit Lock Rule' );
							$( '#fr-rule-modal' ).show();
						} else {
							$('.wrap h1').after('<div class="notice notice-error is-dismissible"><p>Failed to load rule data</p></div>');
						}
					}
				}
			);
		},

		populateRuleForm: function (rule) {
			$( '#fr-rule-id' ).val( rule.id );
			$( '#fr-rule-name' ).val( rule.rule_name );
			$( '#fr-rule-type' ).val( rule.rule_type ).trigger( 'change' );
			$( '#fr-priority' ).val( rule.priority );

			// Parse rule value.
			var ruleValue  = JSON.parse( rule.rule_value );
			var conditions = JSON.parse( rule.lock_conditions );

			// Populate based on rule type.
			switch (rule.rule_type) {
				case 'category':
					ruleValue.forEach(
						function (catId) {
							$( 'input[name="categories[]"][value="' + catId + '"]' ).prop( 'checked', true );
						}
					);
					break;

				case 'tag':
					ruleValue.forEach(
						function (tagId) {
							$( 'input[name="tags[]"][value="' + tagId + '"]' ).prop( 'checked', true );
						}
					);
					break;

				case 'author':
					ruleValue.forEach(
						function (authorId) {
							$( 'input[name="authors[]"][value="' + authorId + '"]' ).prop( 'checked', true );
						}
					);
					break;

				case 'country':
					if (ruleValue.allowed && ruleValue.allowed.length > 0) {
						$( 'input[name="country_type"][value="allow"]' ).prop( 'checked', true );
						$( '#fr-countries' ).val( ruleValue.allowed.join( ', ' ) );
					} else if (ruleValue.blocked && ruleValue.blocked.length > 0) {
						$( 'input[name="country_type"][value="block"]' ).prop( 'checked', true );
						$( '#fr-countries' ).val( ruleValue.blocked.join( ', ' ) );
					}
					break;

				case 'time_window':
					if (ruleValue.start_date) {
						$( '#fr-start-date' ).val( ruleValue.start_date );
					}
					if (ruleValue.end_date) {
						$( '#fr-end-date' ).val( ruleValue.end_date );
					}
					if (ruleValue.start_time) {
						$( '#fr-start-time' ).val( ruleValue.start_time );
					}
					if (ruleValue.end_time) {
						$( '#fr-end-time' ).val( ruleValue.end_time );
					}
					if (ruleValue.allowed_days) {
						ruleValue.allowed_days.forEach(
							function (day) {
								$( 'input[name="allowed_days[]"][value="' + day + '"]' ).prop( 'checked', true );
							}
						);
					}
					break;

				case 'user_role':
					ruleValue.forEach(
						function (role) {
							$( 'input[name="user_roles[]"][value="' + role + '"]' ).prop( 'checked', true );
						}
					);
					break;
			}

			// Populate conditions.
			if (conditions.requires_login) {
				$( 'input[name="requires_login"]' ).prop( 'checked', true );
			}
			if (conditions.requires_subscription) {
				$( 'input[name="requires_subscription"]' ).prop( 'checked', true );
			}
			if (conditions.required_roles) {
				conditions.required_roles.forEach(
					function (role) {
						$( 'input[name="required_roles[]"][value="' + role + '"]' ).prop( 'checked', true );
					}
				);
			}
		},

		showRuleConfig: function () {
			var ruleType = $( this ).val();
			$( '.fr-rule-config' ).hide();
			if (ruleType) {
				$( '#fr-rule-' + ruleType ).show();
			}
		},

		saveRule: function (e) {
			e.preventDefault();

			var formData = new FormData( this );
			formData.append( 'action', 'freedomreader_save_lock_rule' );
			formData.append( 'nonce', freedomreader_admin.nonce );

			// Convert form data to proper format.
			var ruleData = LockRules.serializeRuleData( formData );

			$.ajax(
				{
					url: freedomreader_admin.ajax_url,
					type: 'POST',
					data: ruleData,
					success: function (response) {
						if (response.success) {
							location.reload();
						} else {
							$('.wrap h1').after('<div class="notice notice-error is-dismissible"><p>Error: ' + response.data.message + '</p></div>');
						}
					},
					error: function () {
						$('.wrap h1').after('<div class="notice notice-error is-dismissible"><p>Failed to save rule</p></div>');
					}
				}
			);
		},

		serializeRuleData: function (formData) {
			var data     = {};
			var ruleType = formData.get( 'rule_type' );

			// Basic fields.
			data.action    = 'freedomreader_save_lock_rule';
			data.nonce     = freedomreader_admin.nonce;
			data.rule_id   = formData.get( 'rule_id' );
			data.rule_name = formData.get( 'rule_name' );
			data.rule_type = ruleType;
			data.priority  = formData.get( 'priority' );

			// Rule value based on type.
			var ruleValue = {};

			switch (ruleType) {
				case 'category':
					ruleValue = formData.getAll( 'categories[]' ).map( Number );
					break;

				case 'tag':
					ruleValue = formData.getAll( 'tags[]' ).map( Number );
					break;

				case 'author':
					ruleValue = formData.getAll( 'authors[]' ).map( Number );
					break;

				case 'country':
					var countryType = formData.get( 'country_type' );
					var countries   = formData.get( 'countries' ).split( ',' ).map(
						function (c) {
							return c.trim().toUpperCase();
						}
					).filter(
						function (c) {
							return c.length === 2;
						}
					);

					if (countryType === 'allow') {
						ruleValue.allowed = countries;
						ruleValue.blocked = [];
					} else {
						ruleValue.blocked = countries;
						ruleValue.allowed = [];
					}
					break;

				case 'time_window':
					ruleValue.start_date   = formData.get( 'start_date' );
					ruleValue.end_date     = formData.get( 'end_date' );
					ruleValue.start_time   = formData.get( 'start_time' );
					ruleValue.end_time     = formData.get( 'end_time' );
					ruleValue.allowed_days = formData.getAll( 'allowed_days[]' ).map( Number );
					break;

				case 'user_role':
					ruleValue = formData.getAll( 'user_roles[]' );
					break;
			}

			data.rule_value = JSON.stringify( ruleValue );

			// Lock conditions.
			var conditions = {};
			if (formData.get( 'requires_login' )) {
				conditions.requires_login = true;
			}
			if (formData.get( 'requires_subscription' )) {
				conditions.requires_subscription = true;
			}
			var requiredRoles = formData.getAll( 'required_roles[]' );
			if (requiredRoles.length > 0) {
				conditions.required_roles = requiredRoles;
			}

			data.lock_conditions = JSON.stringify( conditions );

			return data;
		},

		deleteRule: function () {
			if ( ! confirm( 'Are you sure you want to delete this rule?' )) {
				return;
			}

			var ruleId = $( this ).data( 'rule-id' );

			$.ajax(
				{
					url: freedomreader_admin.ajax_url,
					type: 'POST',
					data: {
						action: 'freedomreader_delete_lock_rule',
						nonce: freedomreader_admin.nonce,
						rule_id: ruleId
					},
					success: function (response) {
						if (response.success) {
							$( 'tr[data-rule-id="' + ruleId + '"]' ).fadeOut(
								function () {
									$( this ).remove();
								}
							);
						} else {
							$('.wrap h1').after('<div class="notice notice-error is-dismissible"><p>Failed to delete rule</p></div>');
						}
					}
				}
			);
		},

		toggleRule: function () {
			var ruleId   = $( this ).data( 'rule-id' );
			var isActive = $( this ).is( ':checked' );

			$.ajax(
				{
					url: freedomreader_admin.ajax_url,
					type: 'POST',
					data: {
						action: 'freedomreader_toggle_lock_rule',
						nonce: freedomreader_admin.nonce,
						rule_id: ruleId,
						is_active: isActive ? 1 : 0
					},
					success: function (response) {
						if ( ! response.success) {
							$('.wrap h1').after('<div class="notice notice-error is-dismissible"><p>Failed to update rule status</p></div>');
							// Revert toggle.
							$( this ).prop( 'checked', ! isActive );
						}
					}.bind( this )
				}
			);
		},

		hideModal: function () {
			$( '#fr-rule-modal' ).hide();
		}
	};

	// Initialize when document is ready.
	$( document ).ready(
		function () {
			LockRules.init();
		}
	);

})( jQuery );
