/**
 * FreedomReader Frontend JavaScript
 *
 * @package FreedomReader
 */

(function ($) {
	'use strict';

	var FreedomReader = {

		init: function () {
			this.bindEvents();
			this.initPayPal();
			this.createSuccessModal();
		},

		createSuccessModal: function () {
			var modalHTML                   = `
				< div id                    = "fr-success-modal" class = "fr-modal" style = "display: none;" >
					< div class             = "fr-modal-overlay" > < / div >
					< div class             = "fr-modal-content" >
						< div class         = "fr-success-icon" >
							< svg width     = "64" height = "64" viewBox = "0 0 24 24" fill = "none" >
								< circle cx = "12" cy = "12" r = "10" fill = "#10B981" / >
								< path d    = "m9 12 2 2 4-4" stroke = "white" stroke - width = "2" stroke - linecap = "round" stroke - linejoin = "round" / >
							< / svg >
						< / div >
						< h2 > Payment Successful ! < / h2 >
						< p > Thank you for your purchase. You now have access to this premium content.< / p >
						< button class = "fr-modal-btn" onclick = "FreedomReader.closeSuccessModal()" > Continue Reading < / button >
					< / div >
				< / div >
			`;
			$( 'body' ).append( modalHTML );
		},

		showSuccessModal: function () {
			$( '#fr-success-modal' ).fadeIn( 300 );
			$( 'body' ).addClass( 'fr-modal-open' );
		},

		closeSuccessModal: function () {
			$( '#fr-success-modal' ).fadeOut( 300 );
			$( 'body' ).removeClass( 'fr-modal-open' );
			location.reload();
		},

		bindEvents: function () {
			// Purchase button click.
			$( document ).on( 'click', '.fr-purchase-btn', this.handlePurchaseClick );

			// Subscribe button click.
			$( document ).on( 'click', '.fr-subscribe-btn', this.handleSubscribeClick );

			// Cancel subscription.
			$( document ).on( 'click', '.fr-cancel-subscription', this.handleCancelSubscription );

			// Login required check.
			$( document ).on( 'click', '.fr-purchase-btn, .fr-subscribe-btn', this.checkLoginRequired );
		},

		checkLoginRequired: function (e) {
			if ( ! freedomreader.is_user_logged_in) {
				e.preventDefault();
				FreedomReader.showAuthModal();
				return false;
			}
		},

		handlePurchaseClick: function (e) {
			e.preventDefault();

			var $button = $( this );
			var postId  = $button.data( 'post-id' );
			var price   = $button.data( 'price' );

			if ( ! postId || ! price) {
				FreedomReader.showError( 'Invalid purchase data' );
				return;
			}

			FreedomReader.showProcessing( $button );
			FreedomReader.createPayPalOrder( postId, price, 'single' );
		},

		handleSubscribeClick: function (e) {
			e.preventDefault();

			var $button = $( this );
			var planId  = $button.data( 'plan-id' );

			if ( ! planId) {
				FreedomReader.showError( 'Invalid subscription plan' );
				return;
			}

			FreedomReader.showProcessing( $button );
			FreedomReader.createSubscription( planId );
		},

		handleCancelSubscription: function (e) {
			e.preventDefault();

			if ( ! confirm( freedomreader.messages.confirm_cancel )) {
				return;
			}

			var $button        = $( this );
			var subscriptionId = $button.data( 'subscription-id' );

			FreedomReader.showProcessing( $button );

			$.ajax(
				{
					url: freedomreader.ajax_url,
					type: 'POST',
					data: {
						action: 'freedomreader_cancel_subscription',
						nonce: freedomreader.nonce,
						subscription_id: subscriptionId
					},
					success: function (response) {
						if (response.success) {
							FreedomReader.showSuccess( response.data.message );
							location.reload();
						} else {
							FreedomReader.showError( response.data.message );
							FreedomReader.hideProcessing( $button );
						}
					},
					error: function () {
						FreedomReader.showError( freedomreader.messages.error );
						FreedomReader.hideProcessing( $button );
					}
				}
			);
		},

		createPayPalOrder: function (postId, price, type) {
			$.ajax(
				{
					url: freedomreader.ajax_url,
					type: 'POST',
					dataType: 'json',
					timeout: 30000,
					data: {
						action: 'freedomreader_create_paypal_order',
						nonce: freedomreader.nonce,
						post_id: postId,
						price: price,
						type: type
					},
					success: function (response) {
						if (response.success && response.data && response.data.order_id) {
							FreedomReader.renderPayPalButton( response.data.order_id, postId, type );
						} else {
							var errorMessage = (response.data && response.data.message)
							? response.data.message
							: 'Failed to create order';
							FreedomReader.showError( errorMessage );
							FreedomReader.hideAllProcessing();
						}
					},
					error: function (xhr, status, error) {
						FreedomReader.showError( 'Failed to create PayPal order. Please try again.' );
						FreedomReader.hideAllProcessing();
					}
				}
			);
		},

		createSubscription: function (planId) {
			$.ajax(
				{
					url: freedomreader.ajax_url,
					type: 'POST',
					data: {
						action: 'freedomreader_create_subscription',
						nonce: freedomreader.nonce,
						plan_id: planId
					},
					success: function (response) {
						if (response.success) {
							// Redirect to PayPal for subscription approval.
							window.location.href = response.data.approval_url;
						} else {
							FreedomReader.showError( response.data.message );
							FreedomReader.hideAllProcessing();
						}
					},
					error: function () {
						FreedomReader.showError( freedomreader.messages.error );
						FreedomReader.hideAllProcessing();
					}
				}
			);
		},

		initPayPal: function () {
			if (typeof paypal === 'undefined') {
				return;
			}
			// PayPal will be initialized when needed.
		},

		renderPayPalButton: function (orderId, postId, type) {
			// Create PayPal button container.
			var containerId = 'paypal-button-container-' + Date.now();
			var $container  = $( '<div id="' + containerId + '" class="fr-paypal-container"></div>' );

			// Find the purchase button and replace it temporarily.
			var $purchaseBtn = $( '.fr-purchase-btn[data-post-id="' + postId + '"]' );
			$purchaseBtn.after( $container ).hide();

			// Render PayPal button.
			paypal.Buttons(
				{
					createOrder: function (data, actions) {
						return orderId;
					},

					onApprove: function (data, actions) {
						return FreedomReader.capturePayment( data.orderID, postId, type );
					},

					onError: function (err) {
						var errorMessage = freedomreader.messages.error;

						// Try to extract a more specific error message.
						if (err && typeof err === 'object') {
							if (err.message) {
								errorMessage = err.message;
							} else if (err.details && err.details.length > 0) {
								errorMessage = err.details[0].description || errorMessage;
							}
						}

						FreedomReader.showError( errorMessage );
						FreedomReader.hideAllProcessing();
						$purchaseBtn.show();
						$container.remove();
					},

					onCancel: function (data) {
						FreedomReader.hideAllProcessing();
						$purchaseBtn.show();
						$container.remove();
					}

				}
			).render( '#' + containerId );
		},

		capturePayment: function (orderID, postId, type) {
			return new Promise(
				function (resolve, reject) {
					$.ajax(
						{
							url: freedomreader.ajax_url,
							type: 'POST',
							dataType: 'json',
							timeout: 30000,
							data: {
								action: 'freedomreader_process_payment',
								nonce: freedomreader.nonce,
								paypal_order_id: orderID,
								post_id: postId,
								payment_type: type
							},
							success: function (response) {
								if (response && response.success) {
									if (response.data && response.data.show_success_modal) {
										FreedomReader.showSuccessModal();
									} else if (response.data && response.data.redirect) {
										window.location.href = response.data.redirect;
									} else {
										location.reload();
									}
									resolve();
								} else {
									var errorMessage = (response && response.data && response.data.message)
									? response.data.message
									: 'Payment failed';
									FreedomReader.showError( errorMessage );
									reject( new Error( errorMessage ) );
								}
							},
							error: function (xhr, status, error) {
								FreedomReader.showError( 'Payment processing failed. Please try again.' );
								reject( new Error( 'Payment processing failed' ) );
							}
						}
					);
				}
			);
		},

		showProcessing: function ($element) {
			$element.prop( 'disabled', true );
			$element.data( 'original-text', $element.text() );
			$element.text( freedomreader.messages.processing );

			// Show processing indicator if it exists.
			$element.closest( '.fr-unlock-options' ).find( '.fr-payment-processing' ).show();
		},

		hideProcessing: function ($element) {
			$element.prop( 'disabled', false );
			var originalText = $element.data( 'original-text' );
			if (originalText) {
				$element.text( originalText );
			}

			// Hide processing indicator.
			$element.closest( '.fr-unlock-options' ).find( '.fr-payment-processing' ).hide();
		},

		hideAllProcessing: function () {
			$( '.fr-purchase-btn, .fr-subscribe-btn' ).each(
				function () {
					FreedomReader.hideProcessing( $( this ) );
				}
			);
		},

		showAuthModal: function () {
			// Create modal if it doesn't exist.
			if ( ! $( '#fr-auth-modal' ).length) {
				FreedomReader.createAuthModal();
			}

			// Show modal.
			$( '#fr-auth-modal' ).fadeIn( 300 );
			$( 'body' ).addClass( 'fr-modal-open' );
		},

		hideAuthModal: function () {
			$( '#fr-auth-modal' ).fadeOut( 300 );
			$( 'body' ).removeClass( 'fr-modal-open' );
		},

		createAuthModal: function () {
			var modalHtml       = `
				< div id        = "fr-auth-modal" class = "fr-modal-overlay" >
					< div class = "fr-modal-container" >
						< div class              = "fr-modal-header" >
							< h3 > ${freedomreader.messages.auth_required || 'Authentication Required'} < / h3 >
							< button class       = "fr-modal-close" > & times; < / button >
						< / div >
						< div class              = "fr-modal-body" >
							< div class          = "fr-auth-message" >
								< div class      = "fr-lock-icon" > 🔒 < / div >
								< p > Please log in or create an account to access this premium content.< / p >
							< / div >
							< div class          = "fr-auth-buttons" >
								< a href         = "${freedomreader.login_url}" class = "fr-auth-btn fr-login-btn" >
									< span class = "fr-btn-icon" > 👤 < / span >
									< span class = "fr-btn-text" >
										< strong > Log In < / strong >
										< small > Already have an account ? < / small >
									< / span >
								< / a >
								${freedomreader.registration_enabled ? `
								< a href         = "${freedomreader.register_url}" class = "fr-auth-btn fr-signup-btn" >
									< span class = "fr-btn-icon" > ✨ < / span >
									< span class = "fr-btn-text" >
										< strong > Sign Up < / strong >
										< small > Create a new account < / small >
									< / span >
								< / a >
				` : `
								< div class      = "fr-auth-btn fr-signup-disabled" >
									< span class = "fr-btn-icon" > ⚠️ < / span >
									< span class = "fr-btn-text" >
										< strong > Registration Disabled < / strong >
										< small > Contact admin to create account < / small >
									< / span >
								< / div >
								`}
									< / div >
									< div class  = "fr-auth-benefits" >
									< h4 > Why create an account ? < / h4 >
									< ul >
									< li > ✓ Access premium content < / li >
									< li > ✓ Track your purchases < / li >
									< li > ✓ Manage subscriptions < / li >
									< li > ✓ Download invoices < / li >
									< / ul >
									< / div >
									< / div >
									< / div >
									< / div >
									`;

									$( 'body' ).append( modalHtml );

									// Bind close events.
									$( '#fr-auth-modal .fr-modal-close, #fr-auth-modal .fr-modal-overlay' ).click(
										function (e) {
											if (e.target === this) {
												FreedomReader.hideAuthModal();
											}
										}
									);

			// Prevent modal content clicks from closing modal.
			$( '#fr-auth-modal .fr-modal-container' ).click(
				function (e) {
					e.stopPropagation();
				}
			);

			// ESC key to close.
			$( document ).keyup(
				function (e) {
					if (e.keyCode === 27) {
						FreedomReader.hideAuthModal();
					}
				}
			);
		}
	},

	showError: function ( message ) {
		// Simple error notification.
		var notification = $( '<div class="fr-notification fr-error">' + message + '</div>' );
		$( 'body' ).append( notification );
		notification.fadeIn().delay( 5000 ).fadeOut(
			function () {
				notification.remove();
			}
		);
	},

	showSuccess: function ( message ) {
		// Simple success notification.
		var notification = $( '<div class="fr-notification fr-success">' + message + '</div>' );
		$( 'body' ).append( notification );
		notification.fadeIn().delay( 3000 ).fadeOut(
			function () {
				notification.remove();
			}
		);
	}
};

	// Initialize when document is ready.
	$( document ).ready(
		function () {
			FreedomReader.init();
		}
	);



})( jQuery );
