/**
 * FreedomReader Admin JavaScript
 */

(function ($) {
	'use strict';

	var FreedomReaderAdmin = {

		init: function () {
			this.bindEvents();
			this.initCharts();
		},

		bindEvents: function () {
			// Settings form handling
			$( '.fr-settings-form' ).on( 'submit', this.handleSettingsSubmit );

			// Export data
			$( '.fr-export-data' ).on( 'click', this.exportData );

			// Bulk actions
			$( '.fr-bulk-action' ).on( 'change', this.handleBulkAction );

			// Delete confirmation
			$( '.fr-delete-item' ).on( 'click', this.confirmDelete );

			// Toggle advanced settings
			$( '.fr-toggle-advanced' ).on( 'click', this.toggleAdvancedSettings );
		},

		handleSettingsSubmit: function (e) {
			var $form      = $( this );
			var $submitBtn = $form.find( 'input[type="submit"]' );

			// Show loading state
			$submitBtn.prop( 'disabled', true );
			$submitBtn.data( 'original-value', $submitBtn.val() );
			$submitBtn.val( freedomreader_admin.messages.saving );

			// Validate PayPal credentials if provided
			var clientId     = $form.find( 'input[name="freedomreader_paypal_client_id"]' ).val();
			var clientSecret = $form.find( 'input[name="freedomreader_paypal_client_secret"]' ).val();

			if (clientId && clientSecret) {
				FreedomReaderAdmin.validatePayPalCredentials(
					clientId,
					clientSecret,
					function (valid) {
						if ( ! valid) {
							// Show validation warning without blocking submission
						}
						// Continue with form submission
						$form.off( 'submit' ).submit();
					}
				);

				e.preventDefault();
				return false;
			}
		},

		validatePayPalCredentials: function (clientId, clientSecret, callback) {
			$.ajax(
				{
					url: freedomreader_admin.ajax_url,
					type: 'POST',
					data: {
						action: 'freedomreader_validate_paypal',
						nonce: freedomreader_admin.nonce,
						client_id: clientId,
						client_secret: clientSecret
					},
					success: function (response) {
						callback( response.success );
					},
					error: function () {
						callback( false );
					}
				}
			);
		},

		exportData: function (e) {
			e.preventDefault();

			var $button    = $( this );
			var exportType = $button.data( 'export-type' );
			var dateFrom   = $( '#export-date-from' ).val();
			var dateTo     = $( '#export-date-to' ).val();

			// Build export URL
			var exportUrl = freedomreader_admin.ajax_url +
				'?action=freedomreader_export_data' +
				'&type=' + exportType +
				'&nonce=' + freedomreader_admin.nonce;

			if (dateFrom) {
				exportUrl += '&date_from=' + dateFrom;
			}

			if (dateTo) {
				exportUrl += '&date_to=' + dateTo;
			}

			// Trigger download
			window.location.href = exportUrl;
		},

		handleBulkAction: function () {
			var action = $( this ).val();
			var $form  = $( this ).closest( 'form' );

			if (action === 'delete') {
				if ( ! confirm( freedomreader_admin.messages.confirm_delete )) {
					$( this ).val( '' );
					return false;
				}
			}

			if (action) {
				$form.submit();
			}
		},

		confirmDelete: function (e) {
			if ( ! confirm( freedomreader_admin.messages.confirm_delete )) {
				e.preventDefault();
				return false;
			}
		},

		toggleAdvancedSettings: function (e) {
			e.preventDefault();

			var $button   = $( this );
			var $advanced = $( '.fr-advanced-settings' );

			if ($advanced.is( ':visible' )) {
				$advanced.slideUp();
				$button.text( 'Show Advanced Settings' );
			} else {
				$advanced.slideDown();
				$button.text( 'Hide Advanced Settings' );
			}
		},

		initCharts: function () {
			// Initialize revenue chart if Chart.js is available
			if (typeof Chart !== 'undefined' && $( '#fr-revenue-chart' ).length) {
				this.initRevenueChart();
			}

			// Initialize subscription chart
			if (typeof Chart !== 'undefined' && $( '#fr-subscription-chart' ).length) {
				this.initSubscriptionChart();
			}
		},

		initRevenueChart: function () {
			var ctx       = document.getElementById( 'fr-revenue-chart' ).getContext( '2d' );
			var chartData = window.frRevenueData || {
				labels: [],
				revenue: [],
				purchases: []
			};

			new Chart(
				ctx,
				{
					type: 'line',
					data: {
						labels: chartData.labels,
						datasets: [{
							label: 'Revenue ($)',
							data: chartData.revenue,
							borderColor: '#007cba',
							backgroundColor: 'rgba(0, 124, 186, 0.1)',
							tension: 0.4,
							yAxisID: 'y'
						}, {
							label: 'Purchases',
							data: chartData.purchases,
							borderColor: '#28a745',
							backgroundColor: 'rgba(40, 167, 69, 0.1)',
							tension: 0.4,
							yAxisID: 'y1'
						}]
					},
					options: {
						responsive: true,
						interaction: {
							mode: 'index',
							intersect: false,
						},
						scales: {
							x: {
								display: true,
								title: {
									display: true,
									text: 'Month'
								}
							},
							y: {
								type: 'linear',
								display: true,
								position: 'left',
								title: {
									display: true,
									text: 'Revenue ($)'
								}
							},
							y1: {
								type: 'linear',
								display: true,
								position: 'right',
								title: {
									display: true,
									text: 'Purchases'
								},
								grid: {
									drawOnChartArea: false,
								},
							}
						}
					}
				}
			);
		},

		initSubscriptionChart: function () {
			var ctx       = document.getElementById( 'fr-subscription-chart' ).getContext( '2d' );
			var chartData = window.frSubscriptionData || {
				labels: ['Active', 'Trial', 'Cancelled'],
				data: [0, 0, 0]
			};

			new Chart(
				ctx,
				{
					type: 'doughnut',
					data: {
						labels: chartData.labels,
						datasets: [{
							data: chartData.data,
							backgroundColor: [
							'#28a745',
							'#ffc107',
							'#dc3545'
							],
							borderWidth: 2,
							borderColor: '#fff'
						}]
					},
					options: {
						responsive: true,
						plugins: {
							legend: {
								position: 'bottom'
							}
						}
					}
				}
			);
		}
	};

	// Initialize when document is ready
	$( document ).ready(
		function () {
			FreedomReaderAdmin.init();

			// Settings page tab switching
			$('.nav-tab').click(function(e) {
				e.preventDefault();

				// Remove active class from all tabs and content
				$('.nav-tab').removeClass('nav-tab-active');
				$('.fr-tab-content').removeClass('active');

				// Add active class to clicked tab and show content
				$(this).addClass('nav-tab-active');
				var target = $(this).attr('href');
				$(target).addClass('active');
			});

			// Subscription plan management
			$('#fr-add-plan').click(function() {
				var planId = 'plan_' + Date.now();
				var newPlan = '<div class="fr-plan-row" data-plan-id="' + planId + '">' +
					'<h4>New Plan</h4>' +
					'<table class="form-table">' +
					'<tr>' +
					'<th><label>Plan Name</label></th>' +
					'<td><input type="text" name="subscription_plans[' + planId + '][name]" class="regular-text" /></td>' +
					'</tr>' +
					'<tr>' +
					'<th><label>Price</label></th>' +
					'<td><input type="number" name="subscription_plans[' + planId + '][price]" step="0.01" min="0" class="small-text" /></td>' +
					'</tr>' +
					'<tr>' +
					'<th><label>Billing Interval</label></th>' +
					'<td>' +
					'<select name="subscription_plans[' + planId + '][interval]">' +
					'<option value="month">Monthly</option>' +
					'<option value="year">Yearly</option>' +
					'</select>' +
					'</td>' +
					'</tr>' +
					'</table>' +
					'<button type="button" class="button fr-remove-plan">Remove Plan</button>' +
					'</div>';
				$('#fr-subscription-plans').append(newPlan);
			});

			// Remove subscription plan
			$(document).on('click', '.fr-remove-plan', function() {
				$(this).closest('.fr-plan-row').remove();
			});

			// Add keyword row
			$('#fr-add-keyword').click(function() {
				var newRow = '<div class="fr-keyword-row">' +
					'<input type="text" name="auto_link_keywords[]" placeholder="Keyword" class="regular-text" />' +
					'<input type="number" name="auto_link_post_ids[]" placeholder="Post ID" min="1" />' +
					'<button type="button" class="button fr-remove-keyword">Remove</button>' +
					'</div>';
				$('#fr-auto-link-keywords').append(newRow);
			});

			// Remove keyword row
			$(document).on('click', '.fr-remove-keyword', function() {
				$(this).closest('.fr-keyword-row').remove();
			});

			// Content lock metabox functionality
			$('input[name="fr_enable_lock"]').change(function() {
				if ($(this).is(':checked')) {
					$('.fr-lock-settings').slideDown();
				} else {
					$('.fr-lock-settings').slideUp();
				}
			});

			$('select[name="fr_lock_type"]').change(function() {
				$('.fr-lock-type-settings').hide();
				$('.fr-lock-type-' + $(this).val()).show();
				updatePreview();
			});

			$('input[name="fr_teaser_length"], input[name="fr_price"], textarea[name="fr_custom_message"]').on('input', updatePreview);

			function updatePreview() {
				var lockType = $('select[name="fr_lock_type"]').val();
				var message = '';

				switch(lockType) {
					case 'full':
						message = 'This content is locked. Purchase required to view.';
						break;
					case 'teaser':
						var length = $('input[name="fr_teaser_length"]').val() || 100;
						message = 'Preview: First ' + length + ' characters shown. Purchase to read more.';
						break;
					case 'subscription':
						message = 'This content requires an active subscription.';
						break;
					case 'custom':
						message = $('textarea[name="fr_custom_message"]').val() || 'Custom lock message';
						break;
				}

				$('.fr-preview-message').text(message);
			}

			// Performance check functionality
			$('#fr-performance-check').click(function(e) {
				e.preventDefault();
				var $button = $(this);

				$button.prop('disabled', true).html('<span class="dashicons dashicons-update-alt" style="margin-right: 5px; animation: spin 1s linear infinite;"></span>Running Check...');

				$.ajax({
					url: freedomreader_admin.ajax_url,
					type: 'POST',
					data: {
						action: 'freedomreader_performance_check',
						nonce: freedomreader_admin.nonce
					},
					success: function(response) {
						if (response.success) {
							$button.after('<div class="notice notice-success is-dismissible" style="margin-top:10px;"><p>Performance check completed successfully!</p></div>');
						} else {
							$button.after('<div class="notice notice-error is-dismissible" style="margin-top:10px;"><p>Performance check failed: ' + response.data.message + '</p></div>');
						}
					},
					error: function() {
						$button.after('<div class="notice notice-error is-dismissible" style="margin-top:10px;"><p>Performance check failed due to network error.</p></div>');
					},
					complete: function() {
						$button.prop('disabled', false).html('<span class="dashicons dashicons-performance" style="margin-right: 5px;"></span>Performance Check');
					}
				});
			});
		}
	);

})( jQuery );
