<?php
/**
 * Admin Reports View
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
?>

<div class="wrap">
	<h1><?php esc_html_e( 'Reports & Analytics', 'freedomreader' ); ?></h1>

	<!-- Debug Info -->
	<?php
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Debug parameter for admin page
	if ( current_user_can( 'manage_options' ) && isset( $_GET['debug'] ) ) :
		?>
		<div class="notice notice-warning">
			<h4>Debug Information:</h4>
			<pre style="background: #f1f1f1; padding: 10px; overflow: auto; max-height: 200px;">
		<?php
		echo esc_html( 'Total Purchases: ' . ( $report_data['purchase_analytics']['total_purchases'] ?? 'NULL' ) . "\n" );
		echo esc_html( 'Total Revenue: ' . ( $report_data['purchase_analytics']['total_revenue'] ?? 'NULL' ) . "\n" );
		echo esc_html( 'Revenue Labels: ' . wp_json_encode( $report_data['revenue_data']['labels'] ?? array() ) );
		echo esc_html( 'Revenue Amounts: ' . wp_json_encode( $report_data['revenue_data']['amounts'] ?? array() ) );
		echo esc_html( 'Recent Purchases Count: ' . count( $report_data['recent_activity']['purchases'] ?? array() ) . "\n" );
		?>
			</pre>
		</div>
	<?php endif; ?>

	<!-- Advanced Filters -->
	<div class="fr-report-filters" style="background: #fff; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
		<h3 style="margin-top: 0;"><?php esc_html_e( 'Advanced Filters & Export', 'freedomreader' ); ?></h3>
		<form method="get" id="fr-report-filters-form">
			<input type="hidden" name="page" value="freedomreader-reports" />

			<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px;">
				<div>
					<label for="date_from"><?php esc_html_e( 'From Date', 'freedomreader' ); ?></label>
					<input type="date" name="date_from" id="date_from"
							value="<?php echo esc_attr( isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : '' ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin filter form ?>"
							class="regular-text" />
				</div>

				<div>
					<label for="date_to"><?php esc_html_e( 'To Date', 'freedomreader' ); ?></label>
					<input type="date" name="date_to" id="date_to"
							value="<?php echo esc_attr( isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : '' ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin filter form ?>"
							class="regular-text" />
				</div>

				<div>
					<label for="report_type"><?php esc_html_e( 'Report Type', 'freedomreader' ); ?></label>
					<select name="report_type" id="report_type">
						<option value="all" <?php selected( isset( $_GET['report_type'] ) ? sanitize_text_field( wp_unslash( $_GET['report_type'] ) ) : 'all', 'all' ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin filter form ?>>
							<?php esc_html_e( 'All Data', 'freedomreader' ); ?>
						</option>
						<option value="purchases" <?php selected( isset( $_GET['report_type'] ) ? sanitize_text_field( wp_unslash( $_GET['report_type'] ) ) : 'all', 'purchases' ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin filter form ?>>
							<?php esc_html_e( 'Purchases Only', 'freedomreader' ); ?>
						</option>
						<option value="subscriptions" <?php selected( isset( $_GET['report_type'] ) ? sanitize_text_field( wp_unslash( $_GET['report_type'] ) ) : 'all', 'subscriptions' ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin filter form ?>>
							<?php esc_html_e( 'Subscriptions Only', 'freedomreader' ); ?>
						</option>
						<option value="performance" <?php selected( isset( $_GET['report_type'] ) ? sanitize_text_field( wp_unslash( $_GET['report_type'] ) ) : 'all', 'performance' ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin filter form ?>>
							<?php esc_html_e( 'Performance Metrics', 'freedomreader' ); ?>
						</option>
					</select>
				</div>

				<div>
					<label for="user_filter"><?php esc_html_e( 'User Filter', 'freedomreader' ); ?></label>
					<input type="text" name="user_filter" id="user_filter"
							value="<?php echo esc_attr( isset( $_GET['user_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['user_filter'] ) ) : '' ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin filter form ?>"
							placeholder="<?php esc_attr_e( 'User email or ID', 'freedomreader' ); ?>"
							class="regular-text" />
				</div>
			</div>

			<div style="margin-bottom: 15px;">
				<button type="submit" class="button button-primary">
					<span class="dashicons dashicons-filter" style="margin-right: 5px;"></span>
					<?php esc_html_e( 'Apply Filters', 'freedomreader' ); ?>
				</button>

				<button type="button" class="button" id="fr-reset-filters">
					<span class="dashicons dashicons-dismiss" style="margin-right: 5px;"></span>
					<?php esc_html_e( 'Reset Filters', 'freedomreader' ); ?>
				</button>

				<button type="button" class="button button-secondary" id="fr-export-filtered">
					<span class="dashicons dashicons-download" style="margin-right: 5px;"></span>
					<?php esc_html_e( 'Export Filtered Data', 'freedomreader' ); ?>
				</button>
			</div>
		</form>
	</div>

	<?php if ( empty( $report_data['purchase_analytics']['total_purchases'] ) && empty( $report_data['subscription_analytics']['total_subscriptions'] ) ) : ?>
		<div class="notice notice-info">
			<p>
				<?php esc_html_e( 'No data available yet. Reports will appear here once you have transactions.', 'freedomreader' ); ?>
			</p>
		</div>
	<?php endif; ?>



	<!-- Dashboard Stats -->
	<div class="fr-dashboard-stats">
		<div class="fr-stats-grid">
			<div class="fr-stat-card">
				<div class="fr-stat-label"><?php esc_html_e( 'Total Revenue', 'freedomreader' ); ?></div>
				<div class="fr-stat-value">$<?php echo esc_html( number_format( $report_data['revenue_data']['total_revenue'] ?? 0, 2 ) ); ?></div>
			</div>
			<div class="fr-stat-card">
				<div class="fr-stat-label"><?php esc_html_e( 'Active Subscriptions', 'freedomreader' ); ?></div>
				<div class="fr-stat-value"><?php echo esc_html( $report_data['subscription_analytics']['active_subscriptions'] ?? 0 ); ?></div>
			</div>
			<div class="fr-stat-card">
				<div class="fr-stat-label"><?php esc_html_e( 'Total Purchases', 'freedomreader' ); ?></div>
				<div class="fr-stat-value"><?php echo esc_html( $report_data['purchase_analytics']['total_purchases'] ?? 0 ); ?></div>
			</div>
			<div class="fr-stat-card">
				<div class="fr-stat-label"><?php esc_html_e( 'Monthly Revenue', 'freedomreader' ); ?></div>
				<div class="fr-stat-value">$<?php echo esc_html( number_format( $report_data['subscription_analytics']['mrr'] ?? 0, 2 ) ); ?></div>
			</div>
		</div>
	</div>

	<div class="fr-reports-grid">
		<!-- Revenue Chart -->
		<div class="fr-chart-container">
			<h3><?php esc_html_e( 'Revenue Over Time', 'freedomreader' ); ?></h3>
			<div class="fr-chart-wrapper">
				<canvas id="fr-revenue-chart"></canvas>
			</div>
		</div>

		<!-- Subscription Analytics -->
		<div class="fr-report-card">
			<h3><?php esc_html_e( 'Subscription Analytics', 'freedomreader' ); ?></h3>

			<div class="fr-analytics-stats">
				<div class="fr-stat-item">
					<div class="fr-stat-label"><?php esc_html_e( 'Monthly Recurring Revenue', 'freedomreader' ); ?></div>
					<div class="fr-stat-value">$<?php echo esc_html( number_format( $report_data['subscription_analytics']['mrr'] ?? 0, 2 ) ); ?></div>
				</div>

				<div class="fr-stat-item">
					<div class="fr-stat-label"><?php esc_html_e( 'Churn Rate', 'freedomreader' ); ?></div>
					<div class="fr-stat-value"><?php echo esc_html( number_format( $report_data['subscription_analytics']['churn_rate'] ?? 0, 1 ) ); ?>%</div>
				</div>

				<div class="fr-stat-item">
					<div class="fr-stat-label"><?php esc_html_e( 'Cancelled Subscriptions', 'freedomreader' ); ?></div>
					<div class="fr-stat-value"><?php echo esc_html( $report_data['subscription_analytics']['cancelled_subscriptions'] ?? 0 ); ?></div>
				</div>
			</div>
			
			<!-- Subscription Status Chart -->
			<div class="fr-chart-wrapper" style="height: 200px; margin-top: 20px;">
				<canvas id="fr-subscription-chart"></canvas>
			</div>
		</div>
	</div>
	
	<!-- Top Selling Content -->
	<div class="fr-report-card">
		<h3><?php esc_html_e( 'Top Selling Content', 'freedomreader' ); ?></h3>
		
		<?php if ( ! empty( $report_data['purchase_analytics']['top_content'] ) ) : ?>
			<table class="fr-admin-table">
				<thead>
					<tr>
						<th><?php esc_html_e( 'Content', 'freedomreader' ); ?></th>
						<th><?php esc_html_e( 'Purchases', 'freedomreader' ); ?></th>
						<th><?php esc_html_e( 'Revenue', 'freedomreader' ); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php foreach ( $report_data['purchase_analytics']['top_content'] as $content ) : ?>
						<tr>
							<td>
								<?php if ( ! empty( $content->post_title ) ) : ?>
									<a href="<?php echo esc_url( get_edit_post_link( $content->post_id ) ); ?>" target="_blank">
										<?php echo esc_html( $content->post_title ); ?>
									</a>
								<?php else : ?>
									<em>
									<?php
										/* translators: %d: post ID number */
										printf( esc_html__( 'Post ID: %d (Deleted or Draft)', 'freedomreader' ), esc_html( $content->post_id ) );
									?>
									</em>
								<?php endif; ?>
							</td>
							<td><?php echo esc_html( number_format( $content->purchase_count ) ); ?></td>
							<td>$<?php echo esc_html( number_format( $content->total_revenue, 2 ) ); ?></td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
		<?php else : ?>
			<div class="notice notice-info inline">
				<p><?php esc_html_e( 'No sales data available yet. Sales data will appear here after purchases are made.', 'freedomreader' ); ?></p>
			</div>
		<?php endif; ?>
	</div>
	
	<!-- Recent Activity -->
	<div class="fr-reports-grid">
		<div class="fr-report-card">
			<h3><?php esc_html_e( 'Recent Purchases', 'freedomreader' ); ?></h3>
			
			<?php if ( ! empty( $report_data['recent_activity']['purchases'] ) ) : ?>
				<div class="fr-activity-list">
					<?php foreach ( array_slice( $report_data['recent_activity']['purchases'], 0, 5 ) as $purchase ) : ?>
						<div class="fr-activity-item">
							<div class="fr-activity-content">
								<strong><?php echo esc_html( $purchase->user_name ? $purchase->user_name : 'Unknown User' ); ?></strong>
								<?php esc_html_e( 'purchased', 'freedomreader' ); ?>
								<em><?php echo esc_html( $purchase->post_title ? $purchase->post_title : 'Content' ); ?></em>
							</div>
							<div class="fr-activity-meta">
								$<?php echo number_format( $purchase->amount, 2 ); ?> •
								<?php echo esc_html( human_time_diff( strtotime( $purchase->created_at ), time() ) ); ?> <?php esc_html_e( 'ago', 'freedomreader' ); ?>
							</div>
						</div>
					<?php endforeach; ?>
				</div>
			<?php else : ?>
				<p><?php esc_html_e( 'No recent purchases.', 'freedomreader' ); ?></p>
			<?php endif; ?>
		</div>
		
		<div class="fr-report-card">
			<h3><?php esc_html_e( 'Recent Subscriptions', 'freedomreader' ); ?></h3>
			
			<?php if ( ! empty( $report_data['recent_activity']['subscriptions'] ) ) : ?>
				<div class="fr-activity-list">
					<?php foreach ( array_slice( $report_data['recent_activity']['subscriptions'], 0, 5 ) as $subscription ) : ?>
						<div class="fr-activity-item">
							<div class="fr-activity-content">
								<strong><?php echo esc_html( $subscription->user_name ? $subscription->user_name : 'Unknown User' ); ?></strong>
								<?php esc_html_e( 'subscribed to', 'freedomreader' ); ?>
								<em><?php echo esc_html( $subscription->plan_id ); ?></em>
							</div>
							<div class="fr-activity-meta">
								<span class="fr-status fr-status-<?php echo esc_attr( $subscription->status ); ?>">
									<?php echo esc_html( ucfirst( $subscription->status ) ); ?>
								</span> • 
								<?php echo esc_html( human_time_diff( strtotime( $subscription->created_at ), time() ) ); ?> <?php esc_html_e( 'ago', 'freedomreader' ); ?>
							</div>
						</div>
					<?php endforeach; ?>
				</div>
			<?php else : ?>
				<p><?php esc_html_e( 'No recent subscriptions.', 'freedomreader' ); ?></p>
			<?php endif; ?>
		</div>
	</div>
	
	<!-- Export Section -->
	<div class="fr-report-card">
		<h3><?php esc_html_e( 'Export Data', 'freedomreader' ); ?></h3>
		
		<div class="fr-export-controls">
			<div class="fr-form-row">
				<div class="fr-form-group">
					<label for="export-date-from"><?php esc_html_e( 'From Date', 'freedomreader' ); ?></label>
					<input type="date" id="export-date-from" />
				</div>
				<div class="fr-form-group">
					<label for="export-date-to"><?php esc_html_e( 'To Date', 'freedomreader' ); ?></label>
					<input type="date" id="export-date-to" />
				</div>
				<div class="fr-form-group">
					<button class="button fr-export-data" data-export-type="purchases">
						<?php esc_html_e( 'Export Purchases', 'freedomreader' ); ?>
					</button>
				</div>
				<div class="fr-form-group">
					<button class="button fr-export-data" data-export-type="subscriptions">
						<?php esc_html_e( 'Export Subscriptions', 'freedomreader' ); ?>
					</button>
				</div>
			</div>
		</div>
	</div>
</div>

<?php
// Pass chart data to JavaScript using wp_add_inline_script.
$chart_data_js = '
// Pass chart data to JavaScript.
window.frRevenueData = ' . wp_json_encode( $report_data['revenue_chart'] ) . ';
window.frSubscriptionData = {
	labels: ' . wp_json_encode( array_column( $report_data['subscription_analytics']['status_breakdown'], 'status' ) ) . ',
	data: ' . wp_json_encode( array_column( $report_data['subscription_analytics']['status_breakdown'], 'count' ) ) . '
};';
wp_add_inline_script( 'freedomreader-admin', $chart_data_js );
?>

<?php
// Enqueue inline styles for reports.
$reports_css = '
.fr-reports-grid {
	display: grid;
	grid-template-columns: 2fr 1fr;
	gap: 20px;
	margin-bottom: 30px;
}

.fr-analytics-stats {
	display: grid;
	grid-template-columns: 1fr;
	gap: 15px;
	margin-bottom: 20px;
}

.fr-stat-item {
	background: #f8f9fa;
	padding: 15px;
	border-radius: 6px;
	text-align: center;
}

.fr-stat-label {
	font-size: 12px;
	color: #6c757d;
	text-transform: uppercase;
	letter-spacing: 0.5px;
	margin-bottom: 5px;
}

.fr-stat-value {
	font-size: 24px;
	font-weight: bold;
	color: #2271b1;
}

.fr-activity-list {
	max-height: 300px;
	overflow-y: auto;
}

.fr-activity-item {
	padding: 12px 0;
	border-bottom: 1px solid #f0f0f0;
}

.fr-activity-item:last-child {
	border-bottom: none;
}

.fr-activity-content {
	margin-bottom: 5px;
	line-height: 1.4;
}

.fr-activity-meta {
	font-size: 12px;
	color: #6c757d;
	display: flex;
	align-items: center;
	gap: 8px;
}

.fr-export-controls {
	background: #f8f9fa;
	border: 1px solid #dee2e6;
	border-radius: 6px;
	padding: 15px;
}

.fr-form-row {
	display: flex;
	gap: 15px;
	align-items: end;
	flex-wrap: wrap;
}

.fr-form-group {
	min-width: 150px;
}

.fr-form-group label {
	display: block;
	margin-bottom: 5px;
	font-weight: 500;
	font-size: 13px;
}

.fr-form-group input,
.fr-form-group button {
	padding: 6px 12px;
	border: 1px solid #ced4da;
	border-radius: 4px;
	font-size: 13px;
}

@media (max-width: 1200px) {
	.fr-reports-grid {
		grid-template-columns: 1fr;
	}
}

@media (max-width: 768px) {
	.fr-form-row {
		flex-direction: column;
		align-items: stretch;
	}

	.fr-analytics-stats {
		grid-template-columns: 1fr;
	}
}';
wp_add_inline_style( 'freedomreader-admin', $reports_css );
?>

<?php
// Enqueue chart initialization script.
$chart_init_js = '
jQuery(document).ready(function($) {
	// Revenue Chart.
	if (typeof Chart !== "undefined") {
		var revenueCtx = document.getElementById("fr-revenue-chart");
		if (revenueCtx) {
			var revenueLabels = ' . wp_json_encode( $report_data['revenue_data']['labels'] ?? array() ) . ';
			var revenueData = ' . wp_json_encode( $report_data['revenue_data']['amounts'] ?? array() ) . ';

			console.log("Revenue Labels:", revenueLabels);
			console.log("Revenue Data:", revenueData);

			new Chart(revenueCtx, {
				type: "line",
				data: {
					labels: revenueLabels,
					datasets: [{
						label: "' . esc_js( __( 'Revenue', 'freedomreader' ) ) . '",
						data: revenueData,
						borderColor: "#2271b1",
						backgroundColor: "rgba(34, 113, 177, 0.1)",
						tension: 0.4,
						fill: true
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					scales: {
						y: {
							beginAtZero: true,
							ticks: {
								callback: function(value) {
									return "$" + value.toFixed(2);
								}
							}
						}
					}
				}
			});
		}

		// Subscription Chart.
		var subscriptionCtx = document.getElementById("fr-subscription-chart");
		if (subscriptionCtx) {
			new Chart(subscriptionCtx, {
				type: "doughnut",
				data: {
					labels: ["' . esc_js( __( 'Active', 'freedomreader' ) ) . '", "' . esc_js( __( 'Cancelled', 'freedomreader' ) ) . '"],
					datasets: [{
						data: [
							' . intval( $report_data['subscription_analytics']['active_subscriptions'] ?? 0 ) . ',
							' . intval( $report_data['subscription_analytics']['cancelled_subscriptions'] ?? 0 ) . '
						],
						backgroundColor: ["#28a745", "#dc3545"]
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false
				}
			});
		}
	}
});';
wp_add_inline_script( 'freedomreader-admin', $chart_init_js );
?>
