<?php
/**
 * Lock Rules Admin View
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get existing rules with caching.
$cache_key = 'freedo_lock_rules_admin_view';
$rules     = wp_cache_get( $cache_key );

if ( false === $rules ) {
	global $wpdb;
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query for admin display
	$rules = $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}freedo_lock_rules ORDER BY priority ASC, id ASC" );
	wp_cache_set( $cache_key, $rules, '', 300 ); // Cache for 5 minutes.
}
?>

<div class="wrap">
	<h1><?php esc_html_e( 'Content Lock Rules', 'freedomreader' ); ?></h1>

	<div class="fr-lock-rules-header">
		<div class="fr-header-content">
			<p><?php esc_html_e( 'Create global rules to automatically lock content based on categories, tags, authors, geolocation, time windows, and user roles.', 'freedomreader' ); ?></p>
		</div>
		<div class="fr-header-actions">
			<button class="button button-primary" id="fr-add-rule-btn">
				<?php esc_html_e( 'Add New Rule', 'freedomreader' ); ?>
			</button>
		</div>
	</div>
	
	<!-- Rules List -->
	<div class="fr-rules-list">
		<?php if ( ! empty( $rules ) ) : ?>
			<div class="fr-rules-table-container">
				<table class="wp-list-table widefat fixed striped fr-rules-table">
					<thead>
						<tr>
							<th class="fr-col-name"><?php esc_html_e( 'Rule Name', 'freedomreader' ); ?></th>
							<th class="fr-col-type"><?php esc_html_e( 'Type', 'freedomreader' ); ?></th>
							<th class="fr-col-conditions"><?php esc_html_e( 'Conditions', 'freedomreader' ); ?></th>
							<th class="fr-col-priority"><?php esc_html_e( 'Priority', 'freedomreader' ); ?></th>
							<th class="fr-col-status"><?php esc_html_e( 'Status', 'freedomreader' ); ?></th>
							<th class="fr-col-actions"><?php esc_html_e( 'Actions', 'freedomreader' ); ?></th>
						</tr>
					</thead>
				<tbody>
					<?php foreach ( $rules as $rule ) : ?>
						<tr data-rule-id="<?php echo esc_attr( $rule->id ); ?>">
							<td>
								<strong><?php echo esc_html( $rule->rule_name ); ?></strong>
							</td>
							<td>
								<span class="fr-rule-type fr-rule-type-<?php echo esc_attr( $rule->rule_type ); ?>">
									<?php echo esc_html( ucfirst( str_replace( '_', ' ', $rule->rule_type ) ) ); ?>
								</span>
							</td>
							<td>
								<?php
								$conditions       = json_decode( $rule->lock_conditions, true );
								$condition_labels = array();

								if ( isset( $conditions['requires_login'] ) && $conditions['requires_login'] ) {
									$condition_labels[] = esc_html__( 'Login Required', 'freedomreader' );
								}
								if ( isset( $conditions['requires_subscription'] ) && $conditions['requires_subscription'] ) {
									$condition_labels[] = esc_html__( 'Subscription Required', 'freedomreader' );
								}
								if ( isset( $conditions['required_roles'] ) && ! empty( $conditions['required_roles'] ) ) {
									/* translators: %s: comma-separated list of user roles */
									$condition_labels[] = sprintf( esc_html__( 'Roles: %s', 'freedomreader' ), implode( ', ', $conditions['required_roles'] ) );
								}

								echo esc_html( implode( ', ', $condition_labels ) ? implode( ', ', $condition_labels ) : esc_html__( 'Always Locked', 'freedomreader' ) );
								?>
							</td>
							<td><?php echo esc_html( $rule->priority ); ?></td>
							<td>
								<label class="fr-toggle-switch">
									<input type="checkbox" class="fr-rule-toggle" 
											data-rule-id="<?php echo esc_attr( $rule->id ); ?>"
											<?php checked( $rule->is_active ); ?>>
									<span class="fr-toggle-slider"></span>
								</label>
							</td>
							<td>
								<div class="fr-action-buttons">
									<button class="button button-small fr-edit-rule" data-rule-id="<?php echo esc_attr( $rule->id ); ?>">
										<?php esc_html_e( 'Edit', 'freedomreader' ); ?>
									</button>
									<button class="button button-small fr-delete-rule" data-rule-id="<?php echo esc_attr( $rule->id ); ?>">
										<?php esc_html_e( 'Delete', 'freedomreader' ); ?>
									</button>
								</div>
							</td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
			</div>
		<?php else : ?>
			<div class="fr-no-rules">
				<p><?php esc_html_e( 'No lock rules created yet.', 'freedomreader' ); ?></p>
				<p><?php esc_html_e( 'Create your first rule to automatically lock content based on various criteria.', 'freedomreader' ); ?></p>
			</div>
		<?php endif; ?>
	</div>
</div>

<!-- Rule Editor Modal -->
<div id="fr-rule-modal" class="fr-modal" style="display: none;">
	<div class="fr-modal-content">
		<div class="fr-modal-header">
			<h2 id="fr-modal-title"><?php esc_html_e( 'Add Lock Rule', 'freedomreader' ); ?></h2>
			<button class="fr-modal-close">&times;</button>
		</div>
		
		<form id="fr-rule-form">
			<input type="hidden" id="fr-rule-id" name="rule_id" value="">
			
			<div class="fr-form-row">
				<label for="fr-rule-name"><?php esc_html_e( 'Rule Name', 'freedomreader' ); ?></label>
				<input type="text" id="fr-rule-name" name="rule_name" required>
				<p class="description"><?php esc_html_e( 'A descriptive name for this rule', 'freedomreader' ); ?></p>
			</div>
			
			<div class="fr-form-row">
				<label for="fr-rule-type"><?php esc_html_e( 'Rule Type', 'freedomreader' ); ?></label>
				<select id="fr-rule-type" name="rule_type" required>
					<option value=""><?php esc_html_e( 'Select rule type...', 'freedomreader' ); ?></option>
					<option value="category"><?php esc_html_e( 'Category', 'freedomreader' ); ?></option>
					<option value="tag"><?php esc_html_e( 'Tag', 'freedomreader' ); ?></option>
					<option value="author"><?php esc_html_e( 'Author', 'freedomreader' ); ?></option>
					<option value="country"><?php esc_html_e( 'Country/Geolocation', 'freedomreader' ); ?></option>
					<option value="time_window"><?php esc_html_e( 'Time Window', 'freedomreader' ); ?></option>
					<option value="user_role"><?php esc_html_e( 'User Role', 'freedomreader' ); ?></option>
					<option value="device_type"><?php esc_html_e( 'Device Type', 'freedomreader' ); ?></option>
					<option value="email_verification"><?php esc_html_e( 'Email Verification', 'freedomreader' ); ?></option>
					<option value="purchase_history"><?php esc_html_e( 'Purchase History', 'freedomreader' ); ?></option>
				</select>
			</div>
			
			<!-- Category Rule -->
			<div id="fr-rule-category" class="fr-rule-config" style="display: none;">
				<label><?php esc_html_e( 'Select Categories', 'freedomreader' ); ?></label>
				<div class="fr-checkbox-list">
					<?php
					$categories = get_categories( array( 'hide_empty' => false ) );
					foreach ( $categories as $category ) :
						?>
						<label>
							<input type="checkbox" name="categories[]" value="<?php echo esc_attr( $category->term_id ); ?>">
							<?php echo esc_html( $category->name ); ?>
						</label>
					<?php endforeach; ?>
				</div>
			</div>
			
			<!-- Tag Rule -->
			<div id="fr-rule-tag" class="fr-rule-config" style="display: none;">
				<label><?php esc_html_e( 'Select Tags', 'freedomreader' ); ?></label>
				<div class="fr-checkbox-list">
					<?php
					$post_tags = get_tags( array( 'hide_empty' => false ) );
					foreach ( $post_tags as $post_tag ) :
						?>
						<label>
							<input type="checkbox" name="tags[]" value="<?php echo esc_attr( $post_tag->term_id ); ?>">
							<?php echo esc_html( $post_tag->name ); ?>
						</label>
					<?php endforeach; ?>
				</div>
			</div>
			
			<!-- Author Rule -->
			<div id="fr-rule-author" class="fr-rule-config" style="display: none;">
				<label><?php esc_html_e( 'Select Authors', 'freedomreader' ); ?></label>
				<div class="fr-checkbox-list">
					<?php
					$authors = get_users( array( 'who' => 'authors' ) );
					foreach ( $authors as $author ) :
						?>
						<label>
							<input type="checkbox" name="authors[]" value="<?php echo esc_attr( $author->ID ); ?>">
							<?php echo esc_html( $author->display_name ); ?>
						</label>
					<?php endforeach; ?>
				</div>
			</div>
			
			<!-- Country Rule -->
			<div id="fr-rule-country" class="fr-rule-config" style="display: none;">
				<div class="fr-form-row">
					<label><?php esc_html_e( 'Country Restriction Type', 'freedomreader' ); ?></label>
					<label>
						<input type="radio" name="country_type" value="allow" checked>
						<?php esc_html_e( 'Allow only these countries', 'freedomreader' ); ?>
					</label>
					<label>
						<input type="radio" name="country_type" value="block">
						<?php esc_html_e( 'Block these countries', 'freedomreader' ); ?>
					</label>
				</div>
				
				<div class="fr-form-row">
					<label for="fr-countries"><?php esc_html_e( 'Countries (comma-separated country codes)', 'freedomreader' ); ?></label>
					<input type="text" id="fr-countries" name="countries" placeholder="US, CA, GB, AU">
					<p class="description"><?php esc_html_e( 'Use 2-letter country codes (ISO 3166-1 alpha-2)', 'freedomreader' ); ?></p>
				</div>
			</div>
			
			<!-- Time Window Rule -->
			<div id="fr-rule-time_window" class="fr-rule-config" style="display: none;">
				<div class="fr-form-row">
					<label for="fr-start-date"><?php esc_html_e( 'Start Date (optional)', 'freedomreader' ); ?></label>
					<input type="date" id="fr-start-date" name="start_date">
				</div>
				
				<div class="fr-form-row">
					<label for="fr-end-date"><?php esc_html_e( 'End Date (optional)', 'freedomreader' ); ?></label>
					<input type="date" id="fr-end-date" name="end_date">
				</div>
				
				<div class="fr-form-row">
					<label for="fr-start-time"><?php esc_html_e( 'Start Time (optional)', 'freedomreader' ); ?></label>
					<input type="time" id="fr-start-time" name="start_time">
				</div>
				
				<div class="fr-form-row">
					<label for="fr-end-time"><?php esc_html_e( 'End Time (optional)', 'freedomreader' ); ?></label>
					<input type="time" id="fr-end-time" name="end_time">
				</div>
				
				<div class="fr-form-row">
					<label><?php esc_html_e( 'Allowed Days', 'freedomreader' ); ?></label>
					<div class="fr-checkbox-list fr-days-list">
						<label><input type="checkbox" name="allowed_days[]" value="0"> <?php esc_html_e( 'Sunday', 'freedomreader' ); ?></label>
						<label><input type="checkbox" name="allowed_days[]" value="1"> <?php esc_html_e( 'Monday', 'freedomreader' ); ?></label>
						<label><input type="checkbox" name="allowed_days[]" value="2"> <?php esc_html_e( 'Tuesday', 'freedomreader' ); ?></label>
						<label><input type="checkbox" name="allowed_days[]" value="3"> <?php esc_html_e( 'Wednesday', 'freedomreader' ); ?></label>
						<label><input type="checkbox" name="allowed_days[]" value="4"> <?php esc_html_e( 'Thursday', 'freedomreader' ); ?></label>
						<label><input type="checkbox" name="allowed_days[]" value="5"> <?php esc_html_e( 'Friday', 'freedomreader' ); ?></label>
						<label><input type="checkbox" name="allowed_days[]" value="6"> <?php esc_html_e( 'Saturday', 'freedomreader' ); ?></label>
					</div>
				</div>
			</div>
			
			<!-- User Role Rule -->
			<div id="fr-rule-user_role" class="fr-rule-config" style="display: none;">
				<label><?php esc_html_e( 'Required User Roles', 'freedomreader' ); ?></label>
				<div class="fr-checkbox-list">
					<?php
					global $wp_roles;
					foreach ( $wp_roles->roles as $role_key => $user_role ) :
						?>
						<label>
							<input type="checkbox" name="user_roles[]" value="<?php echo esc_attr( $role_key ); ?>">
							<?php echo esc_html( $user_role['name'] ); ?>
						</label>
					<?php endforeach; ?>
				</div>
			</div>

			<!-- Device Type Rule -->
			<div id="fr-rule-device_type" class="fr-rule-config" style="display: none;">
				<label><?php esc_html_e( 'Lock on these device types:', 'freedomreader' ); ?></label>
				<div class="fr-checkbox-list">
					<label>
						<input type="checkbox" name="locked_devices[]" value="desktop">
						<?php esc_html_e( 'Desktop', 'freedomreader' ); ?>
					</label>
					<label>
						<input type="checkbox" name="locked_devices[]" value="mobile">
						<?php esc_html_e( 'Mobile', 'freedomreader' ); ?>
					</label>
					<label>
						<input type="checkbox" name="locked_devices[]" value="tablet">
						<?php esc_html_e( 'Tablet', 'freedomreader' ); ?>
					</label>
				</div>
				<p class="description"><?php esc_html_e( 'Content will be locked on selected device types.', 'freedomreader' ); ?></p>
			</div>

			<!-- Email Verification Rule -->
			<div id="fr-rule-email_verification" class="fr-rule-config" style="display: none;">
				<p><?php esc_html_e( 'Users must enter their email address to unlock content. Great for lead generation!', 'freedomreader' ); ?></p>
				<div class="fr-form-row">
					<label for="fr-unlock-message"><?php esc_html_e( 'Unlock Message', 'freedomreader' ); ?></label>
					<textarea id="fr-unlock-message" name="unlock_message" rows="3" placeholder="Enter your email to unlock this premium content..."><?php esc_html_e( 'Enter your email to unlock this premium content...', 'freedomreader' ); ?></textarea>
				</div>
			</div>

			<!-- Purchase History Rule -->
			<div id="fr-rule-purchase_history" class="fr-rule-config" style="display: none;">
				<label><?php esc_html_e( 'Required Product Purchases', 'freedomreader' ); ?></label>
				<p class="description"><?php esc_html_e( 'Users who have purchased any of these products will get free access.', 'freedomreader' ); ?></p>

				<div class="fr-form-row">
					<label for="fr-product-ids"><?php esc_html_e( 'Product IDs (comma-separated)', 'freedomreader' ); ?></label>
					<input type="text" id="fr-product-ids" name="product_ids" placeholder="123, 456, 789">
					<p class="description"><?php esc_html_e( 'Enter WooCommerce product IDs or FreedomReader post IDs.', 'freedomreader' ); ?></p>
				</div>

				<?php if ( class_exists( 'WooCommerce' ) ) : ?>
				<div class="fr-form-row">
					<label><?php esc_html_e( 'Popular WooCommerce Products:', 'freedomreader' ); ?></label>
					<div class="fr-product-suggestions">
						<?php
						// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Admin interface, performance acceptable for small datasets
						$products = get_posts(
							array(
								'post_type'      => 'product',
								'posts_per_page' => 10,
								'meta_key'       => 'total_sales', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Admin interface, performance acceptable for small datasets
								'orderby'        => 'meta_value_num',
								'order'          => 'DESC',
							)
						);

						foreach ( $products as $product ) :
							?>
							<label>
								<input type="checkbox" name="suggested_products[]" value="<?php echo esc_attr( $product->ID ); ?>">
								<?php echo esc_html( $product->post_title ); ?> (ID: <?php echo esc_html( $product->ID ); ?>)
							</label>
						<?php endforeach; ?>
					</div>
				</div>
				<?php endif; ?>
			</div>
			
			<!-- Lock Conditions -->
			<div class="fr-form-section">
				<h3><?php esc_html_e( 'Lock Conditions', 'freedomreader' ); ?></h3>
				<p class="description"><?php esc_html_e( 'What conditions must users meet to access the content?', 'freedomreader' ); ?></p>
				
				<div class="fr-form-row">
					<label>
						<input type="checkbox" name="requires_login" value="1">
						<?php esc_html_e( 'Require user to be logged in', 'freedomreader' ); ?>
					</label>
				</div>
				
				<div class="fr-form-row">
					<label>
						<input type="checkbox" name="requires_subscription" value="1">
						<?php esc_html_e( 'Require active subscription', 'freedomreader' ); ?>
					</label>
				</div>
				
				<div class="fr-form-row">
					<label><?php esc_html_e( 'Required User Roles (for access)', 'freedomreader' ); ?></label>
					<div class="fr-checkbox-list">
						<?php foreach ( $wp_roles->roles as $role_key => $user_role ) : ?>
							<label>
								<input type="checkbox" name="required_roles[]" value="<?php echo esc_attr( $role_key ); ?>">
								<?php echo esc_html( $user_role['name'] ); ?>
							</label>
						<?php endforeach; ?>
					</div>
				</div>
			</div>
			
			<div class="fr-form-row">
				<label for="fr-priority"><?php esc_html_e( 'Priority', 'freedomreader' ); ?></label>
				<input type="number" id="fr-priority" name="priority" value="10" min="1" max="100">
				<p class="description"><?php esc_html_e( 'Lower numbers = higher priority. Rules are evaluated in priority order.', 'freedomreader' ); ?></p>
			</div>
			
			<div class="fr-modal-footer">
				<button type="button" class="button" id="fr-cancel-rule"><?php esc_html_e( 'Cancel', 'freedomreader' ); ?></button>
				<button type="submit" class="button button-primary"><?php esc_html_e( 'Save Rule', 'freedomreader' ); ?></button>
			</div>
		</form>
	</div>
</div>

<?php
// Enqueue lock rules styles - this is a large CSS section so we'll add it via wp_add_inline_style.
$lock_rules_css = '
.fr-lock-rules-header {
	display: flex;
	justify-content: space-between;
	align-items: center;
	margin-bottom: 30px;
	padding: 25px;
	background: #fff;
	border: 1px solid #ddd;
	border-radius: 8px;
	box-shadow: 0 1px 3px rgba(0,0,0,0.1);
}

.fr-header-content {
	flex: 1;
}

.fr-header-content p {
	margin: 0;
	font-size: 14px;
	color: #666;
	line-height: 1.5;
}

.fr-header-actions {
	flex-shrink: 0;
	margin-left: 20px;
}

.fr-rules-table-container {
	background: #fff;
	border: 1px solid #ddd;
	border-radius: 8px;
	overflow: hidden;
	box-shadow: 0 1px 3px rgba(0,0,0,0.1);
}

.fr-rules-table {
	margin: 0;
	border: none;
}

.fr-rules-table thead th {
	background: #f8f9fa;
	border-bottom: 2px solid #dee2e6;
	padding: 15px 20px;
	font-weight: 600;
	color: #333;
	font-size: 14px;
}

.fr-rules-table tbody td {
	padding: 15px 20px;
	vertical-align: middle;
	border-bottom: 1px solid #f0f0f0;
}

.fr-rules-table tbody tr:hover {
	background: #f8f9fa;
}

.fr-col-name { width: 25%; }
.fr-col-type { width: 15%; }
.fr-col-conditions { width: 25%; }
.fr-col-priority { width: 10%; }
.fr-col-status { width: 10%; }
.fr-col-actions { width: 15%; }

.fr-action-buttons {
	display: flex;
	gap: 8px;
}

.fr-action-buttons .button {
	padding: 6px 12px;
	font-size: 12px;
	border-radius: 4px;
}

.fr-edit-rule {
	background: #007cba;
	color: #fff;
	border-color: #007cba;
}

.fr-edit-rule:hover {
	background: #005a87;
	border-color: #005a87;
	color: #fff;
}

.fr-delete-rule {
	background: #dc3545;
	color: #fff;
	border-color: #dc3545;
}

.fr-delete-rule:hover {
	background: #c82333;
	border-color: #c82333;
	color: #fff;
}

.fr-rule-type {
	padding: 4px 8px;
	border-radius: 4px;
	font-size: 12px;
	font-weight: 500;
	text-transform: uppercase;
}

.fr-rule-type-category { background: #e3f2fd; color: #1976d2; }
.fr-rule-type-tag { background: #f3e5f5; color: #7b1fa2; }
.fr-rule-type-author { background: #e8f5e8; color: #388e3c; }
.fr-rule-type-country { background: #fff3e0; color: #f57c00; }
.fr-rule-type-time_window { background: #fce4ec; color: #c2185b; }
.fr-rule-type-user_role { background: #e0f2f1; color: #00695c; }
.fr-rule-type-device_type { background: #f3e5f5; color: #7b1fa2; }
.fr-rule-type-email_verification { background: #e8f5e8; color: #388e3c; }
.fr-rule-type-purchase_history { background: #fff8e1; color: #f9a825; }

.fr-toggle-switch {
	position: relative;
	display: inline-block;
	width: 50px;
	height: 24px;
}

.fr-toggle-switch input {
	opacity: 0;
	width: 0;
	height: 0;
}

.fr-toggle-slider {
	position: absolute;
	cursor: pointer;
	top: 0;
	left: 0;
	right: 0;
	bottom: 0;
	background-color: #ccc;
	transition: .4s;
	border-radius: 24px;
}

.fr-toggle-slider:before {
	position: absolute;
	content: "";
	height: 18px;
	width: 18px;
	left: 3px;
	bottom: 3px;
	background-color: white;
	transition: .4s;
	border-radius: 50%;
}

input:checked + .fr-toggle-slider {
	background-color: #2196F3;
}

input:checked + .fr-toggle-slider:before {
	transform: translateX(26px);
}

.fr-modal {
	position: fixed;
	z-index: 999999;
	left: 0;
	top: 0;
	width: 100%;
	height: 100%;
	background-color: rgba(0,0,0,0.6);
	display: flex;
	align-items: center;
	justify-content: center;
}

.fr-modal-content {
	background-color: #fff;
	border-radius: 12px;
	width: 90%;
	max-width: 900px;
	max-height: 90vh;
	overflow-y: auto;
	box-shadow: 0 10px 30px rgba(0,0,0,0.3);
	animation: fr-modal-appear 0.3s ease-out;
}

@keyframes fr-modal-appear {
	from {
		opacity: 0;
		transform: scale(0.9) translateY(-20px);
	}
	to {
		opacity: 1;
		transform: scale(1) translateY(0);
	}
}

.fr-modal-header {
	padding: 25px 30px;
	border-bottom: 1px solid #e5e5e5;
	display: flex;
	justify-content: space-between;
	align-items: center;
	background: #f8f9fa;
	border-radius: 12px 12px 0 0;
}

.fr-modal-header h2 {
	margin: 0;
	font-size: 20px;
	font-weight: 600;
	color: #333;
}

.fr-modal-close {
	background: none;
	border: none;
	font-size: 28px;
	cursor: pointer;
	color: #666;
	padding: 5px;
	border-radius: 4px;
	transition: all 0.2s ease;
}

.fr-modal-close:hover {
	background: #e9ecef;
	color: #333;
}

#fr-rule-form {
	padding: 30px;
}

.fr-form-row {
	margin-bottom: 25px;
}

.fr-form-row label {
	display: block;
	margin-bottom: 8px;
	font-weight: 600;
	color: #333;
	font-size: 14px;
}

.fr-form-row input[type="text"],
.fr-form-row input[type="email"],
.fr-form-row input[type="number"],
.fr-form-row input[type="date"],
.fr-form-row input[type="time"],
.fr-form-row select,
.fr-form-row textarea {
	width: 100%;
	padding: 12px 16px;
	border: 2px solid #e1e5e9;
	border-radius: 8px;
	font-size: 14px;
	transition: border-color 0.2s ease;
	background: #fff;
}

.fr-form-row input:focus,
.fr-form-row select:focus,
.fr-form-row textarea:focus {
	outline: none;
	border-color: #007cba;
	box-shadow: 0 0 0 3px rgba(0, 124, 186, 0.1);
}

.fr-form-row .description {
	margin-top: 8px;
	font-size: 13px;
	color: #666;
	font-style: italic;
}

.fr-rule-config {
	margin: 25px 0;
	padding: 20px;
	background: #f8f9fa;
	border: 1px solid #e9ecef;
	border-radius: 8px;
}

.fr-checkbox-list {
	display: grid;
	grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
	gap: 12px;
	margin-top: 15px;
}

.fr-checkbox-list label {
	display: flex;
	align-items: center;
	padding: 8px 12px;
	background: #fff;
	border: 1px solid #ddd;
	border-radius: 6px;
	cursor: pointer;
	transition: all 0.2s ease;
	font-weight: normal;
	margin-bottom: 0;
}

.fr-checkbox-list label:hover {
	background: #f0f8ff;
	border-color: #007cba;
}

.fr-checkbox-list input[type="checkbox"] {
	margin-right: 8px;
	width: auto;
}

.fr-days-list {
	grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
}

.fr-form-section {
	margin: 35px 0;
	padding: 25px;
	border: 2px solid #e9ecef;
	border-radius: 10px;
	background: #f8f9fa;
}

.fr-form-section h3 {
	margin: 0 0 15px 0;
	color: #333;
	font-size: 16px;
	font-weight: 600;
	padding-bottom: 10px;
	border-bottom: 2px solid #dee2e6;
}

.fr-form-section .description {
	margin-bottom: 20px;
	color: #666;
	font-size: 14px;
}

.fr-modal-footer {
	padding: 25px 30px;
	border-top: 1px solid #e5e5e5;
	text-align: right;
	background: #f8f9fa;
	border-radius: 0 0 12px 12px;
}

.fr-modal-footer .button {
	margin-left: 12px;
	padding: 10px 20px;
	font-size: 14px;
	border-radius: 6px;
	font-weight: 500;
}

.fr-modal-footer .button-primary {
	background: #007cba;
	border-color: #007cba;
	box-shadow: 0 2px 4px rgba(0, 124, 186, 0.2);
}

.fr-modal-footer .button-primary:hover {
	background: #005a87;
	border-color: #005a87;
}

.fr-no-rules {
	text-align: center;
	padding: 40px;
	background: #f9f9f9;
	border-radius: 6px;
}

.fr-product-suggestions {
	max-height: 200px;
	overflow-y: auto;
	border: 1px solid #ddd;
	border-radius: 4px;
	padding: 10px;
	background: #fff;
}

.fr-product-suggestions label {
	display: block;
	padding: 5px 0;
	font-size: 12px;
}

.fr-form-row textarea {
	width: 100%;
	padding: 8px;
	border: 1px solid #ddd;
	border-radius: 4px;
	resize: vertical;
}';
wp_add_inline_style( 'freedomreader-admin', $lock_rules_css );
?>
