<?php
/**
 * Admin functionality for FreedomReader
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin functionality for FreedomReader
 *
 * @package FreedomReader
 */
class FREEDO_Admin {

	/**
	 * Singleton instance
	 *
	 * @var FREEDO_Admin|null
	 */
	private static $instance = null;

	/**
	 * Database instance
	 *
	 * @var FREEDO_Database
	 */
	private $db;

	/**
	 * Get singleton instance
	 *
	 * @return FREEDO_Admin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->db = FREEDO_Database::get_instance();
		$this->init_hooks();
	}

	/**
	 * Initialize admin hooks
	 *
	 * @package FreedomReader
	 */
	private function init_hooks() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
		add_action( 'add_meta_boxes', array( $this, 'add_content_lock_metabox' ) );
		add_action( 'save_post', array( $this, 'save_content_lock_settings' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
		add_filter( 'plugin_action_links_' . FREEDOMREADER_PLUGIN_BASENAME, array( $this, 'add_plugin_action_links' ) );

		add_action( 'wp_ajax_freedomreader_save_lock_rule', array( $this, 'save_lock_rule' ) );
		add_action( 'wp_ajax_freedomreader_test_ajax', array( $this, 'test_ajax' ) );
		add_action( 'wp_ajax_freedomreader_delete_lock_rule', array( $this, 'delete_lock_rule' ) );
		add_action( 'wp_ajax_freedomreader_toggle_lock_rule', array( $this, 'toggle_lock_rule' ) );
		add_action( 'wp_ajax_freedomreader_get_lock_rule', array( $this, 'get_lock_rule' ) );

		add_action( 'wp_ajax_freedomreader_export_data', array( $this, 'export_data' ) );

		add_action( 'wp_ajax_freedomreader_dismiss_support_notice', array( $this, 'dismiss_support_notice' ) );

		add_action( 'wp_ajax_freedomreader_export_all_data', array( $this, 'export_all_data' ) );
		add_action( 'wp_ajax_freedomreader_cleanup_logs', array( $this, 'cleanup_logs' ) );
		add_action( 'wp_ajax_freedomreader_verify_data', array( $this, 'verify_data_integrity' ) );
		add_action( 'wp_ajax_freedomreader_create_backup', array( $this, 'create_manual_backup' ) );
		add_action( 'wp_ajax_freedomreader_import_data', array( $this, 'import_backup_data' ) );
		add_action( 'wp_ajax_freedomreader_performance_check', array( $this, 'run_performance_check' ) );
		add_action( 'wp_ajax_freedomreader_bulk_operations', array( $this, 'handle_bulk_operations' ) );

		add_action( 'wp_ajax_freedomreader_validate_paypal', array( $this, 'validate_paypal_credentials' ) );

		add_action( 'wp_ajax_freedomreader_refund_purchase', array( $this, 'refund_purchase' ) );
	}

	/**
	 * Add admin menu
	 *
	 * @package FreedomReader
	 */
	public function add_admin_menu() {
		add_menu_page(
			esc_html__( 'FreedomReader', 'freedomreader' ),
			esc_html__( 'FreedomReader', 'freedomreader' ),
			'manage_options',
			'freedomreader',
			array( $this, 'admin_dashboard' ),
			'dashicons-lock',
			30
		);

		add_submenu_page(
			'freedomreader',
			esc_html__( 'Dashboard', 'freedomreader' ),
			esc_html__( 'Dashboard', 'freedomreader' ),
			'manage_options',
			'freedomreader',
			array( $this, 'admin_dashboard' )
		);

		add_submenu_page(
			'freedomreader',
			esc_html__( 'Settings', 'freedomreader' ),
			esc_html__( 'Settings', 'freedomreader' ),
			'manage_options',
			'freedomreader-settings',
			array( $this, 'admin_settings' )
		);

		add_submenu_page(
			'freedomreader',
			esc_html__( 'Reports', 'freedomreader' ),
			esc_html__( 'Reports', 'freedomreader' ),
			'manage_options',
			'freedomreader-reports',
			array( $this, 'admin_reports' )
		);

		add_submenu_page(
			'freedomreader',
			esc_html__( 'Purchases', 'freedomreader' ),
			esc_html__( 'Purchases', 'freedomreader' ),
			'manage_options',
			'freedomreader-purchases',
			array( $this, 'admin_purchases' )
		);

		add_submenu_page(
			'freedomreader',
			esc_html__( 'Lock Rules', 'freedomreader' ),
			esc_html__( 'Lock Rules', 'freedomreader' ),
			'manage_options',
			'freedomreader-lock-rules',
			array( $this, 'lock_rules_page' )
		);

		add_submenu_page(
			'freedomreader',
			esc_html__( 'Subscriptions', 'freedomreader' ),
			esc_html__( 'Subscriptions', 'freedomreader' ),
			'manage_options',
			'freedomreader-subscriptions',
			array( $this, 'admin_subscriptions' )
		);

		add_submenu_page(
			'freedomreader',
			esc_html__( 'Documentation', 'freedomreader' ),
			esc_html__( 'Documentation', 'freedomreader' ),
			'manage_options',
			'freedomreader-documentation',
			array( $this, 'admin_documentation' )
		);

		// Bulk Operations menu - temporarily hidden.
		// add_submenu_page(
		// 'freedomreader',
		// esc_html__( 'Bulk Operations', 'freedomreader' ),
		// esc_html__( 'Bulk Operations', 'freedomreader' ),
		// 'manage_options',
		// 'freedomreader-bulk-operations',
		// array( $this, 'admin_bulk_operations' ).
		// );.

		add_submenu_page(
			'freedomreader',
			esc_html__( 'Help & Support', 'freedomreader' ),
			esc_html__( 'Help & Support', 'freedomreader' ),
			'manage_options',
			'freedomreader-help-support',
			array( $this, 'admin_help_support' )
		);
	}

	/**
	 * Enqueue admin scripts and styles
	 *
	 * @param string $hook The current admin page hook.
	 * @package FreedomReader
	 */
	public function enqueue_admin_scripts( $hook ) {
		// Only load on FreedomReader admin pages.
		if ( false === strpos( $hook, 'freedomreader' ) && 'post.php' !== $hook && 'post-new.php' !== $hook ) {
			return;
		}

		wp_enqueue_script(
			'freedomreader-admin',
			FREEDOMREADER_PLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			FREEDOMREADER_VERSION,
			true
		);

		wp_enqueue_style(
			'freedomreader-admin',
			FREEDOMREADER_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			FREEDOMREADER_VERSION
		);

		// Enqueue page-specific scripts and styles.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking current admin page for script loading
		$current_page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';

		if ( 'freedomreader-purchases' === $current_page ) {
			wp_enqueue_script( 'freedomreader-purchases', FREEDOMREADER_PLUGIN_URL . 'admin/assets/js/purchases.js', array( 'jquery' ), FREEDOMREADER_VERSION, true );
			wp_enqueue_style( 'freedomreader-purchases', FREEDOMREADER_PLUGIN_URL . 'admin/assets/css/purchases.css', array(), FREEDOMREADER_VERSION );

			wp_localize_script(
				'freedomreader-purchases',
				'freedomreader_purchases',
				array(
					'nonce'           => wp_create_nonce( 'freedomreader_admin_nonce' ),
					'paypal_mode'     => get_option( 'freedomreader_paypal_mode', 'sandbox' ),
					'confirm_refund'  => esc_html__( 'Are you sure you want to refund this purchase? This action cannot be undone.', 'freedomreader' ),
					'processing_text' => esc_html__( 'Processing...', 'freedomreader' ),
					'refund_success'  => esc_html__( 'Purchase refunded successfully.', 'freedomreader' ),
					'refund_failed'   => esc_html__( 'Refund failed: ', 'freedomreader' ),
					'refund_text'     => esc_html__( 'Refund', 'freedomreader' ),
					'error_occurred'  => esc_html__( 'An error occurred. Please try again.', 'freedomreader' ),
				)
			);
		}

		if ( 'freedomreader-subscriptions' === $current_page ) {
			wp_enqueue_script( 'freedomreader-subscriptions', FREEDOMREADER_PLUGIN_URL . 'admin/assets/js/subscriptions.js', array( 'jquery' ), FREEDOMREADER_VERSION, true );

			wp_localize_script(
				'freedomreader-subscriptions',
				'freedomreader_subscriptions',
				array(
					'nonce'           => wp_create_nonce( 'freedomreader_admin_nonce' ),
					'paypal_mode'     => get_option( 'freedomreader_paypal_mode', 'sandbox' ),
					'confirm_cancel'  => esc_html__( 'Are you sure you want to cancel this subscription?', 'freedomreader' ),
					'processing_text' => esc_html__( 'Processing...', 'freedomreader' ),
					'cancel_success'  => esc_html__( 'Subscription cancelled successfully.', 'freedomreader' ),
					'cancel_failed'   => esc_html__( 'Cancellation failed: ', 'freedomreader' ),
					'cancel_text'     => esc_html__( 'Cancel', 'freedomreader' ),
					'error_occurred'  => esc_html__( 'An error occurred. Please try again.', 'freedomreader' ),
				)
			);
		}

		if ( 'freedomreader-reports' === $current_page ) {

			wp_enqueue_script( 'jquery' );
		}

		if ( 'freedomreader-lock-rules' === $current_page ) {
			wp_enqueue_script(
				'freedomreader-lock-rules',
				FREEDOMREADER_PLUGIN_URL . 'assets/js/lock-rules.js',
				array( 'jquery' ),
				FREEDOMREADER_VERSION,
				true
			);

			wp_localize_script(
				'freedomreader-lock-rules',
				'freedomreader_admin',
				array(
					'ajax_url' => esc_url( admin_url( 'admin-ajax.php' ) ),
					'nonce'    => wp_create_nonce( 'freedomreader_admin_nonce' ),
				)
			);
		}

		// Enqueue admin script and styles for all FreedomReader admin pages.
		wp_enqueue_script(
			'freedomreader-admin',
			FREEDOMREADER_PLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			FREEDOMREADER_VERSION,
			true
		);

		wp_enqueue_style(
			'freedomreader-admin',
			FREEDOMREADER_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			FREEDOMREADER_VERSION
		);

		// Localize script AFTER enqueuing.
		wp_localize_script(
			'freedomreader-admin',
			'freedomreader_admin',
			array(
				'ajax_url' => esc_url( admin_url( 'admin-ajax.php' ) ),
				'nonce'    => wp_create_nonce( 'freedomreader_admin_nonce' ),
				'messages' => array(
					'confirm_delete' => esc_html__( 'Are you sure you want to delete this item?', 'freedomreader' ),
					'saving'         => esc_html__( 'Saving...', 'freedomreader' ),
					'saved'          => esc_html__( 'Saved!', 'freedomreader' ),
					'error'          => esc_html__( 'An error occurred. Please try again.', 'freedomreader' ),
				),
			)
		);

		// Add inline styles for dashboard page.
		if ( 'freedomreader-dashboard' === $current_page ) {
			$dashboard_css = '
				.fr-dashboard-header {
					display: flex;
					justify-content: space-between;
					align-items: center;
					margin-bottom: 20px;
				}
				.fr-dashboard-header h1 {
					margin: 0;
				}
				.fr-support-widget {
					opacity: 0.8;
					transition: opacity 0.3s ease;
				}
				.fr-support-widget:hover {
					opacity: 1;
				}
				#fr-coffee-button-dashboard {
					transform: scale(0.9);
				}
				.fr-dashboard-stats {
					margin-bottom: 30px;
				}
				.fr-stat-cards {
					display: grid;
					grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
					gap: 20px;
					margin-bottom: 30px;
				}
				.fr-stat-card {
					background: #fff;
					border: 1px solid #ddd;
					border-radius: 8px;
					padding: 20px;
					text-align: center;
					box-shadow: 0 2px 4px rgba(0,0,0,0.1);
				}
				.fr-stat-card h3 {
					margin: 0 0 10px 0;
					font-size: 14px;
					color: #666;
					text-transform: uppercase;
					letter-spacing: 0.5px;
				}
				.fr-stat-value {
					font-size: 32px;
					font-weight: bold;
					color: #2271b1;
					margin-bottom: 5px;
				}
				.fr-stat-change {
					font-size: 12px;
					color: #666;
				}
				.fr-dashboard-content {
					display: grid;
					grid-template-columns: 2fr 1fr;
					gap: 30px;
				}
				.fr-dashboard-widget {
					background: #fff;
					border: 1px solid #ddd;
					border-radius: 8px;
					padding: 20px;
					margin-bottom: 20px;
				}
				.fr-dashboard-widget h2 {
					margin: 0 0 15px 0;
					font-size: 18px;
					border-bottom: 1px solid #eee;
					padding-bottom: 10px;
				}
				.fr-quick-actions {
					display: flex;
					gap: 10px;
					flex-wrap: wrap;
				}
				.fr-activity-list {
					list-style: none;
					margin: 0;
					padding: 0;
				}
				.fr-activity-item {
					display: flex;
					align-items: flex-start;
					padding: 10px 0;
					border-bottom: 1px solid #eee;
				}
				.fr-activity-item:last-child {
					border-bottom: none;
				}
				.fr-activity-icon {
					margin-right: 10px;
					font-size: 18px;
				}
				.fr-activity-content {
					flex: 1;
				}
				.fr-activity-meta {
					font-size: 12px;
					color: #666;
					margin-top: 5px;
				}
				.fr-system-status {
					margin-bottom: 10px;
				}
				.fr-status-item {
					display: flex;
					justify-content: space-between;
					align-items: center;
					padding: 8px 0;
					border-bottom: 1px solid #eee;
				}
				.fr-status-item:last-child {
					border-bottom: none;
				}
				.fr-status-label {
					font-weight: 500;
				}
				.fr-status-value {
					font-size: 12px;
					padding: 4px 8px;
					border-radius: 4px;
				}
				.fr-status-ok {
					background: #d4edda;
					color: #155724;
				}
				.fr-status-error {
					background: #f8d7da;
					color: #721c24;
				}
				.fr-plans-list {
					list-style: none;
					margin: 0;
					padding: 0;
				}
				.fr-plan-item {
					display: flex;
					justify-content: space-between;
					align-items: center;
					padding: 8px 0;
					border-bottom: 1px solid #eee;
				}
				.fr-plan-item:last-child {
					border-bottom: none;
				}
				.fr-plan-price {
					font-size: 14px;
					color: #2271b1;
					font-weight: 500;
				}
				.fr-help-links ul {
					list-style: none;
					margin: 0;
					padding: 0;
				}
				.fr-help-links li {
					margin-bottom: 8px;
				}
				.fr-help-links a {
					text-decoration: none;
					color: #2271b1;
				}
				.fr-help-links a:hover {
					text-decoration: underline;
				}
				@media (max-width: 768px) {
					.fr-dashboard-content {
						grid-template-columns: 1fr;
					}
					.fr-stat-cards {
						grid-template-columns: 1fr;
					}
					.fr-quick-actions {
						flex-direction: column;
					}
				}
			';
			wp_add_inline_style( 'freedomreader-admin', $dashboard_css );

			// Add Buy Me a Coffee script for dashboard.
			$dashboard_js = '
				document.addEventListener("DOMContentLoaded", function() {
					var script = document.createElement("script");
					script.type = "text/javascript";
					script.src = "https://cdnjs.buymeacoffee.com/1.0.0/button.prod.min.js";
					script.setAttribute("data-name", "bmc-button");
					script.setAttribute("data-slug", "kiranrs1432");
					script.setAttribute("data-color", "#40DCA5");
					script.setAttribute("data-emoji", "");
					script.setAttribute("data-font", "Cookie");
					script.setAttribute("data-text", "Support");
					script.setAttribute("data-outline-color", "#000000");
					script.setAttribute("data-font-color", "#ffffff");
					script.setAttribute("data-coffee-color", "#FFDD00");

					var container = document.getElementById("fr-coffee-button-dashboard");
					if (container) {
						container.appendChild(script);
					}
				});
			';
			wp_add_inline_script( 'freedomreader-admin', $dashboard_js );
		}
	}

	/**
	 * Admin dashboard page
	 *
	 * @package FreedomReader
	 */
	public function admin_dashboard() {

		$stats = $this->get_dashboard_stats();

		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/dashboard.php';
	}

	/**
	 * Admin settings page
	 *
	 * @package FreedomReader
	 */
	public function admin_settings() {

		if ( isset( $_POST['submit'] ) && isset( $_POST['freedomreader_settings_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['freedomreader_settings_nonce'] ) ), 'freedomreader_settings' ) ) {
			$this->save_settings();
			add_settings_error( 'freedomreader_settings', 'settings_saved', esc_html__( 'Settings saved successfully!', 'freedomreader' ), 'updated' );
		}

		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/settings.php';
	}

	/**
	 * Admin reports page
	 *
	 * @package FreedomReader
	 */
	public function admin_reports() {
		$reports     = FREEDO_Admin_Reports::get_instance();
		$report_data = $reports->get_reports_data();

		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/reports.php';
	}

	/**
	 * Admin purchases page
	 *
	 * @package FreedomReader
	 */
	public function admin_purchases() {
		global $wpdb;

		$per_page = 20;
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Pagination parameter for admin page
		$page   = isset( $_GET['paged'] ) ? max( 1, intval( sanitize_text_field( wp_unslash( $_GET['paged'] ) ) ) ) : 1;
		$offset = ( $page - 1 ) * $per_page;

		$cache_key = 'freedo_purchases_page_' . $page . '_per_page_' . $per_page;
		$purchases = wp_cache_get( $cache_key );

		if ( false === $purchases ) {
			// Get purchases with user and post data.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for admin display
			$purchases = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT p.*, u.display_name, u.user_email, po.post_title
					FROM {$wpdb->prefix}freedo_purchases p
					LEFT JOIN {$wpdb->users} u ON p.user_id = u.ID
					LEFT JOIN {$wpdb->posts} po ON p.post_id = po.ID
					ORDER BY p.created_at DESC
					LIMIT %d OFFSET %d",
					$per_page,
					$offset
				)
			);
			wp_cache_set( $cache_key, $purchases, '', 300 );
		}

		$total_cache_key = 'freedo_purchases_total_count';
		$total_items     = wp_cache_get( $total_cache_key );

		if ( false === $total_items ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table count for admin pagination
			$total_items = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}freedo_purchases" );
			wp_cache_set( $total_cache_key, $total_items, '', 300 );
		}

		$total_pages = ceil( $total_items / $per_page );

		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/purchases.php';
	}

	/**
	 * Admin subscriptions page
	 *
	 * @package FreedomReader
	 */
	public function admin_subscriptions() {
		global $wpdb;

		$per_page = 20;
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Pagination parameter for admin page
		$page   = isset( $_GET['paged'] ) ? max( 1, intval( sanitize_text_field( wp_unslash( $_GET['paged'] ) ) ) ) : 1;
		$offset = ( $page - 1 ) * $per_page;

		$cache_key     = 'freedo_subscriptions_page_' . $page . '_per_page_' . $per_page;
		$subscriptions = wp_cache_get( $cache_key );

		if ( false === $subscriptions ) {

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for admin display
			$subscriptions = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT s.*, u.display_name, u.user_email
					FROM {$wpdb->prefix}freedo_subscriptions s
					LEFT JOIN {$wpdb->users} u ON s.user_id = u.ID
					ORDER BY s.created_at DESC
					LIMIT %d OFFSET %d",
					$per_page,
					$offset
				)
			);
			wp_cache_set( $cache_key, $subscriptions, '', 300 );
		}

		$total_cache_key = 'freedo_subscriptions_total_count';
		$total_items     = wp_cache_get( $total_cache_key );

		if ( false === $total_items ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table count for admin pagination
			$total_items = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions" );
			wp_cache_set( $total_cache_key, $total_items, '', 300 );
		}

		$total_pages = ceil( $total_items / $per_page );

		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/subscriptions.php';
	}

	/**
	 * Display documentation page
	 *
	 * @package FreedomReader
	 */
	public function admin_documentation() {
		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/documentation.php';
	}

	/**
	 * Display bulk operations page
	 *
	 * @package FreedomReader
	 */
	public function admin_bulk_operations() {
		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/bulk-operations.php';
	}

	/**
	 * Display help & support page
	 *
	 * @package FreedomReader
	 */
	public function admin_help_support() {
		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/help-support.php';
	}

	/**
	 * Show support notice occasionally
	 *
	 * @package FreedomReader
	 */
	public function show_support_notice() {
		// Only show on FreedomReader pages.
		$screen = get_current_screen();
		if ( ! $screen || strpos( $screen->id, 'freedomreader' ) === false ) {
			return;
		}

		$last_shown    = get_option( 'freedomreader_support_notice_last_shown', 0 );
		$show_interval = 5 * DAY_IN_SECONDS;

		if ( ( time() - $last_shown ) < $show_interval ) {
			return;
		}

		$dismissed = get_option( 'freedomreader_support_notice_dismissed', 0 );
		if ( ( time() - $dismissed ) < $show_interval ) {
			return;
		}

		?>
		<div class="notice notice-info is-dismissible fr-support-notice" data-notice="support">
			<div style="display: flex; align-items: center; padding: 10px 0;">
				<div style="flex: 1;">
					<h3 style="margin: 0 0 5px 0;"><?php esc_html_e( '☕ Enjoying FreedomReader?', 'freedomreader' ); ?></h3>
					<p style="margin: 0;"><?php esc_html_e( 'If this plugin is helping you monetize your content, consider supporting its development!', 'freedomreader' ); ?></p>
				</div>
				<div style="margin-left: 20px;">
					<div id="fr-coffee-button-notice"></div>
				</div>
			</div>
		</div>
		<?php
		// Enqueue support notice script.
		$support_notice_js = '
		document.addEventListener("DOMContentLoaded", function() {

			var script = document.createElement("script");
			script.type = "text/javascript";
			script.src = "https://cdnjs.buymeacoffee.com/1.0.0/button.prod.min.js";
			script.setAttribute("data-name", "bmc-button");
			script.setAttribute("data-slug", "kiranrs1432");
			script.setAttribute("data-color", "#40DCA5");
			script.setAttribute("data-emoji", "☕");
			script.setAttribute("data-font", "Cookie");
			script.setAttribute("data-text", "Buy me a coffee");
			script.setAttribute("data-outline-color", "#000000");
			script.setAttribute("data-font-color", "#ffffff");
			script.setAttribute("data-coffee-color", "#FFDD00");

			var container = document.getElementById("fr-coffee-button-notice");
			if (container) {
				container.appendChild(script);
			}

			jQuery(document).on("click", ".fr-support-notice .notice-dismiss", function() {
				jQuery.post(ajaxurl, {
					action: "freedo_dismiss_support_notice",
					nonce: "' . esc_js( wp_create_nonce( 'freedo_dismiss_notice' ) ) . '"
				});
			});
		});';
		wp_add_inline_script( 'freedomreader-admin', $support_notice_js );
		?>
		<?php

		// Update last shown time.
		update_option( 'freedomreader_support_notice_last_shown', time() );
	}

	/**
	 * Handle support notice dismissal
	 *
	 * @package FreedomReader
	 */
	public function dismiss_support_notice() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedo_dismiss_notice' ) ) {
			wp_send_json_error( 'Security check failed' );
		}

		update_option( 'freedomreader_support_notice_dismissed', time() );

		wp_send_json_success();
	}

	/**
	 * Add content lock metabox
	 *
	 * @package FreedomReader
	 */
	public function add_content_lock_metabox() {
		$post_types = get_post_types( array( 'public' => true ), 'names' );

		foreach ( $post_types as $post_type ) {
			add_meta_box(
				'freedomreader_content_lock',
				esc_html__( 'FreedomReader Content Lock', 'freedomreader' ),
				array( $this, 'content_lock_metabox_callback' ),
				$post_type,
				'side',
				'high'
			);
		}
	}

	/**
	 * Content lock metabox callback
	 *
	 * @param WP_Post $post The post object.
	 * @package FreedomReader
	 */
	public function content_lock_metabox_callback( $post ) {

		$lock_settings = $this->db->get_content_lock( $post->ID );

		wp_nonce_field( 'freedomreader_content_lock', 'freedomreader_content_lock_nonce' );

		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/content-lock-metabox.php';
	}

	/**
	 * Save content lock settings
	 *
	 * @param int $post_id The post ID.
	 * @package FreedomReader
	 */
	public function save_content_lock_settings( $post_id ) {

		if ( ! isset( $_POST['freedomreader_content_lock_nonce'] ) ||
			! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['freedomreader_content_lock_nonce'] ) ), 'freedomreader_content_lock' ) ) {
			return;
		}

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		$enable_lock = isset( $_POST['freedo_enable_lock'] ) ? 1 : 0;

		if ( 1 === $enable_lock ) {
			$lock_data = array(
				'lock_type'             => sanitize_text_field( wp_unslash( $_POST['freedo_lock_type'] ?? 'full' ) ),
				'price'                 => floatval( wp_unslash( $_POST['freedo_price'] ?? 0 ) ),
				'teaser_length'         => intval( wp_unslash( $_POST['freedo_teaser_length'] ?? 150 ) ),
				'subscription_required' => isset( $_POST['freedo_subscription_required'] ) ? 1 : 0,
			);

			$result = $this->db->save_content_lock( $post_id, $lock_data );

			if ( false === $result ) {
				add_action(
					'admin_notices',
					function () {
						echo '<div class="notice notice-error"><p>Failed to save content lock settings.</p></div>';
					}
				);
			}
		} else {

			global $wpdb;
			$table = $wpdb->prefix . 'freedo_content_locks';
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table delete operation
			$wpdb->delete( $table, array( 'post_id' => $post_id ) );
		}

		wp_cache_delete( 'freedomreader_dashboard_stats' );
		wp_cache_delete( 'freedomreader_content_lock_' . $post_id );
	}

	/**
	 * Get dashboard statistics
	 *
	 * @package FreedomReader
	 */
	private function get_dashboard_stats() {
		// Check cache first.
		$cache_key = 'freedomreader_dashboard_stats';
		$stats     = wp_cache_get( $cache_key );

		if ( false === $stats ) {
			global $wpdb;

			$stats = array();

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for dashboard stats
			$total_revenue          = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT SUM(amount) FROM {$wpdb->prefix}freedo_purchases WHERE status = %s",
					'completed'
				)
			);
			$stats['total_revenue'] = $total_revenue ? $total_revenue : 0;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for dashboard stats
			$monthly_revenue          = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT SUM(amount) FROM {$wpdb->prefix}freedo_purchases
					WHERE status = %s AND MONTH(created_at) = MONTH(NOW()) AND YEAR(created_at) = YEAR(NOW())",
					'completed'
				)
			);
			$stats['monthly_revenue'] = $monthly_revenue ? $monthly_revenue : 0;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for dashboard stats
			$active_subscriptions          = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions WHERE status = %s",
					'active'
				)
			);
			$stats['active_subscriptions'] = $active_subscriptions ? $active_subscriptions : 0;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for dashboard stats
			$trial_subscriptions          = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions WHERE status = %s",
					'trial'
				)
			);
			$stats['trial_subscriptions'] = $trial_subscriptions ? $trial_subscriptions : 0;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for dashboard stats
			$total_purchases          = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_purchases WHERE status = %s",
					'completed'
				)
			);
			$stats['total_purchases'] = $total_purchases ? $total_purchases : 0;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for dashboard stats
			$monthly_purchases          = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_purchases
					WHERE status = %s AND MONTH(created_at) = MONTH(NOW()) AND YEAR(created_at) = YEAR(NOW())",
					'completed'
				)
			);
			$stats['monthly_purchases'] = $monthly_purchases ? $monthly_purchases : 0;

			wp_cache_set( $cache_key, $stats, '', 300 );
		}

		return $stats;
	}

	/**
	 * Save settings
	 *
	 * @package FreedomReader
	 */
	private function save_settings() {
		if ( ! isset( $_POST['freedomreader_settings_nonce'] ) ||
			! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['freedomreader_settings_nonce'] ) ), 'freedomreader_settings' ) ) {
			return;
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$settings = array(
			'paypal_mode',
			'paypal_client_id',
			'paypal_client_secret',
			'currency',
			'trial_days',
			'unlock_button_text',
			'teaser_message',
			'keep_data_on_uninstall',
			'auto_cleanup',
			'export_format',
			'backup_frequency',
			'auto_backup',
			'backup_retention',
			'performance_monitoring',
			'user_activity_logs',
			'email_new_purchases',
			'email_new_subscriptions',
			'email_failed_payments',
			'email_performance_alerts',
			// Content Settings.
			'show_progress_bar',
			'show_reading_time',
			'reading_time_mode',
			'show_ask_author',
			'auto_internal_links',
		);

		foreach ( $settings as $setting ) {
			$field_name = 'freedomreader_' . $setting;
			if ( isset( $_POST[ $field_name ] ) ) {
				$value = sanitize_text_field( wp_unslash( $_POST[ $field_name ] ) );
				update_option( $field_name, $value );
			} else {
				// For checkboxes, if not set in POST, it means unchecked (value = 0).
				$checkbox_fields = array(
					'keep_data_on_uninstall',
					'auto_cleanup',
					'auto_backup',
					'performance_monitoring',
					'user_activity_logs',
					'email_new_purchases',
					'email_new_subscriptions',
					'email_failed_payments',
					'email_performance_alerts',
					'show_progress_bar',
					'show_reading_time',
					'show_ask_author',
					'auto_internal_links',
				);
				if ( in_array( $setting, $checkbox_fields, true ) ) {
					update_option( $field_name, '0' );
				}
			}
		}

		if ( isset( $_POST['subscription_plans'] ) && is_array( $_POST['subscription_plans'] ) ) {
			$plans              = array();
			$subscription_plans = map_deep( wp_unslash( $_POST['subscription_plans'] ), 'sanitize_text_field' );
			foreach ( $subscription_plans as $plan_id => $plan_data ) {
				if ( is_array( $plan_data ) && ! empty( $plan_data['name'] ) ) {
					$plans[ sanitize_key( $plan_id ) ] = array(
						'name'     => sanitize_text_field( $plan_data['name'] ?? '' ),
						'price'    => floatval( $plan_data['price'] ?? 0 ),
						'interval' => sanitize_text_field( $plan_data['interval'] ?? 'month' ),
					);
				}
			}
			update_option( 'freedomreader_subscription_plans', $plans );
		}

		// Save auto-link keywords.
		if ( isset( $_POST['auto_link_keywords'] ) && is_array( $_POST['auto_link_keywords'] ) &&
			isset( $_POST['auto_link_post_ids'] ) && is_array( $_POST['auto_link_post_ids'] ) ) {
			$keywords      = array();
			$keyword_list  = array_map( 'sanitize_text_field', wp_unslash( $_POST['auto_link_keywords'] ) );
			$post_id_list  = array_map( 'intval', wp_unslash( $_POST['auto_link_post_ids'] ) );
			$keyword_count = count( $keyword_list );

			for ( $i = 0; $i < $keyword_count; $i++ ) {
				if ( ! empty( $keyword_list[ $i ] ) && ! empty( $post_id_list[ $i ] ) ) {
					$keywords[ $keyword_list[ $i ] ] = $post_id_list[ $i ];
				}
			}
			update_option( 'freedomreader_auto_link_keywords', $keywords );
		}

		wp_cache_delete( 'freedomreader_dashboard_stats' );
	}

	/**
	 * Add plugin action links
	 *
	 * @param array $links Existing plugin action links.
	 * @return array Modified plugin action links.
	 * @package FreedomReader
	 */
	public function add_plugin_action_links( $links ) {
		$settings_link = '<a href="' . esc_url( admin_url( 'admin.php?page=freedomreader-settings' ) ) . '">' . esc_html__( 'Settings', 'freedomreader' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Admin notices
	 *
	 * @package FreedomReader
	 */
	public function admin_notices() {

		$this->show_support_notice();

		$client_id     = get_option( 'freedomreader_paypal_client_id' );
		$client_secret = get_option( 'freedomreader_paypal_client_secret' );

		if ( empty( $client_id ) || empty( $client_secret ) ) {
			echo '<div class="notice notice-warning is-dismissible">';
			/* translators: %s: URL to the settings page */
			$message = esc_html__( 'FreedomReader: Please configure your PayPal credentials in the <a href="%s">settings page</a> to start accepting payments.', 'freedomreader' );
			echo '<p>' . wp_kses_post( sprintf( $message, esc_url( admin_url( 'admin.php?page=freedomreader-settings' ) ) ) ) . '</p>';
			echo '</div>';
		}

		if ( ! get_option( 'users_can_register' ) ) {
			echo '<div class="notice notice-info is-dismissible">';
			/* translators: %s: URL to the WordPress general settings page */
			$message = esc_html__( 'FreedomReader: User registration is currently disabled. Users won\'t be able to create accounts to purchase content. <a href="%s">Enable user registration</a> in WordPress settings.', 'freedomreader' );
			echo '<p>' . wp_kses_post( sprintf( $message, esc_url( admin_url( 'options-general.php' ) ) ) ) . '</p>';
			echo '</div>';
		}
	}

	/**
	 * Lock rules page
	 *
	 * @package FreedomReader
	 */
	public function lock_rules_page() {
		// Check if database tables exist with caching.
		global $wpdb;
		$cache_key    = 'freedo_table_exists_lock_rules';
		$table_exists = wp_cache_get( $cache_key, 'freedomreader' );

		if ( false === $table_exists ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table existence check with caching
			$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->prefix . 'freedo_lock_rules' ) );
			wp_cache_set( $cache_key, $table_exists, 'freedomreader', HOUR_IN_SECONDS );
		}

		if ( ! $table_exists ) {
			echo '<div class="notice notice-error"><p>';
			echo esc_html__( 'FreedomReader database tables are missing. Please deactivate and reactivate the plugin to create them.', 'freedomreader' );
			echo '</p></div>';
		}

		include FREEDOMREADER_PLUGIN_PATH . 'admin/views/lock-rules.php';
	}

	/**
	 * Test AJAX functionality
	 *
	 * @package FreedomReader
	 */
	public function test_ajax() {
		wp_send_json_success( array( 'message' => 'AJAX is working!' ) );
	}

	/**
	 * Save lock rule via AJAX
	 *
	 * @package FreedomReader
	 */
	public function save_lock_rule() {
		if ( ! isset( $_POST['nonce'] ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'No nonce provided', 'freedomreader' ) ) );
		}

		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid nonce', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		global $wpdb;
		$table = $wpdb->prefix . 'freedo_lock_rules';

		// Check if table exists with caching.
		$cache_key    = 'freedo_table_exists_lock_rules_save';
		$table_exists = wp_cache_get( $cache_key, 'freedomreader' );

		if ( false === $table_exists ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table existence check with caching
			$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );
			wp_cache_set( $cache_key, $table_exists, 'freedomreader', HOUR_IN_SECONDS );
		}
		if ( ! $table_exists ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Database table does not exist. Please deactivate and reactivate the plugin.', 'freedomreader' ) ) );
		}

		$rule_id         = intval( wp_unslash( $_POST['rule_id'] ?? 0 ) );
		$rule_name       = sanitize_text_field( wp_unslash( $_POST['rule_name'] ?? '' ) );
		$rule_type       = sanitize_text_field( wp_unslash( $_POST['rule_type'] ?? '' ) );
		$rule_value      = sanitize_textarea_field( wp_unslash( $_POST['rule_value'] ?? '' ) );
		$lock_conditions = sanitize_textarea_field( wp_unslash( $_POST['lock_conditions'] ?? '' ) );
		$priority        = intval( wp_unslash( $_POST['priority'] ?? 0 ) );

		$data = array(
			'rule_name'       => $rule_name,
			'rule_type'       => $rule_type,
			'rule_value'      => $rule_value,
			'lock_conditions' => $lock_conditions,
			'priority'        => $priority,
			'updated_at'      => current_time( 'mysql' ),
		);

		if ( 0 < $rule_id ) {

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table update operation
			$result = $wpdb->update( $table, $data, array( 'id' => $rule_id ) );
		} else {
			// Create new rule.
			$data['created_at'] = current_time( 'mysql' );
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert operation
			$result = $wpdb->insert( $table, $data );
		}

		if ( false !== $result ) {
			wp_cache_delete( 'freedomreader_active_lock_rules' );
			wp_cache_delete( 'freedomreader_dashboard_stats' );
			wp_cache_delete( 'freedomreader_content_analytics' );
			wp_cache_delete( 'freedomreader_admin_recent_activity' );
			wp_cache_delete( 'freedomreader_lock_rules_admin_view' );
			wp_send_json_success( array( 'message' => esc_html__( 'Rule saved successfully', 'freedomreader' ) ) );
		} else {
			$error_message = $wpdb->last_error ? $wpdb->last_error : esc_html__( 'Failed to save rule', 'freedomreader' );
			wp_send_json_error( array( 'message' => $error_message ) );
		}
	}

	/**
	 * Delete lock rule via AJAX
	 *
	 * @package FreedomReader
	 */
	public function delete_lock_rule() {
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) || ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		global $wpdb;
		$table   = $wpdb->prefix . 'freedo_lock_rules';
		$rule_id = intval( wp_unslash( $_POST['rule_id'] ?? 0 ) );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table delete operation
		$result = $wpdb->delete( $table, array( 'id' => $rule_id ) );

		if ( false !== $result ) {
			wp_cache_delete( 'freedomreader_active_lock_rules' );
			wp_cache_delete( 'freedomreader_dashboard_stats' );
			wp_cache_delete( 'freedomreader_content_analytics' );
			wp_cache_delete( 'freedomreader_admin_recent_activity' );
			wp_send_json_success( array( 'message' => esc_html__( 'Rule deleted successfully', 'freedomreader' ) ) );
		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to delete rule', 'freedomreader' ) ) );
		}
	}

	/**
	 * Toggle lock rule status via AJAX
	 *
	 * @package FreedomReader
	 */
	public function toggle_lock_rule() {
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) || ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		global $wpdb;
		$table     = $wpdb->prefix . 'freedo_lock_rules';
		$rule_id   = intval( wp_unslash( $_POST['rule_id'] ?? 0 ) );
		$is_active = intval( wp_unslash( $_POST['is_active'] ?? 0 ) );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table update operation
		$result = $wpdb->update(
			$table,
			array(
				'is_active'  => $is_active,
				'updated_at' => current_time( 'mysql' ),
			),
			array( 'id' => $rule_id )
		);

		if ( false !== $result ) {
			wp_cache_delete( 'freedomreader_active_lock_rules' );
			wp_cache_delete( 'freedomreader_dashboard_stats' );
			wp_cache_delete( 'freedomreader_content_analytics' );
			wp_cache_delete( 'freedomreader_admin_recent_activity' );
			wp_send_json_success( array( 'message' => esc_html__( 'Rule status updated', 'freedomreader' ) ) );
		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to update rule status', 'freedomreader' ) ) );
		}
	}

	/**
	 * Get lock rule data via AJAX
	 *
	 * @package FreedomReader
	 */
	public function get_lock_rule() {
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) || ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		$rule_id = intval( wp_unslash( $_POST['rule_id'] ?? 0 ) );

		$cache_key = 'freedomreader_lock_rule_' . $rule_id;
		$rule      = wp_cache_get( $cache_key );

		if ( false === $rule ) {
			global $wpdb;
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query for admin functionality
			$rule = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}freedo_lock_rules WHERE id = %d", $rule_id ) );
			wp_cache_set( $cache_key, $rule, '', 300 );
		}

		if ( $rule ) {
			wp_send_json_success( $rule );
		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'Rule not found', 'freedomreader' ) ) );
		}
	}

	/**
	 * Handle export data AJAX request
	 *
	 * @package FreedomReader
	 */
	public function export_data() {

		if ( ! isset( $_GET['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( esc_html__( 'Security check failed', 'freedomreader' ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( esc_html__( 'Permission denied', 'freedomreader' ) );
		}

		$type      = isset( $_GET['type'] ) ? sanitize_text_field( wp_unslash( $_GET['type'] ) ) : 'purchases';
		$date_from = isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : '';
		$date_to   = isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : '';

		if ( ! in_array( $type, array( 'purchases', 'subscriptions' ), true ) ) {
			wp_send_json_error( esc_html__( 'Invalid export type', 'freedomreader' ) );
		}

		$reports = FREEDO_Admin_Reports::get_instance();
		$reports->export_to_csv( $type, $date_from, $date_to );
	}

	/**
	 * Export all plugin data
	 *
	 * @package FreedomReader
	 */
	public function export_all_data() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		global $wpdb;
		$data = array();

		$purchases_table = $wpdb->prefix . 'freedo_purchases';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Admin export functionality, table names cannot be prepared
		$purchases         = $wpdb->get_results( "SELECT * FROM `{$purchases_table}` ORDER BY created_at DESC", ARRAY_A );
		$data['purchases'] = $purchases;

		$subscriptions_table = $wpdb->prefix . 'freedo_subscriptions';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Admin export functionality, table names cannot be prepared
		$subscriptions         = $wpdb->get_results( "SELECT * FROM `{$subscriptions_table}` ORDER BY created_at DESC", ARRAY_A );
		$data['subscriptions'] = $subscriptions;

		$locks_table = $wpdb->prefix . 'freedo_content_locks';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Admin export functionality, table names cannot be prepared
		$locks                 = $wpdb->get_results( "SELECT * FROM `{$locks_table}` ORDER BY created_at DESC", ARRAY_A );
		$data['content_locks'] = $locks;

		$settings         = array(
			'paypal_mode'              => get_option( 'freedomreader_paypal_mode' ),
			'currency'                 => get_option( 'freedomreader_currency' ),
			'teaser_message'           => get_option( 'freedomreader_teaser_message' ),
			'unlock_button_text'       => get_option( 'freedomreader_unlock_button_text' ),
			'subscription_button_text' => get_option( 'freedomreader_subscription_button_text' ),
		);
		$data['settings'] = $settings;

		$export_data = wp_json_encode( $data, JSON_PRETTY_PRINT );
		$filename    = 'freedomreader-export-' . gmdate( 'Y-m-d-H-i-s' ) . '.json';

		header( 'Content-Type: application/json' );
		header( 'Content-Disposition: attachment; filename="' . $filename . '"' );
		header( 'Content-Length: ' . strlen( $export_data ) );

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON export data, already encoded
		echo $export_data;
		wp_die();
	}

	/**
	 * Clean up old logs
	 *
	 * @package FreedomReader
	 */
	public function cleanup_logs() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		global $wpdb;

		$cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( '-6 months' ) );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Admin cleanup functionality
		$deleted = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM `{$wpdb->prefix}freedo_payment_logs` WHERE created_at < %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name cannot be prepared
				$cutoff_date
			)
		);

		if ( false === $deleted ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to clean logs', 'freedomreader' ) ) );
		}

		wp_send_json_success(
			array(
				'message' => sprintf(
					/* translators: %d: Number of log entries deleted */
					esc_html__( 'Successfully cleaned %d old log entries', 'freedomreader' ),
					$deleted
				),
			)
		);
	}

	/**
	 * Verify data integrity
	 *
	 * @package FreedomReader
	 */
	public function verify_data_integrity() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		global $wpdb;
		$issues = array();

		$required_tables = array(
			$wpdb->prefix . 'freedo_purchases',
			$wpdb->prefix . 'freedo_subscriptions',
			$wpdb->prefix . 'freedo_content_locks',
			$wpdb->prefix . 'freedo_payment_logs',
		);

		foreach ( $required_tables as $table ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Admin verification functionality, table names cannot be prepared
			$table_exists = $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" );
			if ( ! $table_exists ) {
				$issues[] = sprintf(
					/* translators: %s: Table name */
					esc_html__( 'Missing table: %s', 'freedomreader' ),
					$table
				);
			}
		}

		// Check for orphaned content locks.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Admin verification functionality
		$orphaned_locks = $wpdb->get_var(
			"SELECT COUNT(*) FROM `{$wpdb->prefix}freedo_content_locks` cl
			 LEFT JOIN {$wpdb->posts} p ON cl.post_id = p.ID
			 WHERE p.ID IS NULL" // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name cannot be prepared
		);

		if ( $orphaned_locks > 0 ) {
			$issues[] = sprintf(
				/* translators: %d: Number of orphaned locks */
				esc_html__( '%d orphaned content locks found', 'freedomreader' ),
				$orphaned_locks
			);
		}

		if ( empty( $issues ) ) {
			wp_send_json_success(
				array(
					'message' => esc_html__( 'Data integrity check passed. No issues found.', 'freedomreader' ),
				)
			);
		} else {
			wp_send_json_success(
				array(
					'message' => esc_html__( 'Data integrity issues found:', 'freedomreader' ) . ' ' . implode( ', ', $issues ),
				)
			);
		}
	}

	/**
	 * Create manual backup
	 *
	 * @package FreedomReader
	 */
	public function create_manual_backup() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		$backup_system = FREEDO_Backup_System::get_instance();
		$backup_file   = $backup_system->create_backup( 'manual' );

		if ( $backup_file ) {
			wp_send_json_success(
				array(
					'message' => esc_html__( 'Backup created successfully!', 'freedomreader' ),
					'file'    => basename( $backup_file ),
				)
			);
		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to create backup', 'freedomreader' ) ) );
		}
	}

	/**
	 * Import backup data
	 *
	 * @package FreedomReader
	 */
	public function import_backup_data() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		if ( ! isset( $_FILES['backup_file'] ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'No backup file provided', 'freedomreader' ) ) );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- File upload array, validated below
		$uploaded_file = $_FILES['backup_file'];
		$overwrite     = isset( $_POST['overwrite'] ) && '1' === sanitize_text_field( wp_unslash( $_POST['overwrite'] ) );

		if ( UPLOAD_ERR_OK !== $uploaded_file['error'] ) {
			wp_send_json_error( array( 'message' => esc_html__( 'File upload error', 'freedomreader' ) ) );
		}

		if ( 'json' !== pathinfo( sanitize_file_name( $uploaded_file['name'] ), PATHINFO_EXTENSION ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid file format. Please upload a JSON backup file.', 'freedomreader' ) ) );
		}

		$backup_system = FREEDO_Backup_System::get_instance();
		$result        = $backup_system->restore_backup( $uploaded_file['tmp_name'], $overwrite );

		if ( $result ) {
			wp_send_json_success( array( 'message' => esc_html__( 'Backup imported successfully!', 'freedomreader' ) ) );
		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to import backup', 'freedomreader' ) ) );
		}
	}

	/**
	 * Run performance check
	 *
	 * @package FreedomReader
	 */
	public function run_performance_check() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		$performance_monitor = FREEDO_Performance_Monitor::get_instance();
		$stats               = $performance_monitor->get_performance_stats( 'day' );

		if ( empty( $stats ) ) {
			wp_send_json_success( array( 'message' => esc_html__( 'No performance data available yet.', 'freedomreader' ) ) );
		} else {
			$message = esc_html__( 'Performance check completed. Recent stats:', 'freedomreader' ) . "\n";
			foreach ( $stats as $stat ) {
				$message .= sprintf(
					"%s: %.2fs avg (%.2fs max) - %d operations\n",
					$stat['operation'],
					$stat['avg_duration'],
					$stat['max_duration'],
					$stat['count']
				);
			}

			wp_send_json_success( array( 'message' => $message ) );
		}
	}

	/**
	 * Handle bulk operations
	 *
	 * @package FreedomReader
	 */
	public function handle_bulk_operations() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		$operation = isset( $_POST['operation'] ) ? sanitize_text_field( wp_unslash( $_POST['operation'] ) ) : '';
		$items     = isset( $_POST['items'] ) ? array_map( 'intval', $_POST['items'] ) : array();

		if ( empty( $operation ) || empty( $items ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid operation or no items selected', 'freedomreader' ) ) );
		}

		$processed = 0;
		global $wpdb;

		switch ( $operation ) {
			case 'delete_locks':
				$locks_table = $wpdb->prefix . 'freedo_content_locks';
				foreach ( $items as $lock_id ) {
					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk operations
					$result = $wpdb->delete( $locks_table, array( 'id' => $lock_id ), array( '%d' ) );
					if ( $result ) {
						++$processed;
					}
				}
				break;

			case 'update_prices':
				$new_price   = isset( $_POST['new_price'] ) ? floatval( $_POST['new_price'] ) : 0;
				$locks_table = $wpdb->prefix . 'freedo_content_locks';
				foreach ( $items as $lock_id ) {
					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk operations
					$result = $wpdb->update(
						$locks_table,
						array( 'price' => $new_price ),
						array( 'id' => $lock_id ),
						array( '%f' ),
						array( '%d' )
					);
					if ( false !== $result ) {
						++$processed;
					}
				}
				break;

			default:
				wp_send_json_error( array( 'message' => esc_html__( 'Unknown operation', 'freedomreader' ) ) );
		}

		wp_send_json_success(
			array(
				'message' => sprintf(
					/* translators: %d: Number of items processed */
					esc_html__( 'Successfully processed %d items', 'freedomreader' ),
					$processed
				),
			)
		);
	}

	/**
	 * Validate PayPal credentials via AJAX
	 *
	 * @package FreedomReader
	 */
	public function validate_paypal_credentials() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		$client_id     = isset( $_POST['client_id'] ) ? sanitize_text_field( wp_unslash( $_POST['client_id'] ) ) : '';
		$client_secret = isset( $_POST['client_secret'] ) ? sanitize_text_field( wp_unslash( $_POST['client_secret'] ) ) : '';

		if ( empty( $client_id ) || empty( $client_secret ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Client ID and Client Secret are required', 'freedomreader' ) ) );
		}

		$mode     = get_option( 'freedomreader_paypal_mode', 'sandbox' );
		$base_url = ( 'live' === $mode ) ? 'https://api.paypal.com' : 'https://api.sandbox.paypal.com';
		$url      = $base_url . '/v1/oauth2/token';

		$args = array(
			'method'  => 'POST',
			'headers' => array(
				'Accept'          => 'application/json',
				'Accept-Language' => 'en_US',
				// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Required for PayPal API authentication
				'Authorization'   => 'Basic ' . base64_encode( $client_id . ':' . $client_secret ),
			),
			'body'    => 'grant_type=client_credentials',
		);

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to connect to PayPal', 'freedomreader' ) ) );
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$body          = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 === $response_code && isset( $body['access_token'] ) ) {
			wp_send_json_success( array( 'message' => esc_html__( 'PayPal credentials are valid', 'freedomreader' ) ) );
		} else {
			$error_message = esc_html__( 'Invalid PayPal credentials', 'freedomreader' );
			if ( isset( $body['error_description'] ) ) {
				$error_message = $body['error_description'];
			}
			wp_send_json_error( array( 'message' => $error_message ) );
		}
	}

	/**
	 * Refund purchase via AJAX
	 *
	 * @package FreedomReader
	 */
	public function refund_purchase() {

		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'freedomreader_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed', 'freedomreader' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied', 'freedomreader' ) ) );
		}

		$purchase_id = isset( $_POST['purchase_id'] ) ? intval( $_POST['purchase_id'] ) : 0;

		if ( ! $purchase_id ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid purchase ID', 'freedomreader' ) ) );
		}

		global $wpdb;
		$table = $wpdb->prefix . 'freedo_purchases';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Purchase lookup for refund
		$purchase = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM `{$wpdb->prefix}freedo_purchases` WHERE id = %d", $purchase_id ) );

		if ( ! $purchase ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Purchase not found', 'freedomreader' ) ) );
		}

		if ( 'refunded' === $purchase->status ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Purchase already refunded', 'freedomreader' ) ) );
		}

		if ( empty( $purchase->paypal_transaction_id ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'No PayPal transaction ID found', 'freedomreader' ) ) );
		}

		$paypal = FREEDO_PayPal::get_instance();
		$result = $paypal->refund_payment( $purchase->paypal_transaction_id );

		if ( $result['success'] ) {

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Status update after refund
			$wpdb->update(
				$table,
				array(
					'status'     => 'refunded',
					'updated_at' => current_time( 'mysql' ),
				),
				array( 'id' => $purchase_id )
			);

			wp_cache_delete( 'freedomreader_dashboard_stats' );
			wp_cache_delete( 'freedomreader_admin_recent_activity' );

			wp_send_json_success( array( 'message' => esc_html__( 'Purchase refunded successfully', 'freedomreader' ) ) );
		} else {
			wp_send_json_error( array( 'message' => $result['message'] ) );
		}
	}
}
