<?php
/**
 * Reports functionality for FreedomReader
 *
 * @package FreedomReader
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Reports functionality for FreedomReader admin
 *
 * Handles admin-specific reporting features including data export and CSV generation.
 *
 * @package FreedomReader
 * @since 1.0.0
 */
class FREEDO_Admin_Reports {
 // phpcs:ignore Squiz.Classes.ClassFileName.NoMatch -- Admin reports class uses different naming convention

	/**
	 * Singleton instance
	 *
	 * @var FREEDO_Admin_Reports|null
	 */
	private static $instance = null;

	/**
	 * Get singleton instance
	 *
	 * @return FREEDO_Admin_Reports
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Constructor.
	}

	/**
	 * Get reports data
	 *
	 * @package FreedomReader
	 */
	public function get_reports_data() {
		$data = array();

		$data['revenue_chart'] = $this->get_revenue_chart_data();

		$data['top_content'] = $this->get_top_selling_content();

		$data['subscription_analytics'] = $this->get_subscription_analytics();

		$data['recent_activity'] = $this->get_recent_activity();

		return $data;
	}

	/**
	 * Get revenue chart data for last 12 months
	 *
	 * @package FreedomReader
	 */
	private function get_revenue_chart_data() {
		$cache_key = 'freedo_revenue_chart_data';
		$results   = wp_cache_get( $cache_key );

		if ( false === $results ) {
			global $wpdb;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Revenue analytics query with caching
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT DATE_FORMAT(created_at, '%%Y-%%m') as month, SUM(amount) as revenue, COUNT(*) as purchases
                     FROM {$wpdb->prefix}freedo_purchases
                     WHERE status = %s AND created_at >= DATE_SUB(NOW(), INTERVAL %d MONTH) 
                     GROUP BY DATE_FORMAT(created_at, '%%Y-%%m') 
                     ORDER BY month ASC",
					'completed',
					12
				)
			);

			wp_cache_set( $cache_key, $results, '', 600 );
		}

		$labels         = array();
		$revenue_data   = array();
		$purchases_data = array();

		for ( $i = 11; $i >= 0; $i-- ) {
			$month    = gmdate( 'Y-m', strtotime( "-$i months" ) );
			$labels[] = gmdate( 'M Y', strtotime( "-$i months" ) );

			$found = false;
			foreach ( $results as $result ) {
				if ( $result->month === $month ) {
					$revenue_data[]   = floatval( $result->revenue );
					$purchases_data[] = intval( $result->purchases );
					$found            = true;
					break;
				}
			}

			if ( ! $found ) {
				$revenue_data[]   = 0;
				$purchases_data[] = 0;
			}
		}

		return array(
			'labels'    => $labels,
			'revenue'   => $revenue_data,
			'purchases' => $purchases_data,
		);
	}

	/**
	 * Get top selling content
	 *
	 * @package FreedomReader
	 */
	private function get_top_selling_content() {
		$cache_key = 'freedo_admin_top_selling_content';
		$results   = wp_cache_get( $cache_key );

		if ( false === $results ) {
			global $wpdb;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Popular content analytics query with caching
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT p.post_id, po.post_title, COUNT(*) as purchase_count, SUM(p.amount) as total_revenue
                     FROM {$wpdb->prefix}freedo_purchases p
                     LEFT JOIN {$wpdb->posts} po ON p.post_id = po.ID 
                     WHERE p.status = %s AND p.post_id IS NOT NULL 
                     GROUP BY p.post_id 
                     ORDER BY purchase_count DESC 
                     LIMIT 10",
					'completed'
				)
			);

			wp_cache_set( $cache_key, $results, '', 600 );
		}

		return $results;
	}

	/**
	 * Get subscription analytics
	 *
	 * @package FreedomReader
	 */
	private function get_subscription_analytics() {
		$cache_key = 'freedo_admin_subscription_analytics';
		$analytics = wp_cache_get( $cache_key );

		if ( false === $analytics ) {
			global $wpdb;

			$analytics = array();

			// Subscription status breakdown.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Subscription analytics query with caching
			$analytics['status_breakdown'] = $wpdb->get_results(
				"SELECT status, COUNT(*) as count
                     FROM {$wpdb->prefix}freedo_subscriptions
                     GROUP BY status"
			);

			// Monthly recurring revenue (MRR).
			$subscription_plans = get_option( 'freedomreader_subscription_plans', array() );
			$mrr                = 0;

			foreach ( $subscription_plans as $plan_id => $plan ) {
				if ( 'month' === $plan['interval'] ) {
					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- MRR calculation query
					$active_count = $wpdb->get_var(
						$wpdb->prepare(
							"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions
                             WHERE plan_id = %s AND status = %s",
							$plan_id,
							'active'
						)
					);
					$mrr         += $active_count * $plan['price'];
				} elseif ( 'year' === $plan['interval'] ) {
					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- MRR calculation query
					$active_count = $wpdb->get_var(
						$wpdb->prepare(
							"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions
                             WHERE plan_id = %s AND status = %s",
							$plan_id,
							'active'
						)
					);
					$mrr         += $active_count * ( $plan['price'] / 12 );
				}
			}

			$analytics['mrr'] = $mrr;

			// Churn rate (cancelled subscriptions in last month).
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Churn rate calculation query
			$total_active = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions WHERE status = %s",
					'active'
				)
			);
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Churn rate calculation query
			$cancelled_last_month = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions
                     WHERE status = %s AND updated_at >= DATE_SUB(NOW(), INTERVAL 1 MONTH)",
					'cancelled'
				)
			);

			$analytics['churn_rate'] = $total_active > 0 ? ( $cancelled_last_month / $total_active ) * 100 : 0;

			// Trial conversion rate.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Trial conversion calculation query
			$total_trials = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions WHERE status IN (%s, %s, %s)",
					'trial',
					'active',
					'cancelled'
				)
			);
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Trial conversion calculation query
			$converted_trials = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->prefix}freedo_subscriptions WHERE status = %s",
					'active'
				)
			);

			$analytics['trial_conversion_rate'] = $total_trials > 0 ? ( $converted_trials / $total_trials ) * 100 : 0;

			wp_cache_set( $cache_key, $analytics, '', 600 );
		}

		return $analytics;
	}

	/**
	 * Get recent activity
	 *
	 * @package FreedomReader
	 */
	private function get_recent_activity() {
		$cache_key = 'freedo_admin_recent_activity';
		$activity  = wp_cache_get( $cache_key );

		if ( false === $activity ) {
			global $wpdb;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Recent activity query with caching
			$recent_purchases = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT p.*, u.display_name, po.post_title
                     FROM {$wpdb->prefix}freedo_purchases p
                     LEFT JOIN {$wpdb->users} u ON p.user_id = u.ID 
                     LEFT JOIN {$wpdb->posts} po ON p.post_id = po.ID 
                     WHERE p.status = %s 
                     ORDER BY p.created_at DESC 
                     LIMIT %d",
					'completed',
					10
				)
			);

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Recent activity query with caching
			$recent_subscriptions = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT s.*, u.display_name
                     FROM {$wpdb->prefix}freedo_subscriptions s
                     LEFT JOIN {$wpdb->users} u ON s.user_id = u.ID 
                     ORDER BY s.created_at DESC 
                     LIMIT %d",
					10
				)
			);

			$activity = array(
				'purchases'     => $recent_purchases,
				'subscriptions' => $recent_subscriptions,
			);

			wp_cache_set( $cache_key, $activity, '', 300 );
		}

		return $activity;
	}

	/**
	 * Export data to CSV
	 *
	 * @param string $type Export type (purchases or subscriptions).
	 * @param string $date_from Start date for export.
	 * @param string $date_to End date for export.
	 * @package FreedomReader
	 */
	public function export_to_csv( $type = 'purchases', $date_from = '', $date_to = '' ) {
		global $wp_filesystem;

		require_once ABSPATH . 'wp-admin/includes/file.php';
		WP_Filesystem();

		$filename = 'freedomreader-' . $type . '-' . gmdate( 'Y-m-d' ) . '.csv';

		$upload_dir = wp_upload_dir();
		$file_path  = $upload_dir['basedir'] . '/' . $filename;

		if ( ! $wp_filesystem->put_contents( $file_path, '', FS_CHMOD_FILE ) ) {
			return new WP_Error( 'file_write_error', 'Unable to create file.' );
		}

		switch ( $type ) {
			case 'purchases':
				$this->export_purchases_csv( $file_path, $date_from, $date_to );
				break;
			case 'subscriptions':
				$this->export_subscriptions_csv( $file_path, $date_from, $date_to );
				break;
		}

		$escaped_filename = esc_attr( $filename );
		header( 'Content-Type: text/csv' );
		header( "Content-Disposition: attachment; filename=\"{$escaped_filename}\"" );
		$content = $wp_filesystem->get_contents( $file_path );
		echo esc_html( $content );
		$wp_filesystem->delete( $file_path );
		exit;
	}

	/**
	 * Export purchases to CSV
	 *
	 * @param string $file_path Path to the CSV file.
	 * @param string $date_from Start date for export.
	 * @param string $date_to End date for export.
	 * @package FreedomReader
	 */
	private function export_purchases_csv( $file_path, $date_from, $date_to ) {
		global $wpdb, $wp_filesystem;

		// CSV headers.
		$headers = array(
			'ID',
			'User',
			'Email',
			'Content',
			'Type',
			'Amount',
			'Currency',
			'Status',
			'Transaction ID',
			'Date',
		);
		$wp_filesystem->put_contents( $file_path, implode( ',', array_map( 'esc_csv', $headers ) ) . "\n" );

		$where_conditions = array( 'p.status = %s' );
		$params           = array( 'completed' );

		if ( $date_from ) {
			$where_conditions[] = 'p.created_at >= %s';
			$params[]           = $date_from;
		}

		if ( $date_to ) {
			$where_conditions[] = 'p.created_at <= %s';
			$params[]           = $date_to . ' 23:59:59';
		}

		$where_clause = 'WHERE ' . implode( ' AND ', $where_conditions );

		$purchases_table = $wpdb->prefix . 'freedo_purchases';
		$users_table     = $wpdb->users;
		$posts_table     = $wpdb->posts;

		$query = "SELECT p.*, u.display_name, u.user_email, po.post_title
                  FROM `{$purchases_table}` p
                  LEFT JOIN `{$users_table}` u ON p.user_id = u.ID
                  LEFT JOIN `{$posts_table}` po ON p.post_id = po.ID
                  {$where_clause}
                  ORDER BY p.created_at DESC";

		if ( ! empty( $params ) ) {
			$prepared_query = call_user_func_array( array( $wpdb, 'prepare' ), array_merge( array( $query ), $params ) );
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared -- CSV export query
			$results = $wpdb->get_results( $prepared_query );
		} else {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared -- CSV export query
			$results = $wpdb->get_results( $query );
		}

		$csv_data = '';
		foreach ( $results as $purchase ) {
			$row       = array(
				$purchase->id,
				$this->esc_csv( $purchase->display_name ? $purchase->display_name : 'N/A' ),
				$this->esc_csv( $purchase->user_email ? $purchase->user_email : 'N/A' ),
				$this->esc_csv( $purchase->post_title ? $purchase->post_title : 'Subscription' ),
				ucfirst( $purchase->purchase_type ),
				$purchase->amount,
				$purchase->currency,
				ucfirst( $purchase->status ),
				$purchase->paypal_transaction_id,
				$purchase->created_at,
			);
			$csv_data .= implode( ',', $row ) . "\n";
		}
		$wp_filesystem->append_contents( $file_path, $csv_data );
	}

	/**
	 * Export subscriptions to CSV
	 *
	 * @param string $file_path Path to the CSV file.
	 * @param string $date_from Start date for export.
	 * @param string $date_to End date for export.
	 * @package FreedomReader
	 */
	private function export_subscriptions_csv( $file_path, $date_from, $date_to ) {
		global $wpdb, $wp_filesystem;

		$headers = array(
			'ID',
			'User',
			'Email',
			'Plan',
			'Status',
			'PayPal Subscription ID',
			'Trial Ends',
			'Next Billing',
			'Created Date',
		);
		$wp_filesystem->put_contents( $file_path, implode( ',', array_map( 'esc_csv', $headers ) ) . "\n" );

		$where_conditions = array();
		$params           = array();

		if ( $date_from ) {
			$where_conditions[] = 's.created_at >= %s';
			$params[]           = $date_from;
		}

		if ( $date_to ) {
			$where_conditions[] = 's.created_at <= %s';
			$params[]           = $date_to . ' 23:59:59';
		}

		$where_clause = ! empty( $where_conditions ) ? 'WHERE ' . implode( ' AND ', $where_conditions ) : '';

		$subscriptions_table = $wpdb->prefix . 'freedo_subscriptions';
		$users_table         = $wpdb->users;

		$query = "SELECT s.*, u.display_name, u.user_email
                  FROM `{$subscriptions_table}` s
                  LEFT JOIN `{$users_table}` u ON s.user_id = u.ID
                  {$where_clause}
                  ORDER BY s.created_at DESC";

		if ( ! empty( $params ) ) {
			$prepared_query = call_user_func_array( array( $wpdb, 'prepare' ), array_merge( array( $query ), $params ) );
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared -- CSV export query
			$results = $wpdb->get_results( $prepared_query );
		} else {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared -- CSV export query
			$results = $wpdb->get_results( $query );
		}

		$subscription_plans = get_option( 'freedomreader_subscription_plans', array() );
		$csv_data           = '';
		foreach ( $results as $subscription ) {
			$plan_name = isset( $subscription_plans[ $subscription->plan_id ] )
				? $subscription_plans[ $subscription->plan_id ]['name']
				: $subscription->plan_id;

			$row       = array(
				$subscription->id,
				$this->esc_csv( $subscription->display_name ? $subscription->display_name : 'N/A' ),
				$this->esc_csv( $subscription->user_email ? $subscription->user_email : 'N/A' ),
				$this->esc_csv( $plan_name ),
				ucfirst( $subscription->status ),
				$subscription->paypal_subscription_id,
				$subscription->trial_ends_at ? $subscription->trial_ends_at : 'N/A',
				$subscription->next_billing_date ? $subscription->next_billing_date : 'N/A',
				$subscription->created_at,
			);
			$csv_data .= implode( ',', $row ) . "\n";
		}
		$wp_filesystem->append_contents( $file_path, $csv_data );
	}
	/**
	 * Custom function to escape CSV data.
	 *
	 * @param string $data The data to escape for CSV.
	 * @return string Escaped CSV data.
	 */
	private function esc_csv( $data ) {
		if ( false !== strpos( $data, ',' ) || false !== strpos( $data, '"' ) || false !== strpos( $data, "\n" ) ) {
			$data = '"' . str_replace( '"', '""', $data ) . '"';
		}
		return $data;
	}
}
