<?php
/**
 * Plugin Name:       FreeChatBots
 * Plugin URI:        https://freechatbots.ai
 * Description:       A.I agent for websites
 * Version:           1.0.0
 * Author:            Intelli Group Pty Ltd
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       FreeChatBots
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('FREECH_RAG_CHATBOT_PATH', plugin_dir_path(__FILE__));
define('FREECH_RAG_CHATBOT_URL', plugin_dir_url(__FILE__));
define('FREECH_RAG_CHATBOT_BACKEND_URL', 'https://wp-ragbot-backend.fly.dev'); 

// define('FREECH_RAG_CHATBOT_BACKEND_URL', 'https://0f46-14-200-177-254.ngrok-free.app'); // Change this for production

// Include necessary files
require_once FREECH_RAG_CHATBOT_PATH . 'includes/admin-menu.php';
require_once FREECH_RAG_CHATBOT_PATH . 'includes/frontend-display.php';
require_once FREECH_RAG_CHATBOT_PATH . 'includes/payment-endpoints.php';

// Register activation hook
register_activation_hook(__FILE__, 'freech_rag_chatbot_activate');

function freech_rag_chatbot_activate() {
    // Create necessary database tables
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}rag_documents (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        title varchar(255) NOT NULL,
        file_path varchar(255) NOT NULL,
        uploaded_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY  (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

// After the activation hook, add this function:
function freech_rag_chatbot_create_icon() {
    $icon_path = FREECH_RAG_CHATBOT_PATH . 'assets/images/chat-icon.png';
    
    // Check if images directory exists, if not create it
    if (!file_exists(dirname($icon_path))) {
        wp_mkdir_p(dirname($icon_path));
    }
    
    // Create SVG chat icon
    $svg = '<?xml version="1.0" encoding="UTF-8"?>
    <svg width="512" height="512" viewBox="0 0 512 512" xmlns="http://www.w3.org/2000/svg">
        <path fill="#ffffff" d="M256 32c-141.4 0-256 93.1-256 208c0 49.6 21.4 95 57 130.7C44.5 421.1 2.7 466 2.2 466.5c-2.2 2.3-2.8 5.7-1.5 8.7 1.3 3 4.1 4.8 7.3 4.8 66.3 0 116-31.8 140.6-51.4 32.7 12.3 69 19.4 107.4 19.4 141.4 0 256-93.1 256-208S397.4 32 256 32z"/>
    </svg>';
    
    // Convert SVG to PNG using Imagick if available
    if (class_exists('Imagick')) {
        try {
            $im = new Imagick();
            $im->readImageBlob($svg);
            $im->setImageFormat("png");
            $im->writeImage($icon_path);
            $im->clear();
            $im->destroy();
            return;
        } catch (Exception $e) {
            // Fall back to default method if Imagick fails
        }
    }
    
    // Fallback: Create a simple PNG using GD
    $im = imagecreatetruecolor(512, 512);
    $white = imagecolorallocate($im, 255, 255, 255);
    
    // Make background transparent
    imagealphablending($im, false);
    $transparency = imagecolorallocatealpha($im, 0, 0, 0, 127);
    imagefill($im, 0, 0, $transparency);
    imagesavealpha($im, true);
    
    // Draw a simple chat bubble
    imagefilledellipse($im, 256, 256, 400, 400, $white);
    imagefilledpolygon($im, [
        150, 400,  // Top left
        50, 500,   // Bottom point
        250, 400   // Top right
    ], 3, $white);
    
    imagepng($im, $icon_path);
    imagedestroy($im);
}

// Add to activation hook
register_activation_hook(__FILE__, function() {
    freech_rag_chatbot_activate();
    freech_rag_chatbot_register_with_backend();
    freech_rag_chatbot_create_icon();
    
    // Add site scanning
    $scan_result = freech_rag_chatbot_scan_site();
    if (!$scan_result) {
        error_log('RAG Chatbot: Failed to scan and upload site content');
    }
});

// Enqueue scripts and styles
add_action('wp_enqueue_scripts', 'freech_rag_chatbot_enqueue_scripts');
function freech_rag_chatbot_enqueue_scripts() {
    wp_enqueue_style('dashicons');
    wp_enqueue_style('rag-chatbot', FREECH_RAG_CHATBOT_URL . 'css/style.css');
    
    wp_localize_script('rag-chatbot', 'ragChatbot', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('rag_chatbot_nonce'),
        'pluginUrl' => FREECH_RAG_CHATBOT_URL,
        'botIcon' => FREECH_RAG_CHATBOT_URL . 'assets/images/chat-icon.png',
        'userIcon' => FREECH_RAG_CHATBOT_URL . 'assets/images/user-icon.png'
    ));
    
    // Debug
    error_log('RAG Chatbot scripts enqueued. Plugin URL: ' . FREECH_RAG_CHATBOT_URL);
}

// Add uninstall hook
register_uninstall_hook(__FILE__, 'freech_rag_chatbot_uninstall');

function freech_rag_chatbot_uninstall() {
    global $wpdb;
    
    // Drop the documents table
    $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}rag_documents");
    
    // Remove the chat icon
    $icon_path = FREECH_RAG_CHATBOT_PATH . 'assets/images/chat-icon.png';
    if (file_exists($icon_path)) {
        wp_delete_file($icon_path);
    }
    
    // Clean up any options if we add them later
    delete_option('rag_chatbot_settings');
}

function freech_rag_chatbot_register_with_backend() {
    $api_key = get_option('freech_rag_chatbot_api_key');
    
    if (!$api_key) {
        $site_url = get_site_url();
        $site_name = get_bloginfo('name');
        $domain = wp_parse_url($site_url, PHP_URL_HOST);
        $admin_email = get_option('admin_email'); // Get the admin email
        
        error_log('RAG Chatbot registration attempt - Domain: ' . $domain . ', Site Name: ' . $site_name . ', Email: ' . $admin_email);
        
        $response = wp_remote_post(FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/register', array(
            'headers' => array(
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'domain' => $domain,
                'name' => $site_name,
                'billing_email' => $admin_email // Include the admin email as billing_email
            )),
            'timeout' => 30,
            'sslverify' => false // Only for development/debugging
        ));
        
        if (is_wp_error($response)) {
            // Handle error
            error_log('RAG Chatbot registration failed: ' . $response->get_error_message());
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (isset($data['api_key'])) {
            update_option('freech_rag_chatbot_api_key', $data['api_key']);
            return $data['api_key'];
        }
        
        return false;
    }
    
    return $api_key;
}

function freech_rag_chatbot_admin_notices() {
    $screen = get_current_screen();
    if ($screen->id !== 'toplevel_page_rag-chatbot-settings') {
        return;
    }
    
    $api_key = get_option('freech_rag_chatbot_api_key');
    if (!$api_key) {
        ?>
        <div class="notice notice-error">
            <p><strong>FreeChatBots registration failed.</strong></p>
            <p>Please check the following:</p>
            <ul style="list-style-type: disc; margin-left: 20px;">
                <li>Is the backend service running at <code><?php echo esc_html(FREECH_RAG_CHATBOT_BACKEND_URL); ?></code>?</li>
                <li>Can your WordPress site reach the backend service?</li>
                <li>Are you using the correct backend URL?</li>
            </ul>
            <?php if ($last_error): ?>
                <p><strong>Last error:</strong> <?php echo esc_html($last_error); ?></p>
            <?php endif; ?>
            <p>You can try re-registering using the button below.</p>
        </div>
        <?php
    }
}

add_action('admin_notices', 'freech_rag_chatbot_admin_notices');

// Add these functions to handle chat sessions and messages
function freech_init_chat_session() {
    // Create a new session
    $api_key = get_option('freech_rag_chatbot_api_key');
    $domain = freech_get_rag_domain();
    $response = wp_remote_post(
        FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/chat/sessions/?domain=' . urlencode($domain),
        array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $api_key
            ),
            'body' => json_encode(array(
                'title' => freech_get_rag_domain()
            ))
        )
    );

    if (!is_wp_error($response)) {
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($data['uuid'])) {
            return $data['uuid'];
        }
    }
    return false;
}

// function freech_send_chat_message($message) {
//     $session_uuid = freech_init_chat_session();
//     if (!$session_uuid) {
//         return array('error' => 'Could not initialize chat session');
//     }

//     $api_key = get_option('freech_rag_chatbot_api_key');
//     $domain = freech_get_rag_domain();
//     $response = wp_remote_post(
//         FREECH_RAG_CHATBOT_BACKEND_URL . "/api/v1/chat/sessions/{$session_uuid}/messages",
//         array(
//             'headers' => array(
//                 'Content-Type' => 'application/json',
//                 'X-API-Key' => $api_key
//             ),
//             'body' => json_encode(array(
//                 'role' => 'user',
//                 'content' => $message,
//                 'reference_message_uuid' => null
//             )),
//             'timeout' => 60  // One minute timeout
//         )
//     );

//     if (is_wp_error($response)) {
//         return array('error' => $response->get_error_message());
//     }

//     $data = json_decode(wp_remote_retrieve_body($response), true);
//     if (isset($data['content'])) {
//         return array(
//             'content' => $data['content'],
//             'message_uuid' => $data['uuid']
//         );
//     }

//     return array('error' => 'Unknown error occurred');
// }

// Add AJAX handler for chat messages
add_action('wp_ajax_rag_chatbot_message', 'freech_handle_chat_message');
add_action('wp_ajax_nopriv_rag_chatbot_message', 'freech_handle_chat_message');

function freech_handle_chat_message() {
    // Check nonce for ALL requests, not just logged-in users
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'rag_chatbot_nonce')) {
        wp_send_json_error(array('error' => 'Security check failed'));
        return;
    }
    
    $message = isset($_POST['message']) ? sanitize_text_field(wp_unslash($_POST['message'])) : '';
    $api_key = get_option('freech_rag_chatbot_api_key');
    $session_uuid = isset($_POST['session_uuid']) ? sanitize_text_field(wp_unslash($_POST['session_uuid'])) : null;
    
    $debug = array(
        'current_session' => $session_uuid,
        'api_key_exists' => !empty($api_key)
    );
    
    // If no session exists, create one
    if (!$session_uuid) {
        $session_url = FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/chat/sessions';
        $request_body = array(
            'title' => freech_get_rag_domain()
        );
        
        $session_response = wp_remote_post(
            $session_url,
            array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'X-API-Key' => $api_key
                ),
                'body' => json_encode($request_body),
                'timeout' => 60
            )
        );

        if (is_wp_error($session_response)) {
            wp_send_json_error(array(
                'error' => 'Failed to create chat session',
                'debug' => array_merge($debug, array(
                    'session_error' => $session_response->get_error_message()
                ))
            ));
            return;
        }

        $session_data = json_decode(wp_remote_retrieve_body($session_response), true);
        $debug['session_response'] = $session_data;
        
        if (!isset($session_data['uuid'])) {
            wp_send_json_error(array(
                'error' => 'Invalid session response',
                'debug' => $debug
            ));
            return;
        }

        $session_uuid = $session_data['uuid'];
        $debug['new_session'] = $session_uuid;
    }

    // Send message using session UUID
    $chat_url = FREECH_RAG_CHATBOT_BACKEND_URL . "/api/v1/chat/sessions/{$session_uuid}/messages";
    
    $response = wp_remote_post(
        $chat_url,
        array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $api_key
            ),
            'body' => json_encode(array(
                'role' => 'user',
                'content' => $message,
                'reference_message_uuid' => null
            )),
            'timeout' => 60
        )
    );

    if (is_wp_error($response)) {
        wp_send_json_error(array(
            'error' => $response->get_error_message(),
            'debug' => $debug
        ));
        return;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);
    
    // If we get a 400 or 404 error, the session might be expired
    if ($status_code == 400 || $status_code == 404) {
        // Create a new session
        $session_url = FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/chat/sessions';
        $request_body = array(
            'title' => freech_get_rag_domain()
        );
        
        $session_response = wp_remote_post(
            $session_url,
            array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'X-API-Key' => $api_key
                ),
                'body' => json_encode($request_body),
                'timeout' => 60
            )
        );

        if (is_wp_error($session_response)) {
            wp_send_json_error(array(
                'error' => 'Failed to create new chat session after expired session',
                'debug' => array_merge($debug, array(
                    'session_error' => $session_response->get_error_message()
                ))
            ));
            return;
        }

        $session_data = json_decode(wp_remote_retrieve_body($session_response), true);
        
        if (!isset($session_data['uuid'])) {
            wp_send_json_error(array(
                'error' => 'Invalid session response when creating new session',
                'debug' => array_merge($debug, array(
                    'session_response' => $session_data
                ))
            ));
            return;
        }

        $session_uuid = $session_data['uuid'];
        $debug['new_session_after_expired'] = $session_uuid;
        
        // Try again with the new session
        $chat_url = FREECH_RAG_CHATBOT_BACKEND_URL . "/api/v1/chat/sessions/{$session_uuid}/messages";
        
        $response = wp_remote_post(
            $chat_url,
            array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'X-API-Key' => $api_key
                ),
                'body' => json_encode(array(
                    'role' => 'user',
                    'content' => $message,
                    'reference_message_uuid' => null
                )),
                'timeout' => 60
            )
        );
        
        if (is_wp_error($response)) {
            wp_send_json_error(array(
                'error' => $response->get_error_message(),
                'debug' => $debug
            ));
            return;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
    }
    
    if (isset($data['content'])) {
        wp_send_json_success(array(
            'content' => $data['content'],
            'message_uuid' => $data['uuid'],
            'session_uuid' => $session_uuid,  // Return session UUID to store in browser
            'debug' => $debug
        ));
    } else {
        wp_send_json_error(array(
            'error' => 'Invalid response from chat service',
            'debug' => array_merge($debug, array(
                'status_code' => $status_code,
                'response_body' => $body
            ))
        ));
    }
}

// Add this debugging function
function freech_rag_chatbot_debug_scripts() {
    global $wp_scripts;
    error_log('Loaded scripts: ' . print_r($wp_scripts->queue, true));
}
add_action('wp_print_scripts', 'freech_rag_chatbot_debug_scripts');

function freech_rag_chatbot_scan_site() {
    $site_data = array();
    
    // Get domain name and site identity
    $domain = wp_parse_url(get_site_url(), PHP_URL_HOST);
    $site_name = get_bloginfo('name');
    $site_desc = get_bloginfo('description');
    
    $site_data[] = "About " . $domain . ", This is about " . ($site_desc ?: "various services and information");
    
    // Helper function to clean content
    function clean_content($content) {
        // First remove script and style elements completely
        $content = preg_replace('/<script\b[^>]*>(.*?)<\/script>/is', '', $content);
        $content = preg_replace('/<style\b[^>]*>(.*?)<\/style>/is', '', $content);
        
        // Remove HTML comments
        $content = preg_replace('/<!--(.|\s)*?-->/', '', $content);
        
        // Get text content from specific meaningful tags first
        preg_match_all('/<h1[^>]*>(.*?)<\/h1>|<h2[^>]*>(.*?)<\/h2>|<h3[^>]*>(.*?)<\/h3>|<p[^>]*>(.*?)<\/p>|<li[^>]*>(.*?)<\/li>/is', $content, $matches);
        
        $meaningful_content = '';
        foreach ($matches[0] as $match) {
            $meaningful_content .= wp_strip_all_tags($match) . ' ';
        }
        
        // If no meaningful content found, fall back to regular strip_tags
        if (empty(trim($meaningful_content))) {
            $meaningful_content = wp_strip_all_tags($content);
        }
        
        // Clean up the extracted content
        $meaningful_content = preg_replace('/\s+/', ' ', $meaningful_content); // Remove extra whitespace
        $meaningful_content = str_replace('&nbsp;', ' ', $meaningful_content); // Remove non-breaking spaces
        $meaningful_content = trim($meaningful_content); // Trim spaces
        
        return $meaningful_content;
    }
    
    // Get homepage content
    $front_page_id = get_option('page_on_front');
    if ($front_page_id) {
        $front_page = get_post($front_page_id);
        $content = clean_content($front_page->post_content);
        if ($content) {
            $site_data[] = "\nMy main offering:";
            $site_data[] = $content;
        }
    }
    
    // Get navigation structure and associated content
    $menus = wp_get_nav_menus();
    if (!empty($menus)) {
        $site_data[] = "\nMy services and sections:";
        foreach ($menus as $menu) {
            $site_data[] = "\n## " . $menu->name;
            $menu_items = wp_get_nav_menu_items($menu->term_id);
            if ($menu_items) {
                foreach ($menu_items as $item) {
                    $site_data[] = "\n### " . clean_content($item->title);
                    
                    // Get content from linked page/post
                    if ($item->object_id) {
                        $linked_post = get_post($item->object_id);
                        if ($linked_post) {
                            $content = clean_content($linked_post->post_content);
                            if ($content) {
                                $site_data[] = "Details: " . $content;
                            }
                        }
                    }
                }
            }
        }
    }
    
    // Get all published pages not in menus
    $pages = get_pages(array(
        'post_status' => 'publish',
        'exclude' => array($front_page_id) // Exclude homepage as it's already processed
    ));
    
    if (!empty($pages)) {
        $site_data[] = "\nAdditional information:";
        foreach ($pages as $page) {
            $content = clean_content($page->post_content);
            if ($content) {
                $site_data[] = "\n### " . $page->post_title;
                $site_data[] = "Content: " . $content;
            }
        }
    }
    
    // Get blog posts by category
    $categories = get_categories(array('hide_empty' => true));
    if (!empty($categories)) {
        $site_data[] = "\nMy content categories:";
        foreach ($categories as $category) {
            $site_data[] = "\n## " . $category->name;
            if ($category->description) {
                $site_data[] = "About: " . clean_content($category->description);
            }
            
            $posts = get_posts(array(
                'category' => $category->term_id,
                'post_status' => 'publish',
                'posts_per_page' => -1
            ));
            
            foreach ($posts as $post) {
                $content = clean_content($post->post_content);
                if ($content) {
                    $site_data[] = "\nArticle: " . $post->post_title;
                    $site_data[] = "Content: " . $content;
                }
            }
        }
    }
    
    // Create temporary file
    $upload_dir = wp_upload_dir();
    $temp_file = $upload_dir['path'] . '/site_content.txt';
    file_put_contents($temp_file, implode("\n", $site_data));
    
    // Prepare file data for freech_handle_document_upload
    $_FILES['rag_document'] = array(
        'name'     => 'site_content.txt',
        'tmp_name' => $temp_file,
        'type'     => 'text/plain',
        'error'    => 0,
        'size'     => filesize($temp_file)
    );
    
    // Create nonce for document upload
    $_POST['rag_upload_nonce'] = wp_create_nonce('rag-upload-nonce');
    
    // Use existing freech_handle_document_upload function
    $uploaded = freech_handle_document_upload();
    
    // Clean up temporary file
    wp_delete_file($temp_file);
    
    return $uploaded;
}

// Add this near the other add_action hooks in your plugin file
add_action('wp_ajax_rag_refresh_tokens', 'freech_rag_refresh_tokens');

// Function to handle the AJAX request to refresh tokens
function freech_rag_refresh_tokens() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'freech_rag_refresh_tokens')) {
    // if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'freech_rag_refresh_tokens')) {
        wp_send_json_error(array('message' => 'Security check failed'));
    }
    
    // Get site info (which will make a fresh API call)
    $site_info = freech_get_rag_site_info();
    
    if (isset($site_info['error'])) {
        wp_send_json_error(array('message' => $site_info['error']));
    } else {
        wp_send_json_success();
    }
    
    wp_die();
}

// Add this function to handle the AJAX request to regenerate site content
function freech_rag_regenerate_site_content() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'rag-regenerate-nonce')) {
    // if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'rag-regenerate-nonce')) {
        wp_send_json_error(array('message' => 'Security check failed.'));
        return;
    }
    
    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'You do not have permission to perform this action.'));
        return;
    }
    
    // Call the scan function
    $result = freech_rag_chatbot_scan_site();
    
    if ($result) {
        wp_send_json_success(array('message' => 'Site content has been successfully regenerated and uploaded to the RAG backend.'));
    } else {
        wp_send_json_error(array('message' => 'Failed to regenerate site content. Please check the logs for more details.'));
    }
}

// Register the AJAX action
add_action('wp_ajax_rag_regenerate_site_content', 'freech_rag_regenerate_site_content');

// Add this function to handle the AJAX request to refresh documents
function freech_rag_refresh_documents() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'rag-refresh-documents-nonce')) {
        wp_send_json_error(array('message' => 'Security check failed.'));
        return;
    }
    
    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'You do not have permission to perform this action.'));
        return;
    }
    
    // Capture output from the display function
    ob_start();
    freech_display_uploaded_documents();
    $html = ob_get_clean();
    
    wp_send_json_success(array('html' => $html));
}

// Register the AJAX action
add_action('wp_ajax_rag_refresh_documents', 'freech_rag_refresh_documents'); 