// Wait for DOM to be ready
document.addEventListener('DOMContentLoaded', function() {
    // console.log('DOM loaded');
    
    var chatIcon = document.getElementById('chat-icon');
    var chatWindow = document.getElementById('chat-window');
    var chatClose = document.getElementById('chat-close');
    var chatMessages = document.getElementById('chat-messages');
    var chatInput = document.getElementById('chat-input');
    var chatSend = document.getElementById('chat-send');
    
    // Toggle chat window on icon click
    if (chatIcon) {
        chatIcon.addEventListener('click', function() {
            // console.log('Icon clicked');
            
            if (chatWindow) {
                // Toggle the chat window
                if (chatWindow.style.display === 'block') {
                    chatWindow.style.display = 'none';
                } else {
                    chatWindow.style.display = 'block';
                    chatWindow.style.visibility = 'visible';
                    chatWindow.style.opacity = '1';
                    
                    // Focus the input field when opening
                    if (chatInput) {
                        setTimeout(function() {
                            chatInput.focus();
                        }, 100);
                    }
                    
                    // Add greeting message when opening the chat
                    // Check if this is the first time opening in this session
                    if (!window.chatGreetingShown) {
                        appendMessage('bot', 'Hello! How can I help you today?');
                        window.chatGreetingShown = true;
                    }
                }
            }
        });
    }
    
    // Add click handler to close button
    if (chatClose) {
        chatClose.addEventListener('click', function() {
            // console.log('Close clicked');
            
            // Force hide the chat window
            if (chatWindow) {
                chatWindow.style.display = 'none';
            }
        });
    }
    
    // Get stored session
    function getChatSession() {
        return localStorage.getItem('rag_chat_session');
    }

    // Store session
    function storeChatSession(sessionUuid) {
        localStorage.setItem('rag_chat_session', sessionUuid);
    }

    // Clear session
    function clearChatSession() {
        localStorage.removeItem('rag_chat_session');
    }
    
    // Send message on button click
    if (chatSend) {
        chatSend.addEventListener('click', function() {
            sendMessage();
        });
    }
    
    // Send message on Enter key
    if (chatInput) {
        chatInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                sendMessage();
            }
        });
    }
    
    function sendMessage() {
        if (!chatInput || !chatMessages) return;
        
        const message = chatInput.value.trim();
        if (!message) return;
        
        // Add user message to chat
        appendMessage('user', message);
        chatInput.value = '';
        
        // Show "typing" indicator
        showTypingIndicator();
        
        // Send to backend with current session
        var xhr = new XMLHttpRequest();
        xhr.open('POST', ragChatbotFrontend.ajaxurl, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4) {
                // Remove typing indicator
                hideTypingIndicator();
                
                if (xhr.status === 200) {
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response.success) {
                            // Store new session if provided
                            if (response.data.session_uuid) {
                                storeChatSession(response.data.session_uuid);
                            }
                            appendMessage('bot', response.data.content);
                        } else {
                            appendMessage('system', 'Error: ' + response.data.error);
                            // Clear session if it's invalid
                            if (response.data.error.includes('Invalid session')) {
                                clearChatSession();
                            }
                        }
                    } catch (e) {
                        appendMessage('system', 'Error processing response');
                        console.error('Parse Error:', e.message);
                        console.error('Response:', xhr.responseText);
                    }
                } else {
                    appendMessage('system', 'Network error. Please try again.');
                    console.error('HTTP Error:', xhr.status);
                    console.error('Response:', xhr.responseText);
                }
            }
        };
        
        var data = 'action=rag_chatbot_message&nonce=' + ragChatbotFrontend.nonce + '&message=' + encodeURIComponent(message) + '&session_uuid=' + encodeURIComponent(getChatSession() || '');
        xhr.send(data);
    }
    
    function showTypingIndicator() {
        if (!chatMessages) return;
        
        // Create typing indicator
        var typingDiv = document.createElement('div');
        typingDiv.id = 'typing-indicator';
        typingDiv.style.textAlign = 'center';
        typingDiv.style.margin = '10px 0';
        typingDiv.style.color = '#888';
        typingDiv.style.fontSize = '14px';
        typingDiv.style.fontStyle = 'italic';
        
        var typingText = document.createElement('span');
        typingText.textContent = 'Sending message';
        typingDiv.appendChild(typingText);
        
        var typingDots = document.createElement('span');
        typingDots.className = 'typing-dots';
        typingDots.textContent = '...';
        typingDiv.appendChild(typingDots);
        
        chatMessages.appendChild(typingDiv);
        chatMessages.scrollTop = chatMessages.scrollHeight;
        
        // Animate the dots
        var dotCount = 0;
        window.typingInterval = setInterval(function() {
            if (!document.getElementById('typing-indicator')) {
                clearInterval(window.typingInterval);
                return;
            }
            
            dotCount = (dotCount + 1) % 4;
            var dotText = '.'.repeat(dotCount);
            typingDots.textContent = dotText;
        }, 500);
    }
    
    function hideTypingIndicator() {
        if (!chatMessages) return;
        
        var typingIndicator = document.getElementById('typing-indicator');
        if (typingIndicator) {
            chatMessages.removeChild(typingIndicator);
        }
        
        if (window.typingInterval) {
            clearInterval(window.typingInterval);
        }
    }
    
    function appendMessage(sender, message) {
        if (!chatMessages) return;
        
        var messageDiv = document.createElement('div');
        messageDiv.className = 'message ' + sender;
        messageDiv.style.display = 'flex';
        messageDiv.style.marginBottom = '15px';
        messageDiv.style.width = '100%';
        
        if (sender === 'user') {
            messageDiv.style.justifyContent = 'flex-end';
        } else if (sender === 'bot') {
            messageDiv.style.justifyContent = 'flex-start';
        } else { // system message
            messageDiv.style.justifyContent = 'center';
        }
        
        var contentDiv = document.createElement('div');
        contentDiv.className = 'message-content';
        
        if (sender === 'user') {
            contentDiv.style.padding = '8px 12px';
            contentDiv.style.borderRadius = '18px';
            contentDiv.style.fontSize = '14px';
            contentDiv.style.maxWidth = '90%';
            contentDiv.style.lineHeight = '1.4';
            contentDiv.style.background = '#0084FF';
            contentDiv.style.color = 'white';
            contentDiv.style.borderBottomRightRadius = '4px';
            contentDiv.textContent = message;
        } else if (sender === 'bot') {
            contentDiv.style.padding = '8px 12px';
            contentDiv.style.borderRadius = '18px';
            contentDiv.style.fontSize = '14px';
            contentDiv.style.maxWidth = '90%';
            contentDiv.style.lineHeight = '1.4';
            contentDiv.style.background = '#F0F0F0';
            contentDiv.style.color = '#000';
            contentDiv.style.borderBottomLeftRadius = '4px';
            // Parse markdown for bot messages
            if (typeof marked !== 'undefined') {
                contentDiv.innerHTML = marked.parse(message);
            } else {
                contentDiv.textContent = message;
            }
        } else { // system message
            contentDiv.style.color = '#888';
            contentDiv.style.fontSize = '12px';
            contentDiv.style.fontStyle = 'italic';
            contentDiv.textContent = message;
        }
        
        messageDiv.appendChild(contentDiv);
        chatMessages.appendChild(messageDiv);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }
}); 