jQuery(document).ready(function($) {
    // Handle plan card selection
    $('.plan-card').on('click', function() {
        $('.plan-card').removeClass('selected');
        $(this).addClass('selected');
    });
    
    // Handle "Purchase" button click
    $('.select-plan').on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $planCard = $(this).closest('.plan-card');
        $('.plan-card').removeClass('selected');
        $planCard.addClass('selected');
        
        var planId = $planCard.data('plan-id');
        var planType = $planCard.data('plan-type');
        var amount = $planCard.data('plan-amount');
        var tokenCount = $planCard.data('plan-tokens');
        
        processPlanCheckout(planId, planType, amount, tokenCount);
    });
    
    // Process plan checkout
    function processPlanCheckout(planId, planType, amount, tokenCount) {
        var $error = $('#payment-error');
        
        // Show loading indicator
        $error.hide();
        $('.select-plan').prop('disabled', true);
        $('.selected .select-plan').html('<span class="dashicons dashicons-update spinning"></span> Processing...');
        
        // Make AJAX request to checkout
        $.ajax({
            url: ragChatbotAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'rag_checkout_payment',
                nonce: ragChatbotAdmin.payment_nonce,
                plan_id: planId,
                plan_type: planType,
                amount: amount,
                token_count: tokenCount
            },
            success: function(response) {
                if (response.success && response.data.checkout_url) {
                    // Open checkout in a new window
                    window.open(response.data.checkout_url, 'stripe_checkout', 'width=600,height=600');
                    
                    // Reset button state
                    $('.select-plan').prop('disabled', false);
                    $('.selected .select-plan').html('<span class="dashicons dashicons-cart"></span> Purchase');
                } else {
                    // Show error
                    $error.text(response.data.message || 'An error occurred').show();
                    $('.select-plan').prop('disabled', false);
                    $('.selected .select-plan').html('<span class="dashicons dashicons-cart"></span> Purchase');
                }
            },
            error: function() {
                // Show generic error
                $error.text('Failed to connect to the server. Please try again.').show();
                $('.select-plan').prop('disabled', false);
                $('.selected .select-plan').html('<span class="dashicons dashicons-cart"></span> Purchase');
            }
        });
    }
    
    // Use event delegation for delete document buttons
    $(document).on('click', '.delete-document', function() {
        var s3Key = $(this).data('s3key');
        deleteDocument(s3Key);
    });
    
    // Process document deletion
    function deleteDocument(s3Key) {
        if (!confirm('Are you sure you want to delete this document?')) {
            return;
        }
        
        // Disable all delete buttons while processing
        $('.delete-document').prop('disabled', true);
        
        // Make AJAX request to delete document
        $.ajax({
            url: ragChatbotAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'rag_delete_document',
                s3_key: s3Key,
                nonce: ragChatbotAdmin.delete_document_nonce
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert('Error deleting document: ' + response.data.message);
                    $('.delete-document').prop('disabled', false);
                }
            },
            error: function() {
                alert('Error deleting document. Please try again.');
                $('.delete-document').prop('disabled', false);
            }
        });
    }
    
    // Handle auto-recharge toggle
    $('#auto-recharge-toggle').on('change', function() {
        if ($(this).is(':checked')) {
            // When enabling, show plan selector if not already enabled
            $('#auto-recharge-options').slideDown();
            
            // If we're showing the current plan view, keep it as is
            if ($('.current-plan').is(':visible')) {
                return;
            }
            
            $('.toggle-status').html('<em>Click "Save Setting" to enable')
                                .removeClass('status-disabled status-enabled')
                                .addClass('status-pending');
        } else {
            // Show spinner while waiting for API response when disabling
            $('.toggle-status').html('<span class="spinner is-active" style="float:none;margin:0;"></span>')
                                .removeClass('status-enabled status-disabled status-pending');
            
            // Make direct API call to disable auto-recharge when toggled off
            $.ajax({
                url: ragChatbotAdmin.backend_url + '/api/v1/site/autorecharge',
                type: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'X-API-Key': ragChatbotAdmin.api_key
                },
                data: JSON.stringify({
                    'autorecharge': false
                }),
                success: function(response) {
                    // Update UI after successful response
                    $('#auto-recharge-options').slideUp();
                    $('.toggle-status').text('Disabled').removeClass('status-enabled status-pending').addClass('status-disabled');
                    
                    // Hide current plan view and show plan selector for next time
                    $('.current-plan').hide();
                    $('.plan-selector').hide();
                    $('#auto-recharge-plan').parent().show();
                },
                error: function(xhr, status, error) {
                    // Revert toggle on error
                    $('#auto-recharge-toggle').prop('checked', true);
                    
                    // If we had a current plan, keep showing it
                    if ($('.current-plan').length) {
                        $('.toggle-status').text('Enabled').removeClass('status-disabled status-pending').addClass('status-enabled');
                    } else {
                        $('.toggle-status').html('<em>Click "Save Setting" to enable</em>')
                                            .removeClass('status-disabled status-enabled')
                                            .addClass('status-pending');
                    }
                    
                    $('<div class="notice notice-error inline"><p>Error disabling auto-recharge: ' + error + '</p></div>')
                        .insertAfter($('#save-auto-recharge'))
                        .delay(3000)
                        .fadeOut();
                }
            });
        }
    });
    
    // Handle "Change Plan" button click
    $('#change-plan').on('click', function() {
        $('.current-plan').hide();
        $('.plan-selector').show();
    });

    // Handle "Cancel" button click
    $('.cancel-link').on('click', function(e) {
        e.preventDefault();
        $('.plan-selector').hide();
        $('.current-plan').show();
    });

    // Handle save auto-recharge settings
    $('#save-auto-recharge').on('click', function() {
        var $button = $(this);
        var enabled = $('#auto-recharge-toggle').is(':checked');
        var planId = $('#auto-recharge-plan').val();
        
        if (!enabled) {
            return; // Don't do anything if toggle is off
        }
        
        // Show spinner in status area while processing
        $('.toggle-status').html('<span class="spinner is-active" style="float:none;margin:0;"></span>')
                            .removeClass('status-enabled status-disabled status-pending');
        
        $button.prop('disabled', true).text('Processing...');
        
        // Make AJAX request to save auto-recharge settings
        $.ajax({
            url: ragChatbotAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'rag_save_autorecharge',
                nonce: ragChatbotAdmin.autorecharge_nonce,
                enabled: enabled,
                plan_id: planId
            },
            success: function(response) {
                $button.prop('disabled', false).text('Save Setting');
                
                if (response.success) {
                    if (response.data.checkout_url) {
                        // Redirect to checkout in the current window
                        window.location.href = response.data.checkout_url;
                    } else {
                        // Successfully enabled
                        $('.toggle-status').text('Enabled')
                                            .removeClass('status-disabled status-pending')
                                            .addClass('status-enabled');
                        
                        // Refresh the page to show current plan
                        location.reload();
                    }
                } else {
                    // Error occurred
                    $('.toggle-status').html('<em>Click "Save Setting" to enable</em>')
                                        .removeClass('status-disabled status-enabled')
                                        .addClass('status-pending');
                                        
                    $('<div class="notice notice-error inline"><p>' + (response.data.message || 'Error saving settings') + '</p></div>')
                        .insertAfter($button)
                        .delay(3000)
                        .fadeOut();
                }
            },
            error: function() {
                $button.prop('disabled', false).text('Save Setting');
                
                // Error occurred
                $('.toggle-status').html('<em>Click "Save Setting" to enable</em>')
                                    .removeClass('status-disabled status-enabled')
                                    .addClass('status-pending');
                                    
                $('<div class="notice notice-error inline"><p>Error communicating with server</p></div>')
                    .insertAfter($button)
                    .delay(3000)
                    .fadeOut();
            }
        });
    });

    $('#regenerate-site-content').on('click', function() {
        var $button = $(this);
        var $status = $('#regenerate-status');
        
        // Disable button and show loading state
        $button.prop('disabled', true).text('Processing...');
        $status.html('<span class="notice notice-info" style="display: block; padding: 10px;">Scanning site content and uploading to RAG backend...</span>').show();
        
        // Make AJAX call to regenerate site content
        $.ajax({
            url: ragChatbotAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'rag_regenerate_site_content',
                nonce: ragChatbotAdmin.regenerate_nonce
            },
            success: function(response) {
                if (response.success) {
                    $status.html('<div class="notice notice-success" style="display: block; padding: 10px;">' + response.data.message + '</div>');
                } else {
                    $status.html('<div class="notice notice-error" style="display: block; padding: 10px;">' + response.data.message + '</div>');
                }
            },
            error: function() {
                $status.html('<div class="notice notice-error" style="display: block; padding: 10px;">An error occurred while processing your request.</div>');
            },
            complete: function() {
                // Re-enable button with original text only (no icon)
                $button.prop('disabled', false).text('Regenerate Site Content');
            }
        });
    });

    $('#refresh-documents').on('click', function() {
        var $button = $(this);
        var $container = $('#documents-container');
        
        // Show loading state
        $button.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Refreshing...');
        
        // Make AJAX call to refresh documents list
        $.ajax({
            url: ragChatbotAdmin.ajaxurl,
            type: 'POST',
            data: {
                action: 'rag_refresh_documents',
                nonce: ragChatbotAdmin.refresh_documents_nonce
            },
            success: function(response) {
                if (response.success) {
                    $container.html(response.data.html);
                } else {
                    alert('Error refreshing documents: ' + response.data.message);
                }
            },
            error: function() {
                alert('An error occurred while refreshing documents.');
            },
            complete: function() {
                // Reset button state
                $button.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Refresh Status');
            }
        });
    });

    // Add file size validation
    $('#rag_document').on('change', function() {
        const file = this.files[0];
        const maxSize = 1 * 1024 * 1024; // 1MB in bytes
        const $error = $('#file-size-error');
        const $submit = $(this).closest('form').find('input[type="submit"]');

        if (file && file.size > maxSize) {
            $error.show();
            $submit.prop('disabled', true);
        } else {
            $error.hide();
            $submit.prop('disabled', false);
        }
    });

    // Add form submission handler
    $('#document-upload-form').on('submit', function() {
        const file = $('#rag_document')[0].files[0];
        const maxSize = 1 * 1024 * 1024; // 1MB in bytes

        if (file && file.size > maxSize) {
            $('#file-size-error').show();
            return false; // Prevent form submission
        }
        return true;
    });
    
});