<?php
if (!defined('ABSPATH')) {
    exit;
}

add_action('admin_menu', 'freech_rag_chatbot_admin_menu');

function freech_rag_chatbot_admin_menu() {
    add_menu_page(
        'FreeChatBots Settings',
        'FreeChatBots',
        'manage_options',
        'freech-rag-chatbot-settings',
        'freech_rag_chatbot_settings_page',
        'dashicons-format-chat'
    );
}

function freech_rag_chatbot_settings_page() {
    // Display messages when returning from Stripe
    if (isset($_GET['payment']) && $_GET['payment'] === 'success') {
        echo '<div class="notice notice-success is-dismissible"><p>Auto-recharge has been enabled.</p></div>';
    } elseif (isset($_GET['payment']) && $_GET['payment'] === 'cancel') {
        echo '<div class="notice notice-warning is-dismissible"><p>Auto-recharge has not been enabled.</p></div>';
    }

    // Handle re-registration if requested
    if (isset($_POST['rag_chatbot_reregister']) && check_admin_referer('rag_chatbot_reregister_nonce')) {
        delete_option('freech_rag_chatbot_api_key');
        $api_key = freech_rag_chatbot_register_with_backend();
    } else {
        $api_key = get_option('freech_rag_chatbot_api_key');
    }

    // Handle file upload if submitted
    if (isset($_FILES['rag_document'])) {
        freech_handle_document_upload();
    }
    
    settings_errors('rag_chatbot_messages');
    
    // Get and display debug info if available
    $debug_info = get_transient('freech_rag_debug_info');
    
    // Get site info including tokens
    $site_info = freech_get_rag_site_info();
    ?>
    <div class="wrap">
        <h1>FreeChatBots Settings</h1>
        
        <div class="rag-api-section">
            <h2>API Key</h2>
            <?php if ($api_key): ?>
                <p>Your unique API key: <code><?php echo esc_html($api_key); ?></code></p>
                <form method="post">
                    <?php wp_nonce_field('rag_chatbot_reregister_nonce'); ?>
                    <input type="submit" name="rag_chatbot_reregister" class="button button-secondary" 
                           value="Re-register with Backend" 
                           onclick="return confirm('Are you sure you want to re-register? This will generate a new API key.');">
                </form>
            <?php else: ?>
                <p class="notice notice-error">No API key found. Please try re-registering with the backend service.</p>
                <form method="post">
                    <?php wp_nonce_field('rag_chatbot_reregister_nonce'); ?>
                    <input type="submit" name="rag_chatbot_reregister" class="button button-primary" 
                           value="Register with Backend">
                </form>
            <?php endif; ?>
        </div>

        <!-- Tokens section -->
        <div class="rag-tokens-section">
            <h2>Tokens</h2>
            <?php if (isset($site_info['error'])): ?>
                <p class="notice notice-error"><?php echo esc_html($site_info['error']); ?></p>
            <?php elseif (isset($site_info['tokens'])): ?>
                <div class="tokens-card">
                    <div class="tokens-header">
                        <div class="tokens-info">
                            <span class="tokens-label">Available Tokens:</span>
                            <span id="tokens-value" class="tokens-value <?php echo $site_info['tokens'] < 0 ? 'tokens-negative' : ($site_info['tokens'] < 1000 ? 'tokens-low' : ''); ?>">
                                <?php echo number_format($site_info['tokens']); ?>
                            </span>
                            <a href="<?php echo esc_url(add_query_arg('refresh', 'true')); ?>" class="refresh-link" title="Refresh">
                                <span class="dashicons dashicons-update"></span>
                            </a>
                        </div>
                    </div>
                    
                    <!-- Plans section -->
                    <div class="plans-section">
                        <h3>Select a Plan</h3>
                        <?php 
                        // Get available plans
                        $plans = freech_get_rag_plans();
                        if (isset($plans['error'])): ?>
                            <p class="notice notice-error"><?php echo esc_html($plans['error']); ?></p>
                        <?php elseif (!empty($plans)): ?>
                            <div class="plan-cards">
                                <?php foreach ($plans as $plan): ?>
                                <div class="plan-card" data-plan-id="<?php echo esc_attr($plan['id']); ?>" data-plan-type="<?php echo esc_attr($plan['type']); ?>" data-plan-amount="<?php echo esc_attr($plan['amount']); ?>" data-plan-tokens="<?php echo esc_attr($plan['token_count']); ?>">
                                    <div class="plan-header">
                                        <div class="plan-name"><?php echo esc_html($plan['name']); ?></div>
                                        <div class="plan-tokens"><?php echo number_format($plan['token_count']); ?> tokens</div>
                                    </div>
                                    
                                    <div class="plan-price">
                                        $<?php echo number_format($plan['amount'], 2); ?>
                                    </div>
                                    
                                    <button class="select-plan button button-primary">
                                        <span class="dashicons dashicons-cart"></span> Purchase
                                    </button>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <div id="payment-error" class="payment-error" style="display: none;"></div>
                        <?php else: ?>
                            <p>No plans available at this time.</p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Auto-recharge section -->
                    <div class="auto-recharge-container">
                        <div class="auto-recharge-toggle">
                            <span class="toggle-label">Auto-Recharge when below 10,000 tokens:</span>
                            <label class="switch">
                                <input type="checkbox" id="auto-recharge-toggle" <?php echo isset($site_info['autorecharge']) && $site_info['autorecharge'] ? 'checked' : ''; ?>>
                                <span class="slider round"></span>
                            </label>
                            <span class="toggle-status <?php echo isset($site_info['autorecharge']) && $site_info['autorecharge'] ? 'status-enabled' : 'status-disabled'; ?>">
                                <?php echo isset($site_info['autorecharge']) && $site_info['autorecharge'] ? 'Enabled' : 'Disabled'; ?>
                            </span>
                        </div>
                        
                        <div id="auto-recharge-options" class="auto-recharge-options" style="<?php echo isset($site_info['autorecharge']) && $site_info['autorecharge'] ? 'display:block;' : 'display:none;'; ?>">
                            <?php if (isset($site_info['autorecharge']) && $site_info['autorecharge'] && isset($site_info['autorecharge_plan_id'])): ?>
                                <!-- Show current active plan when auto-recharge is enabled -->
                                <div class="current-plan">
                                    <strong>Current Plan:</strong>
                                    <?php 
                                    $current_plan_name = "Unknown Plan";
                                    $current_plan_tokens = 0;
                                    $current_plan_amount = 0;
                                    
                                    if (!empty($plans) && !isset($plans['error'])) {
                                        foreach ($plans as $plan) {
                                            if ($plan['id'] == $site_info['autorecharge_plan_id']) {
                                                $current_plan_name = $plan['name'];
                                                $current_plan_tokens = $plan['token_count'];
                                                $current_plan_amount = $plan['amount'];
                                                break;
                                            }
                                        }
                                    }
                                    ?>
                                    <span class="plan-details"><?php echo esc_html($current_plan_name); ?> - <?php echo number_format($current_plan_tokens); ?> tokens ($<?php echo number_format($current_plan_amount, 2); ?>)</span>
                                    <button id="change-plan" class="button">Change Plan</button>
                                </div>
                                <div class="plan-selector" style="display:none;">
                                    <select id="auto-recharge-plan">
                                        <?php if (!empty($plans) && !isset($plans['error'])): ?>
                                            <?php foreach ($plans as $plan): ?>
                                                <option value="<?php echo esc_attr($plan['id']); ?>" 
                                                        data-plan-type="<?php echo esc_attr($plan['type']); ?>"
                                                        data-plan-amount="<?php echo esc_attr($plan['amount']); ?>"
                                                        data-plan-tokens="<?php echo esc_attr($plan['token_count']); ?>"
                                                        <?php echo ($site_info['autorecharge_plan_id'] == $plan['id']) ? 'selected' : ''; ?>>
                                                    <?php echo esc_html($plan['name']); ?> - <?php echo number_format($plan['token_count']); ?> tokens ($<?php echo number_format($plan['amount'], 2); ?>)
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                    <button id="save-auto-recharge">Save Setting</button>
                                    <a href="#" id="cancel-change" class="cancel-link">Cancel</a>
                                </div>
                            <?php else: ?>
                                <!-- Show plan selector when auto-recharge is being enabled -->
                                <select id="auto-recharge-plan">
                                    <?php if (!empty($plans) && !isset($plans['error'])): ?>
                                        <?php foreach ($plans as $plan): ?>
                                            <option value="<?php echo esc_attr($plan['id']); ?>" 
                                                    data-plan-type="<?php echo esc_attr($plan['type']); ?>"
                                                    data-plan-amount="<?php echo esc_attr($plan['amount']); ?>"
                                                    data-plan-tokens="<?php echo esc_attr($plan['token_count']); ?>">
                                                <?php echo esc_html($plan['name']); ?> - <?php echo number_format($plan['token_count']); ?> tokens ($<?php echo number_format($plan['amount'], 2); ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </select>
                                <button id="save-auto-recharge">Save Setting</button>
                                <a href="#" id="cancel-change" class="cancel-link" style="display:none;">Cancel</a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <p>No token information available.</p>
                <a href="<?php echo esc_url(add_query_arg('refresh', 'true')); ?>" class="button button-secondary">Refresh Tokens</a>
            <?php endif; ?>
        </div>

        <div class="rag-upload-section">
            <h2>Upload Documents</h2>
            <form method="post" enctype="multipart/form-data" id="document-upload-form">
                <?php wp_nonce_field('rag-upload-nonce', 'rag_upload_nonce'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Select Document</th>
                        <td>
                            <input type="file" name="rag_document" id="rag_document" accept=".pdf,.doc,.docx,.txt" required>
                            <div class="upload-guidelines notice notice-info inline">
                                <p><strong>Upload Guidelines:</strong></p>
                                <ul>
                                    <li>Maximum file size: 1MB</li>
                                    <li>Supported formats: PDF, DOC, DOCX, TXT</li>
                                    <li>For optimal performance, we recommend uploading no more than 20 documents</li>
                                </ul>
                            </div>
                            <div id="file-size-error" class="notice notice-error inline" style="display: none;">
                                <p>This file is too large (max: 1MB). The file was not uploaded. Please select a smaller file.</p>
                            </div>
                        </td>
                    </tr>
                </table>
                <p class="submit">
                    <input type="submit" class="button button-primary" value="Upload Document">
                </p>
            </form>
        </div>

        <div class="rag-documents-list">
            <h2>Uploaded Documents</h2>
            <div class="documents-header">
                <button id="refresh-documents" class="button button-secondary button-with-icon">
                    <span class="dashicons dashicons-update"></span> Refresh Status
                </button>
            </div>
            <div id="documents-container">
                <?php freech_display_uploaded_documents(); ?>
            </div>
        </div>

        <!-- Moved this section below the Uploaded Documents section -->
        <div class="rag-regenerate-section">
            <h2>Regenerate Site Content</h2>
            <p>⚠️ For best accuracy, please remove any existing site content first before regenerating.</p>
            <p>Use this if you've added new content to your site since installing the plugin and want to update the chatbot's knowledge.</p>
            <button id="regenerate-site-content" class="button button-secondary">
                Regenerate Site Content
            </button>
            <div id="regenerate-status" style="margin-top: 10px; display: none;"></div>
        </div>
    </div>
    
    <?php
    
    // Clear the debug info after displaying
    delete_transient('freech_rag_debug_info');
}

// Helper function to get the domain
function freech_get_rag_domain() {
    return wp_parse_url(get_site_url(), PHP_URL_HOST);
}

// Update document display function
function freech_display_uploaded_documents() {
    $api_key = get_option('freech_rag_chatbot_api_key');
    $domain = freech_get_rag_domain();
    
    if (!$api_key) {
        echo '<p>No API key found. Please register first.</p>';
        return;
    }

    // Fetch documents from API with domain parameter
    $response = wp_remote_get(FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/documents?domain=' . urlencode($domain), array(
        'headers' => array(
            'X-API-Key' => $api_key
        )
    ));

    if (is_wp_error($response)) {
        echo '<p>Error fetching documents: ' . esc_html($response->get_error_message()) . '</p>';
        return;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    if ($status_code !== 200) {
        echo '<p>Error fetching documents. Status code: ' . esc_html($status_code) . '</p>';
        return;
    }

    $documents = json_decode(wp_remote_retrieve_body($response), true);

    if (empty($documents)) {
        echo '<p>No documents uploaded yet.</p>';
        return;
    }

    ?>
    <table class="wp-list-table widefat fixed striped">
        <thead>
            <tr>
                <th>Filename</th>
                <th>Status</th>
                <th>Created At</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($documents as $doc): ?>
                <tr>
                    <td><?php echo esc_html($doc['filename']); ?></td>
                    <td>
                        <?php 
                        $status_class = ($doc['status'] === 'completed' || $doc['status'] === 'indexed') ? 'status-complete' : 'status-pending';
                        echo '<span class="' . esc_attr($status_class) . '">' . esc_html($doc['status']) . '</span>';
                        if ($doc['error']) {
                            echo '<br><span class="error-message">' . esc_html($doc['error']) . '</span>';
                        }
                        ?>
                    </td>
                    <td><?php echo esc_html(wp_date('Y-m-d H:i:s', strtotime($doc['created_at']))); ?></td>
                    <td>
                        <button class="button button-small delete-document" 
                                data-s3key="<?php echo esc_attr($doc['s3_key']); ?>">
                            Delete
                        </button>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <?php
}

// Update document upload handler
function freech_handle_document_upload() {
    if (!isset($_POST['rag_upload_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['rag_upload_nonce'])), 'rag-upload-nonce')) {
    // if (!isset($_POST['rag_upload_nonce']) || !wp_verify_nonce($_POST['rag_upload_nonce'], 'rag-upload-nonce')) {
        wp_die('Security check failed');
    }

    $debug_info = [];
    $debug_info[] = "Starting upload process...";

    if (!isset($_FILES['rag_document']) || 
        !isset($_FILES['rag_document']['error']) || 
        $_FILES['rag_document']['error'] !== UPLOAD_ERR_OK) {
        $debug_info[] = "File upload failed: " . (isset($_FILES['rag_document']['error']) ? $_FILES['rag_document']['error'] : 'No file selected');
        set_transient('freech_rag_debug_info', $debug_info, 60);
        return false;
    }
    // if (!isset($_FILES['rag_document']) || $_FILES['rag_document']['error'] !== UPLOAD_ERR_OK) {
    //     $debug_info[] = "File upload failed: " . ($_FILES['rag_document']['error'] ?? 'No file selected');
    //     set_transient('freech_rag_debug_info', $debug_info, 60);
    //     return false;
    // }

    $file = array();
    if (isset($_FILES['rag_document'])) {
        // Sanitize file data
        $file['name'] = isset($_FILES['rag_document']['name']) ? sanitize_file_name($_FILES['rag_document']['name']) : '';
        $file['type'] = isset($_FILES['rag_document']['type']) ? sanitize_text_field($_FILES['rag_document']['type']) : '';
        $file['tmp_name'] = isset($_FILES['rag_document']['tmp_name']) ? sanitize_text_field($_FILES['rag_document']['tmp_name']) : '';
        $file['error'] = isset($_FILES['rag_document']['error']) ? (int)$_FILES['rag_document']['error'] : 0;
        $file['size'] = isset($_FILES['rag_document']['size']) ? (int)$_FILES['rag_document']['size'] : 0;
    }
    
    // Check file size (1MB limit)
    $max_size = 1 * 1024 * 1024; // 1MB in bytes
    if ($file['size'] > $max_size) {
        add_settings_error(
            'rag_chatbot_messages',
            'rag_chatbot_error',
            'File is too large (max: 1MB). The file was not uploaded. Please select a smaller file.',
            'error'
        );
        return false;
    }

    $api_key = get_option('freech_rag_chatbot_api_key');
    $domain = freech_get_rag_domain();
    
    $debug_info[] = "File name: " . $file['name'];
    $debug_info[] = "Domain: " . $domain;
    $debug_info[] = "API Key (first 5 chars): " . substr($api_key, 0, 5) . "...";

    // Create multipart/form-data boundary
    $boundary = wp_generate_password(24, false);
    
    // Prepare the multipart body
    $payload = '';
    $payload .= '--' . $boundary . "\r\n";
    $payload .= 'Content-Disposition: form-data; name="file"; filename="' . basename($file['name']) . '"' . "\r\n";
    $payload .= 'Content-Type: ' . $file['type'] . "\r\n\r\n";
    $payload .= file_get_contents($file['tmp_name']) . "\r\n";
    $payload .= '--' . $boundary . '--';

    $upload_url = FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/documents/upload?domain=' . urlencode($domain);
    $debug_info[] = "Making API request to: " . $upload_url;

    $response = wp_remote_post($upload_url, array(
        'headers' => array(
            'X-API-Key' => $api_key,
            'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
        ),
        'body' => $payload,
        'timeout' => 60,
    ));

    if (is_wp_error($response)) {
        $error_message = $response->get_error_message();
        $debug_info[] = "Upload failed with error: " . $error_message;
        set_transient('freech_rag_debug_info', $debug_info, 60);
        return false;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    
    $debug_info[] = "Response status code: " . $status_code;
    $debug_info[] = "Response body: " . $body;

    set_transient('freech_rag_debug_info', $debug_info, 60);

    if ($status_code !== 200) {
        add_settings_error(
            'rag_chatbot_messages',
            'rag_chatbot_error',
            'Upload failed: ' . ($body['message'] ?? 'Unknown error'),
            'error'
        );
        return false;
    }

    // Store document info in WordPress database
    global $wpdb;
    $wpdb->insert(
        $wpdb->prefix . 'rag_documents',
        array(
            'title' => sanitize_text_field($file['name']),
            'file_path' => sanitize_text_field($body['file_path'] ?? ''),
            'uploaded_at' => current_time('mysql')
        ),
        array('%s', '%s', '%s')
    );

    add_settings_error(
        'rag_chatbot_messages',
        'rag_chatbot_success',
        'Document uploaded successfully!',
        'success'
    );
    return true;
}

// Update document delete handler
function freech_handle_document_delete() {
    check_ajax_referer('rag_delete_document', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => 'Unauthorized'));
        return;
    }

    $s3_key = sanitize_text_field($_POST['s3_key']);
    $api_key = get_option('freech_rag_chatbot_api_key');
    $domain = freech_get_rag_domain();

    // Use rawurlencode instead of urlencode to properly handle spaces
    $delete_url = FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/documents/' . rawurlencode($s3_key) . '?domain=' . urlencode($domain);

    $response = wp_remote_request(
        $delete_url,
        array(
            'method' => 'DELETE',
            'headers' => array(
                'X-API-Key' => $api_key
            )
        )
    );

    if (is_wp_error($response)) {
        wp_send_json_error(array('message' => $response->get_error_message()));
        return;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    if ($status_code !== 200) {
        wp_send_json_error(array('message' => 'Failed to delete document. Status code: ' . $status_code));
        return;
    }

    wp_send_json_success();
}

add_action('wp_ajax_rag_delete_document', 'freech_handle_document_delete');

// Function to get site info including tokens
function freech_get_rag_site_info() {
    $api_key = get_option('freech_rag_chatbot_api_key');
    $domain = freech_get_rag_domain();
    
    if (!$api_key) {
        return array('error' => 'No API key found. Please register first.');
    }

    // Fetch site info from API
    $response = wp_remote_get(FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/site/info', array(
        'headers' => array(
            'X-API-Key' => $api_key,
            'Content-Type' => 'application/json'
        ),
        'timeout' => 15
    ));

    if (is_wp_error($response)) {
        return array('error' => 'Error fetching site info: ' . $response->get_error_message());
    }

    $status_code = wp_remote_retrieve_response_code($response);
    if ($status_code !== 200) {
        return array('error' => 'Error fetching site info. Status code: ' . $status_code);
    }

    return json_decode(wp_remote_retrieve_body($response), true);
}

// Add this function after freech_get_rag_site_info()
function freech_get_rag_plans() {
    $api_key = get_option('freech_rag_chatbot_api_key');
    
    if (!$api_key) {
        return array('error' => 'No API key found. Please register first.');
    }

    // Fetch plans from API
    $response = wp_remote_get(FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/plans', array(
        'headers' => array(
            'X-API-Key' => $api_key,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json'
        ),
        'timeout' => 15
    ));

    if (is_wp_error($response)) {
        return array('error' => 'Error fetching plans: ' . $response->get_error_message());
    }

    $status_code = wp_remote_retrieve_response_code($response);
    if ($status_code !== 200) {
        return array('error' => 'Error fetching plans. Status code: ' . $status_code);
    }

    return json_decode(wp_remote_retrieve_body($response), true);
}

// Add AJAX handler for payment checkout
add_action('wp_ajax_rag_checkout_payment', 'freech_rag_ajax_checkout_payment');
function freech_rag_ajax_checkout_payment() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'rag_payment_nonce')) {
    // if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'rag_payment_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
    }
    
    // Get amount and token count from request
    $amount = isset($_POST['amount']) ? floatval($_POST['amount']) : 10.00;
    $token_count = isset($_POST['token_count']) ? intval($_POST['token_count']) : 1000;
    
    $api_key = get_option('freech_rag_chatbot_api_key');
    
    if (!$api_key) {
        wp_send_json_error(array('message' => 'No API key found. Please register first.'));
    }
    
    // Create paths for success and cancel pages
    $success_url = rest_url('freechatbots/v1/payment/success');
    $cancel_url = rest_url('freechatbots/v1/payment/cancel');
    
    // Make API call to create checkout session
    $plan_id = isset($_POST['plan_id']) ? absint($_POST['plan_id']) : 0;
    if (!$plan_id) {
        wp_send_json_error(array('message' => 'Invalid plan ID provided.'));
        return;
    }

    $response = wp_remote_post(FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/payments/checkout/plan/' . $plan_id, array(
        'headers' => array(
            'X-API-Key' => $api_key,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json'
        ),
        'body' => json_encode(array(
            'success_url' => $success_url,
            'cancel_url' => $cancel_url
        )),
        'timeout' => 15
    ));
    
    if (is_wp_error($response)) {
        wp_send_json_error(array('message' => 'Error creating checkout: ' . $response->get_error_message()));
    }
    
    $status_code = wp_remote_retrieve_response_code($response);
    if ($status_code !== 200) {
        wp_send_json_error(array('message' => 'Error creating checkout. Status code: ' . $status_code));
    }
    
    $result = json_decode(wp_remote_retrieve_body($response), true);
    
    if (isset($result['checkout_url'])) {
        wp_send_json_success(array('checkout_url' => $result['checkout_url']));
    } else {
        wp_send_json_error(array('message' => 'No checkout URL received from the server.'));
    }
}

// Add this function after freech_rag_ajax_checkout_payment()
function freech_rag_ajax_save_autorecharge() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'rag_autorecharge_nonce')) {
    // if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'rag_autorecharge_nonce')) {
        wp_send_json_error(array('message' => 'Security check failed'));
    }
    
    $enabled = isset($_POST['enabled']) ? (bool)$_POST['enabled'] : false;
    $plan_id = absint($_POST['plan_id']); // Sanitize and validate plan_id as positive integer
    
    $api_key = get_option('freech_rag_chatbot_api_key');
    
    if (!$api_key) {
        wp_send_json_error(array('message' => 'No API key found. Please register first.'));
    }
    
    if ($enabled && $plan_id) {
        // Create paths for success and cancel pages - redirect back to settings page
        $success_url = admin_url('admin.php?page=freech-rag-chatbot-settings&payment=success');
        $cancel_url = admin_url('admin.php?page=freech-rag-chatbot-settings&payment=cancel');
        
        // Make API call to setup auto-recharge with the selected plan
        $response = wp_remote_post(FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/payments/autorecharge/setup/plan/' . $plan_id, array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ),
            'body' => json_encode(array(
                'success_url' => $success_url,
                'cancel_url' => $cancel_url
            )),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => 'Error setting up auto-recharge: ' . $response->get_error_message()));
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            wp_send_json_error(array('message' => 'Error setting up auto-recharge. Status code: ' . $status_code));
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['result']) && !empty($body['result'])) {
            wp_send_json_success(array('checkout_url' => $body['result']));
        } else {
            wp_send_json_error(array('message' => 'Invalid response from server'));
        }
    } else {
        // Just disable auto-recharge if enabled is false
        $response = wp_remote_post(FREECH_RAG_CHATBOT_BACKEND_URL . '/api/v1/site/autorecharge', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'enabled' => false
            )),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => 'Error disabling auto-recharge: ' . $response->get_error_message()));
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            wp_send_json_error(array('message' => 'Error disabling auto-recharge. Status code: ' . $status_code));
        }
        
        wp_send_json_success(array('disabled' => true));
    }
}
add_action('wp_ajax_rag_save_autorecharge', 'freech_rag_ajax_save_autorecharge');

// Enqueue scripts and styles for admin
add_action('admin_enqueue_scripts', 'rag_chatbot_admin_scripts');
function rag_chatbot_admin_scripts($hook) {
    // Only load on our plugin's admin page
    if (strpos($hook, 'freech-rag-chatbot-settings') === false) {
        return;
    }
    
    // Register and enqueue admin styles
    wp_register_style('rag-chatbot-admin-style', FREECH_RAG_CHATBOT_URL . 'css/admin-style.css', array(), '1.0.0');
    wp_enqueue_style('rag-chatbot-admin-style');
    
    // Register and enqueue admin script
    wp_register_script('rag-chatbot-admin-js', FREECH_RAG_CHATBOT_URL . 'js/admin.js', array('jquery'), '1.0.0', true);
    wp_enqueue_script('rag-chatbot-admin-js');
    
    // Pass data to the script
    wp_localize_script('rag-chatbot-admin-js', 'ragChatbotAdmin', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('rag_chatbot_admin_nonce'),
        'payment_nonce' => wp_create_nonce('rag_payment_nonce'),
        'autorecharge_nonce' => wp_create_nonce('rag_autorecharge_nonce'),
        'regenerate_nonce' => wp_create_nonce('rag-regenerate-nonce'),
        'refresh_documents_nonce' => wp_create_nonce('rag-refresh-documents-nonce'),
        'delete_document_nonce' => wp_create_nonce('rag_delete_document'),
        'api_key' => get_option('freech_rag_chatbot_api_key'),
        'backend_url' => FREECH_RAG_CHATBOT_BACKEND_URL
    ));
} 