<?php
/*
Plugin Name: Free CDN Box
Description: Replaces all image URLs in your content with CDN links for faster loading using a free CDN.
Version: 1.2
Author: Anon
License: GPLv2 or later
*/

if ( ! defined( 'ABSPATH' ) ) exit; // 直接アクセスを防ぐ

class fcbox_FreeImageCDN {
    const MAIN_FILE = __FILE__;
    private static $cdn_url = "cdn.statically.io/img";
    private static $image_extensions = array('jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'tiff', 'svg');

    public function __construct() {
        // デフォルト値の設定
        if (get_option('fcbox_enable_cdn') === false) {
            add_option('fcbox_enable_cdn', 1); // CDNをデフォルトで有効にする
        }

        if (get_option('fcbox_minify') === false) {
            add_option('fcbox_minify', 0); // ミニファイをデフォルトでオフにする
        }

        // CDN URL を取得
        self::$cdn_url = get_option('fcbox_cdn_url', self::$cdn_url);

        // 管理メニューを追加
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_styles']); // CSSを読み込む
        add_filter('plugin_row_meta', [$this, 'add_settings_link'], 10, 2); // 設定リンクを追加

        if (!$this->is_local_site()) {
            add_action('wp_enqueue_scripts', [$this, 'dns_prefetch']);
            add_action('template_redirect', [$this, 'start_buffering']);
        }
    }

    public function add_settings_page() {
        add_options_page(
            'Free CDN Box Settings',
            'Free CDN Box',
            'manage_options',
            'free-cdn-box',
            [$this, 'render_settings_page']
        );
    }

    public function register_settings() {
        register_setting('fcbox_options_group', 'fcbox_enable_cdn');
        register_setting('fcbox_options_group', 'fcbox_minify');
        register_setting('fcbox_options_group', 'fcbox_cdn_option'); // CDNオプションの追加
    }

    public function render_settings_page() {
        ?>
        <center>
        <div class="wrap fcbox-settings-page">
            <h1>Free CDN Box 設定</h1>
            <form method="post" action="options.php">
                <?php settings_fields('fcbox_options_group'); ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">CDN設定</th>
                        <td>
                            <div class="radio-option">
                               <tr valign="top">
                        <th scope="row">マルチCDNを活用して画像を最速で配信する</th>
                        <td>
                                <input type="radio" id="fcbox_enable_cdn" name="fcbox_cdn_option" value="multi" <?php checked('multi', get_option('fcbox_cdn_option')); ?> />
                            </div>
                            <div class="radio-option">
                            <tr valign="top">
                        <th scope="row">HTTPのサイトでCDNを利用する</th>
                        <td>
                                <input type="radio" id="fcbox_use_https" name="fcbox_cdn_option" value="https" <?php checked('https', get_option('fcbox_cdn_option')); ?> />
                            </div>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">圧縮設定</th>
                        <td>
                    <tr valign="top">
                        <th scope="row">HTMLを圧縮してWebサイト速度を高速化</th>
                        <td>
                            <div class="checkbox-option">
                                <input type="checkbox" name="fcbox_minify" value="1" <?php checked(1, get_option('fcbox_minify'), true); ?> />
                            </div>
                        </td>
                    </tr>
                </table>
                <input type="submit" class="button-primary submit-button" value="保存する" />
            </form>
        </div>
        <div class="image-container">
                    <h3>マルチCDN</h3>
        <img src="data:image/svg+xml;base64,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" alt="Image 1">
        <img src="data:image/svg+xml;base64,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" alt="Image 2">
        <img src="data:image/svg+xml;base64,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" alt="Image 3">
        </div>
        <?php
    }

    public function add_settings_link($links, $file) {
        if (strpos($file, 'free-cdn-box.php') !== false) {
            $settings_link = '<a href="' . admin_url('options-general.php?page=free-cdn-box') . '">設定</a>';
            array_unshift($links, $settings_link); // リンクの最初に追加
        }
        return $links;
    }

    // DNSプリフェッチ
    public function dns_prefetch() {
        wp_enqueue_script('dns-prefetch-wsrv', '//cdn.statically.io', array(), false, true);
    }

    // バッファリングを開始
    public function start_buffering() {
        ob_start(function($buffer) {
            // ミニファイが有効な場合、バッファをミニファイする
            if (get_option('fcbox_minify')) {
                $buffer = $this->minify($buffer);
            }
            return $this->add_cdn($buffer); // CDNも適用
        });
    }

    // CDN URLをコンテンツに追加
    public function add_cdn($content) {
        // CDNが有効化されているか確認
        if (get_option('fcbox_cdn_option') === 'multi') {
            $cdn_option = 'cdn.statically.io/img';
        } elseif (get_option('fcbox_cdn_option') === 'https') {
            $cdn_option = 'wsrv.nl/?url=http:/';
        } else {
            return $content; // CDNオプションが無効の場合、元のコンテンツを返す
        }

        $pattern = '/(' . preg_quote(content_url(), '/') . '\/[^\s"\']+\.(' . implode('|', self::$image_extensions) . '))/i';

        // 正規表現を使ってURLを置換
        $content = preg_replace_callback($pattern, function($matches) use ($cdn_option) {
            // URLからhttp://とhttps://を排除
            $url_without_protocol = preg_replace('/^https?:\/\//', '', $matches[1]);

            // CDN URLを構築
            return 'https://' . $cdn_option . '/' . $url_without_protocol;
        }, $content);

        return $content;
    }

    // ローカルサイトかどうかを確認
    public function is_local_site() {
        $site_url = site_url();
        $is_local = $site_url && false === strpos($site_url, '.');

        if ('local' === wp_get_environment_type()) {
            $is_local = true;
        }

        $known_local = array(
            '#\.local$#i',
            '#\.localhost$#i',
            '#\.test$#i',
            '#\.docksal$#i',
            '#\.docksal\.site$#i',
            '#\.dev\.cc$#i',
            '#\.lndo\.site$#i',
            '#^https?://127\.0\.0\.1$#',
        );

        if (!$is_local) {
            foreach ($known_local as $url) {
                if (preg_match($url, $site_url)) {
                    $is_local = true;
                    break;
                }
            }
        }

        return $is_local;
    }

    // HTMLミニファイ処理
    private function minify($buffer) {
        return preg_replace('/\s+/', ' ', $buffer);
    }

    // 管理スタイルを読み込む
    public function enqueue_admin_styles() {
        echo '<style>
            .fcbox-settings-page {
                background-color: #f9f9f9;
                border: 1px solid #ddd;
                padding: 20px;
                border-radius: 5px;
                max-width: 600px;
                margin: 20px auto;
            }
            .fcbox-settings-page h1 {
                font-size: 24px;
                margin-bottom: 20px;
            }
            .fcbox-settings-page table.form-table {
                width: 100%;
                border-collapse: collapse;
            }
            .fcbox-settings-page table.form-table th {
                text-align: left;
                padding: 10px;
                width: 55%;
            }
            .fcbox-settings-page table.form-table td {
                padding: 10px;
            }
            .fcbox-settings-page .radio-option,
            .fcbox-settings-page .checkbox-option {
                display: flex;
                align-items: center;
            }
            .fcbox-settings-page input[type="radio"],
            .fcbox-settings-page input[type="checkbox"] {
                transform: scale(1.2);
                margin-right: 10px;
            }
            .fcbox-settings-page .submit-button {
                background-color: #0073aa;
                color: white;
                padding: 10px 20px;
                border: none;
                border-radius: 3px;
                cursor: pointer;
            }
            .fcbox-settings-page .submit-button:hover {
                background-color: #005f8d;
            }
            .image-container {
                gap: 10px;
                padding: 0.5px;
            }
            .image-container img {
                width: 150px;
                height: auto;
                border-radius: 8px;
            }
        </style>';
    }
}

// プラグインを初期化
new fcbox_FreeImageCDN();
