<?php
/**
 * Plugin Name:       Frank Schema Markup Generator
 * Plugin URI:        https://wordpress.org/plugins/frank-schema-markup-generator/
 * Description:       Generate and manage Schema.org JSON-LD structured data markup for your WordPress site. Supports 100+ schema types with 50+ ready-made templates.
 * Version:           1.0.1
 * Author:            FARAZFRANK
 * Author URI:        https://wpfrank.com/
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       frank-schema-markup-generator
 * Domain Path:       /languages
 * Requires at least: 5.0
 * Requires PHP:      7.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Define plugin constants
define( 'SCMG_VERSION', '1.0.1' );
define( 'SCMG_DB_VERSION', '1.0' );
define( 'SCMG_TABLE_NAME', 'scmg_schemas' );
define( 'SCMG_PLUGIN_FILE', __FILE__ );
define( 'SCMG_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SCMG_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Create custom database table on plugin activation.
 */
function scmg_create_database_table() {
    global $wpdb;

    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        post_id bigint(20) UNSIGNED NOT NULL,
        post_type varchar(50) NOT NULL,
        schema_type varchar(50) NOT NULL,
        schema_fields longtext NOT NULL,
        generated_json longtext NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP NOT NULL,
        PRIMARY KEY  (id),
        KEY post_id (post_id),
        KEY post_type (post_type),
        KEY schema_type (schema_type)
    ) $charset_collate;";

    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    dbDelta( $sql );

    // Store database version
    add_option( 'scmg_db_version', SCMG_DB_VERSION );

    // Set activation redirect flag
    add_option( 'scmg_activation_redirect', true );
}
register_activation_hook( __FILE__, 'scmg_create_database_table' );

/**
 * Redirect to plugin page on activation.
 */
function scmg_activation_redirect() {
    // Check if we should redirect
    if ( get_option( 'scmg_activation_redirect', false ) ) {
        // Delete the redirect flag
        delete_option( 'scmg_activation_redirect' );

        // Don't redirect if activating multiple plugins at once
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is a WordPress core parameter for plugin activation, not user input
        if ( isset( $_GET['activate-multi'] ) ) {
            return;
        }

        // Redirect to plugin settings page
        wp_safe_redirect( admin_url( 'admin.php?page=scmg-settings' ) );
        exit;
    }
}
add_action( 'admin_init', 'scmg_activation_redirect' );

/**
 * Enqueue admin styles and scripts.
 *
 * @param string $hook The current admin page hook.
 */
function scmg_enqueue_admin_assets( $hook ) {
    // Only load on our settings page
    if ( 'toplevel_page_scmg-settings' !== $hook ) {
        return;
    }

    // Enqueue CSS
    wp_enqueue_style(
        'scmg-admin-styles',
        plugin_dir_url( __FILE__ ) . 'assets/css/admin-styles.css',
        array(),
        SCMG_VERSION
    );

    // Enqueue JavaScript
    wp_enqueue_script(
        'scmg-admin-scripts',
        plugin_dir_url( __FILE__ ) . 'assets/js/admin-scripts.js',
        array( 'jquery' ),
        SCMG_VERSION,
        true
    );

    // Localize script with AJAX URL and nonce
    wp_localize_script(
        'scmg-admin-scripts',
        'scmgAjax',
        array(
            'ajaxUrl'         => admin_url( 'admin-ajax.php' ),
            'nonce'           => wp_create_nonce( 'scmg_ajax_nonce' ),
            'loadPostsNonce'  => wp_create_nonce( 'scmg_load_posts' ),
        )
    );

    // Enqueue Manage Schemas JavaScript
    wp_enqueue_script(
        'scmg-manage-schemas',
        plugin_dir_url( __FILE__ ) . 'assets/js/manage-schemas.js',
        array( 'jquery' ),
        SCMG_VERSION,
        true
    );

    // Localize Manage Schemas script with AJAX URL, nonces, and i18n strings
    wp_localize_script(
        'scmg-manage-schemas',
        'scmgManageSchemas',
        array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonces'  => array(
                'filter'     => wp_create_nonce( 'scmg_filter_schemas' ),
                'delete'     => wp_create_nonce( 'scmg_delete_schema' ),
                'bulkDelete' => wp_create_nonce( 'scmg_bulk_delete_schemas' ),
            ),
            'i18n'    => array(
                'loading'               => __( 'Loading schemas...', 'frank-schema-markup-generator' ),
                'errorLoading'          => __( 'Error loading schemas.', 'frank-schema-markup-generator' ),
                'showing'               => __( 'Showing', 'frank-schema-markup-generator' ),
                'to'                    => __( 'to', 'frank-schema-markup-generator' ),
                'of'                    => __( 'of', 'frank-schema-markup-generator' ),
                'entries'               => __( 'entries', 'frank-schema-markup-generator' ),
                'confirmDelete'         => __( 'Are you sure you want to delete this schema? This action cannot be undone.', 'frank-schema-markup-generator' ),
                /* translators: %d is the number of schemas to be deleted */
                'confirmDeleteMultiple' => __( 'Are you sure you want to delete %d schemas? This action cannot be undone.', 'frank-schema-markup-generator' ),
                'deleting'              => __( 'Deleting...', 'frank-schema-markup-generator' ),
                'errorDeleting'         => __( 'Error deleting schema.', 'frank-schema-markup-generator' ),
                'successDeleting'       => __( 'Schemas deleted successfully!', 'frank-schema-markup-generator' ),
                'schemasSelected'       => __( 'schema(s) selected', 'frank-schema-markup-generator' ),
                'selectAction'          => __( 'Please select an action.', 'frank-schema-markup-generator' ),
                'selectAtLeastOne'      => __( 'Please select at least one schema.', 'frank-schema-markup-generator' ),
                'copied'                => __( 'Copied!', 'frank-schema-markup-generator' ),
            ),
        )
    );

    // Enqueue Generator Tab JavaScript
    wp_enqueue_script(
        'scmg-generator-scripts',
        plugin_dir_url( __FILE__ ) . 'assets/js/generator-scripts.js',
        array( 'jquery' ),
        SCMG_VERSION,
        true
    );

    // Localize Generator script with AJAX URL, nonces, and i18n strings
    wp_localize_script(
        'scmg-generator-scripts',
        'scmgGenerator',
        array(
            'ajaxUrl'         => admin_url( 'admin-ajax.php' ),
            'manageTabUrl'    => admin_url( 'admin.php?page=scmg-settings&tab=manage' ),
            'generatorTabUrl' => admin_url( 'admin.php?page=scmg-settings&tab=generator' ),
            'nonces'          => array(
                'loadPosts'   => wp_create_nonce( 'scmg_load_posts' ),
                'loadFields'  => wp_create_nonce( 'scmg_ajax_nonce' ),
                'saveSchema'  => wp_create_nonce( 'scmg_ajax_nonce' ),
            ),
            'i18n'            => array(
                'loadingPosts'        => __( 'Loading posts...', 'frank-schema-markup-generator' ),
                'selectPost'          => __( '-- Select a Post/Page --', 'frank-schema-markup-generator' ),
                'noPostsFound'        => __( 'No posts found', 'frank-schema-markup-generator' ),
                'errorLoadingPosts'   => __( 'Error loading posts', 'frank-schema-markup-generator' ),
                'loadingFields'       => __( 'Loading schema fields...', 'frank-schema-markup-generator' ),
                'errorLoadingFields'  => __( 'Error loading schema fields.', 'frank-schema-markup-generator' ),
                'errorMissingParams'  => __( 'Error: Missing required parameters.', 'frank-schema-markup-generator' ),
                'errorPrefix'         => __( 'Error: ', 'frank-schema-markup-generator' ),
                'unknownError'        => __( 'Unknown error', 'frank-schema-markup-generator' ),
                'ajaxError'           => __( 'AJAX Error: ', 'frank-schema-markup-generator' ),
                'completeAllSteps'    => __( 'Please complete all required steps before saving.', 'frank-schema-markup-generator' ),
                'saving'              => __( 'Saving...', 'frank-schema-markup-generator' ),
                'schemaSaved'         => __( 'Schema saved successfully!', 'frank-schema-markup-generator' ),
                'failedToSave'        => __( 'Failed to save schema', 'frank-schema-markup-generator' ),
                'failedToSaveCheck'   => __( 'Failed to save schema. Check console for details.', 'frank-schema-markup-generator' ),
                'confirmReset'        => __( 'Are you sure you want to reset the form? All unsaved changes will be lost.', 'frank-schema-markup-generator' ),
            ),
        )
    );
}
add_action( 'admin_enqueue_scripts', 'scmg_enqueue_admin_assets' );

/**
 * Add settings page to admin menu.
 */
function scmg_add_settings_page() {
    add_menu_page(
        __( 'Schema Markup Generator', 'frank-schema-markup-generator' ), // Page title
        __( 'Schema Markup', 'frank-schema-markup-generator' ), // Menu title
        'manage_options', // Capability
        'scmg-settings', // Menu slug
        'scmg_render_settings_page', // Callback function
        'dashicons-code-standards', // Icon
        65 // Position (after Plugins menu which is at 65)
    );
}
add_action( 'admin_menu', 'scmg_add_settings_page' );

/**
 * Register settings.
 */
function scmg_register_settings() {
    register_setting(
        'scmg_settings_group', // Option group
        'scmg_enabled_post_types', // Option name
        array(
            'type' => 'array',
            'sanitize_callback' => 'scmg_sanitize_post_types',
            'default' => array(),
        )
    );

    register_setting(
        'scmg_settings_group', // Option group
        'scmg_enabled_pages', // Option name
        array(
            'type' => 'array',
            'sanitize_callback' => 'scmg_sanitize_enabled_items',
            'default' => array(),
        )
    );

    register_setting(
        'scmg_settings_group', // Option group
        'scmg_enabled_posts', // Option name
        array(
            'type' => 'array',
            'sanitize_callback' => 'scmg_sanitize_enabled_items',
            'default' => array(),
        )
    );

    register_setting(
        'scmg_settings_group', // Option group
        'scmg_enabled_cpt_items', // Option name
        array(
            'type' => 'array',
            'sanitize_callback' => 'scmg_sanitize_enabled_items',
            'default' => array(),
        )
    );

    register_setting(
        'scmg_settings_group', // Option group
        'scmg_meta_box_type', // Option name
        array(
            'type' => 'array',
            'sanitize_callback' => 'scmg_sanitize_meta_box_type',
            'default' => array(),
        )
    );

    // REMOVED: Dynamic schema registration - no longer needed for WordPress.org compliance
}
add_action( 'admin_init', 'scmg_register_settings' );

/**
 * Sanitize the post types array.
 *
 * @param array $input The input array.
 * @return array The sanitized array.
 */
function scmg_sanitize_post_types( $input ) {
    if ( ! is_array( $input ) ) {
        return array();
    }

    $sanitized = array();
    foreach ( $input as $post_type => $enabled ) {
        $sanitized[ sanitize_key( $post_type ) ] = ( $enabled === '1' ) ? '1' : '0';
    }

    return $sanitized;
}

/**
 * Sanitize the enabled items array (pages, posts, CPT items).
 *
 * @param array $input The input array.
 * @return array The sanitized array.
 */
function scmg_sanitize_enabled_items( $input ) {
    if ( ! is_array( $input ) ) {
        return array();
    }

    $sanitized = array();
    foreach ( $input as $item_id => $enabled ) {
        $sanitized[ absint( $item_id ) ] = ( $enabled === '1' ) ? '1' : '0';
    }

    return $sanitized;
}

/**
 * Sanitize the meta box type array.
 *
 * @param array $input The input array.
 * @return array The sanitized array.
 */
function scmg_sanitize_meta_box_type( $input ) {
    if ( ! is_array( $input ) ) {
        return array();
    }

    $sanitized = array();
    foreach ( $input as $post_type => $type ) {
        $sanitized[ sanitize_key( $post_type ) ] = in_array( $type, array( 'individual', 'dynamic' ), true ) ? $type : 'individual';
    }

    return $sanitized;
}

/**
 * Render the settings page.
 */
function scmg_render_settings_page() {
    // Check user capabilities
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    // Get current tab
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab navigation only, no data modification, value sanitized
    $current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'generator';

    // Get current settings
    $enabled_post_types = get_option( 'scmg_enabled_post_types', array() );
    $enabled_pages = get_option( 'scmg_enabled_pages', array() );
    $enabled_posts = get_option( 'scmg_enabled_posts', array() );
    $enabled_cpt_items = get_option( 'scmg_enabled_cpt_items', array() );

    // Get Pages
    $pages = array();
    $page_obj = get_post_type_object( 'page' );
    if ( $page_obj ) {
        $pages[] = $page_obj;
    }

    // Get Posts
    $posts = array();
    $post_obj = get_post_type_object( 'post' );
    if ( $post_obj ) {
        $posts[] = $post_obj;
    }

    // Get Custom Post Types (non-built-in only)
    $custom_post_types = get_post_types( array( '_builtin' => false ), 'objects' );

    // Get all pages
    $all_pages = get_pages( array( 'number' => 500 ) );

    // Get all posts
    $all_posts = get_posts( array( 'numberposts' => 500, 'post_status' => 'any' ) );

    // Get all CPT items
    $all_cpt_items = array();
    foreach ( $custom_post_types as $cpt ) {
        $cpt_posts = get_posts( array(
            'post_type' => $cpt->name,
            'numberposts' => 500,
            'post_status' => 'any'
        ) );
        // Always add to array, even if empty
        $all_cpt_items[ $cpt->name ] = array(
            'label' => $cpt->labels->name,
            'items' => $cpt_posts
        );
    }

    ?>
    <div class="wrap csg-settings-wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

        <!-- Tab Navigation -->
        <h2 class="nav-tab-wrapper">
            <a href="?page=scmg-settings&tab=generator" class="nav-tab <?php echo 'generator' === $current_tab ? 'nav-tab-active' : ''; ?>">
                <?php esc_html_e( '⚡ Schema Generator', 'frank-schema-markup-generator' ); ?>
            </a>
            <a href="?page=scmg-settings&tab=manage" class="nav-tab <?php echo 'manage' === $current_tab ? 'nav-tab-active' : ''; ?>">
                <?php esc_html_e( '📋 Manage Schemas', 'frank-schema-markup-generator' ); ?>
            </a>
            <a href="?page=scmg-settings&tab=templates" class="nav-tab <?php echo 'templates' === $current_tab ? 'nav-tab-active' : ''; ?>">
                <?php esc_html_e( 'Schema Templates', 'frank-schema-markup-generator' ); ?>
            </a>
            <a href="?page=scmg-settings&tab=how-to-use" class="nav-tab <?php echo 'how-to-use' === $current_tab ? 'nav-tab-active' : ''; ?>">
                <?php esc_html_e( 'How to Use', 'frank-schema-markup-generator' ); ?>
            </a>
            <a href="?page=scmg-settings&tab=faq" class="nav-tab <?php echo 'faq' === $current_tab ? 'nav-tab-active' : ''; ?>">
                <?php esc_html_e( 'FAQ', 'frank-schema-markup-generator' ); ?>
            </a>
            <a href="?page=scmg-settings&tab=donate" class="nav-tab <?php echo 'donate' === $current_tab ? 'nav-tab-active' : ''; ?>">
                <?php esc_html_e( '❤️ Donate', 'frank-schema-markup-generator' ); ?>
            </a>
        </h2>

        <?php
        // Display content based on current tab
        switch ( $current_tab ) {
            case 'manage':
                scmg_render_manage_schemas_tab();
                break;
            case 'templates':
                scmg_render_templates_tab();
                break;
            case 'how-to-use':
                scmg_render_how_to_use_tab();
                break;
            case 'faq':
                scmg_render_faq_tab();
                break;
            case 'donate':
                scmg_render_donate_tab();
                break;
            case 'generator':
            default:
                scmg_render_generator_tab();
                break;
        }
        ?>
    </div>
    <?php
}


/**
 * Render the Schema Generator tab.
 */
function scmg_render_generator_tab() {
    // Check if editing existing schema
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Edit parameter for UI display, verified on save
    $editing_schema_id = isset( $_GET['edit'] ) ? absint( $_GET['edit'] ) : 0;
    $editing_schema = null;

    if ( $editing_schema_id > 0 ) {
        $editing_schema = scmg_get_schema_by_id( $editing_schema_id );
    }

    // Get all public post types excluding attachments
    $all_post_types = get_post_types( array( 'public' => true ), 'objects' );

    // Organize post types: Built-in first, then CPTs
    $post_types = array();
    $custom_post_types = array();

    foreach ( $all_post_types as $post_type ) {
        // Skip attachment (Media)
        if ( $post_type->name === 'attachment' ) {
            continue;
        }

        // Separate built-in and custom post types
        if ( in_array( $post_type->name, array( 'post', 'page' ), true ) ) {
            $post_types[] = $post_type;
        } else {
            $custom_post_types[] = $post_type;
        }
    }

    // Merge: Built-in first, then CPTs
    $post_types = array_merge( $post_types, $custom_post_types );

    // Get schema types
    $schema_types = scmg_get_schema_types();

    ?>
    <div class="scmg-generator-container">
        <h2>
            <?php
            if ( $editing_schema ) {
                esc_html_e( 'Edit Schema', 'frank-schema-markup-generator' );
            } else {
                esc_html_e( 'Schema Generator', 'frank-schema-markup-generator' );
            }
            ?>
        </h2>

        <?php if ( $editing_schema ) : ?>
            <div class="notice notice-warning scmg-notice-warning">
                <p><strong>✏️ <?php esc_html_e( 'Editing Mode:', 'frank-schema-markup-generator' ); ?></strong></p>
                <p><?php esc_html_e( 'You are editing an existing schema. Make your changes below and click "Generate & Save Schema" to update.', 'frank-schema-markup-generator' ); ?></p>
            </div>
        <?php endif; ?>

        <div class="scmg-generator-form">

            <!-- STEP 1: Post Type Selector -->
            <div class="scmg-form-group">
                <label class="scmg-form-label">
                    <span class="scmg-step-badge">1</span>
                    <?php esc_html_e( 'Select Post Type', 'frank-schema-markup-generator' ); ?>
                    <span class="scmg-required">*</span>
                </label>

                <!-- Built-in Post Types -->
                <?php
                $builtin_types = array();
                $cpt_types = array();

                foreach ( $post_types as $post_type ) {
                    if ( in_array( $post_type->name, array( 'post', 'page' ), true ) ) {
                        $builtin_types[] = $post_type;
                    } else {
                        $cpt_types[] = $post_type;
                    }
                }
                ?>

                <?php if ( ! empty( $builtin_types ) ) : ?>
                    <div class="scmg-section-container">
                        <p class="scmg-section-description">
                            <?php esc_html_e( 'Built-in Post Types:', 'frank-schema-markup-generator' ); ?>
                        </p>
                        <div class="scmg-post-type-selector">
                            <?php foreach ( $builtin_types as $post_type ) : ?>
                                <label class="scmg-post-type-label">
                                    <input type="radio" name="scmg_post_type" value="<?php echo esc_attr( $post_type->name ); ?>"
                                        <?php echo ( $editing_schema && get_post_type( $editing_schema->post_id ) === $post_type->name ) ? 'checked' : ''; ?>>
                                    <span>
                                        <?php echo esc_html( $post_type->labels->name ); ?>
                                    </span>
                                </label>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Custom Post Types -->
                <?php if ( ! empty( $cpt_types ) ) : ?>
                    <div>
                        <p class="scmg-section-description">
                            <?php esc_html_e( 'Custom Post Types:', 'frank-schema-markup-generator' ); ?>
                        </p>
                        <div class="scmg-post-type-selector">
                            <?php foreach ( $cpt_types as $post_type ) : ?>
                                <label class="scmg-post-type-label">
                                    <input type="radio" name="scmg_post_type" value="<?php echo esc_attr( $post_type->name ); ?>"
                                        <?php echo ( $editing_schema && get_post_type( $editing_schema->post_id ) === $post_type->name ) ? 'checked' : ''; ?>>
                                    <span>
                                        <?php echo esc_html( $post_type->labels->name ); ?>
                                    </span>
                                </label>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php else : ?>
                    <div class="scmg-info-box">
                        <p>
                            <span class="dashicons dashicons-info"></span>
                            <?php esc_html_e( 'No Custom Post Types found. Only built-in post types (Posts & Pages) are available.', 'frank-schema-markup-generator' ); ?>
                        </p>
                    </div>
                <?php endif; ?>

                <p class="description scmg-description-mt-12">
                    <?php esc_html_e( 'Choose the type of content you want to generate schema for.', 'frank-schema-markup-generator' ); ?>
                </p>
            </div>

            <!-- STEP 2: Post Selector (Hidden until post type selected) -->
            <div id="scmg_step_2_container" class="scmg-form-group-hidden">
                <label for="scmg_post_selector" class="scmg-form-label">
                    <span class="scmg-step-badge">2</span>
                    <?php esc_html_e( 'Select Post/Page', 'frank-schema-markup-generator' ); ?>
                    <span class="scmg-required">*</span>
                </label>

                <!-- Search Box -->
                <div class="scmg-section-container">
                    <input type="text" id="scmg_post_search" class="scmg-search-input" placeholder="<?php esc_attr_e( 'Search by title or enter Post ID...', 'frank-schema-markup-generator' ); ?>">
                    <p class="description scmg-description-mt">
                        <?php esc_html_e( 'Type to search by title, or enter a Post ID directly (e.g., "123").', 'frank-schema-markup-generator' ); ?>
                    </p>
                </div>

                <!-- Post Dropdown -->
                <select id="scmg_post_selector" class="scmg-select-dropdown">
                    <option value=""><?php esc_html_e( '-- Select a Post/Page --', 'frank-schema-markup-generator' ); ?></option>
                </select>
                <p class="description scmg-description-mt">
                    <?php esc_html_e( 'Showing last 20 posts. Use search above to find specific posts.', 'frank-schema-markup-generator' ); ?>
                </p>
            </div>

            <!-- STEP 3: Schema Type Selector (Hidden until post selected) -->
            <div id="scmg_step_3_container" class="scmg-form-group-hidden">
                <label for="scmg_generator_schema_type" class="scmg-form-label">
                    <span class="scmg-step-badge">3</span>
                    <?php esc_html_e( 'Select Schema Type', 'frank-schema-markup-generator' ); ?>
                    <span class="scmg-required">*</span>
                </label>
                <select id="scmg_generator_schema_type" class="scmg-select-dropdown">
                    <option value=""><?php esc_html_e( '-- Select Schema Type --', 'frank-schema-markup-generator' ); ?></option>
                    <?php
                    // Organize schema types by category
                    $categorized_types = array();
                    foreach ( $schema_types as $type_key => $type_label ) {
                        $category = scmg_get_schema_category( $type_key );
                        if ( ! isset( $categorized_types[ $category ] ) ) {
                            $categorized_types[ $category ] = array();
                        }
                        $categorized_types[ $category ][ $type_key ] = $type_label;
                    }

                    // Display schema types grouped by category
                    foreach ( $categorized_types as $category_name => $types ) :
                        ?>
                        <optgroup label="<?php echo esc_attr( $category_name ); ?>">
                            <?php foreach ( $types as $type_key => $type_label ) : ?>
                                <option value="<?php echo esc_attr( $type_key ); ?>" <?php echo ( $editing_schema && $editing_schema->schema_type === $type_key ) ? 'selected' : ''; ?>>
                                    <?php echo esc_html( $type_label ); ?>
                                </option>
                            <?php endforeach; ?>
                        </optgroup>
                    <?php endforeach; ?>
                </select>
                <p class="description scmg-description-mt">
                    <?php
                    printf(
                        /* translators: %d: number of schema types */
                        esc_html__( 'Select the type of schema markup you want to generate. Browse %d+ schema types organized by category.', 'frank-schema-markup-generator' ),
                        count( $schema_types )
                    );
                    ?>
                </p>
            </div>

            <!-- STEP 4: Schema Fields Container (Hidden until schema type selected) -->
            <div id="scmg_step_4_container" class="scmg-hidden">
                <div class="scmg-section-container">
                    <label class="scmg-form-label">
                        <span class="scmg-step-badge">4</span>
                        <?php esc_html_e( 'Fill Schema Fields', 'frank-schema-markup-generator' ); ?>
                        <span class="scmg-required">*</span>
                    </label>
                </div>

                <div id="scmg_generator_fields_container" class="scmg-fields-container">
                    <p class="description scmg-description">
                        <?php esc_html_e( 'Please complete steps 1-3 above to see the form fields.', 'frank-schema-markup-generator' ); ?>
                    </p>
                </div>

                <!-- Action Buttons -->
                <div class="scmg-form-actions">
                    <div class="scmg-form-actions-buttons">
                        <button type="button" id="scmg_generate_save_btn" class="button button-primary button-large scmg-btn-primary-large" disabled>
                            <?php esc_html_e( 'Generate & Save Schema', 'frank-schema-markup-generator' ); ?>
                        </button>
                        <button type="button" id="scmg_reset_form_btn" class="button button-secondary button-large scmg-btn-secondary-large">
                            <?php esc_html_e( 'Reset Form', 'frank-schema-markup-generator' ); ?>
                        </button>
                    </div>
                    <div id="scmg_generator_message" class="scmg-generator-message"></div>
                </div>
            </div>

            <!-- Hidden field for schema ID (used when editing) -->
            <input type="hidden" id="scmg_editing_schema_id" value="<?php echo $editing_schema ? esc_attr( $editing_schema->id ) : ''; ?>">

            <!-- Debug Info (visible in HTML source) -->
            <?php if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) : ?>
            <!-- DEBUG: editing_schema_id = <?php echo absint( $editing_schema_id ); ?> -->
            <!-- DEBUG: editing_schema exists = <?php echo $editing_schema ? 'YES' : 'NO'; ?> -->
            <?php if ( $editing_schema ) : ?>
            <!-- DEBUG: editing_schema->id = <?php echo absint( $editing_schema->id ); ?> -->
            <!-- DEBUG: editing_schema->post_id = <?php echo absint( $editing_schema->post_id ); ?> -->
            <!-- DEBUG: editing_schema->schema_type = <?php echo esc_html( $editing_schema->schema_type ); ?> -->
            <!-- DEBUG: post_type = <?php echo esc_html( get_post_type( $editing_schema->post_id ) ); ?> -->
            <?php endif; ?>
            <?php endif; ?>

            <!-- Data attributes for JavaScript -->
            <div id="scmg_generator_data" class="scmg-data-hidden"
                data-selected-post-id="<?php echo $editing_schema ? absint( $editing_schema->post_id ) : 0; ?>"
                data-selected-schema-type="<?php echo $editing_schema ? esc_attr( $editing_schema->schema_type ) : ''; ?>"
                data-is-editing="<?php echo $editing_schema ? '1' : '0'; ?>"
                <?php if ( $editing_schema ) : ?>
                data-editing-post-type="<?php echo esc_attr( get_post_type( $editing_schema->post_id ) ); ?>"
                data-editing-post-id="<?php echo absint( $editing_schema->post_id ); ?>"
                data-editing-schema-type="<?php echo esc_attr( $editing_schema->schema_type ); ?>"
                data-editing-schema-id="<?php echo absint( $editing_schema->id ); ?>"
                <?php endif; ?>>
            </div>
        </div>
    </div>
    <?php
}

/**
 * Render the Manage Schemas tab.
 */
function scmg_render_manage_schemas_tab() {
    // Get all public post types excluding attachments
    $all_post_types = get_post_types( array( 'public' => true ), 'objects' );
    $post_types = array();

    foreach ( $all_post_types as $post_type ) {
        // Skip attachment (Media)
        if ( $post_type->name === 'attachment' ) {
            continue;
        }
        $post_types[] = $post_type;
    }

    // Get total schemas count
    $total_schemas = scmg_get_schemas_count();

    ?>
    <div class="scmg-manage-schemas-container">
        <h2><?php esc_html_e( 'Manage Schemas', 'frank-schema-markup-generator' ); ?></h2>

        <p class="description scmg-description-mb-20">
            <?php esc_html_e( 'View, edit, delete, or copy all your generated schemas from one place.', 'frank-schema-markup-generator' ); ?>
        </p>

        <?php if ( $total_schemas === 0 ) : ?>
            <div class="notice notice-warning scmg-notice-padding">
                <h3><?php esc_html_e( 'No Schemas Found', 'frank-schema-markup-generator' ); ?></h3>
                <p><?php esc_html_e( 'You haven\'t created any schemas yet. Go to the "Schema Generator" tab to create your first schema.', 'frank-schema-markup-generator' ); ?></p>
                <p>
                    <a href="?page=scmg-settings&tab=generator" class="button button-primary">
                        <?php esc_html_e( 'Create Schema', 'frank-schema-markup-generator' ); ?>
                    </a>
                </p>
            </div>
        <?php else : ?>
            <!-- Info Notice -->
            <div class="notice notice-info scmg-notice-info-border">
                <p>
                    <span class="dashicons dashicons-info scmg-dashicon-info"></span>
                    <strong><?php esc_html_e( 'Duplicate Schema Prevention:', 'frank-schema-markup-generator' ); ?></strong>
                    <?php esc_html_e( 'Posts/pages that already have schemas are automatically hidden from the Schema Generator dropdown. This prevents duplicate schemas. To update an existing schema, use the "Edit" button below.', 'frank-schema-markup-generator' ); ?>
                </p>
            </div>

            <!-- Filters Section -->
            <div class="scmg-filters-section">
                <div>
                    <!-- Post Type Filter -->
                    <div>
                        <label for="scmg_filter_post_type">
                            <?php esc_html_e( 'Filter by Post Type:', 'frank-schema-markup-generator' ); ?>
                        </label>
                        <select id="scmg_filter_post_type">
                            <option value=""><?php esc_html_e( 'All Post Types', 'frank-schema-markup-generator' ); ?></option>

                            <?php
                            // Separate built-in and custom post types
                            $builtin_types = array();
                            $cpt_types = array();

                            foreach ( $post_types as $post_type ) {
                                if ( in_array( $post_type->name, array( 'post', 'page' ), true ) ) {
                                    $builtin_types[] = $post_type;
                                } else {
                                    $cpt_types[] = $post_type;
                                }
                            }
                            ?>

                            <?php if ( ! empty( $builtin_types ) ) : ?>
                                <optgroup label="<?php esc_attr_e( 'Built-in Post Types', 'frank-schema-markup-generator' ); ?>">
                                    <?php foreach ( $builtin_types as $post_type ) : ?>
                                        <option value="<?php echo esc_attr( $post_type->name ); ?>">
                                            <?php echo esc_html( $post_type->labels->name ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </optgroup>
                            <?php endif; ?>

                            <?php if ( ! empty( $cpt_types ) ) : ?>
                                <optgroup label="<?php esc_attr_e( 'Custom Post Types', 'frank-schema-markup-generator' ); ?>">
                                    <?php foreach ( $cpt_types as $post_type ) : ?>
                                        <option value="<?php echo esc_attr( $post_type->name ); ?>">
                                            <?php echo esc_html( $post_type->labels->name ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </optgroup>
                            <?php endif; ?>
                        </select>
                    </div>

                    <!-- Schema Type Filter -->
                    <div>
                        <label for="scmg_filter_schema_type">
                            <?php esc_html_e( 'Filter by Schema Type:', 'frank-schema-markup-generator' ); ?>
                        </label>
                        <select id="scmg_filter_schema_type">
                            <option value=""><?php esc_html_e( 'All Schema Types', 'frank-schema-markup-generator' ); ?></option>
                            <?php
                            $schema_types = scmg_get_schema_types();
                            foreach ( $schema_types as $type_key => $type_label ) :
                            ?>
                                <option value="<?php echo esc_attr( $type_key ); ?>">
                                    <?php echo esc_html( $type_label ); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Search Box -->
                    <div>
                        <label for="scmg_search_schemas">
                            <?php esc_html_e( 'Search:', 'frank-schema-markup-generator' ); ?>
                        </label>
                        <input type="text" id="scmg_search_schemas" placeholder="<?php esc_attr_e( 'Search by post title...', 'frank-schema-markup-generator' ); ?>">
                    </div>

                    <!-- Filter Buttons -->
                    <div class="scmg-filter-buttons">
                        <button type="button" id="scmg_apply_filters" class="button button-primary">
                            <span class="dashicons dashicons-filter"></span>
                            <?php esc_html_e( 'Apply Filters', 'frank-schema-markup-generator' ); ?>
                        </button>
                        <button type="button" id="scmg_reset_filters" class="button button-secondary">
                            <span class="dashicons dashicons-update scmg-dashicon-18"></span>
                            <?php esc_html_e( 'Reset', 'frank-schema-markup-generator' ); ?>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Stats Section -->
            <div class="scmg-schemas-stats">
                <strong><?php esc_html_e( 'Total Schemas:', 'frank-schema-markup-generator' ); ?></strong> <span id="scmg_total_count"><?php echo esc_html( $total_schemas ); ?></span>
                <span id="scmg_filtered_count" class="scmg-filtered-count">
                    <strong><?php esc_html_e( 'Filtered Results:', 'frank-schema-markup-generator' ); ?></strong> <span id="scmg_filtered_number">0</span>
                </span>
            </div>

            <!-- Bulk Actions -->
            <div class="scmg-bulk-actions-container" style="margin-bottom: 15px; display: none;">
                <select id="scmg_bulk_action" class="scmg-bulk-action-select">
                    <option value=""><?php esc_html_e( 'Bulk Actions', 'frank-schema-markup-generator' ); ?></option>
                    <option value="delete"><?php esc_html_e( 'Delete', 'frank-schema-markup-generator' ); ?></option>
                </select>
                <button type="button" id="scmg_apply_bulk_action" class="button button-secondary">
                    <?php esc_html_e( 'Apply', 'frank-schema-markup-generator' ); ?>
                </button>
                <span id="scmg_selected_count" class="scmg-selected-count" style="margin-left: 10px; font-weight: 600;"></span>
            </div>

            <!-- Schemas Table -->
            <div id="scmg_schemas_table_container">
                <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th class="check-column">
                            <input type="checkbox" id="scmg_select_all_schemas" />
                        </th>
                        <th class="scmg-sortable" data-column="id">
                            <?php esc_html_e( 'ID', 'frank-schema-markup-generator' ); ?>
                            <span class="scmg-sort-indicator"></span>
                        </th>
                        <th class="scmg-sortable" data-column="title">
                            <?php esc_html_e( 'Post/Page Title', 'frank-schema-markup-generator' ); ?>
                            <span class="scmg-sort-indicator"></span>
                        </th>
                        <th class="scmg-sortable" data-column="post_type">
                            <?php esc_html_e( 'Post Type', 'frank-schema-markup-generator' ); ?>
                            <span class="scmg-sort-indicator"></span>
                        </th>
                        <th class="scmg-sortable" data-column="schema_type">
                            <?php esc_html_e( 'Schema Type', 'frank-schema-markup-generator' ); ?>
                            <span class="scmg-sort-indicator"></span>
                        </th>
                        <th class="scmg-sortable" data-column="date">
                            <?php esc_html_e( 'Last Updated', 'frank-schema-markup-generator' ); ?>
                            <span class="scmg-sort-indicator"></span>
                        </th>
                        <th><?php esc_html_e( 'Actions', 'frank-schema-markup-generator' ); ?></th>
                    </tr>
                </thead>
                <tbody id="scmg_schemas_tbody">
                    <!-- Schemas will be loaded here via AJAX -->
                    <tr>
                        <td colspan="7" class="scmg-loading-cell">
                            <span class="dashicons dashicons-update spin scmg-loading-spinner-large"></span>
                            <br><br>
                            <span class="scmg-loading-text"><?php esc_html_e( 'Loading schemas...', 'frank-schema-markup-generator' ); ?></span>
                        </td>
                    </tr>
                </tbody>
            </table>

            <!-- Pagination -->
            <div id="scmg_pagination_container">
                <div class="scmg-pagination-info">
                    <span id="scmg_showing_text"><?php esc_html_e( 'Showing 0 to 0 of 0 entries', 'frank-schema-markup-generator' ); ?></span>
                    <span class="scmg-per-page-wrapper">
                        <label for="scmg_per_page"><?php esc_html_e( 'Show:', 'frank-schema-markup-generator' ); ?></label>
                        <select id="scmg_per_page">
                            <option value="10" selected>10</option>
                            <option value="25">25</option>
                            <option value="50">50</option>
                            <option value="100">100</option>
                        </select>
                        <span class="scmg-entries-label"><?php esc_html_e( 'entries', 'frank-schema-markup-generator' ); ?></span>
                    </span>
                </div>
                <div class="scmg-pagination-buttons" id="scmg_pagination_buttons">
                    <!-- Pagination buttons will be inserted here via JavaScript -->
                </div>
            </div>
            </div>
        <?php endif; ?>

        <!-- Schema Code Modal -->
        <div id="scmg-schema-modal" class="scmg-modal scmg-modal-hidden">
            <div class="scmg-modal-content">
                <span class="scmg-modal-close">&times;</span>
                <h2 id="scmg-schema-modal-title"><?php esc_html_e( 'Schema Code', 'frank-schema-markup-generator' ); ?></h2>
                <pre id="scmg-schema-modal-code" class="scmg-modal-code"></pre>
                <div class="scmg-modal-actions">
                    <button type="button" class="button button-primary scmg-modal-copy-schema scmg-modal-btn">
                        <span class="dashicons dashicons-clipboard"></span>
                        <?php esc_html_e( 'Copy to Clipboard', 'frank-schema-markup-generator' ); ?>
                    </button>
                    <button type="button" class="button scmg-modal-close-btn scmg-modal-btn">
                        <span class="dashicons dashicons-no"></span>
                        <?php esc_html_e( 'Close', 'frank-schema-markup-generator' ); ?>
                    </button>
                </div>
            </div>
        </div>
    </div>
    <?php
}

/**
 * Render the Schema Templates tab.
 */
function scmg_render_templates_tab() {
    ?>
    <div class="scmg-templates-container">
        <h2>Schema Templates Library</h2>
        <p class="description scmg-description-mb-20">These templates show the structure of different schema types for reference purposes.</p>

        <?php
        $templates = scmg_get_schema_templates();

        // Organize templates by category
        $categories = array(
            'Organization & Business' => array(),
            'Website & Content' => array(),
            'E-commerce & Products' => array(),
            'Events' => array(),
            'Jobs & Employment' => array(),
            'People & Personal Profiles' => array(),
            'Health & Medical' => array(),
            'Recipes & Food' => array(),
            'Real Estate & Property' => array(),
            'Travel & Transportation' => array(),
            'Religious Places' => array(),
            'Education & Courses' => array(),
            'Media & Entertainment' => array(),
            'Technology & Software' => array(),
            'Visual & Image Content' => array(),
            'Games' => array(),
            'Data & Feeds' => array(),
            'Special Announcements' => array(),
        );

        // Categorize templates using smart categorization
        foreach ( $templates as $template_id => $template ) {
            // Use category from template if set, otherwise auto-detect from type
            if ( isset( $template['category'] ) ) {
                $category = $template['category'];
            } else {
                $category = scmg_get_schema_category( $template['type'] );
            }

            if ( isset( $categories[ $category ] ) ) {
                $categories[ $category ][ $template_id ] = $template;
            }
        }

        // Display templates by category
        foreach ( $categories as $category_name => $category_templates ) :
            if ( empty( $category_templates ) ) {
                continue;
            }
        ?>
            <div class="scmg-template-category">
                <h3 class="scmg-template-category-title">
                    <?php echo esc_html( $category_name ); ?>
                    <span class="scmg-template-count">
                        (<?php echo count( $category_templates ); ?> <?php echo count( $category_templates ) === 1 ? 'template' : 'templates'; ?>)
                    </span>
                </h3>
                <div class="scmg-templates-grid">
                    <?php foreach ( $category_templates as $template_id => $template ) : ?>
                        <div class="scmg-template-card">
                            <div class="scmg-template-header">
                                <h3><?php echo esc_html( $template['name'] ); ?></h3>
                                <span class="scmg-template-type"><?php echo esc_html( $template['type'] ); ?></span>
                            </div>
                            <p class="scmg-template-description"><?php echo esc_html( $template['description'] ); ?></p>
                            <div class="scmg-template-actions">
                                <button type="button" class="button button-primary scmg-copy-template" data-template-id="<?php echo esc_attr( $template_id ); ?>">
                                    Copy Template
                                </button>
                                <button type="button" class="button scmg-view-template" data-template-id="<?php echo esc_attr( $template_id ); ?>">
                                    View Code
                                </button>
                            </div>
                            <pre class="scmg-template-code scmg-template-code-hidden" id="template-<?php echo esc_attr( $template_id ); ?>"><?php echo esc_html( wp_json_encode( $template['schema'], JSON_PRETTY_PRINT ) ); ?></pre>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endforeach; ?>

        <!-- Template Preview Modal -->
        <div id="scmg-template-modal" class="scmg-modal scmg-modal-display-none">
            <div class="scmg-modal-content">
                <span class="scmg-template-modal-close">&times;</span>
                <h2 id="scmg-modal-title">Template Code</h2>
                <pre id="scmg-modal-code"></pre>
                <button type="button" class="button button-primary scmg-template-modal-copy">Copy to Clipboard</button>
            </div>
        </div>
    </div>
    <?php
}

/**
 * Render the How to Use tab.
 */
function scmg_render_how_to_use_tab() {
    ?>
    <div class="scmg-how-to-use">
        <h2>📖 How to Use Schema Markup Generator</h2>

        <div class="scmg-guide-section scmg-warning-box">
            <h3>🎯 Quick Start Guide</h3>
            <p><strong>Follow these 4 simple steps:</strong></p>
            <ol>
                <li><strong>Generate Schema</strong> → Go to "Schema Generator" tab</li>
                <li><strong>Copy Schema Code</strong> → Go to "Manage Schemas" tab and click "Copy" or "View"</li>
                <li><strong>Add to Your Site</strong> → Use a code snippet plugin or theme editor</li>
                <li><strong>Test & Validate</strong> → Use Google Rich Results Test</li>
            </ol>
        </div>

        <div class="scmg-guide-section">
            <h3>⚡ Step 1: Generate Schema Markup</h3>
            <ol>
                <li>Go to <strong>Settings → Schema Box Generator → Schema Generator</strong> tab</li>
                <li>Select a <strong>Post/Page/CPT</strong> from the dropdown (or leave empty for site-wide schema)</li>
                <li>Choose a <strong>Schema Type</strong> from 150+ available types (Article, Product, Event, Recipe, LocalBusiness, etc.)</li>
                <li>Fill in the form fields that appear (the plugin shows relevant fields for your selected type)</li>
                <li>Use <strong>dynamic placeholders</strong> like <code>{{post_title}}</code> or <code>{{author_name}}</code> for automatic data population</li>
                <li>Click <strong>"Generate & Save Schema"</strong> button</li>
                <li>You'll be automatically redirected to the <strong>"Manage Schemas"</strong> tab</li>
            </ol>
            <div class="scmg-guide-example">
                <h4>💡 Example: Article Schema Fields</h4>
                <ul>
                    <li><strong>Headline:</strong> <code>{{post_title}}</code> → Automatically uses your post title</li>
                    <li><strong>Description:</strong> <code>{{post_excerpt}}</code> → Automatically uses post excerpt</li>
                    <li><strong>Author Name:</strong> <code>{{author_name}}</code> → Automatically uses author's display name</li>
                    <li><strong>Publisher Name:</strong> <code>{{site_name}}</code> → Automatically uses your site name</li>
                    <li><strong>Date Published:</strong> <code>{{post_date}}</code> → Automatically uses publication date</li>
                    <li><strong>Image URL:</strong> <code>{{featured_image}}</code> → Automatically uses featured image URL</li>
                </ul>
                <p><em>✅ The plugin will generate proper JSON-LD schema from these fields automatically!</em></p>
            </div>
        </div>

        <div class="scmg-guide-section scmg-tips-box">
            <h3>📋 Step 2: Copy Schema Code (MOST IMPORTANT!)</h3>
            <p><strong>After generating schema, you need to copy the code to use it on your site. Here's how:</strong></p>

            <h4>🔵 Method 1: Quick Copy from Table (Fastest)</h4>
            <ol>
                <li>Go to <strong>"Manage Schemas"</strong> tab</li>
                <li>Find your schema in the table</li>
                <li>Click the <strong>"Copy"</strong> button (📋 icon) in the Actions column</li>
                <li>You'll see a success message: <strong>"✓ Copied!"</strong></li>
                <li>The complete schema code is now in your clipboard!</li>
            </ol>

            <h4>🔵 Method 2: View & Copy from Modal (Recommended for Review)</h4>
            <ol>
                <li>Go to <strong>"Manage Schemas"</strong> tab</li>
                <li>Find your schema in the table</li>
                <li>Click the <strong>"View"</strong> button (👁️ icon) in the Actions column</li>
                <li>A popup modal will open showing the <strong>formatted schema code</strong></li>
                <li>Review the code to make sure it looks correct</li>
                <li>Click the <strong>"Copy to Clipboard"</strong> button inside the modal</li>
                <li>You'll see: <strong>"✓ Copied!"</strong></li>
                <li>Close the modal by clicking X or outside the modal</li>
            </ol>

            <div class="scmg-step-box">
                <h4>📝 What Gets Copied?</h4>
                <p>The copied code includes the complete JSON-LD schema wrapped in a <code>&lt;script&gt;</code> tag:</p>
                <pre class="scmg-modal-code">&lt;script type="application/ld+json"&gt;
{
  "@context": "https://schema.org",
  "@type": "Article",
  "headline": "{{post_title}}",
  "author": {
    "@type": "Person",
    "name": "{{author_name}}"
  },
  ...
}
&lt;/script&gt;</pre>
                <p><em>✅ This is ready to paste directly into your theme or code snippet plugin!</em></p>
            </div>
        </div>

        <div class="scmg-guide-section">
            <h3>🚀 Step 3: Add Schema Code to Your Site</h3>
            <div class="notice notice-warning scmg-notice-info-border">
                <p><strong>⚠️ Important:</strong> This plugin generates schema markup but does NOT automatically insert it into your pages (WordPress.org compliance requirement). You must manually add the copied code to your site using one of the methods below.</p>
            </div>

            <h4><strong>✅ Option 1: Using a Code Snippet Plugin (HIGHLY RECOMMENDED)</strong></h4>
            <p><strong>Why this is the best method:</strong></p>
            <ul>
                <li>✅ Won't be affected by theme updates</li>
                <li>✅ Easy to manage and edit</li>
                <li>✅ Can enable/disable without deleting code</li>
                <li>✅ No need to edit theme files</li>
            </ul>
            <p><strong>Step-by-step instructions:</strong></p>
            <ol>
                <li><strong>Install a code snippet plugin:</strong>
                    <ul>
                        <li><a href="https://wordpress.org/plugins/code-snippets/" target="_blank">Code Snippets</a> (Most popular, recommended)</li>
                        <li><a href="https://wordpress.org/plugins/insert-headers-and-footers/" target="_blank">Insert Headers and Footers</a> (Simple option)</li>
                        <li><a href="https://wordpress.org/plugins/wp-crontrol/" target="_blank">WPCode</a> (Advanced features)</li>
                    </ul>
                </li>
                <li><strong>Copy your schema code</strong> from the "Manage Schemas" tab (using Copy or View button)</li>
                <li><strong>Go to the code snippet plugin settings:</strong>
                    <ul>
                        <li>For "Code Snippets": Go to <strong>Snippets → Add New</strong></li>
                        <li>For "Insert Headers and Footers": Go to <strong>Settings → Insert Headers and Footers</strong></li>
                    </ul>
                </li>
                <li><strong>Paste the schema code</strong> into the header section or create a new snippet</li>
                <li><strong>Configure where to display:</strong>
                    <ul>
                        <li>Select "Run everywhere" for site-wide schema (Organization, WebSite)</li>
                        <li>Select specific pages/posts for content-specific schema (Article, Product)</li>
                    </ul>
                </li>
                <li><strong>Save and activate</strong> the snippet</li>
            </ol>
            <p><em>💡 Pro Tip: Create separate snippets for different schema types (one for Organization, one for Articles, etc.) for better organization!</em></p>

            <h4 class="scmg-mt-30"><strong>⚠️ Option 2: Manual Theme Integration (Advanced Users)</strong></h4>
            <p><strong>Warning:</strong> This method requires editing theme files. Code will be lost if you change themes!</p>
            <ol>
                <li><strong>Copy your schema code</strong> from the "Manage Schemas" tab</li>
                <li>Go to <strong>Appearance → Theme File Editor</strong></li>
                <li>⚠️ <strong>Important:</strong> Create a child theme first if you haven't already (to prevent losing changes on theme updates)</li>
                <li>Open your theme's <code>header.php</code> file from the right sidebar</li>
                <li>Find the <code>&lt;/head&gt;</code> closing tag (usually near the top of the file)</li>
                <li><strong>Paste the schema code</strong> just BEFORE the <code>&lt;/head&gt;</code> tag</li>
                <li>Click <strong>"Update File"</strong> to save</li>
            </ol>
            <p><em>⚠️ Note: If you update or change your theme, you'll need to add the code again!</em></p>

            <h4 class="scmg-mt-30"><strong>🔧 Option 3: Using functions.php (For Developers)</strong></h4>
            <p>Add schema dynamically using WordPress hooks:</p>
            <ol>
                <li>Go to <strong>Appearance → Theme File Editor</strong></li>
                <li>Open <code>functions.php</code> (or create a custom plugin)</li>
                <li>Add this code:</li>
            </ol>
            <pre class="scmg-code-block">// Add schema to wp_head
function my_custom_schema() {
    // Paste your copied schema code here
    echo '&lt;script type="application/ld+json"&gt;
    {
      "@context": "https://schema.org",
      "@type": "Organization",
      "name": "Your Company Name"
    }
    &lt;/script&gt;';
}
add_action('wp_head', 'my_custom_schema');</pre>
            <p><em>💡 This gives you full control over when and where schema appears!</em></p>
        </div>



        <div class="scmg-guide-section">
            <h3>🔍 Step 4: Test & Validate Your Schema</h3>
            <p><strong>After adding schema to your site, it's crucial to test it to ensure it's working correctly!</strong></p>

            <h4>✅ Method 1: Check Page Source (Quick Verification)</h4>
            <ol>
                <li>Visit the page where you added the schema on your live website</li>
                <li>Right-click anywhere on the page and select <strong>"View Page Source"</strong> (or press Ctrl+U / Cmd+U)</li>
                <li>Press <strong>Ctrl+F</strong> (or Cmd+F) to open the search box</li>
                <li>Search for: <code>&lt;script type="application/ld+json"&gt;</code></li>
                <li>Verify your schema code appears in the HTML source</li>
                <li>Check that placeholders like <code>{{post_title}}</code> have been replaced with actual values</li>
            </ol>
            <p><em>✅ If you see your schema code with real values (not placeholders), it's working!</em></p>

            <h4 class="scmg-mt-20">✅ Method 2: Google Rich Results Test (Recommended)</h4>
            <ol>
                <li>Go to <a href="https://search.google.com/test/rich-results" target="_blank"><strong>Google Rich Results Test</strong></a></li>
                <li>Enter your page URL in the search box</li>
                <li>Click <strong>"Test URL"</strong> button</li>
                <li>Wait for Google to analyze your page (takes 10-30 seconds)</li>
                <li>Check the results:
                    <ul>
                        <li><strong>✅ Green checkmark:</strong> Schema is valid and eligible for rich results!</li>
                        <li><strong>⚠️ Yellow warning:</strong> Schema is valid but has optional improvements</li>
                        <li><strong>❌ Red error:</strong> Schema has errors that need to be fixed</li>
                    </ul>
                </li>
                <li>Click on the schema type to see detailed information</li>
                <li>Fix any errors or warnings by editing your schema in the plugin</li>
            </ol>
            <p><em>💡 Pro Tip: Bookmark this tool and test your schema every time you make changes!</em></p>

            <h4 class="scmg-mt-20">✅ Method 3: Schema.org Validator (Advanced)</h4>
            <ol>
                <li>Go to <a href="https://validator.schema.org/" target="_blank"><strong>Schema.org Validator</strong></a></li>
                <li>Paste your schema code or enter your page URL</li>
                <li>Click <strong>"Run Test"</strong></li>
                <li>Review the validation results and fix any errors</li>
            </ol>

            <div class="scmg-warning-box scmg-mt-20">
                <h4>⚠️ Troubleshooting: Schema Not Appearing?</h4>
                <ul>
                    <li>✅ Make sure you've <strong>copied the complete schema code</strong> (including <code>&lt;script&gt;</code> tags)</li>
                    <li>✅ Verify the code snippet plugin is <strong>activated</strong></li>
                    <li>✅ Check that the snippet is set to run on the <strong>correct pages</strong></li>
                    <li>✅ Clear your browser cache and website cache (if using a caching plugin)</li>
                    <li>✅ Check browser console for JavaScript errors (F12 → Console tab)</li>
                    <li>✅ Make sure the schema code is in the <code>&lt;head&gt;</code> section, not the body</li>
                </ul>
            </div>
        </div>

        <div class="scmg-guide-section">
            <h3>📋 Step 5: Manage & Edit Schemas</h3>
            <p><strong>The "Manage Schemas" tab provides powerful tools to organize your schemas:</strong></p>

            <h4>Available Actions:</h4>
            <ul>
                <li><strong>👁️ View:</strong> Opens a modal popup showing the formatted schema code with syntax highlighting</li>
                <li><strong>📋 Copy:</strong> Instantly copies the schema code to your clipboard (fastest method)</li>
                <li><strong>✏️ Edit:</strong> Takes you back to the Schema Generator tab with the form pre-filled for editing</li>
                <li><strong>🗑️ Delete:</strong> Permanently removes the schema from the database (with confirmation)</li>
            </ul>

            <h4 class="scmg-mt-20">Filter & Search:</h4>
            <ul>
                <li><strong>Filter by Post Type:</strong> Show only schemas for Posts, Pages, or specific Custom Post Types</li>
                <li><strong>Filter by Schema Type:</strong> Show only Article, Product, Event, or other schema types</li>
                <li><strong>Search:</strong> Find schemas by post title or content</li>
            </ul>

            <p><em>💡 Tip: Use the Edit button to update schema when your content changes, then copy the updated code!</em></p>
        </div>

        <div class="scmg-guide-section">
            <h3>⚙️ Step 6: Using Dynamic Placeholders (Advanced)</h3>
            <p><strong>Placeholders are powerful variables that automatically populate with your WordPress data!</strong></p>
            <p>Instead of manually entering values, use placeholders to make your schema dynamic and automatically update when content changes.</p>
            <p><em><strong>Note:</strong> The "Example Output" column shows sample data for illustration purposes only. Actual values will be dynamically generated from your WordPress site content.</em></p>
            <table class="widefat">
                <thead>
                    <tr>
                        <th>Placeholder</th>
                        <th>Description</th>
                        <th>Example Output</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><code>{{post_title}}</code></td>
                        <td>Post/Page Title</td>
                        <td>My Awesome Article</td>
                    </tr>
                    <tr>
                        <td><code>{{post_excerpt}}</code></td>
                        <td>Post Excerpt</td>
                        <td>This is a brief summary...</td>
                    </tr>
                    <tr>
                        <td><code>{{post_date}}</code></td>
                        <td>Publication Date (ISO 8601)</td>
                        <td>2025-01-15T10:30:00+00:00</td>
                    </tr>
                    <tr>
                        <td><code>{{post_modified}}</code></td>
                        <td>Last Modified Date (ISO 8601)</td>
                        <td>2025-01-20T14:30:00+00:00</td>
                    </tr>
                    <tr>
                        <td><code>{{author_name}}</code></td>
                        <td>Author Display Name</td>
                        <td>John Doe</td>
                    </tr>
                    <tr>
                        <td><code>{{author_url}}</code></td>
                        <td>Author Archive URL</td>
                        <td>https://example.com/author/john-doe/</td>
                    </tr>
                    <tr>
                        <td><code>{{featured_image}}</code></td>
                        <td>Featured Image URL</td>
                        <td>https://example.com/image.jpg</td>
                    </tr>
                    <tr>
                        <td><code>{{post_url}}</code></td>
                        <td>Post Permalink</td>
                        <td>https://example.com/my-post/</td>
                    </tr>
                    <tr>
                        <td><code>{{site_name}}</code></td>
                        <td>Website Name</td>
                        <td>My Awesome Website</td>
                    </tr>
                    <tr>
                        <td><code>{{site_url}}</code></td>
                        <td>Website URL</td>
                        <td>https://example.com</td>
                    </tr>
                    <tr>
                        <td><code>{{site_description}}</code></td>
                        <td>Website Tagline</td>
                        <td>Just another WordPress site</td>
                    </tr>
                    <tr>
                        <td><code>{{post_category}}</code></td>
                        <td>Post Categories (comma-separated)</td>
                        <td>Technology, WordPress, SEO</td>
                    </tr>
                    <tr>
                        <td><code>{{post_category_first}}</code></td>
                        <td>First Post Category</td>
                        <td>Technology</td>
                    </tr>
                    <tr>
                        <td><code>{{post_tags}}</code></td>
                        <td>Post Tags (comma-separated)</td>
                        <td>tutorial, guide, tips</td>
                    </tr>
                    <tr>
                        <td><code>{{post_id}}</code></td>
                        <td>Post ID</td>
                        <td>123</td>
                    </tr>
                    <tr class="scmg-table-row-highlight">
                        <td colspan="3"><strong>🛒 WooCommerce Placeholders (Only for WooCommerce Products)</strong></td>
                    </tr>
                    <tr>
                        <td><code>{{woo_price}}</code></td>
                        <td>Product Price (current/sale price)</td>
                        <td>49.99</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_regular_price}}</code></td>
                        <td>Product Regular Price</td>
                        <td>59.99</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_sale_price}}</code></td>
                        <td>Product Sale Price</td>
                        <td>49.99</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_currency}}</code></td>
                        <td>Store Currency Code</td>
                        <td>USD</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_sku}}</code></td>
                        <td>Product SKU</td>
                        <td>PROD-12345</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_stock_status}}</code></td>
                        <td>Stock Status (instock/outofstock/onbackorder)</td>
                        <td>instock</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_availability}}</code></td>
                        <td>Schema.org Availability URL</td>
                        <td>https://schema.org/InStock</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_brand}}</code></td>
                        <td>Product Brand (from attributes or custom field)</td>
                        <td>Nike</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_rating}}</code></td>
                        <td>Average Product Rating</td>
                        <td>4.5</td>
                    </tr>
                    <tr>
                        <td><code>{{woo_review_count}}</code></td>
                        <td>Number of Reviews</td>
                        <td>127</td>
                    </tr>
                </tbody>
            </table>
            <div class="scmg-guide-example">
                <h4>Example with Placeholders (Article):</h4>
                <pre>{
  "@context": "https://schema.org",
  "@type": "Article",
  "headline": "{{post_title}}",
  "author": {
    "@type": "Person",
    "name": "{{author_name}}",
    "url": "{{author_url}}"
  },
  "publisher": {
    "@type": "Organization",
    "name": "{{site_name}}",
    "url": "{{site_url}}"
  },
  "datePublished": "{{post_date}}",
  "dateModified": "{{post_modified}}",
  "image": "{{featured_image}}",
  "url": "{{post_url}}",
  "articleSection": "{{post_category_first}}",
  "keywords": "{{post_tags}}"
}</pre>
                <p><em>💡 The placeholders will be automatically replaced with actual values when the page loads.</em></p>
            </div>

            <div class="scmg-guide-example scmg-guide-section-highlight">
                <h4>🛒 Example with WooCommerce Placeholders (Product):</h4>
                <pre>{
  "@context": "https://schema.org",
  "@type": "Product",
  "name": "{{post_title}}",
  "image": "{{featured_image}}",
  "description": "{{post_excerpt}}",
  "sku": "{{woo_sku}}",
  "brand": {
    "@type": "Brand",
    "name": "{{woo_brand}}"
  },
  "offers": {
    "@type": "Offer",
    "url": "{{post_url}}",
    "priceCurrency": "{{woo_currency}}",
    "price": "{{woo_price}}",
    "availability": "{{woo_availability}}"
  },
  "aggregateRating": {
    "@type": "AggregateRating",
    "ratingValue": "{{woo_rating}}",
    "reviewCount": "{{woo_review_count}}"
  }
}</pre>
                <p><em>🛒 WooCommerce placeholders automatically pull data from your WooCommerce products - no manual entry needed!</em></p>
                <p><em>💡 If WooCommerce is not active or post is not a product, these placeholders will be empty.</em></p>
            </div>
        </div>



        <div class="scmg-guide-section scmg-guide-section-blue">
            <h3>🎯 Step 7: Use Schema Templates (Time Saver!)</h3>
            <p><strong>Don't want to create schema from scratch? Use our ready-made templates!</strong></p>

            <ol>
                <li>Go to <strong>"Schema Templates Library"</strong> tab</li>
                <li>Browse 150+ templates organized by category:
                    <ul>
                        <li>Organization & Business</li>
                        <li>Website & Content</li>
                        <li>E-commerce & Products</li>
                        <li>Events, Jobs, People, Health, Recipes, Real Estate, Travel, Education, Media, and more!</li>
                    </ul>
                </li>
                <li>Find the template you need (e.g., "Article", "Product", "LocalBusiness")</li>
                <li>Click <strong>"View Code"</strong> to preview the template in a modal popup</li>
                <li>Click <strong>"Copy Template"</strong> to copy the template code</li>
                <li>Paste it into the Schema Generator form or directly into your code snippet plugin</li>
                <li>Customize the values or use placeholders like <code>{{post_title}}</code></li>
            </ol>

            <p><em>✅ Templates are pre-validated and follow Google's guidelines - just customize and use!</em></p>
        </div>

        <div class="scmg-guide-section csg-guide-tips">
            <h3>💡 Pro Tips & Best Practices</h3>
            <ul>
                <li><strong>✅ Always Validate:</strong> Test with Google's Rich Results Test tool before going live</li>
                <li><strong>📋 Use Templates First:</strong> Start with templates to avoid syntax errors and save time</li>
                <li><strong>🎯 Keep It Relevant:</strong> Only add schema types that match your actual content</li>
                <li><strong>🔄 Use Placeholders:</strong> Dynamic placeholders keep your schema automatically updated</li>
                <li><strong>📝 One Schema Per Content Type:</strong> Don't mix Article and Product schema on the same page</li>
                <li><strong>🔍 Check Required Fields:</strong> Make sure all required fields are filled (marked with *)</li>
                <li><strong>💾 Backup Before Editing:</strong> Copy existing schema before making major changes</li>
                <li><strong>🚀 Use Code Snippet Plugin:</strong> Easier to manage than editing theme files</li>
                <li><strong>📊 Monitor Performance:</strong> Check Google Search Console for schema-related issues</li>
                <li><strong>🔧 Update When Content Changes:</strong> Edit schema when you update post titles, dates, or other info</li>
                <li><strong>⚡ Clear Cache:</strong> Clear browser and site cache after adding/updating schema</li>
                <li><strong>📱 Test on Mobile:</strong> Verify schema works on mobile devices too</li>
            </ul>
        </div>

        <div class="scmg-guide-section scmg-guide-section-bordered">
            <h3>🎓 Common Use Cases</h3>

            <h4>📰 For Bloggers & Content Creators:</h4>
            <ul>
                <li>Use <strong>Article</strong> schema for blog posts</li>
                <li>Use <strong>BlogPosting</strong> for personal blog content</li>
                <li>Use <strong>NewsArticle</strong> for news content</li>
                <li>Use <strong>HowTo</strong> for tutorial posts</li>
                <li>Use <strong>FAQPage</strong> for FAQ pages</li>
            </ul>

            <h4>🛒 For E-commerce Sites (WooCommerce Supported!):</h4>
            <ul>
                <li>Use <strong>Product</strong> schema for product pages - <em>WooCommerce data auto-populated!</em></li>
                <li>Use placeholders: <code>{{woo_price}}</code>, <code>{{woo_currency}}</code>, <code>{{woo_sku}}</code>, <code>{{woo_availability}}</code></li>
                <li>Use <strong>Offer</strong> for pricing and availability</li>
                <li>Use <strong>AggregateRating</strong> for product reviews - <code>{{woo_rating}}</code>, <code>{{woo_review_count}}</code></li>
                <li>Use <strong>Organization</strong> for your store info</li>
                <li><em>💡 WooCommerce products automatically get price, SKU, stock status, brand, and ratings!</em></li>
            </ul>

            <h4>🏢 For Local Businesses:</h4>
            <ul>
                <li>Use <strong>LocalBusiness</strong> for your business info</li>
                <li>Use <strong>Organization</strong> for company details</li>
                <li>Use <strong>Service</strong> for services offered</li>
                <li>Use <strong>Event</strong> for business events</li>
            </ul>

            <h4>🍳 For Recipe Sites:</h4>
            <ul>
                <li>Use <strong>Recipe</strong> schema for recipe posts</li>
                <li>Include cooking time, ingredients, nutrition info</li>
                <li>Add ratings and reviews</li>
            </ul>

            <h4>📅 For Event Sites:</h4>
            <ul>
                <li>Use <strong>Event</strong> schema for events</li>
                <li>Include date, time, location, ticket info</li>
                <li>Use specific types: MusicEvent, SportsEvent, TheaterEvent, etc.</li>
            </ul>
        </div>

        <div class="scmg-guide-section csg-guide-resources">
            <h3>📖 Helpful Resources & Tools</h3>
            <div class="scmg-resources-grid">
                <div class="scmg-resource-card">
                    <h4>🔍 Testing Tools</h4>
                    <ul>
                        <li><a href="https://search.google.com/test/rich-results" target="_blank">Google Rich Results Test</a></li>
                        <li><a href="https://validator.schema.org/" target="_blank">Schema.org Validator</a></li>
                        <li><a href="https://search.google.com/search-console" target="_blank">Google Search Console</a></li>
                    </ul>
                </div>
                <div class="scmg-resource-card">
                    <h4>📚 Documentation</h4>
                    <ul>
                        <li><a href="https://schema.org/" target="_blank">Schema.org Official Docs</a></li>
                        <li><a href="https://developers.google.com/search/docs/appearance/structured-data/intro-structured-data" target="_blank">Google Structured Data Guide</a></li>
                        <li><a href="https://developers.google.com/search/docs/appearance/structured-data/search-gallery" target="_blank">Google Search Gallery</a></li>
                    </ul>
                </div>
                <div class="scmg-resource-card">
                    <h4>🔌 Recommended Plugins</h4>
                    <ul>
                        <li><a href="https://wordpress.org/plugins/code-snippets/" target="_blank">Code Snippets</a></li>
                        <li><a href="https://wordpress.org/plugins/insert-headers-and-footers/" target="_blank">Insert Headers and Footers</a></li>
                        <li><a href="https://wordpress.org/plugins/wp-crontrol/" target="_blank">WPCode</a></li>
                    </ul>
                </div>
            </div>
        </div>

        <div class="scmg-guide-section scmg-guide-section-info">
            <h3>❓ Need More Help?</h3>
            <p>If you're stuck or have questions:</p>
            <ul>
                <li>📖 Check the <strong>"FAQ"</strong> tab for common questions and answers</li>
                <li>📋 Browse the <strong>"Schema Templates Library"</strong> for ready-made examples</li>
                <li>🔍 Test your schema with Google Rich Results Test to identify issues</li>
                <li>📚 Visit Schema.org for detailed documentation on each schema type</li>
                <li>💬 Ask for help in WordPress forums or support channels</li>
            </ul>
        </div>
    </div>
    <?php
}

/**
 * Render the FAQ tab.
 */
function scmg_render_faq_tab() {
    ?>
    <div class="scmg-faq">
        <h2>Frequently Asked Questions</h2>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ What is Schema Markup?</h3>
            <div class="scmg-faq-answer">
                <p>Schema markup (also known as structured data) is code that helps search engines understand your content better. It provides additional context about your pages, which can lead to rich results in search engines like Google.</p>
                <p><strong>Benefits:</strong></p>
                <ul>
                    <li>Enhanced search results with rich snippets</li>
                    <li>Better click-through rates</li>
                    <li>Improved SEO performance</li>
                    <li>More visibility in search results</li>
                </ul>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ What is JSON-LD?</h3>
            <div class="scmg-faq-answer">
                <p>JSON-LD (JavaScript Object Notation for Linked Data) is the recommended format by Google for adding structured data to web pages. It's easy to implement and doesn't interfere with your page's HTML.</p>
                <p>This plugin uses JSON-LD format to add schema markup to your WordPress site.</p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Do I need to add schema to every page?</h3>
            <div class="scmg-faq-answer">
                <p>No, you only need to add schema to pages where it makes sense. For example:</p>
                <ul>
                    <li><strong>Article schema</strong> for blog posts</li>
                    <li><strong>Product schema</strong> for product pages</li>
                    <li><strong>FAQ schema</strong> for FAQ pages</li>
                    <li><strong>Local Business schema</strong> for contact/about pages</li>
                </ul>
                <p>Focus on your most important pages first.</p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ How do I know if my schema is working?</h3>
            <div class="scmg-faq-answer">
                <p>Follow these steps to verify:</p>
                <ol>
                    <li>Visit your page in a browser</li>
                    <li>Right-click and select "View Page Source"</li>
                    <li>Search for <code>&lt;script type="application/ld+json"&gt;</code></li>
                    <li>Your schema should appear in the HTML</li>
                    <li>Test with <a href="https://search.google.com/test/rich-results" target="_blank">Google Rich Results Test</a></li>
                </ol>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Can I use multiple schemas on one page?</h3>
            <div class="scmg-faq-answer">
                <p>Yes! You can add multiple schema types to a single page. For example, an article page might have:</p>
                <ul>
                    <li>Article schema for the main content</li>
                    <li>BreadcrumbList schema for navigation</li>
                    <li>Organization schema for your company</li>
                </ul>
                <p>Simply include multiple schema objects in your JSON-LD code, or use an array format.</p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Where do I generate schemas?</h3>
            <div class="scmg-faq-answer">
                <p>Go to <strong>Settings → Schema Box Generator → Schema Generator</strong> tab. From there:</p>
                <ol>
                    <li>Select a post/page/CPT from the dropdown (or leave empty for site-wide schema)</li>
                    <li>Choose a schema type from 150+ available options</li>
                    <li>Fill in the form fields (use placeholders like <code>{{post_title}}</code> for dynamic data)</li>
                    <li>Click "Generate & Save Schema"</li>
                </ol>
                <p>All your schemas are stored in the database and can be managed from the "Manage Schemas" tab.</p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ How do I copy the schema code?</h3>
            <div class="scmg-faq-answer">
                <p>There are two easy ways to copy your schema code:</p>
                <p><strong>Method 1: Quick Copy (Fastest)</strong></p>
                <ol>
                    <li>Go to <strong>"Manage Schemas"</strong> tab</li>
                    <li>Find your schema in the table</li>
                    <li>Click the <strong>"Copy"</strong> button (📋 icon)</li>
                    <li>You'll see "✓ Copied!" - the code is now in your clipboard!</li>
                </ol>
                <p><strong>Method 2: View & Copy (Recommended for Review)</strong></p>
                <ol>
                    <li>Go to <strong>"Manage Schemas"</strong> tab</li>
                    <li>Click the <strong>"View"</strong> button (👁️ icon)</li>
                    <li>A popup modal will show the formatted schema code</li>
                    <li>Review the code to make sure it looks correct</li>
                    <li>Click <strong>"Copy to Clipboard"</strong> button in the modal</li>
                </ol>
                <p><em>💡 The copied code includes the complete <code>&lt;script type="application/ld+json"&gt;</code> wrapper and is ready to paste!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ How do I edit or delete a schema?</h3>
            <div class="scmg-faq-answer">
                <p>Go to <strong>Settings → Schema Box Generator → Manage Schemas</strong> tab. You'll see a table with all your schemas and these action buttons:</p>
                <ul>
                    <li><strong>👁️ View:</strong> Opens a modal popup showing the formatted schema code</li>
                    <li><strong>📋 Copy:</strong> Instantly copies the schema code to your clipboard</li>
                    <li><strong>✏️ Edit:</strong> Takes you back to the Schema Generator tab with the form pre-filled for editing</li>
                    <li><strong>🗑️ Delete:</strong> Permanently removes the schema from the database (asks for confirmation)</li>
                </ul>
                <p><em>💡 Tip: After editing a schema, remember to copy the updated code and replace it in your theme/code snippet plugin!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Why doesn't the plugin automatically add schema to my pages?</h3>
            <div class="scmg-faq-answer">
                <p>For <strong>WordPress.org compliance and security reasons</strong>, this plugin generates schema markup but does NOT automatically insert it into your pages.</p>
                <p><strong>How it works:</strong></p>
                <ol>
                    <li>Plugin generates schema based on your form inputs and saves it to the database</li>
                    <li>You copy the generated schema using the <strong>"Copy"</strong> or <strong>"View"</strong> button in the Manage Schemas tab</li>
                    <li>You manually add it to your site using a code snippet plugin (recommended) or theme editor</li>
                </ol>
                <p><strong>Why this approach?</strong></p>
                <ul>
                    <li>✅ Follows WordPress.org security guidelines and plugin review requirements</li>
                    <li>✅ Gives you full control over where and when schema appears</li>
                    <li>✅ Prevents potential security vulnerabilities from arbitrary code execution</li>
                    <li>✅ More transparent and predictable behavior</li>
                    <li>✅ Easier to debug and troubleshoot</li>
                    <li>✅ Works with any theme or page builder</li>
                </ul>
                <p><em>💡 Recommended: Use a code snippet plugin like <a href="https://wordpress.org/plugins/code-snippets/" target="_blank">"Code Snippets"</a> or <a href="https://wordpress.org/plugins/insert-headers-and-footers/" target="_blank">"Insert Headers and Footers"</a> for easy management without editing theme files!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ What are the placeholders like {{post_title}}?</h3>
            <div class="scmg-faq-answer">
                <p>Placeholders are <strong>dynamic variables</strong> that automatically populate with your WordPress post/page data. Instead of manually entering the same information for every post, use placeholders to make your schema dynamic!</p>
                <p><strong>Available Placeholders:</strong></p>
                <ul>
                    <li><code>{{post_title}}</code> - Post/Page title</li>
                    <li><code>{{post_excerpt}}</code> - Post excerpt or auto-generated summary</li>
                    <li><code>{{post_date}}</code> - Publication date (ISO 8601 format)</li>
                    <li><code>{{post_modified}}</code> - Last modified date (ISO 8601 format)</li>
                    <li><code>{{author_name}}</code> - Author's display name</li>
                    <li><code>{{author_url}}</code> - Author archive URL</li>
                    <li><code>{{featured_image}}</code> - Featured image URL</li>
                    <li><code>{{post_url}}</code> - Post permalink</li>
                    <li><code>{{site_name}}</code> - Website name (from Settings → General)</li>
                    <li><code>{{site_url}}</code> - Website URL</li>
                    <li><code>{{site_description}}</code> - Website tagline</li>
                    <li><code>{{post_category}}</code> - All post categories (comma-separated)</li>
                    <li><code>{{post_category_first}}</code> - First post category</li>
                    <li><code>{{post_tags}}</code> - All post tags (comma-separated)</li>
                    <li><code>{{post_id}}</code> - Post ID number</li>
                </ul>
                <p><strong>Benefits:</strong></p>
                <ul>
                    <li>✅ Saves time - no need to manually enter data for each post</li>
                    <li>✅ Automatically updates when you change post content</li>
                    <li>✅ One schema template works for all posts of the same type</li>
                    <li>✅ Reduces errors from manual data entry</li>
                </ul>
                <p><em>💡 See the "How to Use" tab for a complete list with examples!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ How many schema types are available?</h3>
            <div class="scmg-faq-answer">
                <p>This plugin supports <strong>150+ schema types</strong> organized into 18 categories!</p>
                <p><strong>Categories include:</strong></p>
                <ul class="scmg-two-column-list">
                    <li>Organization & Business</li>
                    <li>Website & Content</li>
                    <li>E-commerce & Products</li>
                    <li>Events</li>
                    <li>Jobs & Employment</li>
                    <li>People & Personal Profiles</li>
                    <li>Health & Medical</li>
                    <li>Recipes & Food</li>
                    <li>Real Estate & Property</li>
                    <li>Travel & Transportation</li>
                    <li>Religious Places</li>
                    <li>Education & Courses</li>
                    <li>Media & Entertainment</li>
                    <li>Technology & Software</li>
                    <li>Visual & Image Content</li>
                    <li>Games</li>
                    <li>Data & Feeds</li>
                    <li>Special Announcements</li>
                </ul>
                <p><em>💡 Check the "Schema Templates Library" tab to browse all available templates organized by category!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Where can I find ready-made schema templates?</h3>
            <div class="scmg-faq-answer">
                <p>Go to the <strong>"Schema Templates Library"</strong> tab to access 150+ ready-to-use templates!</p>
                <p><strong>How to use templates:</strong></p>
                <ol>
                    <li>Browse templates by category (Organization, Content, E-commerce, Events, etc.)</li>
                    <li>Click <strong>"View Code"</strong> to preview the template in a modal popup</li>
                    <li>Click <strong>"Copy Template"</strong> to copy the template code</li>
                    <li>Paste it into the Schema Generator form or directly into your code snippet plugin</li>
                    <li>Customize the values or use placeholders like <code>{{post_title}}</code></li>
                </ol>
                <p><em>✅ All templates are pre-validated and follow Google's structured data guidelines!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Will this plugin slow down my website?</h3>
            <div class="scmg-faq-answer">
                <p>No! The plugin is very lightweight and only adds a small JSON-LD script to your pages. This has minimal impact on page load time and can actually improve your SEO performance.</p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Can I use this with other SEO plugins?</h3>
            <div class="scmg-faq-answer">
                <p>Yes! This plugin works alongside other SEO plugins like Yoast SEO, Rank Math, or All in One SEO. However, be careful not to add duplicate schema markup. If your SEO plugin already adds schema for certain content types, you may want to disable it there to avoid conflicts.</p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ How long does it take for Google to show rich results?</h3>
            <div class="scmg-faq-answer">
                <p>After adding schema markup:</p>
                <ol>
                    <li>Google needs to crawl your page (can take days to weeks)</li>
                    <li>Your schema must be valid and error-free</li>
                    <li>Your content must meet Google's quality guidelines</li>
                    <li>Rich results are not guaranteed, even with valid schema</li>
                </ol>
                <p>Be patient and focus on creating quality content with proper schema markup.</p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Where can I find schema examples?</h3>
            <div class="scmg-faq-answer">
                <p>Great resources for schema examples:</p>
                <ul>
                    <li><strong>This Plugin:</strong> Check the "Schema Templates" tab for ready-to-use templates</li>
                    <li><a href="https://schema.org/" target="_blank">Schema.org</a> - Official documentation with examples</li>
                    <li><a href="https://developers.google.com/search/docs/appearance/structured-data/search-gallery" target="_blank">Google Search Gallery</a> - Examples for different content types</li>
                    <li><a href="https://jsonld.com/examples/" target="_blank">JSON-LD Examples</a> - Community examples</li>
                </ul>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ What's the difference between "View" and "Copy" buttons?</h3>
            <div class="scmg-faq-answer">
                <p>Both buttons help you get your schema code, but they work differently:</p>
                <p><strong>📋 Copy Button (Quick Method):</strong></p>
                <ul>
                    <li>Instantly copies the schema code to your clipboard</li>
                    <li>Shows "✓ Copied!" confirmation</li>
                    <li>Fastest way to copy schema</li>
                    <li>Best when you already know the schema is correct</li>
                </ul>
                <p><strong>👁️ View Button (Review Method):</strong></p>
                <ul>
                    <li>Opens a modal popup showing the formatted schema code</li>
                    <li>Lets you review the code before copying</li>
                    <li>Includes a "Copy to Clipboard" button inside the modal</li>
                    <li>Best when you want to verify the schema first</li>
                </ul>
                <p><em>💡 Use "Copy" for speed, use "View" when you want to double-check the code!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Can I use this plugin with page builders (Elementor, Divi, etc.)?</h3>
            <div class="scmg-faq-answer">
                <p>Yes! This plugin works with <strong>all page builders</strong> including Elementor, Divi, Beaver Builder, WPBakery, and others.</p>
                <p><strong>How it works:</strong></p>
                <ol>
                    <li>Create your page with your page builder as usual</li>
                    <li>Generate schema for that page using this plugin</li>
                    <li>Copy the schema code from "Manage Schemas" tab</li>
                    <li>Add it to your site using a code snippet plugin</li>
                </ol>
                <p>The schema is added to the HTML <code>&lt;head&gt;</code> section, so it works regardless of which page builder you use!</p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Is this plugin compatible with Gutenberg and Classic Editor?</h3>
            <div class="scmg-faq-answer">
                <p>Yes! The plugin works with <strong>both Gutenberg (Block Editor) and Classic Editor</strong>.</p>
                <p>Since version 2.5.0, schema generation is done from the plugin settings page (not from the post editor), so it works the same way regardless of which editor you use.</p>
                <p><em>✅ Compatible with WordPress 5.0+ and all modern WordPress versions!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Will this plugin conflict with my SEO plugin (Yoast, Rank Math, etc.)?</h3>
            <div class="scmg-faq-answer">
                <p>This plugin is designed to work <strong>alongside</strong> other SEO plugins like Yoast SEO, Rank Math, All in One SEO, and others.</p>
                <p><strong>Important considerations:</strong></p>
                <ul>
                    <li>⚠️ <strong>Avoid duplicate schema:</strong> If your SEO plugin already adds Article schema, don't add it again with this plugin</li>
                    <li>✅ <strong>Disable conflicting schema:</strong> Most SEO plugins let you disable specific schema types - disable the ones you're adding with this plugin</li>
                    <li>✅ <strong>Use for different types:</strong> Use your SEO plugin for basic schema, use this plugin for advanced types (Recipe, Event, Product, etc.)</li>
                    <li>✅ <strong>Test for duplicates:</strong> Use Google Rich Results Test to check for duplicate schema</li>
                </ul>
                <p><em>💡 Pro Tip: This plugin gives you more control and supports 150+ schema types that most SEO plugins don't offer!</em></p>
            </div>
        </div>

        <div class="scmg-faq-item">
            <h3 class="scmg-faq-question">❓ Does this plugin support WooCommerce?</h3>
            <div class="scmg-faq-answer">
                <p><strong>Yes! Full WooCommerce support with automatic data integration!</strong></p>
                <p><strong>🛒 WooCommerce Features:</strong></p>
                <ul>
                    <li>✅ <strong>Automatic Product Data:</strong> Price, SKU, stock status, currency automatically pulled from WooCommerce</li>
                    <li>✅ <strong>10 WooCommerce Placeholders:</strong> Use <code>{{woo_price}}</code>, <code>{{woo_currency}}</code>, <code>{{woo_sku}}</code>, <code>{{woo_availability}}</code>, <code>{{woo_brand}}</code>, <code>{{woo_rating}}</code>, <code>{{woo_review_count}}</code>, and more!</li>
                    <li>✅ <strong>Product Schema Template:</strong> Pre-configured Product template with WooCommerce placeholders</li>
                    <li>✅ <strong>Ratings & Reviews:</strong> Automatically includes aggregate ratings and review counts</li>
                    <li>✅ <strong>Stock Status:</strong> Converts WooCommerce stock status to Schema.org availability (InStock, OutOfStock, BackOrder)</li>
                    <li>✅ <strong>Brand Support:</strong> Pulls brand from product attributes or custom fields</li>
                </ul>
                <p><strong>How to use with WooCommerce:</strong></p>
                <ol>
                    <li>Go to Schema Generator tab</li>
                    <li>Select "Product" as post type</li>
                    <li>Choose your WooCommerce product</li>
                    <li>Select "Product" schema type</li>
                    <li>Use WooCommerce placeholders like <code>{{woo_price}}</code>, <code>{{woo_sku}}</code></li>
                    <li>Generate & Save - All WooCommerce data will be automatically populated!</li>
                </ol>
                <p><em>💡 Pro Tip: Check the "How to Use" tab → Step 6 for a complete list of WooCommerce placeholders with examples!</em></p>
            </div>
        </div>

        <div class="scmg-faq-section csg-faq-support">
            <h3>🆘 Still Need Help?</h3>
            <p>If you have questions not covered here:</p>
            <ul>
                <li>Check the <strong>"How to Use"</strong> tab for detailed instructions</li>
                <li>Review the <strong>"Schema Templates"</strong> for examples</li>
                <li>Test your schema with <a href="https://search.google.com/test/rich-results" target="_blank">Google Rich Results Test</a></li>
                <li>Visit <a href="https://schema.org/" target="_blank">Schema.org</a> for official documentation</li>
            </ul>
        </div>
    </div>
    <?php
}

/**
 * Render the Donate tab.
 */
function scmg_render_donate_tab() {
    ?>
    <div class="scmg-donate">
        <div class="scmg-donate-hero">
            <div class="scmg-donate-icon">☕</div>
            <h2>Support Schema Markup Generator</h2>
            <p class="scmg-donate-tagline">Help keep this plugin free and actively maintained!</p>
        </div>

        <div class="scmg-donate-content">
            <div class="scmg-donate-main">
                <div class="scmg-donate-section">
                    <h3>👋 Hello There!</h3>
                    <p>Thank you for using <strong>Schema Markup Generator</strong>! I'm thrilled that this plugin is helping you improve your website's SEO and search engine visibility.</p>
                    <p>This plugin is completely <strong>free</strong> and <strong>open-source</strong>, and I'm committed to keeping it that way. However, developing and maintaining quality WordPress plugins takes considerable time and effort.</p>
                </div>

                <div class="scmg-donate-section csg-donate-features">
                    <h3>✨ What You Get (For Free!)</h3>
                    <ul>
                        <li>✅ <strong>150+ Schema Types</strong> - Comprehensive schema type support across 18 categories</li>
                        <li>✅ <strong>150+ Ready-Made Templates</strong> - Pre-validated JSON-LD templates organized by category</li>
                        <li>✅ <strong>15+ Dynamic Placeholders</strong> - Auto-populate with post/site data</li>
                        <li>✅ <strong>Centralized Management</strong> - Generate, view, edit, copy, and delete schemas from one place</li>
                        <li>✅ <strong>View & Copy Features</strong> - Modal popup to review schema code before copying</li>
                        <li>✅ <strong>Filter & Search</strong> - Easily find schemas by post type, schema type, or search</li>
                        <li>✅ <strong>Comprehensive Guide</strong> - Detailed step-by-step instructions with examples</li>
                        <li>✅ <strong>Extensive FAQ Section</strong> - Answers to common questions and troubleshooting</li>
                        <li>✅ <strong>WordPress.org Compliant</strong> - Follows all security and quality guidelines</li>
                        <li>✅ <strong>Regular Updates</strong> - Bug fixes, new features, and new schema types</li>
                        <li>✅ <strong>Clean Code</strong> - Well-documented, secure, and optimized</li>
                        <li>✅ <strong>No Ads or Upsells</strong> - Pure functionality, no premium versions</li>
                        <li>✅ <strong>Works with All Themes</strong> - Compatible with any WordPress theme or page builder</li>
                        <li>✅ <strong>SEO Plugin Compatible</strong> - Works alongside Yoast, Rank Math, and others</li>
                    </ul>
                </div>

                <div class="scmg-donate-section csg-donate-why">
                    <h3>💝 Why Donate?</h3>
                    <p>Your donation helps me:</p>
                    <ul>
                        <li>☕ <strong>Stay Motivated</strong> - Coffee fuels coding sessions!</li>
                        <li>🔧 <strong>Maintain the Plugin</strong> - Regular updates and bug fixes</li>
                        <li>✨ <strong>Add New Features</strong> - Visual builder, validation, and more</li>
                        <li>📚 <strong>Improve Documentation</strong> - Better guides and examples</li>
                        <li>🆘 <strong>Provide Support</strong> - Help users with issues</li>
                        <li>🌐 <strong>Keep It Free</strong> - No premium versions or paywalls</li>
                    </ul>
                </div>

                <div class="scmg-donate-cta">
                    <h3>☕ Buy Me a Cup of Coffee</h3>
                    <p>If this plugin has saved you time or helped improve your SEO, consider buying me a coffee! Even a small donation makes a big difference and shows your appreciation.</p>

                    <div class="scmg-donate-buttons">
                        <a href="https://paypal.me/buymecupofcoffee?locale.x=en_GB&country.x=IN" target="_blank" class="scmg-donate-button csg-donate-button-primary">
                            <span class="scmg-donate-button-icon">☕</span>
                            <span class="scmg-donate-button-text">
                                <strong>Buy Me a Coffee</strong>
                                <small>via PayPal</small>
                            </span>
                        </a>
                    </div>

                    <p class="scmg-donate-note">
                        <em>💡 Tip: Any amount is appreciated! Whether it's $5 for a coffee or $10 for lunch, your support means the world to me.</em>
                    </p>
                </div>

                <div class="scmg-donate-section csg-donate-alternatives">
                    <h3>🌟 Other Ways to Support</h3>
                    <p>Not able to donate? No problem! Here are other ways you can help:</p>
                    <ul>
                        <li>⭐ <strong>Rate the Plugin</strong> - Leave a 5-star review on WordPress.org</li>
                        <li>📢 <strong>Spread the Word</strong> - Tell others about this plugin</li>
                        <li>🐛 <strong>Report Bugs</strong> - Help improve the plugin by reporting issues</li>
                        <li>💡 <strong>Suggest Features</strong> - Share your ideas for improvements</li>
                        <li>📝 <strong>Write a Blog Post</strong> - Share your experience using the plugin</li>
                        <li>🔗 <strong>Share on Social Media</strong> - Help others discover this tool</li>
                    </ul>
                </div>

                <div class="scmg-donate-section csg-donate-thanks">
                    <h3>🙏 Thank You!</h3>
                    <p>Whether you donate or not, thank you for using <strong>Schema Markup Generator</strong>. Your support—in any form—keeps me motivated to continue developing and improving this plugin.</p>
                    <p>If you have any questions, suggestions, or just want to say hi, feel free to reach out!</p>
                    <p><strong>Happy Schema Building! 🚀</strong></p>
                </div>
            </div>

            <div class="scmg-donate-sidebar">
                <div class="scmg-donate-stats">
                    <h4>📊 Plugin Stats</h4>
                    <div class="scmg-stat-item">
                        <span class="scmg-stat-number">100+</span>
                        <span class="scmg-stat-label">Schema Types</span>
                    </div>
                    <div class="scmg-stat-item">
                        <span class="scmg-stat-number">50+</span>
                        <span class="scmg-stat-label">Ready Templates</span>
                    </div>
                    <div class="scmg-stat-item">
                        <span class="scmg-stat-number">25+</span>
                        <span class="scmg-stat-label">Dynamic Placeholders</span>
                    </div>
                    <div class="scmg-stat-item">
                        <span class="scmg-stat-number">15+</span>
                        <span class="scmg-stat-label">Schema Categories</span>
                    </div>
                    <div class="scmg-stat-item">
                        <span class="scmg-stat-number">100%</span>
                        <span class="scmg-stat-label">Free & Open Source</span>
                    </div>
                </div>

                <div class="scmg-donate-testimonial">
                    <h4>💬 What Users Say</h4>
                    <blockquote>
                        <p>"This plugin is exactly what I needed! Simple, powerful, and completely free. The schema templates saved me hours of work."</p>
                        <cite>— Happy User</cite>
                    </blockquote>
                </div>

                <div class="scmg-donate-quick-donate">
                    <h4>⚡ Quick Donate</h4>
                    <p>Show your appreciation with a quick donation:</p>
                    <a href="https://paypal.me/buymecupofcoffee?locale.x=en_GB&country.x=IN" target="_blank" class="scmg-donate-button csg-donate-button-secondary">
                        ☕ Donate Now
                    </a>
                </div>

                <div class="scmg-donate-social">
                    <h4>🌐 Stay Connected</h4>
                    <p>Follow for updates and new features!</p>
                    <div class="scmg-social-links">
                        <a href="https://twitter.com/WPFrank2" target="_blank" class="scmg-social-link" title="Twitter">🐦</a>
                        <a href="https://www.facebook.com/wpfrankfaraz/" class="scmg-social-link" target="_blank" title="Facebook">📘</a>
                        <a href="https://wpfrank.com/" class="scmg-social-link" target="_blank" title="Website">🌐</a>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php
}

/**
 * Get category for a schema type.
 *
 * @param string $schema_type The schema type.
 * @return string The category name.
 */
function scmg_get_schema_category( $schema_type ) {
    $category_map = array(
        // Organization & Business
        'Organization' => 'Organization & Business',
        'LocalBusiness' => 'Organization & Business',
        'Corporation' => 'Organization & Business',
        'NGO' => 'Organization & Business',
        'GovernmentOrganization' => 'Organization & Business',
        'EducationalOrganization' => 'Organization & Business',
        'MedicalOrganization' => 'Organization & Business',
        'SportsOrganization' => 'Organization & Business',
        'Store' => 'Organization & Business',

        // Website & Content
        'WebSite' => 'Website & Content',
        'WebPage' => 'Website & Content',
        'Article' => 'Website & Content',
        'Report' => 'Website & Content',
        'NewsArticle' => 'Website & Content',
        'BlogPosting' => 'Website & Content',
        'TechArticle' => 'Website & Content',
        'Guide' => 'Website & Content',
        'DiscussionForumPosting' => 'Website & Content',
        'LiveBlogPosting' => 'Website & Content',
        'FAQPage' => 'Website & Content',
        'HowTo' => 'Website & Content',
        'BreadcrumbList' => 'Website & Content',
        'ItemList' => 'Website & Content',
        'CollectionPage' => 'Website & Content',
        'QAPage' => 'Website & Content',
        'Comment' => 'Website & Content',
        'SiteNavigationElement' => 'Website & Content',
        'Certification' => 'Website & Content',

        // E-commerce & Products
        'Product' => 'E-commerce & Products',
        'ProductGroup' => 'E-commerce & Products',
        'Offer' => 'E-commerce & Products',
        'AggregateOffer' => 'E-commerce & Products',
        'Brand' => 'E-commerce & Products',
        'Review' => 'E-commerce & Products',
        'CriticReview' => 'E-commerce & Products',
        'AggregateRating' => 'E-commerce & Products',
        'Service' => 'E-commerce & Products',
        'FinancialProduct' => 'E-commerce & Products',
        'MemberProgram' => 'E-commerce & Products',

        // Events
        'Event' => 'Events',
        'BusinessEvent' => 'Events',
        'EducationEvent' => 'Events',
        'Festival' => 'Events',
        'MusicEvent' => 'Events',
        'SportsEvent' => 'Events',
        'TheaterEvent' => 'Events',
        'VisualArtsEvent' => 'Events',
        'ExhibitionEvent' => 'Events',
        'CourseInstance' => 'Events',

        // Jobs & Employment
        'JobPosting' => 'Jobs & Employment',
        'Occupation' => 'Jobs & Employment',
        'EmployeeRole' => 'Jobs & Employment',
        'WorkBasedProgram' => 'Jobs & Employment',

        // People & Personal Profiles
        'Person' => 'People & Personal Profiles',
        'ProfilePage' => 'People & Personal Profiles',
        'Author' => 'People & Personal Profiles',
        'Celebrity' => 'People & Personal Profiles',
        'Teacher' => 'People & Personal Profiles',
        'Parent' => 'People & Personal Profiles',
        'Patient' => 'People & Personal Profiles',
        'Musician' => 'People & Personal Profiles',
        'Actor' => 'People & Personal Profiles',
        'Athlete' => 'People & Personal Profiles',
        'Politician' => 'People & Personal Profiles',

        // Health & Medical
        'MedicalCondition' => 'Health & Medical',
        'Hospital' => 'Health & Medical',
        'Physician' => 'Health & Medical',
        'Clinic' => 'Health & Medical',
        'MedicalProcedure' => 'Health & Medical',
        'MedicalTherapy' => 'Health & Medical',
        'Drug' => 'Health & Medical',
        'DietarySupplement' => 'Health & Medical',
        'ExercisePlan' => 'Health & Medical',

        // Recipes & Food
        'Recipe' => 'Recipes & Food',
        'Cookbook' => 'Recipes & Food',
        'Menu' => 'Recipes & Food',
        'Restaurant' => 'Recipes & Food',
        'CafeOrCoffeeShop' => 'Recipes & Food',
        'FoodEstablishment' => 'Recipes & Food',
        'Bakery' => 'Recipes & Food',
        'BarOrPub' => 'Recipes & Food',
        'FastFoodRestaurant' => 'Recipes & Food',
        'IceCreamShop' => 'Recipes & Food',
        'FoodService' => 'Recipes & Food',

        // Real Estate & Property
        'RealEstateListing' => 'Real Estate & Property',
        'Apartment' => 'Real Estate & Property',
        'House' => 'Real Estate & Property',
        'SingleFamilyResidence' => 'Real Estate & Property',
        'VacationRental' => 'Real Estate & Property',

        // Travel & Transportation
        'Flight' => 'Travel & Transportation',
        'Trip' => 'Travel & Transportation',
        'TouristTrip' => 'Travel & Transportation',
        'TouristAttraction' => 'Travel & Transportation',
        'TouristDestination' => 'Travel & Transportation',
        'LandmarksOrHistoricalBuildings' => 'Travel & Transportation',
        'TaxiService' => 'Travel & Transportation',

        // Religious Places
        'HinduTemple' => 'Religious Places',
        'Church' => 'Religious Places',
        'Mosque' => 'Religious Places',

        // Education & Courses
        'Course' => 'Education & Courses',
        'School' => 'Education & Courses',
        'CollegeOrUniversity' => 'Education & Courses',
        'LearningResource' => 'Education & Courses',

        // Media & Entertainment
        'Book' => 'Media & Entertainment',
        'Movie' => 'Media & Entertainment',
        'TVSeries' => 'Media & Entertainment',
        'MusicAlbum' => 'Media & Entertainment',
        'MusicPlaylist' => 'Media & Entertainment',
        'VideoObject' => 'Media & Entertainment',
        'AudioObject' => 'Media & Entertainment',
        'BroadcastService' => 'Media & Entertainment',
        'CableOrSatelliteService' => 'Media & Entertainment',
        'MusicRecording' => 'Media & Entertainment',
        'PodcastEpisode' => 'Media & Entertainment',

        // Technology & Software
        'SoftwareApplication' => 'Technology & Software',
        'MobileApplication' => 'Technology & Software',
        'WebAPI' => 'Technology & Software',

        // Visual & Image Content
        'ImageObject' => 'Visual & Image Content',
        'ImageGallery' => 'Visual & Image Content',
        'MediaGallery' => 'Visual & Image Content',
        'Photograph' => 'Visual & Image Content',

        // Games
        'VideoGame' => 'Games',
        'Game' => 'Games',

        // Data & Feeds
        'DataFeed' => 'Data & Feeds',

        // Special Announcements
        'SpecialAnnouncement' => 'Special Announcements',
    );

    return isset( $category_map[ $schema_type ] ) ? $category_map[ $schema_type ] : 'Website & Content';
}

/**
 * Get schema templates.
 *
 * @return array Array of schema templates.
 */
function scmg_get_schema_templates() {
    return array(
        'article' => array(
            'name' => 'Article',
            'type' => 'Article',
            'description' => 'For blog posts, news articles, and editorial content.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Article',
                'headline' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                    'url' => '{{author_url}}',
                ),
                'publisher' => array(
                    '@type' => 'Organization',
                    'name' => '{{site_name}}',
                    'logo' => array(
                        '@type' => 'ImageObject',
                        'url' => '{{site_logo}}',
                    ),
                ),
                'datePublished' => '{{post_date}}',
                'dateModified' => '{{post_modified}}',
                'mainEntityOfPage' => array(
                    '@type' => 'WebPage',
                    '@id' => '{{post_url}}',
                ),
            ),
        ),
        'product' => array(
            'name' => 'Product',
            'type' => 'Product',
            'description' => 'For e-commerce product pages with pricing and availability. WooCommerce auto-supported!',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Product',
                'name' => '{{post_title}}',
                'image' => '{{featured_image}}',
                'description' => '{{post_excerpt}}',
                'sku' => '{{woo_sku}}',
                'brand' => array(
                    '@type' => 'Brand',
                    'name' => '{{woo_brand}}',
                ),
                'offers' => array(
                    '@type' => 'Offer',
                    'url' => '{{post_url}}',
                    'priceCurrency' => '{{woo_currency}}',
                    'price' => '{{woo_price}}',
                    'availability' => '{{woo_availability}}',
                ),
                'aggregateRating' => array(
                    '@type' => 'AggregateRating',
                    'ratingValue' => '{{woo_rating}}',
                    'reviewCount' => '{{woo_review_count}}',
                ),
            ),
        ),
        'local_business' => array(
            'name' => 'Local Business',
            'type' => 'LocalBusiness',
            'description' => 'For local businesses with physical location and contact info.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'LocalBusiness',
                'name' => 'Business Name',
                'image' => '{{featured_image}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'streetAddress' => '123 Main Street',
                    'addressLocality' => 'City',
                    'addressRegion' => 'State',
                    'postalCode' => '12345',
                    'addressCountry' => 'US',
                ),
                'telephone' => '+1-555-555-5555',
                'url' => '{{post_url}}',
                'openingHours' => 'Mo-Fr 09:00-17:00',
            ),
        ),
        'faq' => array(
            'name' => 'FAQ',
            'type' => 'FAQPage',
            'description' => 'For FAQ pages with questions and answers.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'FAQPage',
                'mainEntity' => array(
                    array(
                        '@type' => 'Question',
                        'name' => 'What is your return policy?',
                        'acceptedAnswer' => array(
                            '@type' => 'Answer',
                            'text' => 'We offer a 30-day return policy on all products.',
                        ),
                    ),
                    array(
                        '@type' => 'Question',
                        'name' => 'How long does shipping take?',
                        'acceptedAnswer' => array(
                            '@type' => 'Answer',
                            'text' => 'Standard shipping takes 5-7 business days.',
                        ),
                    ),
                ),
            ),
        ),
        'howto' => array(
            'name' => 'How-To',
            'type' => 'HowTo',
            'description' => 'For step-by-step guides and tutorials.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'HowTo',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'totalTime' => 'PT30M',
                'step' => array(
                    array(
                        '@type' => 'HowToStep',
                        'name' => 'Step 1',
                        'text' => 'Description of step 1',
                        'image' => '{{featured_image}}',
                    ),
                    array(
                        '@type' => 'HowToStep',
                        'name' => 'Step 2',
                        'text' => 'Description of step 2',
                        'image' => '{{featured_image}}',
                    ),
                ),
            ),
        ),
        'recipe' => array(
            'name' => 'Recipe',
            'type' => 'Recipe',
            'description' => 'For cooking recipes with ingredients and instructions.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Recipe',
                'name' => '{{post_title}}',
                'image' => '{{featured_image}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                    'url' => '{{author_url}}',
                ),
                'datePublished' => '{{post_date}}',
                'dateModified' => '{{post_modified}}',
                'description' => '{{post_excerpt}}',
                'prepTime' => 'PT20M',
                'cookTime' => 'PT30M',
                'totalTime' => 'PT50M',
                'recipeYield' => '4 servings',
                'recipeCategory' => '{{post_category_first}}',
                'keywords' => '{{post_tags}}',
                'recipeIngredient' => array(
                    '2 cups flour',
                    '1 cup sugar',
                    '3 eggs',
                ),
                'recipeInstructions' => array(
                    array(
                        '@type' => 'HowToStep',
                        'text' => 'Mix flour and sugar',
                    ),
                    array(
                        '@type' => 'HowToStep',
                        'text' => 'Add eggs and mix well',
                    ),
                ),
            ),
        ),
        'event' => array(
            'name' => 'Event',
            'type' => 'Event',
            'description' => 'For events, conferences, and webinars.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Event',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'startDate' => '2025-12-01T19:00:00',
                'endDate' => '2025-12-01T22:00:00',
                'location' => array(
                    '@type' => 'Place',
                    'name' => 'Event Venue',
                    'address' => array(
                        '@type' => 'PostalAddress',
                        'streetAddress' => '123 Event Street',
                        'addressLocality' => 'City',
                        'addressRegion' => 'State',
                        'postalCode' => '12345',
                        'addressCountry' => 'US',
                    ),
                ),
                'offers' => array(
                    '@type' => 'Offer',
                    'url' => '{{post_url}}',
                    'price' => '50.00',
                    'priceCurrency' => 'USD',
                    'availability' => 'https://schema.org/InStock',
                ),
            ),
        ),
        'video' => array(
            'name' => 'Video',
            'type' => 'VideoObject',
            'description' => 'For video content with duration and thumbnail.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'VideoObject',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'thumbnailUrl' => '{{featured_image}}',
                'uploadDate' => '{{post_date}}',
                'duration' => 'PT5M30S',
                'contentUrl' => 'YOUR_VIDEO_URL',
                'embedUrl' => 'YOUR_EMBED_URL',
            ),
        ),
        'organization' => array(
            'name' => 'Organization',
            'type' => 'Organization',
            'description' => 'For company/organization information.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Organization',
                'name' => '{{site_name}}',
                'url' => '{{site_url}}',
                'logo' => '{{site_logo}}',
                'description' => '{{site_description}}',
                'contactPoint' => array(
                    '@type' => 'ContactPoint',
                    'telephone' => '+1-555-555-5555',
                    'contactType' => 'Customer Service',
                ),
                'sameAs' => array(
                    'https://facebook.com/yourpage',
                    'https://twitter.com/yourhandle',
                    'https://linkedin.com/company/yourcompany',
                ),
            ),
        ),
        'breadcrumb' => array(
            'name' => 'Breadcrumb',
            'type' => 'BreadcrumbList',
            'description' => 'For navigation breadcrumbs.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'BreadcrumbList',
                'itemListElement' => array(
                    array(
                        '@type' => 'ListItem',
                        'position' => 1,
                        'name' => 'Home',
                        'item' => '{{site_url}}',
                    ),
                    array(
                        '@type' => 'ListItem',
                        'position' => 2,
                        'name' => 'Category',
                        'item' => 'YOUR_CATEGORY_URL',
                    ),
                    array(
                        '@type' => 'ListItem',
                        'position' => 3,
                        'name' => '{{post_title}}',
                        'item' => '{{post_url}}',
                    ),
                ),
            ),
        ),
        'review' => array(
            'name' => 'Review',
            'type' => 'Review',
            'description' => 'For product or service reviews with ratings.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Review',
                'itemReviewed' => array(
                    '@type' => 'Product',
                    'name' => '{{post_title}}',
                    'image' => '{{featured_image}}',
                ),
                'reviewRating' => array(
                    '@type' => 'Rating',
                    'ratingValue' => '4.5',
                    'bestRating' => '5',
                    'worstRating' => '1',
                ),
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                    'url' => '{{author_url}}',
                ),
                'reviewBody' => '{{post_excerpt}}',
                'datePublished' => '{{post_date}}',
                'publisher' => array(
                    '@type' => 'Organization',
                    'name' => '{{site_name}}',
                ),
            ),
        ),
        'course' => array(
            'name' => 'Course',
            'type' => 'Course',
            'description' => 'For online courses and educational content.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Course',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'provider' => array(
                    '@type' => 'Organization',
                    'name' => '{{site_name}}',
                    'sameAs' => '{{site_url}}',
                ),
                'offers' => array(
                    '@type' => 'Offer',
                    'category' => 'Paid',
                    'price' => '99.00',
                    'priceCurrency' => 'USD',
                ),
                'hasCourseInstance' => array(
                    '@type' => 'CourseInstance',
                    'courseMode' => 'online',
                    'courseWorkload' => 'PT10H',
                ),
            ),
        ),
        'job_posting' => array(
            'name' => 'Job Posting',
            'type' => 'JobPosting',
            'description' => 'For job listings and career opportunities.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'JobPosting',
                'title' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'datePosted' => '{{post_date}}',
                'hiringOrganization' => array(
                    '@type' => 'Organization',
                    'name' => '{{site_name}}',
                    'sameAs' => '{{site_url}}',
                    'logo' => '{{site_logo}}',
                ),
                'jobLocation' => array(
                    '@type' => 'Place',
                    'address' => array(
                        '@type' => 'PostalAddress',
                        'streetAddress' => '123 Main Street',
                        'addressLocality' => 'City',
                        'addressRegion' => 'State',
                        'postalCode' => '12345',
                        'addressCountry' => 'US',
                    ),
                ),
                'employmentType' => 'FULL_TIME',
                'baseSalary' => array(
                    '@type' => 'MonetaryAmount',
                    'currency' => 'USD',
                    'value' => array(
                        '@type' => 'QuantitativeValue',
                        'value' => 50000,
                        'unitText' => 'YEAR',
                    ),
                ),
            ),
        ),
        'book' => array(
            'name' => 'Book',
            'type' => 'Book',
            'description' => 'For books, ebooks, and publications.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Book',
                'name' => '{{post_title}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                ),
                'image' => '{{featured_image}}',
                'description' => '{{post_excerpt}}',
                'isbn' => '978-3-16-148410-0',
                'numberOfPages' => 250,
                'publisher' => array(
                    '@type' => 'Organization',
                    'name' => 'Publisher Name',
                ),
                'datePublished' => '{{post_date}}',
                'bookFormat' => 'https://schema.org/Paperback',
            ),
        ),
        'software' => array(
            'name' => 'Software Application',
            'type' => 'SoftwareApplication',
            'description' => 'For software, apps, and digital tools.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'SoftwareApplication',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'operatingSystem' => 'Windows, macOS, Linux',
                'applicationCategory' => 'BusinessApplication',
                'offers' => array(
                    '@type' => 'Offer',
                    'price' => '0',
                    'priceCurrency' => 'USD',
                ),
                'aggregateRating' => array(
                    '@type' => 'AggregateRating',
                    'ratingValue' => '4.5',
                    'ratingCount' => '100',
                ),
            ),
        ),
        'restaurant' => array(
            'name' => 'Restaurant',
            'type' => 'Restaurant',
            'description' => 'For restaurants, cafes, and dining establishments.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Restaurant',
                'name' => '{{post_title}}',
                'image' => '{{featured_image}}',
                'description' => '{{post_excerpt}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'streetAddress' => '123 Restaurant Street',
                    'addressLocality' => 'City',
                    'addressRegion' => 'State',
                    'postalCode' => '12345',
                    'addressCountry' => 'US',
                ),
                'telephone' => '+1-555-555-5555',
                'servesCuisine' => 'Italian',
                'priceRange' => '$$',
                'openingHours' => 'Mo-Su 11:00-22:00',
                'acceptsReservations' => 'True',
            ),
        ),
        'music' => array(
            'name' => 'Music Recording',
            'type' => 'MusicRecording',
            'description' => 'For songs, albums, and music content.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'MusicRecording',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'byArtist' => array(
                    '@type' => 'MusicGroup',
                    'name' => 'Artist Name',
                ),
                'inAlbum' => array(
                    '@type' => 'MusicAlbum',
                    'name' => 'Album Name',
                ),
                'duration' => 'PT3M30S',
                'datePublished' => '{{post_date}}',
            ),
        ),
        'podcast' => array(
            'name' => 'Podcast',
            'type' => 'PodcastEpisode',
            'description' => 'For podcast episodes and audio content.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'PodcastEpisode',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'url' => '{{post_url}}',
                'datePublished' => '{{post_date}}',
                'partOfSeries' => array(
                    '@type' => 'PodcastSeries',
                    'name' => 'YOUR_PODCAST_SERIES_NAME',
                    'url' => 'YOUR_PODCAST_URL',
                ),
                'associatedMedia' => array(
                    '@type' => 'MediaObject',
                    'contentUrl' => 'YOUR_EPISODE_MP3_URL',
                    'duration' => 'PT45M',
                ),
            ),
        ),
        'news_article' => array(
            'name' => 'News Article',
            'type' => 'NewsArticle',
            'description' => 'For news articles and press releases.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'NewsArticle',
                'headline' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'datePublished' => '{{post_date}}',
                'dateModified' => '{{post_date}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                ),
                'publisher' => array(
                    '@type' => 'Organization',
                    'name' => '{{site_name}}',
                    'logo' => array(
                        '@type' => 'ImageObject',
                        'url' => '{{site_logo}}',
                    ),
                ),
                'articleSection' => 'News',
            ),
        ),
        'medical' => array(
            'name' => 'Medical Condition',
            'type' => 'MedicalCondition',
            'description' => 'For medical conditions and health information.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'MedicalCondition',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'possibleTreatment' => array(
                    '@type' => 'MedicalTherapy',
                    'name' => 'Treatment Name',
                ),
                'riskFactor' => array(
                    '@type' => 'MedicalRiskFactor',
                    'name' => 'Risk Factor',
                ),
            ),
        ),
        'apartment' => array(
            'name' => 'Apartment',
            'type' => 'Apartment',
            'description' => 'For apartment listings and rental properties.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Apartment',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'streetAddress' => 'Street Address',
                    'addressLocality' => 'City',
                    'addressRegion' => 'State',
                    'postalCode' => 'Postal Code',
                    'addressCountry' => 'Country',
                ),
                'numberOfRooms' => '3',
                'floorSize' => array(
                    '@type' => 'QuantitativeValue',
                    'value' => '1000',
                    'unitCode' => 'SQF',
                ),
            ),
        ),
        'house' => array(
            'name' => 'House',
            'type' => 'House',
            'description' => 'For house listings and real estate properties.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'House',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'streetAddress' => 'Street Address',
                    'addressLocality' => 'City',
                    'addressRegion' => 'State',
                    'postalCode' => 'Postal Code',
                    'addressCountry' => 'Country',
                ),
                'numberOfRooms' => '4',
            ),
        ),
        'singlefamilyresidence' => array(
            'name' => 'Single Family Residence',
            'type' => 'SingleFamilyResidence',
            'description' => 'For single family home listings.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'SingleFamilyResidence',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'streetAddress' => 'Street Address',
                    'addressLocality' => 'City',
                    'addressRegion' => 'State',
                    'postalCode' => 'Postal Code',
                    'addressCountry' => 'Country',
                ),
            ),
        ),
        'vacationrental' => array(
            'name' => 'Vacation Rental',
            'type' => 'VacationRental',
            'description' => 'For vacation rental properties.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'VacationRental',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'addressLocality' => 'City',
                    'addressRegion' => 'State',
                    'addressCountry' => 'Country',
                ),
            ),
        ),
        'audioobject' => array(
            'name' => 'Audio Object',
            'type' => 'AudioObject',
            'description' => 'For audio files, podcasts, and music.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'AudioObject',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'contentUrl' => '{{post_url}}',
                'duration' => 'PT10M',
                'encodingFormat' => 'audio/mpeg',
            ),
        ),
        'techarticle' => array(
            'name' => 'Tech Article',
            'type' => 'TechArticle',
            'description' => 'For technical articles and tutorials.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'TechArticle',
                'headline' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                ),
                'publisher' => array(
                    '@type' => 'Organization',
                    'name' => '{{site_name}}',
                    'logo' => array(
                        '@type' => 'ImageObject',
                        'url' => '{{site_logo}}',
                    ),
                ),
                'datePublished' => '{{post_date}}',
            ),
        ),
        'qapage' => array(
            'name' => 'Q&A Page',
            'type' => 'QAPage',
            'description' => 'For question and answer pages.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'QAPage',
                'mainEntity' => array(
                    '@type' => 'Question',
                    'name' => '{{post_title}}',
                    'text' => '{{post_excerpt}}',
                    'answerCount' => 1,
                    'acceptedAnswer' => array(
                        '@type' => 'Answer',
                        'text' => 'Answer text here',
                    ),
                ),
            ),
        ),
        'tvseries' => array(
            'name' => 'TV Series',
            'type' => 'TVSeries',
            'description' => 'For TV shows and series.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'TVSeries',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'numberOfSeasons' => '1',
                'numberOfEpisodes' => '10',
            ),
        ),
        'softwareapplication' => array(
            'name' => 'Software Application',
            'type' => 'SoftwareApplication',
            'description' => 'For software and desktop applications.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'SoftwareApplication',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'operatingSystem' => 'Windows, macOS, Linux',
                'applicationCategory' => 'BusinessApplication',
                'offers' => array(
                    '@type' => 'Offer',
                    'price' => '0',
                    'priceCurrency' => 'USD',
                ),
            ),
        ),
        'mobileapplication' => array(
            'name' => 'Mobile Application',
            'type' => 'MobileApplication',
            'description' => 'For mobile apps (iOS, Android).',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'MobileApplication',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'operatingSystem' => 'iOS, Android',
                'applicationCategory' => 'GameApplication',
                'offers' => array(
                    '@type' => 'Offer',
                    'price' => '0',
                    'priceCurrency' => 'USD',
                ),
            ),
        ),
        'webpage' => array(
            'name' => 'Web Page',
            'type' => 'WebPage',
            'description' => 'For general web pages.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'WebPage',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'url' => '{{post_url}}',
                'datePublished' => '{{post_date}}',
                'dateModified' => '{{post_modified}}',
            ),
        ),
        'videogame' => array(
            'name' => 'Video Game',
            'type' => 'VideoGame',
            'description' => 'For video games.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'VideoGame',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'gamePlatform' => 'PC, PlayStation, Xbox',
            ),
        ),
        'game' => array(
            'name' => 'Game',
            'type' => 'Game',
            'description' => 'For general games.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Game',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
            ),
        ),
        'service' => array(
            'name' => 'Service',
            'type' => 'Service',
            'description' => 'For services offered by businesses.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Service',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'provider' => array(
                    '@type' => 'Organization',
                    'name' => '{{site_name}}',
                ),
                'serviceType' => 'Service Type',
                'areaServed' => 'Service Area',
            ),
        ),
        'trip' => array(
            'name' => 'Trip',
            'type' => 'Trip',
            'description' => 'For travel itineraries and trips.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Trip',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
            ),
        ),
        'touristattraction' => array(
            'name' => 'Tourist Attraction',
            'type' => 'TouristAttraction',
            'description' => 'For tourist attractions and landmarks.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'TouristAttraction',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'addressLocality' => 'City',
                    'addressCountry' => 'Country',
                ),
            ),
        ),
        'touristdestination' => array(
            'name' => 'Tourist Destination',
            'type' => 'TouristDestination',
            'description' => 'For tourist destinations.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'TouristDestination',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
            ),
        ),
        'landmarksorhistoricalbuildings' => array(
            'name' => 'Landmarks or Historical Buildings',
            'type' => 'LandmarksOrHistoricalBuildings',
            'description' => 'For historical landmarks and buildings.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'LandmarksOrHistoricalBuildings',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'image' => '{{featured_image}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'addressLocality' => 'City',
                    'addressCountry' => 'Country',
                ),
            ),
        ),
        'hindutemple' => array(
            'name' => 'Hindu Temple',
            'type' => 'HinduTemple',
            'description' => 'For Hindu temples and places of worship.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'HinduTemple',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'addressLocality' => 'City',
                    'addressCountry' => 'Country',
                ),
            ),
        ),
        'church' => array(
            'name' => 'Church',
            'type' => 'Church',
            'description' => 'For churches and Christian places of worship.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Church',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'addressLocality' => 'City',
                    'addressCountry' => 'Country',
                ),
            ),
        ),
        'mosque' => array(
            'name' => 'Mosque',
            'type' => 'Mosque',
            'description' => 'For mosques and Islamic places of worship.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Mosque',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'address' => array(
                    '@type' => 'PostalAddress',
                    'addressLocality' => 'City',
                    'addressCountry' => 'Country',
                ),
            ),
        ),
        'profilepage' => array(
            'name' => 'Profile Page',
            'type' => 'ProfilePage',
            'description' => 'For user profile pages.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'ProfilePage',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'mainEntity' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                ),
            ),
        ),
        'guide' => array(
            'name' => 'Guide',
            'type' => 'Guide',
            'description' => 'For guides and tutorials.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Guide',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                ),
            ),
        ),
        'musicplaylist' => array(
            'name' => 'Music Playlist',
            'type' => 'MusicPlaylist',
            'description' => 'For music playlists.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'MusicPlaylist',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'numTracks' => '10',
            ),
        ),
        'musicalbum' => array(
            'name' => 'Music Album',
            'type' => 'MusicAlbum',
            'description' => 'For music albums.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'MusicAlbum',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'byArtist' => array(
                    '@type' => 'MusicGroup',
                    'name' => 'Artist Name',
                ),
            ),
        ),
        'productgroup' => array(
            'name' => 'Product Group',
            'type' => 'ProductGroup',
            'description' => 'For product groups and collections.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'ProductGroup',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'url' => '{{post_url}}',
            ),
        ),
        'imagegallery' => array(
            'name' => 'Image Gallery',
            'type' => 'ImageGallery',
            'description' => 'For image galleries.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'ImageGallery',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
            ),
        ),
        'mediagallery' => array(
            'name' => 'Media Gallery',
            'type' => 'MediaGallery',
            'description' => 'For media galleries.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'MediaGallery',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
            ),
        ),
        'criticreview' => array(
            'name' => 'Critic Review',
            'type' => 'CriticReview',
            'description' => 'For professional critic reviews.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'CriticReview',
                'name' => '{{post_title}}',
                'reviewBody' => '{{post_excerpt}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                ),
                'reviewRating' => array(
                    '@type' => 'Rating',
                    'ratingValue' => '5',
                    'bestRating' => '5',
                ),
            ),
        ),
        'datafeed' => array(
            'name' => 'Data Feed',
            'type' => 'DataFeed',
            'description' => 'For data feeds.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'DataFeed',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
            ),
        ),
        'specialannouncement' => array(
            'name' => 'Special Announcement (COVID-19)',
            'type' => 'SpecialAnnouncement',
            'description' => 'For special announcements related to COVID-19.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'SpecialAnnouncement',
                'name' => '{{post_title}}',
                'text' => '{{post_excerpt}}',
                'datePosted' => '{{post_date}}',
                'category' => 'https://www.wikidata.org/wiki/Q81068910',
            ),
        ),
        'certification' => array(
            'name' => 'Certification',
            'type' => 'Certification',
            'description' => 'For certifications and credentials.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'Certification',
                'name' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
            ),
        ),
        'discussionforumposting' => array(
            'name' => 'Discussion Forum Posting',
            'type' => 'DiscussionForumPosting',
            'description' => 'For forum posts and discussions.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'DiscussionForumPosting',
                'headline' => '{{post_title}}',
                'text' => '{{post_excerpt}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                ),
                'datePublished' => '{{post_date}}',
            ),
        ),
        'liveblogposting' => array(
            'name' => 'Live Blog Posting',
            'type' => 'LiveBlogPosting',
            'description' => 'For live blog posts and updates.',
            'schema' => array(
                '@context' => 'https://schema.org',
                '@type' => 'LiveBlogPosting',
                'headline' => '{{post_title}}',
                'description' => '{{post_excerpt}}',
                'author' => array(
                    '@type' => 'Person',
                    'name' => '{{author_name}}',
                ),
                'publisher' => array(
                    '@type' => 'Organization',
                    'name' => '{{site_name}}',
                ),
                'datePublished' => '{{post_date}}',
            ),
        ),
    );
}

/**
 * Check if meta box should be shown for current post.
 *
 * @param int $post_id The post ID.
 * @param string $post_type The post type.
 * @return bool Whether to show the meta box.
 */
function scmg_should_show_meta_box( $post_id, $post_type ) {
    // Get enabled post types from settings
    $enabled_post_types = get_option( 'scmg_enabled_post_types', array() );

    // If post type is not enabled, don't show
    if ( ! isset( $enabled_post_types[ $post_type ] ) || $enabled_post_types[ $post_type ] !== '1' ) {
        return false;
    }

    // Check individual item settings
    if ( $post_type === 'page' ) {
        $enabled_pages = get_option( 'scmg_enabled_pages', array() );
        // If no pages are specifically enabled, show for all pages
        if ( empty( $enabled_pages ) ) {
            return true;
        }
        // Check if this specific page is enabled
        return isset( $enabled_pages[ $post_id ] ) && $enabled_pages[ $post_id ] === '1';
    } elseif ( $post_type === 'post' ) {
        $enabled_posts = get_option( 'scmg_enabled_posts', array() );
        // If no posts are specifically enabled, show for all posts
        if ( empty( $enabled_posts ) ) {
            return true;
        }
        // Check if this specific post is enabled
        return isset( $enabled_posts[ $post_id ] ) && $enabled_posts[ $post_id ] === '1';
    } else {
        // Custom post type
        $enabled_cpt_items = get_option( 'scmg_enabled_cpt_items', array() );
        // If no CPT items are specifically enabled, show for all CPT items
        if ( empty( $enabled_cpt_items ) ) {
            return true;
        }
        // Check if this specific CPT item is enabled
        return isset( $enabled_cpt_items[ $post_id ] ) && $enabled_cpt_items[ $post_id ] === '1';
    }
}

/**
 * Replace placeholders in schema with actual post data.
 *
 * @param string $schema The schema string with placeholders.
 * @param int    $post_id The post ID.
 * @return string The schema with replaced placeholders.
 */
function scmg_replace_schema_placeholders( $schema, $post_id ) {
    $post = get_post( $post_id );
    if ( ! $post ) {
        return $schema;
    }

    // Get post data
    $post_title = get_the_title( $post_id );
    $post_excerpt = has_excerpt( $post_id ) ? get_the_excerpt( $post_id ) : wp_trim_words( $post->post_content, 55, '...' );
    $post_date = get_the_date( 'c', $post_id ); // ISO 8601 format
    $post_modified = get_the_modified_date( 'c', $post_id );
    $post_url = get_permalink( $post_id );

    // Get author data
    $author_id = $post->post_author;
    $author_name = get_the_author_meta( 'display_name', $author_id );
    $author_url = get_author_posts_url( $author_id );

    // Get featured image
    $featured_image = '';
    if ( has_post_thumbnail( $post_id ) ) {
        $featured_image = get_the_post_thumbnail_url( $post_id, 'full' );
    }

    // Get site data
    $site_name = get_bloginfo( 'name' );
    $site_url = get_bloginfo( 'url' );
    $site_description = get_bloginfo( 'description' );

    // Get site logo
    $site_logo = '';
    $custom_logo_id = get_theme_mod( 'custom_logo' );
    if ( $custom_logo_id ) {
        $site_logo = wp_get_attachment_image_url( $custom_logo_id, 'full' );
    }

    // Get categories (for posts)
    $categories = array();
    $category_names = array();
    $post_categories = get_the_category( $post_id );
    if ( ! empty( $post_categories ) ) {
        foreach ( $post_categories as $cat ) {
            $categories[] = $cat->name;
            $category_names[] = $cat->name;
        }
    }
    $post_category = ! empty( $category_names ) ? implode( ', ', $category_names ) : '';
    $post_category_first = ! empty( $category_names ) ? $category_names[0] : '';

    // Get tags
    $tags = array();
    $post_tags = get_the_tags( $post_id );
    if ( ! empty( $post_tags ) ) {
        foreach ( $post_tags as $tag ) {
            $tags[] = $tag->name;
        }
    }
    $post_tags_string = ! empty( $tags ) ? implode( ', ', $tags ) : '';

    // WooCommerce support - Get product data if WooCommerce is active and post is a product
    $woo_price = '';
    $woo_currency = '';
    $woo_sku = '';
    $woo_stock_status = '';
    $woo_availability = '';
    $woo_brand = '';
    $woo_rating = '';
    $woo_review_count = '';
    $woo_sale_price = '';
    $woo_regular_price = '';

    if ( class_exists( 'WooCommerce' ) && 'product' === $post->post_type ) {
        $product = wc_get_product( $post_id );

        if ( $product ) {
            // Get price
            $woo_price = $product->get_price();
            $woo_regular_price = $product->get_regular_price();
            $woo_sale_price = $product->get_sale_price();

            // Get currency
            $woo_currency = get_woocommerce_currency();

            // Get SKU
            $woo_sku = $product->get_sku();

            // Get stock status
            $woo_stock_status = $product->get_stock_status();

            // Convert stock status to Schema.org availability
            switch ( $woo_stock_status ) {
                case 'instock':
                    $woo_availability = 'https://schema.org/InStock';
                    break;
                case 'outofstock':
                    $woo_availability = 'https://schema.org/OutOfStock';
                    break;
                case 'onbackorder':
                    $woo_availability = 'https://schema.org/BackOrder';
                    break;
                default:
                    $woo_availability = 'https://schema.org/InStock';
            }

            // Get brand (from product attributes or custom field)
            $brand_attribute = $product->get_attribute( 'brand' );
            if ( ! empty( $brand_attribute ) ) {
                $woo_brand = $brand_attribute;
            } else {
                // Try to get from custom field
                $brand_meta = get_post_meta( $post_id, '_product_brand', true );
                if ( ! empty( $brand_meta ) ) {
                    $woo_brand = $brand_meta;
                }
            }

            // Get rating and review count
            $woo_rating = $product->get_average_rating();
            $woo_review_count = $product->get_review_count();
        }
    }

    // Create replacement array
    $replacements = array(
        '{{post_title}}'         => esc_html( $post_title ),
        '{{post_excerpt}}'       => esc_html( $post_excerpt ),
        '{{post_date}}'          => $post_date,
        '{{post_modified}}'      => $post_modified,
        '{{post_url}}'           => esc_url( $post_url ),
        '{{author_name}}'        => esc_html( $author_name ),
        '{{author_url}}'         => esc_url( $author_url ),
        '{{featured_image}}'     => esc_url( $featured_image ),
        '{{site_name}}'          => esc_html( $site_name ),
        '{{site_url}}'           => esc_url( $site_url ),
        '{{site_description}}'   => esc_html( $site_description ),
        '{{site_logo}}'          => esc_url( $site_logo ),
        '{{post_category}}'      => esc_html( $post_category ),
        '{{post_category_first}}' => esc_html( $post_category_first ),
        '{{post_tags}}'          => esc_html( $post_tags_string ),
        '{{post_id}}'            => $post_id,
        // WooCommerce placeholders
        '{{woo_price}}'          => esc_html( $woo_price ),
        '{{woo_currency}}'       => esc_html( $woo_currency ),
        '{{woo_sku}}'            => esc_html( $woo_sku ),
        '{{woo_stock_status}}'   => esc_html( $woo_stock_status ),
        '{{woo_availability}}'   => esc_url( $woo_availability ),
        '{{woo_brand}}'          => esc_html( $woo_brand ),
        '{{woo_rating}}'         => esc_html( $woo_rating ),
        '{{woo_review_count}}'   => esc_html( $woo_review_count ),
        '{{woo_sale_price}}'     => esc_html( $woo_sale_price ),
        '{{woo_regular_price}}' => esc_html( $woo_regular_price ),
    );

    // Replace placeholders
    $schema = str_replace( array_keys( $replacements ), array_values( $replacements ), $schema );

    return $schema;
}

/**
 *
 * Schema generation is now handled through the plugin settings page
 * instead of individual post/page edit screens. This provides better
 * centralized management and WordPress.org compliance.
 */

// ============================================================================
// DATABASE CRUD OPERATIONS
// ============================================================================

/**
 * Save a new schema to the database.
 *
 * @param int    $post_id       The post ID.
 * @param string $post_type     The post type.
 * @param string $schema_type   The schema type.
 * @param array  $schema_fields The schema fields.
 * @param string $generated_json The generated JSON-LD.
 * @return int|false The inserted row ID or false on failure.
 */
function scmg_save_schema( $post_id, $post_type, $schema_type, $schema_fields, $generated_json ) {
    global $wpdb;

    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;

    // Check if table exists
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking table existence requires direct query
    $table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) );

    if ( ! $table_exists ) {
        // Table doesn't exist, create it
        scmg_create_database_table();
    }

    // Check if schema already exists for this post
    $existing = scmg_get_schema_by_post_id( $post_id );

    if ( $existing ) {
        // Update existing schema
        return scmg_update_schema( $existing->id, $post_id, $post_type, $schema_type, $schema_fields, $generated_json );
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query, no WP API available
    $result = $wpdb->insert(
        $table_name,
        array(
            'post_id'        => $post_id,
            'post_type'      => $post_type,
            'schema_type'    => $schema_type,
            'schema_fields'  => maybe_serialize( $schema_fields ),
            'generated_json' => $generated_json,
        ),
        array( '%d', '%s', '%s', '%s', '%s' )
    );

    if ( $result === false ) {
        return false;
    }

    return $wpdb->insert_id;
}

/**
 * Update an existing schema in the database.
 *
 * @param int    $id            The schema ID.
 * @param int    $post_id       The post ID.
 * @param string $post_type     The post type.
 * @param string $schema_type   The schema type.
 * @param array  $schema_fields The schema fields.
 * @param string $generated_json The generated JSON-LD.
 * @return int|false The number of rows updated or false on failure.
 */
function scmg_update_schema( $id, $post_id, $post_type, $schema_type, $schema_fields, $generated_json ) {
    global $wpdb;

    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct query, no WP API available
    $result = $wpdb->update(
        $table_name,
        array(
            'post_id'        => $post_id,
            'post_type'      => $post_type,
            'schema_type'    => $schema_type,
            'schema_fields'  => maybe_serialize( $schema_fields ),
            'generated_json' => $generated_json,
        ),
        array( 'id' => $id ),
        array( '%d', '%s', '%s', '%s', '%s' ),
        array( '%d' )
    );

    if ( $result === false ) {
        return false;
    }

    return $result;
}

/**
 * Delete a schema from the database.
 *
 * @param int $id The schema ID.
 * @return int|false The number of rows deleted or false on failure.
 */
function scmg_delete_schema( $id ) {
    global $wpdb;

    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct query, no WP API available
    $result = $wpdb->delete(
        $table_name,
        array( 'id' => $id ),
        array( '%d' )
    );

    return $result;
}

/**
 * Get a schema by ID.
 *
 * @param int $id The schema ID.
 * @return object|null The schema object or null if not found.
 */
function scmg_get_schema_by_id( $id ) {
    global $wpdb;

    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct query
    $schema = $wpdb->get_row(
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $table_name is safe (wpdb prefix + constant)
        $wpdb->prepare( "SELECT * FROM {$table_name} WHERE id = %d", $id )
    );

    if ( $schema && ! empty( $schema->schema_fields ) ) {
        $schema->schema_fields = maybe_unserialize( $schema->schema_fields );
    }

    return $schema;
}

/**
 * Get a schema by post ID.
 *
 * @param int $post_id The post ID.
 * @return object|null The schema object or null if not found.
 */
function scmg_get_schema_by_post_id( $post_id ) {
    global $wpdb;

    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct query
    $schema = $wpdb->get_row(
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $table_name is safe (wpdb prefix + constant)
        $wpdb->prepare( "SELECT * FROM {$table_name} WHERE post_id = %d", $post_id )
    );

    if ( $schema && ! empty( $schema->schema_fields ) ) {
        $schema->schema_fields = maybe_unserialize( $schema->schema_fields );
    }

    return $schema;
}

/**
 * Get all schemas from the database.
 *
 * @param array $args Optional. Query arguments (limit, offset, orderby, order).
 * @return array Array of schema objects.
 */
function scmg_get_all_schemas( $args = array() ) {
    global $wpdb;

    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;

    $defaults = array(
        'limit'   => 100,
        'offset'  => 0,
        'orderby' => 'updated_at',
        'order'   => 'DESC',
    );

    $args = wp_parse_args( $args, $defaults );

    // Sanitize orderby and order to prevent SQL injection
    $allowed_orderby = array( 'id', 'post_id', 'schema_type', 'created_at', 'updated_at' );
    $orderby = in_array( $args['orderby'], $allowed_orderby, true ) ? $args['orderby'] : 'id';
    $order = strtoupper( $args['order'] ) === 'ASC' ? 'ASC' : 'DESC';

    $query = $wpdb->prepare(
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $table_name is safe (wpdb prefix + constant), $orderby and $order are sanitized above (whitelist)
        "SELECT * FROM {$table_name} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d",
        $args['limit'],
        $args['offset']
    );

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Query is prepared above with sanitized $orderby (whitelist) and $order (strict check)
    $schemas = $wpdb->get_results( $query );

    // Unserialize schema_fields for each schema
    foreach ( $schemas as $schema ) {
        if ( ! empty( $schema->schema_fields ) ) {
            $schema->schema_fields = maybe_unserialize( $schema->schema_fields );
        }
    }

    return $schemas;
}

/**
 * Get total count of schemas.
 *
 * @return int Total number of schemas.
 */
function scmg_get_schemas_count() {
    global $wpdb;

    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table requires direct query, $table_name is safe (wpdb prefix + constant)
    return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
}

/**
 * Get generated schema for a specific post.
 *
 * This function generates the schema markup based on the post's structured fields.
 * It does NOT output anything - it only returns the generated schema.
 *
 * @param int $post_id Post ID.
 * @return string|false Generated JSON-LD schema markup, or false if no schema.
 */
function scmg_get_generated_schema_for_post( $post_id ) {
	if ( ! $post_id ) {
		return false;
	}

	// Get structured schema from post meta
	$schema_type = get_post_meta( $post_id, '_csg_schema_type', true );
	$schema_fields = get_post_meta( $post_id, '_csg_schema_fields', true );

	if ( empty( $schema_type ) || ! is_array( $schema_fields ) ) {
		return false;
	}

	// Generate schema from structured fields
	$schema_array = scmg_generate_schema_from_fields( $schema_type, $schema_fields, $post_id );

	if ( empty( $schema_array ) || ! is_array( $schema_array ) ) {
		return false;
	}

	// Return formatted JSON-LD
	return '<script type="application/ld+json">' . "\n" .
	       wp_json_encode( $schema_array, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) .
	       "\n" . '</script>';
}

/**
 * Get available schema types.
 *
 * @return array Array of schema types with their labels.
 */
function scmg_get_schema_types() {
    return array(
        // Organization & Business
        'Organization'                  => __( 'Organization', 'frank-schema-markup-generator' ),
        'LocalBusiness'                 => __( 'Local Business', 'frank-schema-markup-generator' ),
        'Corporation'                   => __( 'Corporation', 'frank-schema-markup-generator' ),
        'NGO'                           => __( 'NGO (Non-Governmental Organization)', 'frank-schema-markup-generator' ),
        'GovernmentOrganization'        => __( 'Government Organization', 'frank-schema-markup-generator' ),
        'EducationalOrganization'       => __( 'Educational Organization', 'frank-schema-markup-generator' ),
        'MedicalOrganization'           => __( 'Medical Organization', 'frank-schema-markup-generator' ),
        'SportsOrganization'            => __( 'Sports Organization', 'frank-schema-markup-generator' ),
        'Store'                         => __( 'Store', 'frank-schema-markup-generator' ),

        // Website & Content
        'WebSite'                       => __( 'Website', 'frank-schema-markup-generator' ),
        'WebPage'                       => __( 'Web Page', 'frank-schema-markup-generator' ),
        'Article'                       => __( 'Article', 'frank-schema-markup-generator' ),
        'Report'                        => __( 'Report', 'frank-schema-markup-generator' ),
        'NewsArticle'                   => __( 'News Article', 'frank-schema-markup-generator' ),
        'BlogPosting'                   => __( 'Blog Posting', 'frank-schema-markup-generator' ),
        'TechArticle'                   => __( 'Tech Article', 'frank-schema-markup-generator' ),
        'Guide'                         => __( 'Guide', 'frank-schema-markup-generator' ),
        'DiscussionForumPosting'        => __( 'Discussion Forum Posting', 'frank-schema-markup-generator' ),
        'LiveBlogPosting'               => __( 'Live Blog Posting', 'frank-schema-markup-generator' ),
        'FAQPage'                       => __( 'FAQ Page', 'frank-schema-markup-generator' ),
        'HowTo'                         => __( 'How-To', 'frank-schema-markup-generator' ),
        'BreadcrumbList'                => __( 'Breadcrumb', 'frank-schema-markup-generator' ),
        'ItemList'                      => __( 'Item List', 'frank-schema-markup-generator' ),
        'CollectionPage'                => __( 'Collection Page', 'frank-schema-markup-generator' ),
        'QAPage'                        => __( 'Q&A Page', 'frank-schema-markup-generator' ),
        'Comment'                       => __( 'Comment', 'frank-schema-markup-generator' ),
        'SiteNavigationElement'         => __( 'Site Navigation Element', 'frank-schema-markup-generator' ),
        'Certification'                 => __( 'Certification', 'frank-schema-markup-generator' ),

        // E-commerce & Products
        'Product'                       => __( 'Product', 'frank-schema-markup-generator' ),
        'ProductGroup'                  => __( 'Product Group', 'frank-schema-markup-generator' ),
        'Offer'                         => __( 'Offer', 'frank-schema-markup-generator' ),
        'AggregateOffer'                => __( 'Aggregate Offer', 'frank-schema-markup-generator' ),
        'Brand'                         => __( 'Brand', 'frank-schema-markup-generator' ),
        'Review'                        => __( 'Review', 'frank-schema-markup-generator' ),
        'CriticReview'                  => __( 'Critic Review', 'frank-schema-markup-generator' ),
        'AggregateRating'               => __( 'Aggregate Rating', 'frank-schema-markup-generator' ),
        'Service'                       => __( 'Service', 'frank-schema-markup-generator' ),
        'FinancialProduct'              => __( 'Financial Product', 'frank-schema-markup-generator' ),
        'MemberProgram'                 => __( 'Member Program', 'frank-schema-markup-generator' ),

        // Events
        'Event'                         => __( 'Event', 'frank-schema-markup-generator' ),
        'BusinessEvent'                 => __( 'Business Event', 'frank-schema-markup-generator' ),
        'EducationEvent'                => __( 'Education Event', 'frank-schema-markup-generator' ),
        'Festival'                      => __( 'Festival', 'frank-schema-markup-generator' ),
        'MusicEvent'                    => __( 'Music Event', 'frank-schema-markup-generator' ),
        'SportsEvent'                   => __( 'Sports Event', 'frank-schema-markup-generator' ),
        'TheaterEvent'                  => __( 'Theater Event', 'frank-schema-markup-generator' ),
        'VisualArtsEvent'               => __( 'Visual Arts Event', 'frank-schema-markup-generator' ),
        'ExhibitionEvent'               => __( 'Exhibition Event', 'frank-schema-markup-generator' ),
        'CourseInstance'                => __( 'Course Instance', 'frank-schema-markup-generator' ),

        // Jobs & Employment
        'JobPosting'                    => __( 'Job Posting', 'frank-schema-markup-generator' ),
        'Occupation'                    => __( 'Occupation', 'frank-schema-markup-generator' ),
        'EmployeeRole'                  => __( 'Employee Role', 'frank-schema-markup-generator' ),
        'WorkBasedProgram'              => __( 'Work Based Program', 'frank-schema-markup-generator' ),

        // People & Personal Profiles
        'Person'                        => __( 'Person', 'frank-schema-markup-generator' ),
        'ProfilePage'                   => __( 'Profile Page', 'frank-schema-markup-generator' ),
        'Author'                        => __( 'Author', 'frank-schema-markup-generator' ),
        'Celebrity'                     => __( 'Celebrity', 'frank-schema-markup-generator' ),
        'Teacher'                       => __( 'Teacher', 'frank-schema-markup-generator' ),
        'Parent'                        => __( 'Parent', 'frank-schema-markup-generator' ),
        'Patient'                       => __( 'Patient', 'frank-schema-markup-generator' ),
        'Musician'                      => __( 'Musician', 'frank-schema-markup-generator' ),
        'Actor'                         => __( 'Actor', 'frank-schema-markup-generator' ),
        'Athlete'                       => __( 'Athlete', 'frank-schema-markup-generator' ),
        'Politician'                    => __( 'Politician', 'frank-schema-markup-generator' ),

        // Health & Medical
        'MedicalCondition'              => __( 'Medical Condition', 'frank-schema-markup-generator' ),
        'Hospital'                      => __( 'Hospital', 'frank-schema-markup-generator' ),
        'Physician'                     => __( 'Physician', 'frank-schema-markup-generator' ),
        'Clinic'                        => __( 'Clinic', 'frank-schema-markup-generator' ),
        'MedicalProcedure'              => __( 'Medical Procedure', 'frank-schema-markup-generator' ),
        'MedicalTherapy'                => __( 'Medical Therapy', 'frank-schema-markup-generator' ),
        'Drug'                          => __( 'Drug', 'frank-schema-markup-generator' ),
        'DietarySupplement'             => __( 'Dietary Supplement', 'frank-schema-markup-generator' ),
        'ExercisePlan'                  => __( 'Exercise Plan', 'frank-schema-markup-generator' ),

        // Recipes & Food
        'Recipe'                        => __( 'Recipe', 'frank-schema-markup-generator' ),
        'Cookbook'                      => __( 'Cookbook', 'frank-schema-markup-generator' ),
        'Menu'                          => __( 'Menu', 'frank-schema-markup-generator' ),
        'Restaurant'                    => __( 'Restaurant', 'frank-schema-markup-generator' ),
        'CafeOrCoffeeShop'              => __( 'Cafe or Coffee Shop', 'frank-schema-markup-generator' ),
        'FoodEstablishment'             => __( 'Food Establishment', 'frank-schema-markup-generator' ),
        'Bakery'                        => __( 'Bakery', 'frank-schema-markup-generator' ),
        'BarOrPub'                      => __( 'Bar or Pub', 'frank-schema-markup-generator' ),
        'FastFoodRestaurant'            => __( 'Fast Food Restaurant', 'frank-schema-markup-generator' ),
        'IceCreamShop'                  => __( 'Ice Cream Shop', 'frank-schema-markup-generator' ),
        'FoodService'                   => __( 'Food Service', 'frank-schema-markup-generator' ),

        // Real Estate & Property
        'RealEstateListing'             => __( 'Real Estate Listing', 'frank-schema-markup-generator' ),
        'Apartment'                     => __( 'Apartment', 'frank-schema-markup-generator' ),
        'House'                         => __( 'House', 'frank-schema-markup-generator' ),
        'SingleFamilyResidence'         => __( 'Single Family Residence', 'frank-schema-markup-generator' ),
        'VacationRental'                => __( 'Vacation Rental', 'frank-schema-markup-generator' ),

        // Travel & Transportation
        'Flight'                        => __( 'Flight', 'frank-schema-markup-generator' ),
        'Trip'                          => __( 'Trip', 'frank-schema-markup-generator' ),
        'TouristTrip'                   => __( 'Tourist Trip', 'frank-schema-markup-generator' ),
        'TouristAttraction'             => __( 'Tourist Attraction', 'frank-schema-markup-generator' ),
        'TouristDestination'            => __( 'Tourist Destination', 'frank-schema-markup-generator' ),
        'LandmarksOrHistoricalBuildings' => __( 'Landmarks or Historical Buildings', 'frank-schema-markup-generator' ),
        'TaxiService'                   => __( 'Taxi Service', 'frank-schema-markup-generator' ),

        // Religious Places
        'HinduTemple'                   => __( 'Hindu Temple', 'frank-schema-markup-generator' ),
        'Church'                        => __( 'Church', 'frank-schema-markup-generator' ),
        'Mosque'                        => __( 'Mosque', 'frank-schema-markup-generator' ),

        // Education & Courses
        'Course'                        => __( 'Course', 'frank-schema-markup-generator' ),
        'School'                        => __( 'School', 'frank-schema-markup-generator' ),
        'CollegeOrUniversity'           => __( 'College or University', 'frank-schema-markup-generator' ),
        'LearningResource'              => __( 'Learning Resource', 'frank-schema-markup-generator' ),

        // Media & Entertainment
        'Book'                          => __( 'Book', 'frank-schema-markup-generator' ),
        'Movie'                         => __( 'Movie', 'frank-schema-markup-generator' ),
        'TVSeries'                      => __( 'TV Series', 'frank-schema-markup-generator' ),
        'MusicAlbum'                    => __( 'Music Album', 'frank-schema-markup-generator' ),
        'MusicPlaylist'                 => __( 'Music Playlist', 'frank-schema-markup-generator' ),
        'VideoObject'                   => __( 'Video', 'frank-schema-markup-generator' ),
        'AudioObject'                   => __( 'Audio', 'frank-schema-markup-generator' ),
        'BroadcastService'              => __( 'Broadcast Service', 'frank-schema-markup-generator' ),
        'CableOrSatelliteService'       => __( 'Cable or Satellite Service', 'frank-schema-markup-generator' ),

        // Technology & Software
        'SoftwareApplication'           => __( 'Software Application', 'frank-schema-markup-generator' ),
        'MobileApplication'             => __( 'Mobile Application', 'frank-schema-markup-generator' ),
        'WebAPI'                        => __( 'Web API', 'frank-schema-markup-generator' ),

        // Visual & Image Content
        'ImageObject'                   => __( 'Image Object', 'frank-schema-markup-generator' ),
        'ImageGallery'                  => __( 'Image Gallery', 'frank-schema-markup-generator' ),
        'MediaGallery'                  => __( 'Media Gallery', 'frank-schema-markup-generator' ),
        'Photograph'                    => __( 'Photograph', 'frank-schema-markup-generator' ),

        // Games
        'VideoGame'                     => __( 'Video Game', 'frank-schema-markup-generator' ),
        'Game'                          => __( 'Game', 'frank-schema-markup-generator' ),

        // Data & Feeds
        'DataFeed'                      => __( 'Data Feed', 'frank-schema-markup-generator' ),

        // Special Announcements
        'SpecialAnnouncement'           => __( 'Special Announcement (COVID-19)', 'frank-schema-markup-generator' ),
    );
}

/**
 * Get schema field definitions for a specific schema type.
 *
 * @param string $schema_type The schema type.
 * @return array Array of field definitions.
 */
function scmg_get_schema_field_definitions( $schema_type ) {
    $fields = array();

    switch ( $schema_type ) {
        case 'Article':
        case 'BlogPosting':
        case 'NewsArticle':
            $fields = array(
                'headline' => array(
                    'label'       => __( 'Headline', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_title}}',
                    'description' => __( 'The headline of the article. Use {{post_title}} for automatic post title.', 'frank-schema-markup-generator' ),
                ),
                'description' => array(
                    'label'       => __( 'Description', 'frank-schema-markup-generator' ),
                    'type'        => 'textarea',
                    'required'    => false,
                    'placeholder' => '{{post_excerpt}}',
                    'description' => __( 'A short description of the article.', 'frank-schema-markup-generator' ),
                ),
                'author_name' => array(
                    'label'       => __( 'Author Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{author_name}}',
                    'description' => __( 'The name of the author.', 'frank-schema-markup-generator' ),
                ),
                'publisher_name' => array(
                    'label'       => __( 'Publisher Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{site_name}}',
                    'description' => __( 'The name of the publisher.', 'frank-schema-markup-generator' ),
                ),
                'publisher_logo' => array(
                    'label'       => __( 'Publisher Logo URL', 'frank-schema-markup-generator' ),
                    'type'        => 'url',
                    'required'    => false,
                    'placeholder' => '{{site_logo}}',
                    'description' => __( 'URL of the publisher logo image.', 'frank-schema-markup-generator' ),
                ),
                'date_published' => array(
                    'label'       => __( 'Date Published', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_date}}',
                    'description' => __( 'Publication date in ISO 8601 format.', 'frank-schema-markup-generator' ),
                ),
                'date_modified' => array(
                    'label'       => __( 'Date Modified', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '{{post_modified}}',
                    'description' => __( 'Last modified date in ISO 8601 format.', 'frank-schema-markup-generator' ),
                ),
                'image' => array(
                    'label'       => __( 'Image URL', 'frank-schema-markup-generator' ),
                    'type'        => 'url',
                    'required'    => false,
                    'placeholder' => '{{featured_image}}',
                    'description' => __( 'URL of the article image.', 'frank-schema-markup-generator' ),
                ),
            );
            break;

        case 'Product':
            $fields = array(
                'name' => array(
                    'label'       => __( 'Product Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_title}}',
                    'description' => __( 'The name of the product.', 'frank-schema-markup-generator' ),
                ),
                'description' => array(
                    'label'       => __( 'Description', 'frank-schema-markup-generator' ),
                    'type'        => 'textarea',
                    'required'    => false,
                    'placeholder' => '{{post_excerpt}}',
                    'description' => __( 'A description of the product.', 'frank-schema-markup-generator' ),
                ),
                'image' => array(
                    'label'       => __( 'Image URL', 'frank-schema-markup-generator' ),
                    'type'        => 'url',
                    'required'    => false,
                    'placeholder' => '{{featured_image}}',
                    'description' => __( 'URL of the product image.', 'frank-schema-markup-generator' ),
                ),
                'brand' => array(
                    'label'       => __( 'Brand', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'The brand of the product.', 'frank-schema-markup-generator' ),
                ),
                'sku' => array(
                    'label'       => __( 'SKU', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'The Stock Keeping Unit.', 'frank-schema-markup-generator' ),
                ),
                'price' => array(
                    'label'       => __( 'Price', 'frank-schema-markup-generator' ),
                    'type'        => 'number',
                    'required'    => false,
                    'placeholder' => '0.00',
                    'description' => __( 'The price of the product.', 'frank-schema-markup-generator' ),
                ),
                'currency' => array(
                    'label'       => __( 'Currency', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => 'USD',
                    'description' => __( 'Currency code (e.g., USD, EUR, GBP).', 'frank-schema-markup-generator' ),
                ),
            );
            break;

        case 'LocalBusiness':
            $fields = array(
                'name' => array(
                    'label'       => __( 'Business Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '',
                    'description' => __( 'The name of the business.', 'frank-schema-markup-generator' ),
                ),
                'description' => array(
                    'label'       => __( 'Description', 'frank-schema-markup-generator' ),
                    'type'        => 'textarea',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'A description of the business.', 'frank-schema-markup-generator' ),
                ),
                'address_street' => array(
                    'label'       => __( 'Street Address', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'Street address of the business.', 'frank-schema-markup-generator' ),
                ),
                'address_city' => array(
                    'label'       => __( 'City', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'City where the business is located.', 'frank-schema-markup-generator' ),
                ),
                'address_region' => array(
                    'label'       => __( 'State/Region', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'State or region.', 'frank-schema-markup-generator' ),
                ),
                'address_postal' => array(
                    'label'       => __( 'Postal Code', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'Postal or ZIP code.', 'frank-schema-markup-generator' ),
                ),
                'address_country' => array(
                    'label'       => __( 'Country', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'Country code (e.g., US, GB, IN).', 'frank-schema-markup-generator' ),
                ),
                'telephone' => array(
                    'label'       => __( 'Telephone', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'Business phone number.', 'frank-schema-markup-generator' ),
                ),
                'price_range' => array(
                    'label'       => __( 'Price Range', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '$$',
                    'description' => __( 'Price range (e.g., $, $$, $$$, $$$$).', 'frank-schema-markup-generator' ),
                ),
            );
            break;

        case 'Event':
            $fields = array(
                'name' => array(
                    'label'       => __( 'Event Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_title}}',
                    'description' => __( 'The name of the event.', 'frank-schema-markup-generator' ),
                ),
                'description' => array(
                    'label'       => __( 'Description', 'frank-schema-markup-generator' ),
                    'type'        => 'textarea',
                    'required'    => false,
                    'placeholder' => '{{post_excerpt}}',
                    'description' => __( 'A description of the event.', 'frank-schema-markup-generator' ),
                ),
                'start_date' => array(
                    'label'       => __( 'Start Date', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '2025-12-31T19:00:00',
                    'description' => __( 'Event start date and time (ISO 8601 format).', 'frank-schema-markup-generator' ),
                ),
                'end_date' => array(
                    'label'       => __( 'End Date', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '2025-12-31T23:00:00',
                    'description' => __( 'Event end date and time (ISO 8601 format).', 'frank-schema-markup-generator' ),
                ),
                'location_name' => array(
                    'label'       => __( 'Location Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'Name of the event location.', 'frank-schema-markup-generator' ),
                ),
                'location_address' => array(
                    'label'       => __( 'Location Address', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'Full address of the event location.', 'frank-schema-markup-generator' ),
                ),
            );
            break;

        case 'TechArticle':
        case 'DiscussionForumPosting':
        case 'LiveBlogPosting':
            // Same as Article but with specific @type
            $fields = array(
                'headline' => array(
                    'label'       => __( 'Headline', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_title}}',
                    'description' => __( 'The headline of the article.', 'frank-schema-markup-generator' ),
                ),
                'description' => array(
                    'label'       => __( 'Description', 'frank-schema-markup-generator' ),
                    'type'        => 'textarea',
                    'required'    => false,
                    'placeholder' => '{{post_excerpt}}',
                    'description' => __( 'A short description.', 'frank-schema-markup-generator' ),
                ),
                'author_name' => array(
                    'label'       => __( 'Author Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{author_name}}',
                    'description' => __( 'The name of the author.', 'frank-schema-markup-generator' ),
                ),
                'publisher_name' => array(
                    'label'       => __( 'Publisher Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{site_name}}',
                    'description' => __( 'The name of the publisher.', 'frank-schema-markup-generator' ),
                ),
                'date_published' => array(
                    'label'       => __( 'Date Published', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_date}}',
                    'description' => __( 'Publication date.', 'frank-schema-markup-generator' ),
                ),
                'image' => array(
                    'label'       => __( 'Image URL', 'frank-schema-markup-generator' ),
                    'type'        => 'url',
                    'required'    => false,
                    'placeholder' => '{{featured_image}}',
                    'description' => __( 'URL of the article image.', 'frank-schema-markup-generator' ),
                ),
            );
            break;

        case 'ProductGroup':
            $fields = array(
                'name' => array(
                    'label'       => __( 'Product Group Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_title}}',
                    'description' => __( 'The name of the product group.', 'frank-schema-markup-generator' ),
                ),
                'description' => array(
                    'label'       => __( 'Description', 'frank-schema-markup-generator' ),
                    'type'        => 'textarea',
                    'required'    => false,
                    'placeholder' => '{{post_excerpt}}',
                    'description' => __( 'Description of the product group.', 'frank-schema-markup-generator' ),
                ),
                'url' => array(
                    'label'       => __( 'Product Group URL', 'frank-schema-markup-generator' ),
                    'type'        => 'url',
                    'required'    => false,
                    'placeholder' => '{{post_url}}',
                    'description' => __( 'URL of the product group page.', 'frank-schema-markup-generator' ),
                ),
                'image' => array(
                    'label'       => __( 'Image URL', 'frank-schema-markup-generator' ),
                    'type'        => 'url',
                    'required'    => false,
                    'placeholder' => '{{featured_image}}',
                    'description' => __( 'URL of the product group image.', 'frank-schema-markup-generator' ),
                ),
            );
            break;

        case 'Service':
            $fields = array(
                'name' => array(
                    'label'       => __( 'Service Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_title}}',
                    'description' => __( 'The name of the service.', 'frank-schema-markup-generator' ),
                ),
                'description' => array(
                    'label'       => __( 'Description', 'frank-schema-markup-generator' ),
                    'type'        => 'textarea',
                    'required'    => false,
                    'placeholder' => '{{post_excerpt}}',
                    'description' => __( 'Description of the service.', 'frank-schema-markup-generator' ),
                ),
                'provider_name' => array(
                    'label'       => __( 'Provider Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '{{site_name}}',
                    'description' => __( 'Name of the service provider.', 'frank-schema-markup-generator' ),
                ),
                'service_type' => array(
                    'label'       => __( 'Service Type', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'Type of service offered.', 'frank-schema-markup-generator' ),
                ),
                'area_served' => array(
                    'label'       => __( 'Area Served', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => false,
                    'placeholder' => '',
                    'description' => __( 'Geographic area where service is provided.', 'frank-schema-markup-generator' ),
                ),
            );
            break;

        default:
            // For other schema types, provide basic fields
            $fields = array(
                'name' => array(
                    'label'       => __( 'Name', 'frank-schema-markup-generator' ),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => '{{post_title}}',
                    'description' => __( 'The name or title.', 'frank-schema-markup-generator' ),
                ),
                'description' => array(
                    'label'       => __( 'Description', 'frank-schema-markup-generator' ),
                    'type'        => 'textarea',
                    'required'    => false,
                    'placeholder' => '{{post_excerpt}}',
                    'description' => __( 'A description.', 'frank-schema-markup-generator' ),
                ),
            );
            break;
    }

    return apply_filters( 'scmg_schema_field_definitions', $fields, $schema_type );
}

/**
 * Render schema fields based on schema type.
 *
 * @param string  $schema_type   The schema type.
 * @param array   $schema_fields Saved field values.
 * @param WP_Post $post          The post object.
 */
function scmg_render_schema_fields( $schema_type, $schema_fields, $post ) {
    $field_definitions = scmg_get_schema_field_definitions( $schema_type );

    if ( empty( $field_definitions ) ) {
        echo '<p>' . esc_html__( 'No fields available for this schema type.', 'frank-schema-markup-generator' ) . '</p>';
        return;
    }

    echo '<div class="scmg-schema-fields">';
    echo '<h4>' . esc_html__( 'Schema Fields', 'frank-schema-markup-generator' ) . '</h4>';
    echo '<p class="description scmg-description-mb-15">' . esc_html__( 'Fill in the fields below. You can use placeholders like {{post_title}}, {{author_name}}, {{site_name}}, etc.', 'frank-schema-markup-generator' ) . '</p>';

    foreach ( $field_definitions as $field_key => $field_def ) {
        $field_value = isset( $schema_fields[ $field_key ] ) ? $schema_fields[ $field_key ] : '';
        $field_name = 'SCMG_schema_fields[' . esc_attr( $field_key ) . ']';
        $field_id = 'scmg_field_' . esc_attr( $field_key );
        $required_mark = ! empty( $field_def['required'] ) ? ' <span class="scmg-field-required">*</span>' : '';

        echo '<div class="scmg-field-group">';
        echo '<label for="' . esc_attr( $field_id ) . '">' . esc_html( $field_def['label'] ) . $required_mark . '</label>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

        switch ( $field_def['type'] ) {
            case 'textarea':
                echo '<textarea name="' . esc_attr( $field_name ) . '" id="' . esc_attr( $field_id ) . '" placeholder="' . esc_attr( $field_def['placeholder'] ) . '">' . esc_textarea( $field_value ) . '</textarea>';
                break;

            case 'url':
                echo '<input type="url" name="' . esc_attr( $field_name ) . '" id="' . esc_attr( $field_id ) . '" value="' . esc_attr( $field_value ) . '" placeholder="' . esc_attr( $field_def['placeholder'] ) . '" />';
                break;

            case 'number':
                echo '<input type="number" step="0.01" name="' . esc_attr( $field_name ) . '" id="' . esc_attr( $field_id ) . '" value="' . esc_attr( $field_value ) . '" placeholder="' . esc_attr( $field_def['placeholder'] ) . '" />';
                break;

            case 'date':
                echo '<input type="date" name="' . esc_attr( $field_name ) . '" id="' . esc_attr( $field_id ) . '" value="' . esc_attr( $field_value ) . '" placeholder="' . esc_attr( $field_def['placeholder'] ) . '" />';
                break;

            case 'text':
            default:
                echo '<input type="text" name="' . esc_attr( $field_name ) . '" id="' . esc_attr( $field_id ) . '" value="' . esc_attr( $field_value ) . '" placeholder="' . esc_attr( $field_def['placeholder'] ) . '" />';
                break;
        }

        if ( ! empty( $field_def['description'] ) ) {
            echo '<p class="description">' . esc_html( $field_def['description'] ) . '</p>';
        }

        echo '</div>';
    }

    echo '</div>';
}

// ============================================================================
// AJAX HANDLERS
// ============================================================================

/**
 * AJAX handler to load schema fields for the generator form.
 */
function scmg_ajax_load_schema_fields() {
    check_ajax_referer( 'scmg_ajax_nonce', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Permission denied.', 'frank-schema-markup-generator' ) ) );
    }

    $schema_type = isset( $_POST['schema_type'] ) ? sanitize_text_field( wp_unslash( $_POST['schema_type'] ) ) : '';
    $post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;
    $schema_id = isset( $_POST['schema_id'] ) ? absint( $_POST['schema_id'] ) : 0;

    if ( empty( $schema_type ) || empty( $post_id ) ) {
        wp_send_json_error( array( 'message' => __( 'Invalid parameters.', 'frank-schema-markup-generator' ) ) );
    }

    $post = get_post( $post_id );
    if ( ! $post ) {
        wp_send_json_error( array( 'message' => __( 'Post not found.', 'frank-schema-markup-generator' ) ) );
    }

    // Get saved fields from database if editing
    $schema_fields = array();
    if ( $schema_id > 0 ) {
        $schema = scmg_get_schema_by_id( $schema_id );

        if ( $schema && is_array( $schema->schema_fields ) ) {
            $schema_fields = $schema->schema_fields;
        }
    }

    // Render fields
    ob_start();
    scmg_render_schema_fields( $schema_type, $schema_fields, $post );
    $html = ob_get_clean();

    wp_send_json_success( array( 'html' => $html ) );
}
add_action( 'wp_ajax_scmg_load_schema_fields', 'scmg_ajax_load_schema_fields' );

/**
 * AJAX handler to check if schema exists for a post.
 */
function scmg_ajax_check_existing_schema() {
    check_ajax_referer( 'scmg_ajax_nonce', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Permission denied.', 'frank-schema-markup-generator' ) ) );
    }

    $post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

    if ( empty( $post_id ) ) {
        wp_send_json_error( array( 'message' => __( 'Invalid post ID.', 'frank-schema-markup-generator' ) ) );
    }

    $existing = scmg_get_schema_by_post_id( $post_id );

    if ( $existing ) {
        wp_send_json_success( array(
            'exists' => true,
            'schema' => array(
                'id'          => $existing->id,
                'schema_type' => $existing->schema_type,
                'updated_at'  => mysql2date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $existing->updated_at ),
            ),
        ) );
    } else {
        wp_send_json_success( array( 'exists' => false ) );
    }
}
add_action( 'wp_ajax_scmg_check_existing_schema', 'scmg_ajax_check_existing_schema' );

/**
 * AJAX handler to save schema.
 */
function scmg_ajax_save_schema() {
    check_ajax_referer( 'scmg_ajax_nonce', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Permission denied.', 'frank-schema-markup-generator' ) ) );
    }

    $post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;
    $schema_type = isset( $_POST['schema_type'] ) ? sanitize_text_field( wp_unslash( $_POST['schema_type'] ) ) : '';
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitization happens below with array_map and sanitize_text_field
    $schema_fields_raw = isset( $_POST['schema_fields'] ) ? wp_unslash( $_POST['schema_fields'] ) : array();
    $schema_id = isset( $_POST['schema_id'] ) ? absint( $_POST['schema_id'] ) : 0;

    if ( empty( $post_id ) || empty( $schema_type ) ) {
        wp_send_json_error( array( 'message' => __( 'Invalid parameters.', 'frank-schema-markup-generator' ) ) );
    }

    $post = get_post( $post_id );
    if ( ! $post ) {
        wp_send_json_error( array( 'message' => __( 'Post not found.', 'frank-schema-markup-generator' ) ) );
    }

    // Sanitize schema fields
    $sanitized_fields = array();
    if ( is_array( $schema_fields_raw ) ) {
        foreach ( $schema_fields_raw as $field_key => $field_value ) {
            $sanitized_key = sanitize_key( $field_key );
            if ( is_array( $field_value ) ) {
                $sanitized_fields[ $sanitized_key ] = array_map( 'sanitize_text_field', $field_value );
            } else {
                $sanitized_fields[ $sanitized_key ] = sanitize_text_field( $field_value );
            }
        }
    }

    // Generate JSON-LD
    $schema_array = scmg_generate_schema_from_fields( $schema_type, $sanitized_fields, $post_id );
    $generated_json = '<script type="application/ld+json">' . "\n" .
                      wp_json_encode( $schema_array, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) .
                      "\n" . '</script>';

    // Check if updating existing schema
    $is_update = false;
    if ( $schema_id > 0 ) {
        // Editing mode - update specific schema by ID
        $result = scmg_update_schema( $schema_id, $post_id, $post->post_type, $schema_type, $sanitized_fields, $generated_json );
        $is_update = true;
    } else {
        // New schema - save to database
        $result = scmg_save_schema( $post_id, $post->post_type, $schema_type, $sanitized_fields, $generated_json );
        $existing = scmg_get_schema_by_post_id( $post_id );
        $is_update = ( $existing !== null && $existing->id == $result );
    }

    if ( $result === false ) {
        global $wpdb;
        $error_message = __( 'Failed to save schema.', 'frank-schema-markup-generator' );
        if ( $wpdb->last_error ) {
            $error_message .= ' Database error: ' . $wpdb->last_error;
        }
        wp_send_json_error( array( 'message' => $error_message ) );
    }

    wp_send_json_success( array(
        'message' => $is_update ? __( 'Schema updated successfully!', 'frank-schema-markup-generator' ) : __( 'Schema saved successfully!', 'frank-schema-markup-generator' ),
        'schema_id' => $result,
        'updated' => $is_update,
    ) );
}
add_action( 'wp_ajax_scmg_save_schema', 'scmg_ajax_save_schema' );

/**
 * AJAX handler to delete schema.
 */
function scmg_ajax_delete_schema() {
    // Check nonce - frontend sends 'scmg_delete_schema' nonce
    check_ajax_referer( 'scmg_delete_schema', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Permission denied.', 'frank-schema-markup-generator' ) ) );
    }

    $schema_id = isset( $_POST['schema_id'] ) ? absint( $_POST['schema_id'] ) : 0;

    if ( empty( $schema_id ) ) {
        wp_send_json_error( array( 'message' => __( 'Invalid schema ID.', 'frank-schema-markup-generator' ) ) );
    }

    $result = scmg_delete_schema( $schema_id );

    if ( $result === false ) {
        wp_send_json_error( array( 'message' => __( 'Failed to delete schema.', 'frank-schema-markup-generator' ) ) );
    }

    wp_send_json_success( array( 'message' => __( 'Schema deleted successfully!', 'frank-schema-markup-generator' ) ) );
}
add_action( 'wp_ajax_scmg_delete_schema', 'scmg_ajax_delete_schema' );

/**
 * AJAX handler to bulk delete schemas.
 */
function scmg_ajax_bulk_delete_schemas() {
    // Check nonce
    check_ajax_referer( 'scmg_bulk_delete_schemas', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Permission denied.', 'frank-schema-markup-generator' ) ) );
    }

    $schema_ids = isset( $_POST['schema_ids'] ) && is_array( $_POST['schema_ids'] ) ? array_map( 'absint', $_POST['schema_ids'] ) : array();

    if ( empty( $schema_ids ) ) {
        wp_send_json_error( array( 'message' => __( 'No schemas selected.', 'frank-schema-markup-generator' ) ) );
    }

    $deleted_count = 0;
    $failed_count = 0;

    foreach ( $schema_ids as $schema_id ) {
        $result = scmg_delete_schema( $schema_id );
        if ( $result !== false ) {
            $deleted_count++;
        } else {
            $failed_count++;
        }
    }

    if ( $deleted_count > 0 ) {
        $message = sprintf(
            // translators: %d is the number of schemas deleted
            _n( '%d schema deleted successfully!', '%d schemas deleted successfully!', $deleted_count, 'frank-schema-markup-generator' ),
            $deleted_count
        );

        if ( $failed_count > 0 ) {
            $message .= ' ' . sprintf(
                // translators: %d is the number of schemas that failed to delete
                _n( '%d schema failed to delete.', '%d schemas failed to delete.', $failed_count, 'frank-schema-markup-generator' ),
                $failed_count
            );
        }

        wp_send_json_success( array( 'message' => $message ) );
    } else {
        wp_send_json_error( array( 'message' => __( 'Failed to delete schemas.', 'frank-schema-markup-generator' ) ) );
    }
}
add_action( 'wp_ajax_scmg_bulk_delete_schemas', 'scmg_ajax_bulk_delete_schemas' );

/**
 * AJAX handler to get schema by ID.
 */
function scmg_ajax_get_schema() {
    check_ajax_referer( 'scmg_ajax_nonce', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Permission denied.', 'frank-schema-markup-generator' ) ) );
    }

    $schema_id = isset( $_POST['schema_id'] ) ? absint( $_POST['schema_id'] ) : 0;

    if ( empty( $schema_id ) ) {
        wp_send_json_error( array( 'message' => __( 'Invalid schema ID.', 'frank-schema-markup-generator' ) ) );
    }

    $schema = scmg_get_schema_by_id( $schema_id );

    if ( ! $schema ) {
        wp_send_json_error( array( 'message' => __( 'Schema not found.', 'frank-schema-markup-generator' ) ) );
    }

    wp_send_json_success( array( 'schema' => $schema ) );
}
add_action( 'wp_ajax_scmg_get_schema', 'scmg_ajax_get_schema' );

/**
 * AJAX handler to load posts by post type with search.
 */
function scmg_ajax_load_posts() {
    check_ajax_referer( 'scmg_load_posts', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Permission denied.', 'frank-schema-markup-generator' ) ) );
    }

    $post_type = isset( $_POST['post_type'] ) ? sanitize_key( $_POST['post_type'] ) : '';
    $search = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ) : '';
    $limit = isset( $_POST['limit'] ) ? absint( $_POST['limit'] ) : 20;
    $include_post_id = isset( $_POST['include_post_id'] ) ? absint( $_POST['include_post_id'] ) : 0;

    if ( empty( $post_type ) ) {
        wp_send_json_error( array( 'message' => __( 'Invalid post type.', 'frank-schema-markup-generator' ) ) );
    }

    $args = array(
        'post_type'      => $post_type,
        'posts_per_page' => $limit,
        'post_status'    => 'any',
        'orderby'        => 'date',
        'order'          => 'DESC',
    );

    // If search query provided
    if ( ! empty( $search ) ) {
        // Check if search is numeric (Post ID search)
        if ( is_numeric( $search ) ) {
            $args['p'] = absint( $search );
            $args['posts_per_page'] = 1;
        } else {
            // Title search
            $args['s'] = $search;
        }
    }

    $posts = get_posts( $args );
    $posts_data = array();

    // Get all post IDs that already have schemas
    global $wpdb;
    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $table_name is safe, constructed from $wpdb->prefix and constant
    $existing_post_ids = $wpdb->get_col( "SELECT post_id FROM {$table_name}" );
    $existing_post_ids = array_map( 'intval', $existing_post_ids );

    // If include_post_id is provided and not in results, add it
    if ( $include_post_id > 0 ) {
        $found = false;
        foreach ( $posts as $post ) {
            if ( $post->ID === $include_post_id ) {
                $found = true;
                break;
            }
        }

        if ( ! $found ) {
            $include_post = get_post( $include_post_id );
            if ( $include_post && $include_post->post_type === $post_type ) {
                // Add to beginning of array
                array_unshift( $posts, $include_post );
            }
        }
    }

    foreach ( $posts as $post ) {
        // Skip posts that already have schemas (unless it's the include_post_id for editing)
        if ( in_array( $post->ID, $existing_post_ids, true ) && $post->ID !== $include_post_id ) {
            continue;
        }

        $status_label = '';
        if ( $post->post_status !== 'publish' ) {
            $status_label = ' (' . ucfirst( $post->post_status ) . ')';
        }

        $posts_data[] = array(
            'ID'           => $post->ID,
            'post_title'   => $post->post_title ? $post->post_title : '(No Title)',
            'status_label' => $status_label,
        );
    }

    wp_send_json_success( array(
        'posts' => $posts_data,
        'total_available' => count( $posts_data ),
    ) );
}
add_action( 'wp_ajax_scmg_load_posts', 'scmg_ajax_load_posts' );

/**
 * AJAX handler to filter schemas.
 */
function scmg_ajax_filter_schemas() {
    check_ajax_referer( 'scmg_filter_schemas', 'nonce' );

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => __( 'Permission denied.', 'frank-schema-markup-generator' ) ) );
    }

    global $wpdb;
    $table_name = $wpdb->prefix . SCMG_TABLE_NAME;

    $post_type = isset( $_POST['post_type'] ) ? sanitize_key( $_POST['post_type'] ) : '';
    $schema_type = isset( $_POST['schema_type'] ) ? sanitize_key( $_POST['schema_type'] ) : '';
    $search = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ) : '';
    $sort_column = isset( $_POST['sort_column'] ) ? sanitize_key( $_POST['sort_column'] ) : 'id';
    $sort_order = isset( $_POST['sort_order'] ) && in_array( $_POST['sort_order'], array( 'asc', 'desc' ), true ) ? sanitize_key( $_POST['sort_order'] ) : 'desc';
    $page = isset( $_POST['page'] ) ? absint( $_POST['page'] ) : 1;
    $per_page = isset( $_POST['per_page'] ) ? absint( $_POST['per_page'] ) : 10;

    // Map sort columns to database columns
    $column_map = array(
        'id'          => 'id',
        'title'       => 'post_id',
        'post_type'   => 'post_type',
        'schema_type' => 'schema_type',
        'date'        => 'updated_at',
    );

    $order_by_column = isset( $column_map[ $sort_column ] ) ? $column_map[ $sort_column ] : 'id';

    // Calculate offset
    $offset = ( $page - 1 ) * $per_page;

    // Build query
    $where_conditions = array();
    $prepare_args = array();

    if ( ! empty( $post_type ) ) {
        $where_conditions[] = 'post_type = %s';
        $prepare_args[] = $post_type;
    }

    if ( ! empty( $schema_type ) ) {
        $where_conditions[] = 'schema_type = %s';
        $prepare_args[] = $schema_type;
    }

    if ( ! empty( $search ) ) {
        $where_conditions[] = 'post_id IN (SELECT ID FROM ' . $wpdb->posts . ' WHERE post_title LIKE %s)';
        $prepare_args[] = '%' . $wpdb->esc_like( $search ) . '%';
    }

    // Get total count first
    if ( ! empty( $where_conditions ) ) {
        $where_clause = implode( ' AND ', $where_conditions );
        $count_query = $wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare -- $table_name is safe (wpdb prefix + constant), $where_clause contains placeholders that are prepared with $prepare_args
            "SELECT COUNT(*) FROM {$table_name} WHERE {$where_clause}",
            ...$prepare_args
        );
    } else {
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $table_name is safe, constructed from $wpdb->prefix and constant
        $count_query = "SELECT COUNT(*) FROM {$table_name}";
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Query prepared above with $wpdb->prepare(), $where_clause uses placeholders, search sanitized with sanitize_text_field and esc_like
    $total_count = $wpdb->get_var( $count_query );

    // Get paginated results
    if ( ! empty( $where_conditions ) ) {
        $where_clause = implode( ' AND ', $where_conditions );
        $limit_args = array_merge( $prepare_args, array( $per_page, $offset ) );
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare -- Variables are safe: $table_name from wpdb, $where_clause uses placeholders, $order_by_column and $sort_order are sanitized
        $sql = "SELECT * FROM {$table_name} WHERE {$where_clause} ORDER BY {$order_by_column} {$sort_order} LIMIT %d OFFSET %d";
        $query = call_user_func_array( array( $wpdb, 'prepare' ), array_merge( array( $sql ), $limit_args ) );
    } else {
        $query = $wpdb->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Variables are safe: $table_name from wpdb, $order_by_column and $sort_order are sanitized
            "SELECT * FROM {$table_name} ORDER BY {$order_by_column} {$sort_order} LIMIT %d OFFSET %d",
            array( $per_page, $offset )
        );
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Query prepared above with $wpdb->prepare(), $order_by_column from whitelist, $sort_order validated against array
    $schemas = $wpdb->get_results( $query );

    if ( empty( $schemas ) ) {
        $html = '<tr><td colspan="7" class="scmg-no-results-cell">';
        $html .= '<span class="dashicons dashicons-info scmg-no-results-icon"></span>';
        $html .= esc_html__( 'No schemas found matching your filters.', 'frank-schema-markup-generator' );
        $html .= '</td></tr>';
        wp_send_json_success( array( 'html' => $html, 'count' => 0 ) );
    }

    // Generate HTML
    $html = '';
    $schema_types = scmg_get_schema_types();

    foreach ( $schemas as $schema ) {
        $post = get_post( $schema->post_id );
        $post_title = $post ? $post->post_title : __( '(Post not found)', 'frank-schema-markup-generator' );
        $post_type_obj = get_post_type_object( $schema->post_type );
        $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : $schema->post_type;
        $schema_type_label = isset( $schema_types[ $schema->schema_type ] ) ? $schema_types[ $schema->schema_type ] : $schema->schema_type;

        $html .= '<tr data-schema-id="' . esc_attr( $schema->id ) . '" class="scmg-table-row-hover">';
        $html .= '<th class="check-column"><input type="checkbox" class="scmg-schema-checkbox" value="' . esc_attr( $schema->id ) . '" /></th>';
        $html .= '<td class="scmg-table-cell-id">' . esc_html( $schema->id ) . '</td>';
        $html .= '<td class="scmg-table-cell">';
        $html .= '<strong class="scmg-table-cell-title">' . esc_html( $post_title ) . '</strong>';
        $html .= '</td>';
        $html .= '<td class="scmg-table-cell scmg-table-cell-meta">' . esc_html( $post_type_label ) . '</td>';
        $html .= '<td class="scmg-table-cell"><span class="scmg-schema-type-badge">' . esc_html( $schema_type_label ) . '</span></td>';
        $html .= '<td class="scmg-table-cell scmg-table-cell-meta">' . esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $schema->updated_at ) ) ) . '</td>';
        $html .= '<td class="scmg-table-cell">';
        $html .= '<div class="scmg-actions-wrapper">';
        $html .= '<button type="button" class="button scmg-view-schema-btn scmg-action-btn" data-schema-id="' . esc_attr( $schema->id ) . '" title="' . esc_attr__( 'View Schema Code', 'frank-schema-markup-generator' ) . '">';
        $html .= '<span class="dashicons dashicons-visibility scmg-action-icon"></span>';
        $html .= '</button>';
        $html .= '<button type="button" class="button scmg-copy-schema-btn scmg-action-btn" data-schema-id="' . esc_attr( $schema->id ) . '" title="' . esc_attr__( 'Copy Schema to Clipboard', 'frank-schema-markup-generator' ) . '">';
        $html .= '<span class="dashicons dashicons-clipboard scmg-action-icon"></span>';
        $html .= '</button>';
        $html .= '<a href="?page=scmg-settings&tab=generator&edit=' . esc_attr( $schema->id ) . '" class="button scmg-action-btn" title="' . esc_attr__( 'Edit Schema', 'frank-schema-markup-generator' ) . '">';
        $html .= '<span class="dashicons dashicons-edit scmg-action-icon"></span>';
        $html .= '</a>';
        $html .= '<button type="button" class="button scmg-delete-schema-btn scmg-action-btn-delete" data-schema-id="' . esc_attr( $schema->id ) . '" title="' . esc_attr__( 'Delete Schema', 'frank-schema-markup-generator' ) . '">';
        $html .= '<span class="dashicons dashicons-trash scmg-action-icon"></span>';
        $html .= '</button>';
        $html .= '</div>';
        $html .= '<textarea class="scmg-schema-json-hidden scmg-hidden-textarea" data-schema-id="' . esc_attr( $schema->id ) . '">' . esc_textarea( $schema->generated_json ) . '</textarea>';
        $html .= '</td>';
        $html .= '</tr>';
    }

    wp_send_json_success( array(
        'html' => $html,
        'count' => count( $schemas ),
        'total' => intval( $total_count )
    ) );
}
add_action( 'wp_ajax_scmg_filter_schemas', 'scmg_ajax_filter_schemas' );

/**
 * Generate JSON-LD schema from structured fields.
 *
 * @param string $schema_type   The schema type.
 * @param array  $schema_fields The field values.
 * @param int    $post_id       The post ID.
 * @return array The generated schema array.
 */
function scmg_generate_schema_from_fields( $schema_type, $schema_fields, $post_id ) {
    if ( empty( $schema_type ) || ! is_array( $schema_fields ) ) {
        return array();
    }

    // Replace placeholders in all field values
    foreach ( $schema_fields as $key => $value ) {
        if ( is_string( $value ) ) {
            $schema_fields[ $key ] = scmg_replace_schema_placeholders( $value, $post_id );
        }
    }

    $schema = array(
        '@context' => 'https://schema.org',
        '@type'    => $schema_type,
    );

    // Build schema based on type
    switch ( $schema_type ) {
        case 'Article':
        case 'BlogPosting':
        case 'NewsArticle':
            if ( ! empty( $schema_fields['headline'] ) ) {
                $schema['headline'] = $schema_fields['headline'];
            }
            if ( ! empty( $schema_fields['description'] ) ) {
                $schema['description'] = $schema_fields['description'];
            }
            if ( ! empty( $schema_fields['author_name'] ) ) {
                $schema['author'] = array(
                    '@type' => 'Person',
                    'name'  => $schema_fields['author_name'],
                );
            }
            if ( ! empty( $schema_fields['publisher_name'] ) ) {
                $schema['publisher'] = array(
                    '@type' => 'Organization',
                    'name'  => $schema_fields['publisher_name'],
                );
                if ( ! empty( $schema_fields['publisher_logo'] ) ) {
                    $schema['publisher']['logo'] = array(
                        '@type' => 'ImageObject',
                        'url'   => esc_url_raw( $schema_fields['publisher_logo'] ),
                    );
                }
            }
            if ( ! empty( $schema_fields['date_published'] ) ) {
                $schema['datePublished'] = $schema_fields['date_published'];
            }
            if ( ! empty( $schema_fields['date_modified'] ) ) {
                $schema['dateModified'] = $schema_fields['date_modified'];
            }
            if ( ! empty( $schema_fields['image'] ) ) {
                $schema['image'] = esc_url_raw( $schema_fields['image'] );
            }
            break;

        case 'Product':
            if ( ! empty( $schema_fields['name'] ) ) {
                $schema['name'] = $schema_fields['name'];
            }
            if ( ! empty( $schema_fields['description'] ) ) {
                $schema['description'] = $schema_fields['description'];
            }
            if ( ! empty( $schema_fields['image'] ) ) {
                $schema['image'] = esc_url_raw( $schema_fields['image'] );
            }
            if ( ! empty( $schema_fields['brand'] ) ) {
                $schema['brand'] = array(
                    '@type' => 'Brand',
                    'name'  => $schema_fields['brand'],
                );
            }
            if ( ! empty( $schema_fields['sku'] ) ) {
                $schema['sku'] = $schema_fields['sku'];
            }
            if ( ! empty( $schema_fields['price'] ) && ! empty( $schema_fields['currency'] ) ) {
                $schema['offers'] = array(
                    '@type'         => 'Offer',
                    'price'         => $schema_fields['price'],
                    'priceCurrency' => $schema_fields['currency'],
                );
            }
            break;

        case 'LocalBusiness':
            if ( ! empty( $schema_fields['name'] ) ) {
                $schema['name'] = $schema_fields['name'];
            }
            if ( ! empty( $schema_fields['description'] ) ) {
                $schema['description'] = $schema_fields['description'];
            }

            // Build address
            $has_address = false;
            $address = array( '@type' => 'PostalAddress' );
            if ( ! empty( $schema_fields['address_street'] ) ) {
                $address['streetAddress'] = $schema_fields['address_street'];
                $has_address = true;
            }
            if ( ! empty( $schema_fields['address_city'] ) ) {
                $address['addressLocality'] = $schema_fields['address_city'];
                $has_address = true;
            }
            if ( ! empty( $schema_fields['address_region'] ) ) {
                $address['addressRegion'] = $schema_fields['address_region'];
                $has_address = true;
            }
            if ( ! empty( $schema_fields['address_postal'] ) ) {
                $address['postalCode'] = $schema_fields['address_postal'];
                $has_address = true;
            }
            if ( ! empty( $schema_fields['address_country'] ) ) {
                $address['addressCountry'] = $schema_fields['address_country'];
                $has_address = true;
            }
            if ( $has_address ) {
                $schema['address'] = $address;
            }

            if ( ! empty( $schema_fields['telephone'] ) ) {
                $schema['telephone'] = $schema_fields['telephone'];
            }
            if ( ! empty( $schema_fields['price_range'] ) ) {
                $schema['priceRange'] = $schema_fields['price_range'];
            }
            break;

        case 'Event':
            if ( ! empty( $schema_fields['name'] ) ) {
                $schema['name'] = $schema_fields['name'];
            }
            if ( ! empty( $schema_fields['description'] ) ) {
                $schema['description'] = $schema_fields['description'];
            }
            if ( ! empty( $schema_fields['start_date'] ) ) {
                $schema['startDate'] = $schema_fields['start_date'];
            }
            if ( ! empty( $schema_fields['end_date'] ) ) {
                $schema['endDate'] = $schema_fields['end_date'];
            }
            if ( ! empty( $schema_fields['location_name'] ) || ! empty( $schema_fields['location_address'] ) ) {
                $schema['location'] = array(
                    '@type' => 'Place',
                );
                if ( ! empty( $schema_fields['location_name'] ) ) {
                    $schema['location']['name'] = $schema_fields['location_name'];
                }
                if ( ! empty( $schema_fields['location_address'] ) ) {
                    $schema['location']['address'] = $schema_fields['location_address'];
                }
            }
            break;

        default:
            // For other types, use basic name and description
            if ( ! empty( $schema_fields['name'] ) ) {
                $schema['name'] = $schema_fields['name'];
            }
            if ( ! empty( $schema_fields['description'] ) ) {
                $schema['description'] = $schema_fields['description'];
            }
            break;
    }

    return apply_filters( 'scmg_generated_schema', $schema, $schema_type, $schema_fields, $post_id );
}
