/**
 * Manage Schemas Tab - JavaScript
 * 
 * Handles all functionality for the Manage Schemas tab including:
 * - Schema listing with AJAX
 * - Sorting and pagination
 * - Filtering and search
 * - Bulk actions (delete)
 * - Individual schema actions (view, copy, delete)
 * - Modal functionality
 * 
 * @package WP_Frank_Schema_Markup_Generator_Tool
 * @since 2.4.0
 */

jQuery(document).ready(function($) {
    'use strict';

    // Sorting variables
    var currentSortColumn = 'id';
    var currentSortOrder = 'desc';

    // Pagination variables
    var currentPage = 1;
    var perPage = 10;
    var totalSchemas = 0;

    // Load schemas on page load
    loadSchemas();

    // Apply filters
    $('#scmg_apply_filters').on('click', function() {
        currentPage = 1; // Reset to first page
        loadSchemas();
    });

    // Reset filters
    $('#scmg_reset_filters').on('click', function() {
        $('#scmg_filter_post_type').val('');
        $('#scmg_filter_schema_type').val('');
        $('#scmg_search_schemas').val('');
        currentPage = 1; // Reset to first page
        loadSchemas();
    });

    // Per page change
    $('#scmg_per_page').on('change', function() {
        perPage = parseInt($(this).val());
        currentPage = 1; // Reset to first page
        loadSchemas();
    });

    // Enter key on search
    $('#scmg_search_schemas').on('keypress', function(e) {
        if (e.which === 13) {
            loadSchemas();
        }
    });

    // Column sorting
    $('.scmg-sortable').on('click', function() {
        var column = $(this).data('column');

        // Toggle sort order if same column, otherwise default to desc
        if (currentSortColumn === column) {
            currentSortOrder = (currentSortOrder === 'asc') ? 'desc' : 'asc';
        } else {
            currentSortColumn = column;
            currentSortOrder = 'desc';
        }

        // Update sort indicators
        $('.scmg-sortable').removeClass('sorted-asc sorted-desc');
        $('.scmg-sort-indicator').html('');

        $(this).addClass('sorted-' + currentSortOrder);
        $(this).find('.scmg-sort-indicator').html(
            currentSortOrder === 'asc'
                ? ' <span class="dashicons dashicons-arrow-up-alt2 scmg-dashicon-14"></span>'
                : ' <span class="dashicons dashicons-arrow-down-alt2 scmg-dashicon-14"></span>'
        );

        loadSchemas();
    });

    /**
     * Load schemas via AJAX
     */
    function loadSchemas() {
        var postType = $('#scmg_filter_post_type').val();
        var schemaType = $('#scmg_filter_schema_type').val();
        var search = $('#scmg_search_schemas').val();

        $('#scmg_schemas_tbody').html('<tr><td colspan="7" class="scmg-loading-cell"><span class="dashicons dashicons-update spin scmg-loading-spinner-large"></span><br>' + scmgManageSchemas.i18n.loading + '</td></tr>');

        $.post(scmgManageSchemas.ajaxUrl, {
            action: 'scmg_filter_schemas',
            post_type: postType,
            schema_type: schemaType,
            search: search,
            sort_column: currentSortColumn,
            sort_order: currentSortOrder,
            page: currentPage,
            per_page: perPage,
            nonce: scmgManageSchemas.nonces.filter
        }, function(response) {
            if (response.success) {
                $('#scmg_schemas_tbody').html(response.data.html);

                // Update total count
                totalSchemas = response.data.total;

                // Update counts
                if (postType || schemaType || search) {
                    $('#scmg_filtered_count').show();
                    $('#scmg_filtered_number').text(response.data.total);
                } else {
                    $('#scmg_filtered_count').hide();
                }

                // Update pagination
                updatePagination();

                // Re-bind event handlers
                bindSchemaActions();
            } else {
                $('#scmg_schemas_tbody').html('<tr><td colspan="7" class="scmg-error-message">' + scmgManageSchemas.i18n.errorLoading + '</td></tr>');
            }
        });
    }

    /**
     * Update pagination controls
     */
    function updatePagination() {
        var totalPages = Math.ceil(totalSchemas / perPage);
        var startEntry = totalSchemas === 0 ? 0 : ((currentPage - 1) * perPage) + 1;
        var endEntry = Math.min(currentPage * perPage, totalSchemas);

        // Update showing text
        $('#scmg_showing_text').text(scmgManageSchemas.i18n.showing + ' ' + startEntry + ' ' + scmgManageSchemas.i18n.to + ' ' + endEntry + ' ' + scmgManageSchemas.i18n.of + ' ' + totalSchemas + ' ' + scmgManageSchemas.i18n.entries);

        // Build pagination buttons
        var paginationHTML = '';

        if (totalPages > 1) {
            // Previous button
            paginationHTML += '<button class="button scmg-page-btn" data-page="' + (currentPage - 1) + '" ' + (currentPage === 1 ? 'disabled' : '') + '>';
            paginationHTML += '<span class="dashicons dashicons-arrow-left-alt2 scmg-dashicon-16"></span>';
            paginationHTML += '</button>';

            // Page numbers
            var startPage = Math.max(1, currentPage - 2);
            var endPage = Math.min(totalPages, currentPage + 2);

            // First page
            if (startPage > 1) {
                paginationHTML += '<button class="button scmg-page-btn" data-page="1">1</button>';
                if (startPage > 2) {
                    paginationHTML += '<span class="scmg-pagination-ellipsis">...</span>';
                }
            }

            // Page number buttons
            for (var i = startPage; i <= endPage; i++) {
                var isActive = (i === currentPage);
                paginationHTML += '<button class="button scmg-page-btn ' + (isActive ? 'button-primary' : '') + '" data-page="' + i + '">' + i + '</button>';
            }

            // Last page
            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    paginationHTML += '<span class="scmg-pagination-ellipsis">...</span>';
                }
                paginationHTML += '<button class="button scmg-page-btn" data-page="' + totalPages + '">' + totalPages + '</button>';
            }

            // Next button
            paginationHTML += '<button class="button scmg-page-btn" data-page="' + (currentPage + 1) + '" ' + (currentPage === totalPages ? 'disabled' : '') + '>';
            paginationHTML += '<span class="dashicons dashicons-arrow-right-alt2 scmg-dashicon-16"></span>';
            paginationHTML += '</button>';
        }

        $('#scmg_pagination_buttons').html(paginationHTML);

        // Bind pagination button clicks
        $('.scmg-page-btn').off('click').on('click', function() {
            if (!$(this).prop('disabled')) {
                currentPage = parseInt($(this).data('page'));
                loadSchemas();
            }
        });
    }

    /**
     * Bind schema action buttons
     */
    function bindSchemaActions() {
        // View schema code in modal
        $('.scmg-view-schema-btn').off('click').on('click', function() {
            var schemaId = $(this).data('schema-id');
            var $textarea = $('.scmg-schema-json-hidden[data-schema-id="' + schemaId + '"]');
            var schemaCode = $textarea.val();

            // Show modal with schema code
            $('#scmg-schema-modal-code').text(schemaCode);
            $('#scmg-schema-modal').fadeIn(200);

            // Store schema ID for copy button
            $('#scmg-schema-modal').data('current-schema-id', schemaId);
        });

        // Copy schema to clipboard
        $('.scmg-copy-schema-btn').off('click').on('click', function() {
            var schemaId = $(this).data('schema-id');
            var $textarea = $('.scmg-schema-json-hidden[data-schema-id="' + schemaId + '"]');
            var $button = $(this);
            var originalText = $button.html();

            $textarea.select();
            document.execCommand('copy');

            $button.html('<span class="dashicons dashicons-yes scmg-dashicon-16"></span>');
            $button.css('background-color', '#00a32a').css('color', '#fff');

            setTimeout(function() {
                $button.html(originalText);
                $button.css('background-color', '').css('color', '');
            }, 2000);

            window.getSelection().removeAllRanges();
        });

        // Delete schema
        $('.scmg-delete-schema-btn').off('click').on('click', function() {
            if (!confirm(scmgManageSchemas.i18n.confirmDelete)) {
                return;
            }

            var schemaId = $(this).data('schema-id');
            var $row = $(this).closest('tr');
            var $button = $(this);
            var originalText = $button.html();

            $button.prop('disabled', true).html('<span class="dashicons dashicons-update spin scmg-btn-icon"></span> ' + scmgManageSchemas.i18n.deleting);

            $.post(scmgManageSchemas.ajaxUrl, {
                action: 'scmg_delete_schema',
                schema_id: schemaId,
                nonce: scmgManageSchemas.nonces.delete
            }, function(response) {
                if (response.success) {
                    $row.fadeOut(300, function() {
                        $(this).remove();

                        // Reload schemas
                        loadSchemas();

                        // Update total count
                        var currentTotal = parseInt($('#scmg_total_count').text());
                        $('#scmg_total_count').text(currentTotal - 1);
                    });
                } else {
                    alert(response.data.message || scmgManageSchemas.i18n.errorDeleting);
                    $button.prop('disabled', false).html(originalText);
                }
            });
        });
    }

    // Select All Checkbox
    $('#scmg_select_all_schemas').on('change', function() {
        var isChecked = $(this).prop('checked');
        $('.scmg-schema-checkbox').prop('checked', isChecked);
        updateBulkActionsUI();
    });

    // Individual Checkbox
    $(document).on('change', '.scmg-schema-checkbox', function() {
        updateBulkActionsUI();

        // Update "Select All" checkbox state
        var totalCheckboxes = $('.scmg-schema-checkbox').length;
        var checkedCheckboxes = $('.scmg-schema-checkbox:checked').length;
        $('#scmg_select_all_schemas').prop('checked', totalCheckboxes === checkedCheckboxes);
    });

    /**
     * Update Bulk Actions UI
     */
    function updateBulkActionsUI() {
        var checkedCount = $('.scmg-schema-checkbox:checked').length;

        if (checkedCount > 0) {
            $('.scmg-bulk-actions-container').slideDown(200);
            $('#scmg_selected_count').text(checkedCount + ' ' + scmgManageSchemas.i18n.schemasSelected);
        } else {
            $('.scmg-bulk-actions-container').slideUp(200);
            $('#scmg_selected_count').text('');
        }
    }

    // Apply Bulk Action
    $('#scmg_apply_bulk_action').on('click', function() {
        var action = $('#scmg_bulk_action').val();
        var selectedIds = [];

        $('.scmg-schema-checkbox:checked').each(function() {
            selectedIds.push($(this).val());
        });

        if (!action) {
            alert(scmgManageSchemas.i18n.selectAction);
            return;
        }

        if (selectedIds.length === 0) {
            alert(scmgManageSchemas.i18n.selectAtLeastOne);
            return;
        }

        if (action === 'delete') {
            var confirmMessage = selectedIds.length === 1
                ? scmgManageSchemas.i18n.confirmDelete
                : scmgManageSchemas.i18n.confirmDeleteMultiple.replace('%d', selectedIds.length);

            if (!confirm(confirmMessage)) {
                return;
            }

            // Disable button
            var $button = $(this);
            var originalText = $button.text();
            $button.prop('disabled', true).text(scmgManageSchemas.i18n.deleting);

            // Delete schemas
            $.post(scmgManageSchemas.ajaxUrl, {
                action: 'scmg_bulk_delete_schemas',
                schema_ids: selectedIds,
                nonce: scmgManageSchemas.nonces.bulkDelete
            }, function(response) {
                if (response.success) {
                    // Reload schemas
                    loadSchemas();

                    // Update total count
                    var currentTotal = parseInt($('#scmg_total_count').text());
                    $('#scmg_total_count').text(currentTotal - selectedIds.length);

                    // Reset bulk actions
                    $('#scmg_bulk_action').val('');
                    $('#scmg_select_all_schemas').prop('checked', false);
                    $('.scmg-bulk-actions-container').slideUp(200);

                    // Show success message
                    alert(response.data.message || scmgManageSchemas.i18n.successDeleting);
                } else {
                    alert(response.data.message || scmgManageSchemas.i18n.errorDeleting);
                    $button.prop('disabled', false).text(originalText);
                }
            }).fail(function() {
                alert(scmgManageSchemas.i18n.errorDeleting);
                $button.prop('disabled', false).text(originalText);
            });
        }
    });

    // Modal close handlers
    $('.scmg-modal-close, .scmg-modal-close-btn').on('click', function() {
        $('#scmg-schema-modal').fadeOut(200);
    });

    // Close modal on outside click
    $('#scmg-schema-modal').on('click', function(e) {
        if ($(e.target).is('#scmg-schema-modal')) {
            $(this).fadeOut(200);
        }
    });

    // Copy schema from modal
    $('.scmg-modal-copy-schema').on('click', function() {
        var schemaId = $('#scmg-schema-modal').data('current-schema-id');
        var $textarea = $('.scmg-schema-json-hidden[data-schema-id="' + schemaId + '"]');
        var $button = $(this);
        var originalHtml = $button.html();

        // Copy to clipboard
        $textarea.select();
        document.execCommand('copy');
        window.getSelection().removeAllRanges();

        // Show success feedback
        $button.html('<span class="dashicons dashicons-yes"></span> ' + scmgManageSchemas.i18n.copied);
        $button.css('background-color', '#00a32a');

        setTimeout(function() {
            $button.html(originalHtml);
            $button.css('background-color', '');
        }, 2000);
    });
});

