/**
 * Schema Generator Tab - JavaScript
 *
 * Handles all functionality for the Schema Generator tab including:
 * - Post type selection
 * - Post/page selection
 * - Schema type selection
 * - Schema fields loading
 * - Form submission
 * - Edit mode functionality
 *
 * @package WP_Frank_Schema_Markup_Generator_Tool
 * @since 2.5.0
 */

jQuery(document).ready(function($) {
    'use strict';

    var generatorData = $('#scmg_generator_data');

    if (generatorData.length === 0) {
        return;
    }

    var selectedPostType = '';
    var selectedPostId = parseInt(generatorData.data('selected-post-id')) || 0;
    var selectedSchemaType = generatorData.data('selected-schema-type') || '';
    var editingSchemaId = 0;
    var searchTimeout = null;

    // If editing, pre-select post type and load posts
    var isEditing = generatorData.data('is-editing');
    if (isEditing === '1' || isEditing === 1 || isEditing === true) {
        var editingPostType = generatorData.data('editing-post-type') || '';
        var editingPostId = parseInt(generatorData.data('editing-post-id')) || 0;
        var editingSchemaType = generatorData.data('editing-schema-type') || '';
        editingSchemaId = parseInt(generatorData.data('editing-schema-id')) || 0;

        // Set selected values
        selectedPostType = editingPostType;
        selectedPostId = editingPostId;
        selectedSchemaType = editingSchemaType;

        // Show all steps
        $('#scmg_step_2_container').removeClass('scmg-hidden').show();
        $('#scmg_step_3_container').removeClass('scmg-hidden').show();
        $('#scmg_step_4_container').removeClass('scmg-hidden').show().css('display', 'block');

        // Pre-select post type
        $('input[name="scmg_post_type"][value="' + editingPostType + '"]').prop('checked', true);
        $('input[name="scmg_post_type"][value="' + editingPostType + '"]').closest('label').css({
            'border-color': '#2271b1',
            'background': '#f0f6fc'
        });

        // Load posts and pre-select the editing post
        loadPostsForEditing(editingPostType, editingPostId, function() {
            // Pre-select schema type
            var schemaDropdown = $('#scmg_generator_schema_type');
            schemaDropdown.val(editingSchemaType);
            selectedSchemaType = editingSchemaType;

            // Show loading message in fields container
            $('#scmg_generator_fields_container').html('<div class="scmg-loading">' + scmgGenerator.i18n.loadingFields + '</div>');

            // Load schema fields with existing data
            loadSchemaFieldsForEditing(editingSchemaType, editingPostId, editingSchemaId);
        });
    }

    // STEP 1: Post Type Selection
    $('input[name="scmg_post_type"]').on('change', function() {
        var postType = $(this).val();
        selectedPostType = postType;

        // Reset subsequent steps
        selectedPostId = 0;
        selectedSchemaType = '';
        $('#scmg_step_3_container').hide();
        $('#scmg_step_4_container').hide();
        $('#scmg_generator_schema_type').val('');
        $('#scmg_generator_fields_container').html('');

        // Visual feedback
        $('input[name="scmg_post_type"]').closest('label').css({
            'border-color': '#c3c4c7',
            'background': '#fff'
        });
        $(this).closest('label').css({
            'border-color': '#2271b1',
            'background': '#f0f6fc'
        });

        // Show step 2
        $('#scmg_step_2_container').show();

        // Load posts for selected post type
        loadPosts(postType);
    });

    /**
     * Load posts for a given post type
     */
    function loadPosts(postType, searchQuery) {
        var $selector = $('#scmg_post_selector');
        $selector.html('<option value="">' + scmgGenerator.i18n.loadingPosts + '</option>');
        $selector.prop('disabled', true);

        $.post(scmgGenerator.ajaxUrl, {
            action: 'scmg_load_posts',
            post_type: postType,
            search: searchQuery || '',
            nonce: scmgGenerator.nonces.loadPosts
        }, function(response) {
            if (response.success && response.data.posts) {
                var options = '<option value="">' + scmgGenerator.i18n.selectPost + '</option>';

                if (response.data.posts.length === 0) {
                    options = '<option value="">' + scmgGenerator.i18n.noPostsFound + '</option>';
                } else {
                    $.each(response.data.posts, function(index, post) {
                        options += '<option value="' + post.ID + '">' + post.post_title + post.status_label + '</option>';
                    });
                }

                $selector.html(options);
                $selector.prop('disabled', false);
            } else {
                $selector.html('<option value="">' + scmgGenerator.i18n.errorLoadingPosts + '</option>');
                $selector.prop('disabled', false);
            }
        }).fail(function() {
            $selector.html('<option value="">' + scmgGenerator.i18n.errorLoadingPosts + '</option>');
            $selector.prop('disabled', false);
        });
    }

    /**
     * Load posts for editing (with pre-selection)
     */
    function loadPostsForEditing(postType, postId, callback) {
        var $selector = $('#scmg_post_selector');
        $selector.html('<option value="">' + scmgGenerator.i18n.loadingPosts + '</option>');
        $selector.prop('disabled', true);

        $.post(scmgGenerator.ajaxUrl, {
            action: 'scmg_load_posts',
            post_type: postType,
            include_post_id: postId,
            nonce: scmgGenerator.nonces.loadPosts
        }, function(response) {
            if (response.success && response.data.posts) {
                var options = '<option value="">' + scmgGenerator.i18n.selectPost + '</option>';

                $.each(response.data.posts, function(index, post) {
                    var selected = (post.ID == postId) ? ' selected' : '';
                    options += '<option value="' + post.ID + '"' + selected + '>' + post.post_title + post.status_label + '</option>';
                });

                $selector.html(options);
                $selector.prop('disabled', false);

                // Set the value without triggering change (to avoid loop)
                if (postId) {
                    $selector.val(postId);
                }

                // Call callback if provided
                if (typeof callback === 'function') {
                    callback();
                }
            } else {
                $selector.html('<option value="">' + scmgGenerator.i18n.errorLoadingPosts + '</option>');
                $selector.prop('disabled', false);
            }
        }).fail(function() {
            $selector.html('<option value="">' + scmgGenerator.i18n.errorLoadingPosts + '</option>');
            $selector.prop('disabled', false);
        });
    }

    // STEP 2: Post Selection (Dropdown)
    $('#scmg_post_selector').on('change', function() {
        selectedPostId = parseInt($(this).val()) || 0;

        if (selectedPostId > 0) {
            // Reset subsequent steps
            selectedSchemaType = '';
            $('#scmg_step_4_container').hide();
            $('#scmg_generator_schema_type').val('');
            $('#scmg_generator_fields_container').html('');

            // Show step 3
            $('#scmg_step_3_container').show();
        } else {
            // Hide step 3 if no post selected
            $('#scmg_step_3_container').hide();
            $('#scmg_step_4_container').hide();
        }
    });

    // Search functionality for posts
    $('#scmg_post_search').on('keyup', function() {
        var searchQuery = $(this).val().trim();

        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(function() {
            if (selectedPostType) {
                loadPosts(selectedPostType, searchQuery);
            }
        }, 500); // 500ms debounce
    });

    // STEP 3: Schema Type Selection
    $('#scmg_generator_schema_type').on('change', function() {
        selectedSchemaType = $(this).val();

        if (!selectedPostId || !selectedSchemaType) {
            $('#scmg_step_4_container').hide();
            return;
        }

        // Show step 4
        $('#scmg_step_4_container').show();

        // Load schema fields
        loadSchemaFields(selectedSchemaType, selectedPostId);
    });

    /**
     * Load schema fields
     */
    function loadSchemaFields(schemaType, postId) {
        $('#scmg_generator_fields_container').html('<div class="scmg-loading">' + scmgGenerator.i18n.loadingFields + '</div>');

        // Disable save button while loading
        $('#scmg_generate_save_btn').prop('disabled', true);

        $.post(scmgGenerator.ajaxUrl, {
            action: 'scmg_load_schema_fields',
            schema_type: schemaType,
            post_id: postId,
            nonce: scmgGenerator.nonces.loadFields
        }, function(response) {
            if (response.success) {
                $('#scmg_generator_fields_container').html(response.data.html);
                // Enable save button after fields loaded
                $('#scmg_generate_save_btn').prop('disabled', false);
            } else {
                $('#scmg_generator_fields_container').html('<p style="color: #d63638;">' + scmgGenerator.i18n.errorLoadingFields + '</p>');
            }
        });
    }

    /**
     * Load schema fields for editing
     */
    function loadSchemaFieldsForEditing(schemaType, postId, schemaId) {
        if (!schemaType || !postId) {
            $('#scmg_generator_fields_container').html('<p style="color: #d63638;">' + scmgGenerator.i18n.errorMissingParams + '</p>');
            return;
        }

        $('#scmg_generator_fields_container').html('<div class="scmg-loading">' + scmgGenerator.i18n.loadingFields + '</div>');

        // Disable save button while loading
        $('#scmg_generate_save_btn').prop('disabled', true);

        $.post(scmgGenerator.ajaxUrl, {
            action: 'scmg_load_schema_fields',
            schema_type: schemaType,
            post_id: postId,
            schema_id: schemaId,
            nonce: scmgGenerator.nonces.loadFields
        }, function(response) {
            if (response.success) {
                $('#scmg_generator_fields_container').html(response.data.html);
                // Enable save button after fields loaded
                $('#scmg_generate_save_btn').prop('disabled', false);
            } else {
                $('#scmg_generator_fields_container').html('<p style="color: #d63638;">' + scmgGenerator.i18n.errorPrefix + (response.data && response.data.message ? response.data.message : scmgGenerator.i18n.unknownError) + '</p>');
            }
        }).fail(function(xhr, status, error) {
            $('#scmg_generator_fields_container').html('<p style="color: #d63638;">' + scmgGenerator.i18n.ajaxError + status + ' - ' + error + '</p>');
        });
    }

    // Save Schema Button
    $(document).on('click', '#scmg_generate_save_btn', function(e) {
        e.preventDefault();

        var button = $(this);
        var originalText = button.text();

        // Validate required fields
        if (!selectedPostId || !selectedSchemaType) {
            alert(scmgGenerator.i18n.completeAllSteps);
            return;
        }

        // Disable button
        button.prop('disabled', true).text(scmgGenerator.i18n.saving);

        // Collect schema fields from the form
        var schemaFields = {};
        $('#scmg_generator_fields_container').find('input, select, textarea').each(function() {
            var field = $(this);
            var fieldName = field.attr('name');

            // Match both 'schema_fields[key]' and 'SCMG_schema_fields[key]'
            if (fieldName && (fieldName.indexOf('schema_fields[') !== -1 || fieldName.indexOf('SCMG_schema_fields[') !== -1)) {
                // Extract field key from name using regex
                var match = fieldName.match(/\[([^\]]+)\]/);
                if (match && match[1]) {
                    var fieldKey = match[1];
                    var fieldValue = field.val();
                    schemaFields[fieldKey] = fieldValue;
                }
            }
        });

        // Prepare data for AJAX
        var postData = {
            action: 'scmg_save_schema',
            nonce: scmgGenerator.nonces.saveSchema,
            post_id: selectedPostId,
            schema_type: selectedSchemaType,
            schema_fields: schemaFields
        };

        // Add schema_id if editing
        if (editingSchemaId > 0) {
            postData.schema_id = editingSchemaId;
        }

        $.post(scmgGenerator.ajaxUrl, postData, function(response) {
            if (response.success) {
                // Show inline success message
                var messageHtml = '<div class="notice notice-success scmg-inline-notice" style="margin: 15px 0; padding: 12px; display: flex; align-items: center;">' +
                    '<span class="dashicons dashicons-yes-alt" style="color: #00a32a; font-size: 20px; margin-right: 8px;"></span>' +
                    '<strong>' + (response.data.message || scmgGenerator.i18n.schemaSaved) + '</strong>' +
                    '</div>';

                $('#scmg_generator_message').html(messageHtml);

                // Scroll to message
                $('html, body').animate({
                    scrollTop: $('#scmg_generator_message').offset().top - 100
                }, 500);

                // Redirect to Manage tab after 2 seconds
                setTimeout(function() {
                    window.location.href = scmgGenerator.manageTabUrl;
                }, 2000);
            } else {
                // Show inline error message
                var errorHtml = '<div class="notice notice-error scmg-inline-notice" style="margin: 15px 0; padding: 12px; display: flex; align-items: center;">' +
                    '<span class="dashicons dashicons-warning" style="color: #d63638; font-size: 20px; margin-right: 8px;"></span>' +
                    '<strong>' + scmgGenerator.i18n.errorPrefix + (response.data.message || scmgGenerator.i18n.failedToSave) + '</strong>' +
                    '</div>';

                $('#scmg_generator_message').html(errorHtml);
                button.prop('disabled', false).text(originalText);

                // Scroll to message
                $('html, body').animate({
                    scrollTop: $('#scmg_generator_message').offset().top - 100
                }, 500);
            }
        }).fail(function() {
            // Show inline error message
            var errorHtml = '<div class="notice notice-error scmg-inline-notice" style="margin: 15px 0; padding: 12px; display: flex; align-items: center;">' +
                '<span class="dashicons dashicons-warning" style="color: #d63638; font-size: 20px; margin-right: 8px;"></span>' +
                '<strong>' + scmgGenerator.i18n.errorPrefix + scmgGenerator.i18n.failedToSaveCheck + '</strong>' +
                '</div>';

            $('#scmg_generator_message').html(errorHtml);
            button.prop('disabled', false).text(originalText);

            // Scroll to message
            $('html, body').animate({
                scrollTop: $('#scmg_generator_message').offset().top - 100
            }, 500);
        });
    });

    // Reset Form Button
    $(document).on('click', '#scmg_reset_form_btn', function(e) {
        e.preventDefault();

        if (confirm(scmgGenerator.i18n.confirmReset)) {
            // Reload the page to reset everything
            window.location.href = scmgGenerator.generatorTabUrl;
        }
    });
});

