/**
 * Custom Schema Box Generator - Admin Scripts
 *
 * @package Custom Schema Box Generator
 * @version 1.0.0
 */

(function($) {
    'use strict';

    $(document).ready(function() {

        /**
         * Select All functionality for checkboxes
         * Using event delegation to handle dynamically loaded content
         */
        $(document).on('click', '.csg-select-all-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();

            // Don't do anything if button is disabled
            if ($(this).hasClass('csg-disabled')) {
                return false;
            }

            var container = $(this).closest('.csg-items-list-container');
            var checkboxes = container.find('.csg-items-list input[type="checkbox"]');

            // Check all checkboxes
            checkboxes.prop('checked', true).trigger('change');

            // Update button states
            updateButtonStates();

            return false;
        });

        /**
         * Deselect All functionality for checkboxes
         * Using event delegation to handle dynamically loaded content
         */
        $(document).on('click', '.csg-deselect-all-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();

            // Don't do anything if button is disabled
            if ($(this).hasClass('csg-disabled')) {
                return false;
            }

            var container = $(this).closest('.csg-items-list-container');
            var checkboxes = container.find('.csg-items-list input[type="checkbox"]');

            // Uncheck all checkboxes
            checkboxes.prop('checked', false).trigger('change');

            // Update button states
            updateButtonStates();

            return false;
        });

        /**
         * Update button states based on checkbox selection
         */
        function updateButtonStates() {
            $('.csg-items-list-container').each(function() {
                var container = $(this);
                var checkboxes = container.find('.csg-items-list input[type="checkbox"]');
                var checkedCount = checkboxes.filter(':checked').length;
                var totalCount = checkboxes.length;

                var selectBtn = container.find('.csg-select-all-btn');
                var deselectBtn = container.find('.csg-deselect-all-btn');

                // Remove disabled class and reset styles first
                selectBtn.removeClass('csg-disabled').css({'opacity': '1', 'cursor': 'pointer', 'pointer-events': 'auto'});
                deselectBtn.removeClass('csg-disabled').css({'opacity': '1', 'cursor': 'pointer', 'pointer-events': 'auto'});

                if (checkedCount === totalCount && totalCount > 0) {
                    // All checked - disable Select All
                    selectBtn.addClass('csg-disabled').css({'opacity': '0.5', 'cursor': 'not-allowed'});
                } else if (checkedCount === 0) {
                    // None checked - disable Deselect All
                    deselectBtn.addClass('csg-disabled').css({'opacity': '0.5', 'cursor': 'not-allowed'});
                }
                // If some are checked, both buttons remain enabled
            });
        }

        // Update button states on page load
        setTimeout(function() {
            updateButtonStates();
        }, 100);

        // Update button states when checkboxes change - using event delegation
        $(document).on('change', '.csg-items-list input[type="checkbox"]', function() {
            updateButtonStates();
        });

        /**
         * Template functionality
         */

        // Copy template to clipboard
        $('.scmg-copy-template').on('click', function(e) {
            e.preventDefault();
            var templateId = $(this).data('template-id');
            var templateCode = $('#template-' + templateId).text();

            // Copy to clipboard
            copyToClipboard(templateCode);

            // Show success message
            var button = $(this);
            var originalText = button.text();
            button.text('✓ Copied!').prop('disabled', true);

            setTimeout(function() {
                button.text(originalText).prop('disabled', false);
            }, 2000);
        });

        // View template code in modal
        $('.scmg-view-template').on('click', function(e) {
            e.preventDefault();
            var templateId = $(this).data('template-id');
            var templateCode = $('#template-' + templateId).text();
            var templateName = $(this).closest('.scmg-template-card').find('h3').text();

            $('#scmg-modal-title').text(templateName + ' - Template Code');
            $('#scmg-modal-code').text(templateCode);
            $('#scmg-template-modal').fadeIn(200);
        });

        // Close template modal
        $('.scmg-template-modal-close').on('click', function() {
            $('#scmg-template-modal').fadeOut(200);
        });

        // Close template modal when clicking outside
        $('#scmg-template-modal').on('click', function(e) {
            if ($(e.target).is('#scmg-template-modal')) {
                $('#scmg-template-modal').fadeOut(200);
            }
        });

        // Copy from template modal
        $('.scmg-template-modal-copy').on('click', function(e) {
            e.preventDefault();
            var code = $('#scmg-modal-code').text();
            copyToClipboard(code);

            var button = $(this);
            var originalText = button.text();
            button.text('✓ Copied!').prop('disabled', true);

            setTimeout(function() {
                button.text(originalText).prop('disabled', false);
            }, 2000);
        });

        /**
         * FAQ accordion functionality
         */
        $('.csg-faq-question').on('click', function() {
            var answer = $(this).next('.csg-faq-answer');
            var allAnswers = $('.csg-faq-answer');

            // Close all other answers
            allAnswers.not(answer).slideUp();

            // Toggle current answer
            answer.slideToggle();
        });

        // Open first FAQ by default
        $('.csg-faq-item:first-child .csg-faq-answer').show();

        /**
         * REMOVED: Meta box type toggle (Individual/Dynamic) - no longer needed for WordPress.org compliance
         */

        /**
         * Toggle meta box type container when enabling/disabling post type
         */
        $('.csg-enable-toggle').on('change', function() {
            var postType = $(this).data('post-type');
            var isEnabled = $(this).val() === '1';
            var container = $('.csg-meta-box-type-container[data-post-type="' + postType + '"]');

            if (isEnabled) {
                container.slideDown(300);
            } else {
                container.slideUp(300);
            }
        });

        /**
         * Click to copy placeholder to clipboard
         */
        $(document).on('click', '.csg-placeholder-list li', function() {
            var placeholder = $(this).text();
            var $this = $(this);

            // Copy to clipboard
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(placeholder).then(function() {
                    // Show feedback
                    var originalBg = $this.css('background-color');
                    $this.css('background-color', '#00a32a');
                    $this.css('color', '#fff');

                    setTimeout(function() {
                        $this.css('background-color', '');
                        $this.css('color', '');
                    }, 500);
                }).catch(function() {
                    console.log('Failed to copy placeholder');
                });
            }
        });

        /**
         * Helper function to copy text to clipboard
         */
        function copyToClipboard(text) {
            // Modern clipboard API
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(text).catch(function(err) {
                    // Fallback to old method
                    fallbackCopyToClipboard(text);
                });
            } else {
                // Fallback for older browsers
                fallbackCopyToClipboard(text);
            }
        }

        /**
         * Fallback copy method for older browsers
         */
        function fallbackCopyToClipboard(text) {
            var textArea = document.createElement('textarea');
            textArea.value = text;
            textArea.style.position = 'fixed';
            textArea.style.top = '0';
            textArea.style.left = '0';
            textArea.style.width = '2em';
            textArea.style.height = '2em';
            textArea.style.padding = '0';
            textArea.style.border = 'none';
            textArea.style.outline = 'none';
            textArea.style.boxShadow = 'none';
            textArea.style.background = 'transparent';
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();

            try {
                document.execCommand('copy');
            } catch (err) {
                console.error('Failed to copy text: ', err);
            }

            document.body.removeChild(textArea);
        }

    });

})(jQuery);

