<?php
/**
 * Dashboard Template
 *
 * @package FormRankLS
 */

if (!defined('ABSPATH')) {
    exit;
}

use FormRankLS\Admin\Upgrade_Notice;

// Get learning system status
$scoring_engine = new \FormRankLS\Core\Local_Scoring_Engine();
$learning_stats = $scoring_engine->get_learning_stats();
$learning_status = [
    'mode' => $learning_stats['scoring_mode'],
    'total_leads' => $learning_stats['total_leads'],
    'total_conversions' => $learning_stats['total_conversions'],
    'conversion_rate' => $learning_stats['conversion_rate'],
];

// Calculate progress
if ($learning_status['total_leads'] < 50) {
    $learning_status['mode_label'] = __('Cold Start', 'formrank-lead-scoring');
    $learning_status['mode_icon'] = '🌱';
    $learning_status['mode_color'] = '#3b82f6';
    $learning_status['progress_current'] = $learning_status['total_leads'];
    $learning_status['progress_target'] = 50;
    $learning_status['progress_type'] = 'leads';
    $learning_status['next_mode'] = __('Enhanced Rules', 'formrank-lead-scoring');
} elseif ($learning_status['total_conversions'] < 10) {
    $learning_status['mode_label'] = __('Enhanced Rules', 'formrank-lead-scoring');
    $learning_status['mode_icon'] = '📊';
    $learning_status['mode_color'] = '#f59e0b';
    $learning_status['progress_current'] = $learning_status['total_conversions'];
    $learning_status['progress_target'] = 10;
    $learning_status['progress_type'] = 'conversions';
    $learning_status['next_mode'] = __('Learning Mode', 'formrank-lead-scoring');
} else {
    $learning_status['mode_label'] = __('Learning Mode', 'formrank-lead-scoring');
    $learning_status['mode_icon'] = '🧠';
    $learning_status['mode_color'] = '#10b981';
    $learning_status['progress_current'] = null;
    $learning_status['progress_target'] = null;
    $learning_status['progress_type'] = null;
    $learning_status['next_mode'] = null;
}

$progress_percent = $learning_status['progress_target']
    ? min(100, round(($learning_status['progress_current'] / $learning_status['progress_target']) * 100))
    : 100;
$progress_remaining = $learning_status['progress_target']
    ? max(0, $learning_status['progress_target'] - $learning_status['progress_current'])
    : 0;
?>
<div class="wrap formrank-wrap">
    <h1><?php esc_html_e('FormRank Dashboard', 'formrank-lead-scoring'); ?></h1>

    <?php Upgrade_Notice::render_top_banner('dashboard'); ?>

    <?php if (!empty($demo_lead_count)) : ?>
        <div class="formrank-demo-banner" id="formrank-demo-banner">
            <p>
                <span class="dashicons dashicons-info"></span>
                <?php
                printf(
                    /* translators: %d: number of demo leads */
                    esc_html__('You\'re viewing %d demo leads. Remove them once you start receiving real form submissions.', 'formrank-lead-scoring'),
                    $demo_lead_count
                );
                ?>
            </p>
            <button type="button" class="button" id="formrank-remove-demo-data">
                <?php esc_html_e('Remove Demo Data', 'formrank-lead-scoring'); ?>
            </button>
        </div>
    <?php endif; ?>

    <!-- Stats Cards -->
    <div class="formrank-stats-grid">
        <div class="formrank-stat-card">
            <div class="formrank-stat-content">
                <div class="formrank-stat-value" style="color: #ef4444;"><?php echo esc_html($stats['leads_by_label']['hot']); ?></div>
                <div class="formrank-stat-label"><?php esc_html_e('Hot Leads', 'formrank-lead-scoring'); ?></div>
            </div>
        </div>

        <div class="formrank-stat-card">
            <div class="formrank-stat-content">
                <div class="formrank-stat-value" style="color: #f59e0b;"><?php echo esc_html($stats['leads_by_label']['warm']); ?></div>
                <div class="formrank-stat-label"><?php esc_html_e('Warm Leads', 'formrank-lead-scoring'); ?></div>
            </div>
        </div>

        <div class="formrank-stat-card">
            <div class="formrank-stat-content">
                <div class="formrank-stat-value" style="color: #3b82f6;"><?php echo esc_html($stats['total_leads']); ?></div>
                <div class="formrank-stat-label"><?php esc_html_e('Total Leads', 'formrank-lead-scoring'); ?></div>
            </div>
        </div>

        <div class="formrank-stat-card">
            <div class="formrank-stat-content">
                <div class="formrank-stat-value" style="color: #10b981;"><?php echo esc_html($stats['avg_score'] ?: '-'); ?></div>
                <div class="formrank-stat-label"><?php esc_html_e('Avg Score', 'formrank-lead-scoring'); ?></div>
            </div>
        </div>
    </div>

    <!-- Learning System Status -->
    <div class="formrank-card formrank-learning-quick-status" style="margin-bottom: 20px;">
        <div class="formrank-learning-header">
            <h3 class="formrank-card-title">
                <span class="dashicons dashicons-chart-line"></span>
                <?php esc_html_e('Learning System', 'formrank-lead-scoring'); ?>
            </h3>
            <span class="formrank-mode-badge formrank-mode-<?php echo esc_attr($learning_status['mode']); ?>">
                <?php echo esc_html($learning_status['mode_icon'] . ' ' . $learning_status['mode_label']); ?>
            </span>
        </div>

        <div class="formrank-learning-grid" style="grid-template-columns: repeat(3, 1fr);">
            <div class="formrank-stat-card" style="margin-bottom: 0;">
                <div class="formrank-stat-icon formrank-icon-info">
                    <span class="dashicons dashicons-groups"></span>
                </div>
                <div class="formrank-stat-content">
                    <div class="formrank-stat-value"><?php echo esc_html(number_format($learning_status['total_leads'])); ?></div>
                    <div class="formrank-stat-label"><?php esc_html_e('Total Leads', 'formrank-lead-scoring'); ?></div>
                </div>
            </div>

            <div class="formrank-stat-card" style="margin-bottom: 0;">
                <div class="formrank-stat-icon formrank-icon-success">
                    <span class="dashicons dashicons-yes-alt"></span>
                </div>
                <div class="formrank-stat-content">
                    <div class="formrank-stat-value"><?php echo esc_html(number_format($learning_status['total_conversions'])); ?></div>
                    <div class="formrank-stat-label"><?php esc_html_e('Conversions', 'formrank-lead-scoring'); ?></div>
                </div>
            </div>

            <div class="formrank-stat-card" style="margin-bottom: 0;">
                <div class="formrank-stat-icon formrank-icon-hot">
                    <span class="dashicons dashicons-chart-line"></span>
                </div>
                <div class="formrank-stat-content">
                    <div class="formrank-stat-value"><?php echo esc_html($learning_status['conversion_rate']); ?>%</div>
                    <div class="formrank-stat-label"><?php esc_html_e('Conversion Rate', 'formrank-lead-scoring'); ?></div>
                </div>
            </div>
        </div>

        <?php if ($learning_status['progress_target']) : ?>
        <div class="formrank-learning-progress" style="margin-top: 15px; margin-bottom: 0;">
            <div class="formrank-progress-header">
                <span class="formrank-progress-text">
                    <?php
                    printf(
                        /* translators: %s: next mode name */
                        esc_html__('Progress to %s', 'formrank-lead-scoring'),
                        esc_html($learning_status['next_mode'])
                    );
                    ?>
                </span>
                <span class="formrank-progress-count">
                    <?php
                    printf(
                        /* translators: 1: current count, 2: target count, 3: type (leads/conversions) */
                        esc_html__('%1$d / %2$d %3$s', 'formrank-lead-scoring'),
                        intval($learning_status['progress_current']),
                        intval($learning_status['progress_target']),
                        esc_html($learning_status['progress_type'])
                    );
                    ?>
                </span>
            </div>
            <div class="formrank-usage-meter" style="padding: 0;">
                <div class="formrank-usage-bar">
                    <div class="formrank-usage-fill"
                         style="width: <?php echo esc_attr($progress_percent); ?>%;
                                background: linear-gradient(90deg, <?php echo esc_attr($learning_status['mode_color']); ?>, <?php echo esc_attr($learning_status['mode_color']); ?>dd);">
                    </div>
                </div>
            </div>
            <p class="formrank-progress-hint" style="margin-bottom: 0;">
                <?php
                if ($progress_remaining > 0) {
                    printf(
                        /* translators: 1: remaining count, 2: type (leads/conversions), 3: next mode name */
                        esc_html__('%1$d more %2$s needed to unlock %3$s', 'formrank-lead-scoring'),
                        esc_html($progress_remaining),
                        esc_html($learning_status['progress_type']),
                        esc_html($learning_status['next_mode'])
                    );
                }
                ?>
            </p>
        </div>
        <?php endif; ?>

        <p style="margin: 15px 0 0 0; text-align: right;">
            <a href="<?php echo esc_url(admin_url('admin.php?page=formrank-lead-scoring-conversions')); ?>" class="button">
                <?php esc_html_e('View Conversion Analytics', 'formrank-lead-scoring'); ?> &rarr;
            </a>
        </p>
    </div>

    <!-- Main Content Grid -->
    <div class="formrank-dashboard-grid">
        <!-- Recent Leads -->
        <div class="formrank-card">
            <h2><?php esc_html_e('Recent Leads', 'formrank-lead-scoring'); ?></h2>

            <?php if (empty($stats['recent_leads'])) : ?>
                <p class="formrank-empty"><?php esc_html_e('No leads captured yet.', 'formrank-lead-scoring'); ?></p>
            <?php else : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Score', 'formrank-lead-scoring'); ?></th>
                            <th><?php esc_html_e('Name', 'formrank-lead-scoring'); ?></th>
                            <th><?php esc_html_e('Email', 'formrank-lead-scoring'); ?></th>
                            <th><?php esc_html_e('Date', 'formrank-lead-scoring'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($stats['recent_leads'] as $lead) : ?>
                            <tr>
                                <td>
                                    <?php
                                    $colors = ['hot' => '#ef4444', 'warm' => '#f59e0b', 'neutral' => '#10b981', 'cool' => '#3b82f6', 'cold' => '#6b7280'];
                                    $color = $colors[$lead['score_label'] ?? ''] ?? '#6b7280';
                                    ?>
                                    <strong style="color: <?php echo esc_attr($color); ?>">
                                        <?php echo esc_html($lead['score'] ?? '-'); ?>
                                    </strong>
                                </td>
                                <td>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=formrank-lead-scoring-lead&id=' . $lead['id'])); ?>">
                                        <?php echo esc_html($lead['name'] ?: __('Unknown', 'formrank-lead-scoring')); ?>
                                    </a>
                                </td>
                                <td><?php echo esc_html($lead['email'] ?: '-'); ?></td>
                                <td>
                                    <?php
                                    if ($lead['created_at']) {
                                        echo esc_html(human_time_diff(strtotime($lead['created_at']), current_time('timestamp')) . ' ' . __('ago', 'formrank-lead-scoring'));
                                    }
                                    ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <p style="margin-top: 15px;">
                    <a href="<?php echo esc_url(admin_url('admin.php?page=formrank-lead-scoring')); ?>" class="button">
                        <?php esc_html_e('View All Leads', 'formrank-lead-scoring'); ?>
                    </a>
                </p>
            <?php endif; ?>
        </div>

        <!-- Usage Stats -->
        <div class="formrank-card">
            <h2><?php esc_html_e('Usage This Month', 'formrank-lead-scoring'); ?></h2>

            <div class="formrank-usage-meter">
                <?php
                $current = $stats['usage']['current_month'];
                $limit = $stats['usage']['limit'];
                $is_unlimited = $limit === 'unlimited';
                $percentage = $is_unlimited ? 0 : min(100, ($current / $limit) * 100);
                ?>

                <div class="formrank-usage-bar">
                    <div class="formrank-usage-fill" style="width: <?php echo esc_attr($percentage); ?>%;"></div>
                </div>

                <div class="formrank-usage-text">
                    <?php if ($is_unlimited) : ?>
                        <strong><?php echo esc_html($current); ?></strong> <?php esc_html_e('leads scored (unlimited)', 'formrank-lead-scoring'); ?>
                    <?php else : ?>
                        <strong><?php echo esc_html($current); ?></strong> / <?php echo esc_html($limit); ?> <?php esc_html_e('leads scored', 'formrank-lead-scoring'); ?>
                    <?php endif; ?>
                </div>

                <?php if (!$stats['usage']['is_pro']) : ?>
                    <p class="formrank-upgrade-notice">
                        <?php esc_html_e('Upgrade to Pro for unlimited leads.', 'formrank-lead-scoring'); ?>
                    </p>
                <?php endif; ?>
            </div>

            <!-- Score Distribution -->
            <h3 style="margin-top: 20px;"><?php esc_html_e('Score Distribution', 'formrank-lead-scoring'); ?></h3>

            <div class="formrank-distribution">
                <?php
                $labels = [
                    'hot' => ['name' => __('Hot', 'formrank-lead-scoring'), 'color' => '#ef4444'],
                    'warm' => ['name' => __('Warm', 'formrank-lead-scoring'), 'color' => '#f59e0b'],
                    'neutral' => ['name' => __('Neutral', 'formrank-lead-scoring'), 'color' => '#10b981'],
                    'cool' => ['name' => __('Cool', 'formrank-lead-scoring'), 'color' => '#3b82f6'],
                    'cold' => ['name' => __('Cold', 'formrank-lead-scoring'), 'color' => '#6b7280']
                ];

                $total = max(1, array_sum($stats['leads_by_label']));

                foreach ($labels as $key => $label) :
                    $count = $stats['leads_by_label'][$key] ?? 0;
                    $pct = round(($count / $total) * 100);
                ?>
                    <div class="formrank-dist-row">
                        <span class="formrank-dist-label"><?php echo esc_html($label['name']); ?></span>
                        <div class="formrank-dist-bar">
                            <div class="formrank-dist-fill" style="width: <?php echo esc_attr($pct); ?>%; background: <?php echo esc_attr($label['color']); ?>;"></div>
                        </div>
                        <span class="formrank-dist-count"><?php echo esc_html($count); ?></span>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
    <p style="text-align: center; color: #9ca3af; font-size: 12px; margin-top: 24px;">
        <?php
        printf(
            /* translators: %s: FormRank Lead Scoring link */
            esc_html__('Powered by %s', 'formrank-lead-scoring'),
            '<a href="https://formrankwp.com" style="color: #9ca3af; text-decoration: underline;">FormRank</a>'
        );
        ?>
    </p>
</div>
