<?php
/**
 * WPForms Integration
 *
 * @package FormRankLS\Integrations
 */

namespace FormRankLS\Integrations;

if (!defined('ABSPATH')) {
    exit;
}

class WPForms extends Integration_Base {

    protected $plugin_name = 'wpforms';

    /**
     * Capture WPForms submission
     *
     * Hook: wpforms_process_complete
     *
     * @param array $fields Submitted form fields
     * @param array $entry Entry data
     * @param array $form_data Form configuration
     * @param int $entry_id Entry ID
     */
    public function capture_submission($fields, $entry, $form_data, $entry_id) {
        // Check if this form should be scored
        if (!$this->should_score_form($form_data['id'])) {
            return;
        }

        // Extract lead data from fields
        $lead_data = $this->extract_lead_data($fields, $form_data);

        // Save and score the lead
        $this->process_lead($lead_data, [
            'form_plugin' => 'wpforms',
            'form_id' => $form_data['id'],
            'entry_id' => $entry_id,
            'source_url' => wp_get_referer() ?: home_url()
        ]);
    }

    /**
     * Extract structured data from WPForms fields
     */
    private function extract_lead_data(array $fields, array $form_data): array {
        $lead_data = [];
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        foreach ($fields as $field_id => $field) {
            $field_name = sanitize_text_field($field['name'] ?? "field_{$field_id}");
            $field_value = $this->get_field_value($field);

            // Store all fields
            if (!empty($field_value)) {
                $lead_data[$field_name] = $field_value;
            }

            // Try to identify common fields
            $field_type = $field['type'] ?? '';
            $field_name_lower = strtolower($field_name);

            // Email detection
            if ($field_type === 'email' || strpos($field_name_lower, 'email') !== false) {
                $email = is_array($field_value) ? ($field_value['primary'] ?? '') : $field_value;
            }

            // Name detection
            if ($field_type === 'name' || in_array($field_name_lower, ['name', 'full name', 'your name', 'full_name'])) {
                if (is_array($field['value_raw'] ?? null)) {
                    // WPForms name field returns array with first/last
                    $name_parts = $field['value_raw'];
                    $name = trim(
                        ($name_parts['first'] ?? '') . ' ' .
                        ($name_parts['middle'] ?? '') . ' ' .
                        ($name_parts['last'] ?? '')
                    );
                } else {
                    $name = $field_value;
                }
            }

            // First name + Last name detection
            if (in_array($field_name_lower, ['first name', 'first_name', 'firstname'])) {
                $first_name = $field_value;
            }
            if (in_array($field_name_lower, ['last name', 'last_name', 'lastname'])) {
                $last_name = $field_value;
            }

            // Company detection
            if (strpos($field_name_lower, 'company') !== false ||
                strpos($field_name_lower, 'business') !== false ||
                strpos($field_name_lower, 'organization') !== false) {
                $company = $field_value;
            }

            // Phone detection
            if ($field_type === 'phone' || strpos($field_name_lower, 'phone') !== false) {
                $phone = $field_value;
            }
        }

        // Combine first + last name if full name wasn't found directly
        if (empty($name) && (!empty($first_name) || !empty($last_name))) {
            $name = trim(($first_name ?? '') . ' ' . ($last_name ?? ''));
        }

        return [
            'fields' => $lead_data,
            'email' => $email,
            'name' => $name,
            'company' => $company,
            'phone' => $phone
        ];
    }

    /**
     * Get field value handling different field types
     */
    private function get_field_value(array $field): string {
        $value = $field['value'] ?? '';

        // Handle array values
        if (is_array($value)) {
            return implode(', ', array_filter($value));
        }

        return sanitize_textarea_field($value);
    }

    /**
     * Get list of available WPForms forms
     */
    public static function get_available_forms(): array {
        if (!function_exists('wpforms')) {
            return [];
        }

        $forms = wpforms()->form->get('', [
            'orderby' => 'title',
            'order' => 'ASC'
        ]);

        if (empty($forms)) {
            return [];
        }

        $form_list = [];

        foreach ($forms as $form) {
            $form_list[$form->ID] = [
                'id' => $form->ID,
                'title' => $form->post_title,
                'created' => $form->post_date
            ];
        }

        return $form_list;
    }
}
