<?php
/**
 * Base Integration Class - shared functionality for all form integrations
 *
 * @package FormRankLS\Integrations
 */

namespace FormRankLS\Integrations;

if (!defined('ABSPATH')) {
    exit;
}

use FormRankLS\Core\Local_Scoring_Engine;
use FormRankLS\Core\Webhook_Handler;

class Integration_Base {

    protected $plugin_name = 'base';
    protected $scoring_engine;
    protected $webhook_handler;

    public function __construct() {
        // Use local scoring engine (no external API required)
        $this->scoring_engine = new Local_Scoring_Engine();
        $this->webhook_handler = new Webhook_Handler();
    }

    /**
     * Check if form should be scored
     */
    protected function should_score_form(int $form_id): bool {
        $settings = get_option('formrank_settings', []);
        $enabled_forms = $settings['enabled_forms'][$this->plugin_name] ?? [];

        // If no forms specified, score all forms
        if (empty($enabled_forms)) {
            return true;
        }

        return in_array($form_id, $enabled_forms);
    }

    /**
     * Process a captured lead
     */
    protected function process_lead(array $lead_data, array $meta): void {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Sanitize all data
        $lead_data = $this->sanitize_lead_data($lead_data);

        // Insert lead record
        $insert_data = [
            'form_plugin' => sanitize_text_field($meta['form_plugin']),
            'form_id' => absint($meta['form_id']),
            'entry_id' => isset($meta['entry_id']) ? absint($meta['entry_id']) : null,
            'email' => sanitize_email($lead_data['email'] ?? ''),
            'name' => sanitize_text_field($lead_data['name'] ?? ''),
            'company' => sanitize_text_field($lead_data['company'] ?? ''),
            'phone' => sanitize_text_field($lead_data['phone'] ?? ''),
            'form_data' => wp_json_encode($lead_data['fields'] ?? []),
            'source_url' => esc_url_raw($meta['source_url'] ?? ''),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(substr(wp_unslash($_SERVER['HTTP_USER_AGENT']), 0, 512)) : '',
            'ip_address' => $this->get_client_ip(),
            'status' => 'new',
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        ];

        $wpdb->insert($table, $insert_data);
        $lead_id = $wpdb->insert_id;

        if (!$lead_id) {
            error_log('FormRank: Failed to insert lead record');
            return;
        }

        // Score the lead asynchronously if possible, otherwise sync
        if (function_exists('wp_schedule_single_event') && !defined('DOING_CRON')) {
            wp_schedule_single_event(time(), 'formrank_score_lead', [$lead_id]);
        } else {
            $this->score_lead($lead_id);
        }
    }

    /**
     * Score a specific lead (called via cron or directly)
     */
    public function score_lead(int $lead_id): void {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Get lead data
        $lead = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $lead_id
        ), ARRAY_A);

        if (!$lead) {
            return;
        }

        // Score the lead
        $form_data = json_decode($lead['form_data'], true) ?: [];
        $score_result = $this->scoring_engine->score_lead($form_data);

        // Update lead record
        $wpdb->update($table, [
            'score' => $score_result['score'],
            'score_label' => $score_result['label'],
            'score_reasoning' => $score_result['reasoning'],
            'score_factors' => wp_json_encode($score_result['factors'] ?? []),
            'scoring_method' => 'local',
            'scored_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        ], ['id' => $lead_id]);

        // Trigger notifications and webhooks
        $this->trigger_notifications($lead_id, $score_result);
        $this->webhook_handler->trigger($lead_id, $score_result);

        // Increment usage counter
        $this->increment_usage();
    }

    /**
     * Sanitize lead data recursively
     */
    protected function sanitize_lead_data(array $data): array {
        $sanitized = [];

        foreach ($data as $key => $value) {
            $key = sanitize_key($key);

            if (is_array($value)) {
                $sanitized[$key] = $this->sanitize_lead_data($value);
            } elseif (is_email($value)) {
                $sanitized[$key] = sanitize_email($value);
            } else {
                $sanitized[$key] = sanitize_textarea_field($value);
            }
        }

        return $sanitized;
    }

    /**
     * Increment usage counter
     */
    protected function increment_usage(): void {
        $usage = get_option('formrank_usage_stats', []);
        $month = gmdate('Y-m');
        $usage[$month] = ($usage[$month] ?? 0) + 1;

        // Keep only last 3 months
        $usage = array_slice($usage, -3, 3, true);

        update_option('formrank_usage_stats', $usage);
    }

    /**
     * Get current usage stats
     */
    public function get_usage_stats(): array {
        $usage = get_option('formrank_usage_stats', []);
        $month = gmdate('Y-m');

        return [
            'current_month' => $usage[$month] ?? 0,
            'limit' => $this->has_pro_license() ? 'unlimited' : 50,
            'is_pro' => $this->has_pro_license(),
            'history' => $usage
        ];
    }

    /**
     * Check if Pro license is active
     */
    protected function has_pro_license(): bool {
        return apply_filters('formrank_ls_is_pro', false);
    }

    /**
     * Get client IP address
     */
    protected function get_client_ip(): string {
        $ip_keys = [
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_X_FORWARDED_FOR',      // Load balancers
            'HTTP_X_REAL_IP',            // Nginx proxy
            'REMOTE_ADDR'                // Direct connection
        ];

        foreach ($ip_keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = explode(',', sanitize_text_field(wp_unslash($_SERVER[$key])))[0];
                $ip = trim($ip);

                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }

        return '';
    }

    /**
     * Trigger email notifications for hot leads
     */
    protected function trigger_notifications(int $lead_id, array $score_result): void {
        $settings = get_option('formrank_settings', []);

        // Email notification
        if (!empty($settings['notifications']['email_enabled'])) {
            $threshold = $settings['notifications']['hot_lead_threshold'] ?? 80;

            if ($score_result['score'] >= $threshold) {
                $this->send_hot_lead_email($lead_id, $score_result);
            }
        }

        // Slack notification
        if (!empty($settings['notifications']['slack_webhook'])) {
            $threshold = $settings['notifications']['hot_lead_threshold'] ?? 80;

            if ($score_result['score'] >= $threshold) {
                $this->webhook_handler->send_slack_notification($lead_id, $score_result);
            }
        }
    }

    /**
     * Send hot lead notification email
     */
    protected function send_hot_lead_email(int $lead_id, array $score_result): void {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $lead = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $lead_id
        ), ARRAY_A);

        if (!$lead) {
            return;
        }

        $settings = get_option('formrank_settings', []);
        $recipients = $settings['notifications']['email_recipients'] ?? get_option('admin_email');

        $subject = sprintf(
            /* translators: 1: Lead name or email, 2: Lead score */
            __('[FormRank] Hot Lead Alert: %1$s (Score: %2$d)', 'formrank-lead-scoring'),
            $lead['name'] ?: $lead['email'] ?: __('New Lead', 'formrank-lead-scoring'),
            $score_result['score']
        );

        // Load email template
        ob_start();
        $template_path = FORMRANK_LS_PATH . 'templates/emails/hot-lead-alert.php';

        if (file_exists($template_path)) {
            include $template_path;
        } else {
            // Fallback to simple email
            echo wp_kses_post($this->get_fallback_email_content($lead, $score_result));
        }

        $message = ob_get_clean();

        $headers = ['Content-Type: text/html; charset=UTF-8'];

        wp_mail($recipients, $subject, $message, $headers);
    }

    /**
     * Fallback email content if template is missing
     */
    private function get_fallback_email_content(array $lead, array $score_result): string {
        $view_url = admin_url("admin.php?page=formrank-lead-scoring-lead&id={$lead['id']}");
        $form_data = json_decode($lead['form_data'], true) ?: [];

        $html = '<html><body style="font-family: Arial, sans-serif;">';
        $html .= '<h2>Hot Lead Alert!</h2>';
        $html .= '<p>A new high-scoring lead has been captured:</p>';
        $html .= '<table style="border-collapse: collapse; width: 100%;">';
        $html .= sprintf('<tr><td style="padding: 8px; border: 1px solid #ddd;"><strong>Score</strong></td><td style="padding: 8px; border: 1px solid #ddd;">%d (%s)</td></tr>', intval($score_result['score']), esc_html($score_result['label']));
        $html .= sprintf('<tr><td style="padding: 8px; border: 1px solid #ddd;"><strong>Name</strong></td><td style="padding: 8px; border: 1px solid #ddd;">%s</td></tr>', esc_html($lead['name'] ?: 'N/A'));
        $html .= sprintf('<tr><td style="padding: 8px; border: 1px solid #ddd;"><strong>Email</strong></td><td style="padding: 8px; border: 1px solid #ddd;">%s</td></tr>', esc_html($lead['email'] ?: 'N/A'));
        $html .= sprintf('<tr><td style="padding: 8px; border: 1px solid #ddd;"><strong>Company</strong></td><td style="padding: 8px; border: 1px solid #ddd;">%s</td></tr>', esc_html($lead['company'] ?: 'N/A'));
        $html .= sprintf('<tr><td style="padding: 8px; border: 1px solid #ddd;"><strong>Reasoning</strong></td><td style="padding: 8px; border: 1px solid #ddd;">%s</td></tr>', esc_html($score_result['reasoning']));
        $html .= '</table>';
        $html .= sprintf('<p><a href="%s" style="background: #2563eb; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block; margin-top: 15px;">View Lead Details</a></p>', esc_url($view_url));
        $html .= '</body></html>';

        return $html;
    }
}
