<?php
/**
 * Gravity Forms Integration
 *
 * @package FormRankLS\Integrations
 */

namespace FormRankLS\Integrations;

if (!defined('ABSPATH')) {
    exit;
}

class Gravity_Forms extends Integration_Base {

    protected $plugin_name = 'gravity';

    /**
     * Capture Gravity Forms submission
     *
     * Hook: gform_after_submission
     *
     * @param array $entry The entry that was just created
     * @param array $form The form used to create the entry
     */
    public function capture_submission($entry, $form) {
        // Check if this form should be scored
        if (!$this->should_score_form($form['id'])) {
            return;
        }

        // Extract lead data from entry
        $lead_data = $this->extract_lead_data($entry, $form);

        // Save and score the lead
        $this->process_lead($lead_data, [
            'form_plugin' => 'gravity',
            'form_id' => $form['id'],
            'entry_id' => $entry['id'],
            'source_url' => $entry['source_url'] ?? wp_get_referer()
        ]);
    }

    /**
     * Extract structured data from Gravity Forms entry
     */
    private function extract_lead_data(array $entry, array $form): array {
        $lead_data = [];
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        foreach ($form['fields'] as $field) {
            $field_id = $field->id;
            $field_label = sanitize_text_field($field->label);
            $field_type = $field->type;

            // Get field value (handle sub-fields for complex types)
            $value = $this->get_field_value($entry, $field);

            if (!empty($value)) {
                $lead_data[$field_label] = $value;
            }

            $field_label_lower = strtolower($field_label);

            // Email detection
            if ($field_type === 'email') {
                $email = $value;
            }

            // Name detection - by type or label
            if ($field_type === 'name') {
                $name = $value;
            } elseif (in_array($field_label_lower, ['name', 'full name', 'your name', 'full_name'])) {
                $name = $value;
            }

            // First/Last name detection by label
            if (in_array($field_label_lower, ['first name', 'first_name', 'firstname'])) {
                $first_name = $value;
            }
            if (in_array($field_label_lower, ['last name', 'last_name', 'lastname', 'surname'])) {
                $last_name = $value;
            }

            // Company detection
            if (strpos($field_label_lower, 'company') !== false ||
                strpos($field_label_lower, 'business') !== false ||
                strpos($field_label_lower, 'organization') !== false) {
                $company = $value;
            }

            // Phone detection
            if ($field_type === 'phone') {
                $phone = $value;
            }
        }

        // Combine first + last name if full name wasn't found directly
        if (empty($name) && (!empty($first_name) || !empty($last_name))) {
            $name = trim(($first_name ?? '') . ' ' . ($last_name ?? ''));
        }

        return [
            'fields' => $lead_data,
            'email' => $email,
            'name' => $name,
            'company' => $company,
            'phone' => $phone
        ];
    }

    /**
     * Get field value handling different field types
     */
    private function get_field_value(array $entry, $field): string {
        $field_id = $field->id;
        $field_type = $field->type;

        // Handle complex field types (name, address, etc.)
        if (in_array($field_type, ['name', 'address'])) {
            $inputs = $field->inputs ?? [];
            $parts = [];

            foreach ($inputs as $input) {
                $input_id = $input['id'];
                $value = $entry[$input_id] ?? '';
                if (!empty($value)) {
                    $parts[] = sanitize_text_field($value);
                }
            }

            return implode(' ', $parts);
        }

        // Handle checkbox/multi-select
        if ($field_type === 'checkbox' || $field_type === 'multiselect') {
            $choices = [];
            $inputs = $field->inputs ?? [];

            foreach ($inputs as $input) {
                $input_id = $input['id'];
                $value = $entry[$input_id] ?? '';
                if (!empty($value)) {
                    $choices[] = sanitize_text_field($value);
                }
            }

            return implode(', ', $choices);
        }

        // Standard field
        $value = $entry[$field_id] ?? '';

        return sanitize_textarea_field($value);
    }

    /**
     * Get list of available Gravity Forms
     */
    public static function get_available_forms(): array {
        if (!class_exists('GFAPI')) {
            return [];
        }

        $forms = \GFAPI::get_forms();

        if (empty($forms)) {
            return [];
        }

        $form_list = [];

        foreach ($forms as $form) {
            $form_list[$form['id']] = [
                'id' => $form['id'],
                'title' => $form['title'],
                'is_active' => $form['is_active']
            ];
        }

        return $form_list;
    }
}
