<?php
/**
 * Formidable Forms Integration
 *
 * @package FormRankLS\Integrations
 */

namespace FormRankLS\Integrations;

if (!defined('ABSPATH')) {
    exit;
}

class Formidable extends Integration_Base {

    protected $plugin_name = 'formidable';

    /**
     * Capture Formidable Forms submission
     *
     * Hook: frm_after_create_entry
     *
     * @param int   $entry_id The entry ID
     * @param int   $form_id  The form ID
     */
    public function capture_submission( $entry_id, $form_id ) {
        // Check if this form should be scored
        if ( ! $this->should_score_form( $form_id ) ) {
            return;
        }

        // Get entry data
        $entry = \FrmEntry::getOne( $entry_id );

        if ( ! $entry ) {
            return;
        }

        // Get form fields
        $form = \FrmForm::getOne( $form_id );

        if ( ! $form ) {
            return;
        }

        // Extract lead data from entry
        $lead_data = $this->extract_lead_data( $entry_id, $form_id );

        // Save and score the lead
        $this->process_lead( $lead_data, [
            'form_plugin' => 'formidable',
            'form_id'     => $form_id,
            'entry_id'    => $entry_id,
            'source_url'  => wp_get_referer() ?: home_url(),
        ] );
    }

    /**
     * Extract structured data from Formidable entry
     */
    private function extract_lead_data( int $entry_id, int $form_id ): array {
        global $wpdb;

        $lead_data = [];
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        // Get all fields for this form
        $fields_table = $wpdb->prefix . 'frm_fields';
        $meta_table = $wpdb->prefix . 'frm_item_metas';

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $results = $wpdb->get_results( $wpdb->prepare(
            "SELECT f.id as field_id, f.name, f.type, f.field_key, m.meta_value
            FROM {$fields_table} f
            LEFT JOIN {$meta_table} m ON f.id = m.field_id AND m.item_id = %d
            WHERE f.form_id = %d
            ORDER BY f.field_order ASC",
            $entry_id,
            $form_id
        ), ARRAY_A );

        foreach ( $results as $field ) {
            $field_name = sanitize_text_field( $field['name'] );
            $field_value = $field['meta_value'];
            $field_type = $field['type'];

            // Handle serialized values (checkboxes, etc.)
            if ( is_serialized( $field_value ) ) {
                $unserialized = maybe_unserialize( $field_value );
                if ( is_array( $unserialized ) ) {
                    $field_value = implode( ', ', array_filter( $unserialized ) );
                }
            }

            if ( empty( $field_value ) ) {
                continue;
            }

            $lead_data[ $field_name ] = sanitize_textarea_field( $field_value );

            $field_name_lower = strtolower( $field_name );
            $field_key_lower = strtolower( $field['field_key'] );

            // Email detection
            if ( $field_type === 'email' || strpos( $field_name_lower, 'email' ) !== false ) {
                $email = $field_value;
            }

            // Name detection - by type, key, or label (with exclusions for false positives)
            if ( $field_type === 'name' ||
                 in_array( $field_key_lower, [ 'name', 'full_name', 'your_name' ], true ) ||
                 ( strpos( $field_name_lower, 'name' ) !== false &&
                   strpos( $field_name_lower, 'last' ) === false &&
                   strpos( $field_name_lower, 'first' ) === false &&
                   strpos( $field_name_lower, 'company' ) === false &&
                   strpos( $field_name_lower, 'business' ) === false ) ) {
                $name = $field_value;
            }

            // First/Last name detection
            if ( strpos( $field_name_lower, 'first' ) !== false && strpos( $field_name_lower, 'name' ) !== false ) {
                $first_name = $field_value;
            }
            if ( ( strpos( $field_name_lower, 'last' ) !== false && strpos( $field_name_lower, 'name' ) !== false ) ||
                 $field_name_lower === 'surname' ) {
                $last_name = $field_value;
            }

            // Company detection
            if ( strpos( $field_name_lower, 'company' ) !== false ||
                 strpos( $field_name_lower, 'business' ) !== false ||
                 strpos( $field_name_lower, 'organization' ) !== false ) {
                $company = $field_value;
            }

            // Phone detection
            if ( $field_type === 'phone' ||
                 strpos( $field_name_lower, 'phone' ) !== false ||
                 strpos( $field_name_lower, 'mobile' ) !== false ) {
                $phone = $field_value;
            }
        }

        // Combine first + last name if full name wasn't found directly
        if ( empty( $name ) && ( ! empty( $first_name ) || ! empty( $last_name ) ) ) {
            $name = trim( ( $first_name ?? '' ) . ' ' . ( $last_name ?? '' ) );
        }

        return [
            'fields'  => $lead_data,
            'email'   => $email,
            'name'    => $name,
            'company' => $company,
            'phone'   => $phone,
        ];
    }

    /**
     * Get list of available Formidable Forms
     */
    public static function get_available_forms(): array {
        if ( ! class_exists( 'FrmForm' ) ) {
            return [];
        }

        global $wpdb;
        $table = $wpdb->prefix . 'frm_forms';

        $forms = $wpdb->get_results(
            "SELECT id, name, status FROM {$table} WHERE status = 'published' ORDER BY name ASC",
            ARRAY_A
        );

        if ( empty( $forms ) ) {
            return [];
        }

        $form_list = [];

        foreach ( $forms as $form ) {
            $form_list[ $form['id'] ] = [
                'id'     => $form['id'],
                'title'  => $form['name'],
                'status' => $form['status'],
            ];
        }

        return $form_list;
    }

    /**
     * Check if Formidable Forms is active
     */
    public static function is_active(): bool {
        return class_exists( 'FrmForm' );
    }
}
