<?php
/**
 * Fluent Forms Integration
 *
 * @package FormRankLS\Integrations
 */

namespace FormRankLS\Integrations;

if (!defined('ABSPATH')) {
    exit;
}

class Fluent_Forms extends Integration_Base {

    protected $plugin_name = 'fluent';

    /**
     * Capture Fluent Forms submission
     *
     * Hook: fluentform/submission_inserted
     *
     * @param int   $submission_id The submission ID
     * @param array $form_data     The submitted form data
     * @param array $form          The form configuration
     */
    public function capture_submission( $submission_id, $form_data, $form ) {
        // Check if this form should be scored
        if ( ! $this->should_score_form( $form->id ) ) {
            return;
        }

        // Extract lead data from submission
        $lead_data = $this->extract_lead_data( $form_data, $form );

        // Get submission meta
        global $wpdb;
        $table = $wpdb->prefix . 'fluentform_submissions';
        $submission = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $submission_id
        ), ARRAY_A );

        // Save and score the lead
        $this->process_lead( $lead_data, [
            'form_plugin' => 'fluent',
            'form_id'     => $form->id,
            'entry_id'    => $submission_id,
            'source_url'  => $submission['source_url'] ?? wp_get_referer() ?: home_url(),
        ] );
    }

    /**
     * Extract structured data from Fluent Forms submission
     */
    private function extract_lead_data( array $form_data, $form ): array {
        $lead_data = [];
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        // Build field map from form configuration (key => label/element type)
        $field_map = $this->build_field_map( $form );

        foreach ( $form_data as $field_key => $field_value ) {
            // Skip internal fields
            if ( strpos( $field_key, '__' ) === 0 ) {
                continue;
            }

            $clean_key = sanitize_text_field( $field_key );

            // Handle arrays (checkboxes, multi-select)
            if ( is_array( $field_value ) ) {
                $field_value = implode( ', ', array_filter( $field_value ) );
            }

            if ( empty( $field_value ) ) {
                continue;
            }

            $lead_data[ $clean_key ] = sanitize_textarea_field( $field_value );

            $field_key_lower = strtolower( $field_key );
            $field_info = $field_map[ $field_key ] ?? [];
            $field_label = $field_info['label'] ?? '';
            $field_element = $field_info['element'] ?? '';

            // Email detection
            if ( $field_element === 'input_email' ||
                 strpos( $field_key_lower, 'email' ) !== false ||
                 filter_var( $field_value, FILTER_VALIDATE_EMAIL ) ) {
                $email = $field_value;
            }

            // Name detection - Fluent Forms composite name field (names array)
            if ( $field_key === 'names' && is_array( $form_data['names'] ) ) {
                $name_parts = $form_data['names'];
                $name = trim(
                    ( $name_parts['first_name'] ?? '' ) . ' ' .
                    ( $name_parts['middle_name'] ?? '' ) . ' ' .
                    ( $name_parts['last_name'] ?? '' )
                );
                // Collapse multiple spaces from empty middle name
                $name = preg_replace( '/\s+/', ' ', $name );
            } elseif ( $field_element === 'input_name' && is_array( $form_data[ $field_key ] ?? null ) ) {
                // Composite name field with custom element name
                $name_parts = $form_data[ $field_key ];
                $name = trim(
                    ( $name_parts['first_name'] ?? '' ) . ' ' .
                    ( $name_parts['middle_name'] ?? '' ) . ' ' .
                    ( $name_parts['last_name'] ?? '' )
                );
                $name = preg_replace( '/\s+/', ' ', $name );
            } elseif ( in_array( $field_key_lower, [ 'name', 'full_name', 'your_name', 'fullname' ], true ) ) {
                $name = $field_value;
            } elseif ( empty( $name ) && ! empty( $field_label ) &&
                       strpos( $field_label, 'name' ) !== false &&
                       strpos( $field_label, 'last' ) === false &&
                       strpos( $field_label, 'first' ) === false &&
                       strpos( $field_label, 'company' ) === false &&
                       strpos( $field_label, 'business' ) === false ) {
                // Fallback: check field label for "name"
                $name = $field_value;
            }

            // First/Last name detection - by key or label
            if ( $field_key_lower === 'first_name' ||
                 ( ! empty( $field_label ) && strpos( $field_label, 'first' ) !== false && strpos( $field_label, 'name' ) !== false ) ) {
                $first_name = $field_value;
            }
            if ( $field_key_lower === 'last_name' ||
                 ( ! empty( $field_label ) && ( ( strpos( $field_label, 'last' ) !== false && strpos( $field_label, 'name' ) !== false ) || $field_label === 'surname' ) ) ) {
                $last_name = $field_value;
            }

            // Company detection - by key or label
            if ( strpos( $field_key_lower, 'company' ) !== false ||
                 strpos( $field_key_lower, 'business' ) !== false ||
                 strpos( $field_key_lower, 'organization' ) !== false ||
                 strpos( $field_label, 'company' ) !== false ||
                 strpos( $field_label, 'business' ) !== false ||
                 strpos( $field_label, 'organization' ) !== false ) {
                $company = $field_value;
            }

            // Phone detection - by key or label
            if ( strpos( $field_key_lower, 'phone' ) !== false ||
                 strpos( $field_key_lower, 'mobile' ) !== false ||
                 strpos( $field_key_lower, 'tel' ) !== false ||
                 strpos( $field_label, 'phone' ) !== false ||
                 strpos( $field_label, 'mobile' ) !== false ) {
                $phone = $field_value;
            }
        }

        // Combine first + last name if full name wasn't found directly
        if ( empty( $name ) && ( ! empty( $first_name ) || ! empty( $last_name ) ) ) {
            $name = trim( ( $first_name ?? '' ) . ' ' . ( $last_name ?? '' ) );
        }

        return [
            'fields'  => $lead_data,
            'email'   => $email,
            'name'    => $name,
            'company' => $company,
            'phone'   => $phone,
        ];
    }

    /**
     * Build a map of field keys to their labels and element types from form config
     */
    private function build_field_map( $form ): array {
        $field_map = [];

        $form_fields = json_decode( $form->form_fields ?? '', true );
        if ( empty( $form_fields['fields'] ) ) {
            return $field_map;
        }

        foreach ( $form_fields['fields'] as $field ) {
            $name = $field['attributes']['name'] ?? '';
            $label = strtolower( trim( $field['settings']['label'] ?? '' ) );
            $element = $field['element'] ?? '';

            if ( ! empty( $name ) ) {
                $field_map[ $name ] = [
                    'label'   => $label,
                    'element' => $element,
                ];
            }
        }

        return $field_map;
    }

    /**
     * Get list of available Fluent Forms
     */
    public static function get_available_forms(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'fluentform_forms';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
            return [];
        }

        $forms = $wpdb->get_results(
            "SELECT id, title, status FROM {$table} WHERE status = 'published' ORDER BY title ASC",
            ARRAY_A
        );

        if ( empty( $forms ) ) {
            return [];
        }

        $form_list = [];

        foreach ( $forms as $form ) {
            $form_list[ $form['id'] ] = [
                'id'     => $form['id'],
                'title'  => $form['title'],
                'status' => $form['status'],
            ];
        }

        return $form_list;
    }

    /**
     * Check if Fluent Forms is active
     */
    public static function is_active(): bool {
        return defined( 'FLUENTFORM' ) || function_exists( 'fluentform' );
    }
}
