<?php
/**
 * Contact Form 7 Integration
 *
 * @package FormRankLS\Integrations
 */

namespace FormRankLS\Integrations;

if (!defined('ABSPATH')) {
    exit;
}

class CF7 extends Integration_Base {

    protected $plugin_name = 'cf7';

    /**
     * Capture Contact Form 7 submission
     *
     * Hook: wpcf7_before_send_mail
     *
     * @param \WPCF7_ContactForm $contact_form The contact form instance
     */
    public function capture_submission($contact_form) {
        $form_id = $contact_form->id();

        // Check if this form should be scored
        if (!$this->should_score_form($form_id)) {
            return;
        }

        // Get submission data
        $submission = \WPCF7_Submission::get_instance();

        if (!$submission) {
            return;
        }

        // Extract lead data from submission
        $lead_data = $this->extract_lead_data($submission, $contact_form);

        // Save and score the lead
        $this->process_lead($lead_data, [
            'form_plugin' => 'cf7',
            'form_id' => $form_id,
            'entry_id' => null, // CF7 doesn't have entry IDs by default
            'source_url' => $submission->get_meta('url') ?: wp_get_referer()
        ]);
    }

    /**
     * Extract structured data from CF7 submission
     */
    private function extract_lead_data($submission, $contact_form): array {
        $posted_data = $submission->get_posted_data();
        $lead_data = [];
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        // Get form tags to understand field types
        $tags = $contact_form->scan_form_tags();
        $tag_types = [];

        foreach ($tags as $tag) {
            $tag_types[$tag->name] = $tag->basetype;
        }

        // Extract field labels from form template
        $tag_labels = $this->extract_cf7_labels($contact_form);

        foreach ($posted_data as $field_name => $value) {
            // Skip special CF7 fields
            if (strpos($field_name, '_wpcf7') === 0) {
                continue;
            }

            // Handle array values
            if (is_array($value)) {
                $value = implode(', ', array_filter($value));
            }

            $value = sanitize_textarea_field($value);

            if (empty($value)) {
                continue;
            }

            $clean_name = ucwords(str_replace(['_', '-'], ' ', $field_name));
            $lead_data[$clean_name] = $value;

            $field_name_lower = strtolower($field_name);
            $field_type = $tag_types[$field_name] ?? '';
            $field_label = $tag_labels[$field_name] ?? '';

            // Email detection
            if ($field_type === 'email' || strpos($field_name_lower, 'email') !== false) {
                $email = $value;
            }

            // Name detection - check tag name
            if (in_array($field_name_lower, ['your-name', 'name', 'full-name', 'fullname', 'your_name'])) {
                $name = $value;
            }

            // Name detection - check tag label as fallback
            if (empty($name) && !empty($field_label) &&
                strpos($field_label, 'name') !== false &&
                strpos($field_label, 'last') === false &&
                strpos($field_label, 'first') === false &&
                strpos($field_label, 'company') === false &&
                strpos($field_label, 'business') === false) {
                $name = $value;
            }

            // First/Last name detection - by tag name or label
            if (in_array($field_name_lower, ['first-name', 'firstname', 'first_name']) ||
                (!empty($field_label) && strpos($field_label, 'first') !== false && strpos($field_label, 'name') !== false)) {
                $first_name = $value;
            }
            if (in_array($field_name_lower, ['last-name', 'lastname', 'last_name']) ||
                (!empty($field_label) && ((strpos($field_label, 'last') !== false && strpos($field_label, 'name') !== false) || $field_label === 'surname'))) {
                $last_name = $value;
            }

            // Company detection - by tag name or label
            if (strpos($field_name_lower, 'company') !== false ||
                strpos($field_name_lower, 'business') !== false ||
                strpos($field_name_lower, 'organization') !== false ||
                strpos($field_label, 'company') !== false ||
                strpos($field_label, 'business') !== false ||
                strpos($field_label, 'organization') !== false) {
                $company = $value;
            }

            // Phone detection - by type, tag name, or label
            if ($field_type === 'tel' || strpos($field_name_lower, 'phone') !== false ||
                strpos($field_label, 'phone') !== false || strpos($field_label, 'mobile') !== false) {
                $phone = $value;
            }
        }

        // Combine first + last name if full name wasn't found directly
        if (empty($name) && (!empty($first_name) || !empty($last_name))) {
            $name = trim(($first_name ?? '') . ' ' . ($last_name ?? ''));
        }

        // Smart fallback: if name still empty, find the first text field
        // that isn't identified as email, phone, company, subject, or message
        if (empty($name)) {
            foreach ($tags as $tag) {
                if (empty($tag->name) || $tag->basetype !== 'text') {
                    continue;
                }

                $tag_lower = strtolower($tag->name);

                // Skip fields already identified as other types
                if (strpos($tag_lower, 'email') !== false ||
                    strpos($tag_lower, 'phone') !== false ||
                    strpos($tag_lower, 'tel') !== false ||
                    strpos($tag_lower, 'company') !== false ||
                    strpos($tag_lower, 'business') !== false ||
                    strpos($tag_lower, 'organization') !== false ||
                    strpos($tag_lower, 'subject') !== false ||
                    strpos($tag_lower, 'message') !== false ||
                    strpos($tag_lower, 'url') !== false ||
                    strpos($tag_lower, 'website') !== false) {
                    continue;
                }

                $tag_value = $posted_data[$tag->name] ?? '';
                if (is_array($tag_value)) {
                    $tag_value = implode(' ', $tag_value);
                }
                $tag_value = trim($tag_value);

                if (!empty($tag_value)) {
                    $name = sanitize_textarea_field($tag_value);
                    break;
                }
            }
        }

        return [
            'fields' => $lead_data,
            'email' => $email,
            'name' => $name,
            'company' => $company,
            'phone' => $phone
        ];
    }

    /**
     * Extract field labels from CF7 form template
     */
    private function extract_cf7_labels($contact_form): array {
        $labels = [];
        $form_content = $contact_form->prop('form');

        if (empty($form_content)) {
            return $labels;
        }

        // Pattern 1: <label> wrapping a form tag - <label>Label Text [type* tag-name]</label>
        if (preg_match_all(
            '/<label[^>]*>\s*([^<\[]+?)\s*\[\s*[a-z][a-z0-9*]*\s+([a-z0-9_-]+)/i',
            $form_content,
            $matches,
            PREG_SET_ORDER
        )) {
            foreach ($matches as $match) {
                $label_text = strtolower(trim($match[1]));
                $tag_name = $match[2];
                if (!empty($label_text) && !empty($tag_name)) {
                    $labels[$tag_name] = $label_text;
                }
            }
        }

        // Pattern 2: <label>Text</label> followed by [type* tag-name] (non-wrapping)
        if (preg_match_all(
            '/<label[^>]*>\s*([^<]+?)\s*<\/label>\s*(?:<br\s*\/?>|[\r\n])?\s*\[\s*[a-z][a-z0-9*]*\s+([a-z0-9_-]+)/i',
            $form_content,
            $matches,
            PREG_SET_ORDER
        )) {
            foreach ($matches as $match) {
                $label_text = strtolower(trim($match[1]));
                $tag_name = $match[2];
                if (!empty($label_text) && !empty($tag_name) && !isset($labels[$tag_name])) {
                    $labels[$tag_name] = $label_text;
                }
            }
        }

        // Pattern 3: Plain text before a form tag (e.g., "Name:\n[text* your-name]")
        if (preg_match_all(
            '/(?:^|>)\s*([A-Za-z][A-Za-z\s]{1,30}?)\s*:?\s*(?:<br\s*\/?>|[\r\n])\s*\[\s*[a-z][a-z0-9*]*\s+([a-z0-9_-]+)/mi',
            $form_content,
            $matches,
            PREG_SET_ORDER
        )) {
            foreach ($matches as $match) {
                $label_text = strtolower(trim($match[1]));
                $tag_name = $match[2];
                if (!empty($label_text) && !empty($tag_name) && !isset($labels[$tag_name])) {
                    $labels[$tag_name] = $label_text;
                }
            }
        }

        return $labels;
    }

    /**
     * Get list of available Contact Form 7 forms
     */
    public static function get_available_forms(): array {
        if (!class_exists('WPCF7_ContactForm')) {
            return [];
        }

        $forms = \WPCF7_ContactForm::find(['posts_per_page' => -1]);

        if (empty($forms)) {
            return [];
        }

        $form_list = [];

        foreach ($forms as $form) {
            $form_list[$form->id()] = [
                'id' => $form->id(),
                'title' => $form->title()
            ];
        }

        return $form_list;
    }
}
