<?php
/**
 * Demo lead templates for the setup wizard
 *
 * Provides realistic sample leads across 6 industries so new users
 * can see how FormRank scoring works before real submissions arrive.
 *
 * @package FormRankLS\Data
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Get the list of available demo industries.
 *
 * @return array Keyed by slug. Each entry has label, desc, and icon (dashicon name without prefix).
 */
function formrank_get_demo_industries(): array {
    return array(
        'services' => array(
            'label' => 'Professional Services',
            'desc'  => 'Consulting, Agency, Marketing',
            'icon'  => 'building',
        ),
        'ecommerce' => array(
            'label' => 'E-Commerce',
            'desc'  => 'Online Store, Products',
            'icon'  => 'cart',
        ),
        'healthcare' => array(
            'label' => 'Healthcare',
            'desc'  => 'Medical, Dental, Wellness',
            'icon'  => 'heart',
        ),
        'realestate' => array(
            'label' => 'Real Estate',
            'desc'  => 'Agent, Property, Rentals',
            'icon'  => 'admin-home',
        ),
        'saas' => array(
            'label' => 'SaaS / Technology',
            'desc'  => 'Software, Apps, IT Services',
            'icon'  => 'laptop',
        ),
        'other' => array(
            'label' => 'General Business',
            'desc'  => 'General Business',
            'icon'  => 'portfolio',
        ),
    );
}

/**
 * Get all demo templates keyed by industry slug.
 *
 * @return array Each key is an industry slug, value is the array of 10 leads.
 */
function formrank_get_all_demo_templates(): array {
    $industries = formrank_get_demo_industries();
    $templates  = array();

    foreach ( array_keys( $industries ) as $slug ) {
        $templates[ $slug ] = formrank_get_demo_leads( $slug );
    }

    return $templates;
}

/**
 * Get demo leads for a specific industry.
 *
 * Each industry returns exactly 10 leads with the following score distribution:
 *   2 hot (80-100), 2 warm (60-79), 2 neutral (40-59), 2 cool (20-39), 2 cold (1-19).
 *
 * @param string $industry Industry slug.
 * @return array List of lead arrays, empty if slug is unrecognised.
 */
function formrank_get_demo_leads( string $industry ): array {
    $leads = array(
        // -----------------------------------------------------------------
        // PROFESSIONAL SERVICES
        // -----------------------------------------------------------------
        'services' => array(
            // --- Hot #1 ---
            array(
                'name'            => 'Rachel Nguyen',
                'email'           => 'r.nguyen.demo@example.com',
                'company'         => 'Apex Brands Inc.',
                'phone'           => '555-8301',
                'form_data'       => array(
                    'Name'      => 'Rachel Nguyen',
                    'Email'     => 'r.nguyen.demo@example.com',
                    'Company'   => 'Apex Brands Inc.',
                    'Phone'     => '555-8301',
                    'Budget'    => '$15,000 - $25,000',
                    'Timeline'  => 'Within 2 weeks',
                    'Service'   => 'Brand Strategy & Digital Marketing',
                    'Message'   => 'We are launching a new product line next month and need a full brand strategy with digital campaigns ready to go. Our board approved a Q1 budget of $20K for this initiative. Can we schedule a call this week?',
                ),
                'score'           => 94,
                'score_label'     => 'hot',
                'score_reasoning' => 'Corporate email from a decision-maker with an approved budget of $20K and an urgent two-week timeline. The detailed message shows high purchase intent and readiness.',
                'score_factors'   => array(
                    'intent'     => 38,
                    'fit'        => 28,
                    'quality'    => 18,
                    'engagement' => 10,
                ),
                'status'          => 'contacted',
            ),
            // --- Hot #2 ---
            array(
                'name'            => 'David Park',
                'email'           => 'd.park.demo@example.com',
                'company'         => 'Meridian Consulting Group',
                'phone'           => '555-8302',
                'form_data'       => array(
                    'Name'      => 'David Park',
                    'Email'     => 'd.park.demo@example.com',
                    'Company'   => 'Meridian Consulting Group',
                    'Phone'     => '555-8302',
                    'Budget'    => '$10,000 - $15,000',
                    'Timeline'  => 'Immediately',
                    'Service'   => 'SEO & Content Marketing',
                    'Message'   => 'Our organic traffic dropped 40% after a Google algorithm update. We need an SEO audit and recovery plan ASAP. We have a $12K monthly retainer ready. Please send your proposal by end of day Friday.',
                ),
                'score'           => 88,
                'score_label'     => 'hot',
                'score_reasoning' => 'Urgent SEO recovery need with a confirmed monthly retainer budget. The specific problem description and tight deadline indicate immediate buying intent.',
                'score_factors'   => array(
                    'intent'     => 36,
                    'fit'        => 26,
                    'quality'    => 17,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Warm #1 ---
            array(
                'name'            => 'Maria Santos',
                'email'           => 'm.santos.demo@example.com',
                'company'         => 'GreenLeaf Solutions',
                'phone'           => '555-8303',
                'form_data'       => array(
                    'Name'      => 'Maria Santos',
                    'Email'     => 'm.santos.demo@example.com',
                    'Company'   => 'GreenLeaf Solutions',
                    'Phone'     => '555-8303',
                    'Budget'    => '$5,000 - $10,000',
                    'Timeline'  => 'Next quarter',
                    'Service'   => 'Website Redesign',
                    'Message'   => 'We have been considering a website refresh for a while. Our current site is about 5 years old and we would like something modern with better lead capture. Looking for proposals from a few agencies.',
                ),
                'score'           => 72,
                'score_label'     => 'warm',
                'score_reasoning' => 'Clear project need with a reasonable budget, but the next-quarter timeline and multi-agency evaluation suggest no immediate urgency.',
                'score_factors'   => array(
                    'intent'     => 28,
                    'fit'        => 22,
                    'quality'    => 14,
                    'engagement' => 8,
                ),
                'status'          => 'new',
            ),
            // --- Warm #2 ---
            array(
                'name'            => 'James Whitfield',
                'email'           => 'j.whitfield.demo@example.com',
                'company'         => 'Whitfield & Associates',
                'phone'           => '555-8304',
                'form_data'       => array(
                    'Name'      => 'James Whitfield',
                    'Email'     => 'j.whitfield.demo@example.com',
                    'Company'   => 'Whitfield & Associates',
                    'Phone'     => '555-8304',
                    'Budget'    => '$3,000 - $5,000',
                    'Timeline'  => '1-2 months',
                    'Service'   => 'Social Media Management',
                    'Message'   => 'Our law firm wants to start building a social media presence. We have no current strategy but understand it is important for client acquisition. What packages do you offer?',
                ),
                'score'           => 65,
                'score_label'     => 'warm',
                'score_reasoning' => 'Professional firm with a defined budget and reasonable timeline. Asking for package options indicates genuine interest but an early-stage evaluation.',
                'score_factors'   => array(
                    'intent'     => 24,
                    'fit'        => 21,
                    'quality'    => 13,
                    'engagement' => 7,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #1 ---
            array(
                'name'            => 'Karen Lee',
                'email'           => 'k.lee.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Karen Lee',
                    'Email'   => 'k.lee.demo@example.com',
                    'Service' => 'General Inquiry',
                    'Message' => 'How much do your marketing services cost? Do you offer any free consultations?',
                ),
                'score'           => 48,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Price-focused inquiry with no company details, budget, or timeline. Free-consultation request suggests early research phase.',
                'score_factors'   => array(
                    'intent'     => 18,
                    'fit'        => 14,
                    'quality'    => 10,
                    'engagement' => 6,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #2 ---
            array(
                'name'            => 'Tom Bradley',
                'email'           => 't.bradley.demo@example.com',
                'company'         => 'Bradley Freelance',
                'phone'           => '555-8306',
                'form_data'       => array(
                    'Name'      => 'Tom Bradley',
                    'Email'     => 't.bradley.demo@example.com',
                    'Company'   => 'Bradley Freelance',
                    'Phone'     => '555-8306',
                    'Budget'    => 'Not sure yet',
                    'Timeline'  => 'No rush',
                    'Service'   => 'Graphic Design',
                    'Message'   => 'I might need a logo and some business cards. Just exploring options right now.',
                ),
                'score'           => 42,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Small freelance project with an undefined budget and no timeline urgency. Exploratory inquiry that may not convert to a paying engagement.',
                'score_factors'   => array(
                    'intent'     => 16,
                    'fit'        => 12,
                    'quality'    => 9,
                    'engagement' => 5,
                ),
                'status'          => 'new',
            ),
            // --- Cool #1 ---
            array(
                'name'            => 'Alex Rivera',
                'email'           => 'a.rivera.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Alex Rivera',
                    'Email'   => 'a.rivera.demo@example.com',
                    'Message' => 'Do you do free work for nonprofits? We have zero budget.',
                ),
                'score'           => 28,
                'score_label'     => 'cool',
                'score_reasoning' => 'Explicitly states zero budget and is seeking pro-bono work. No company details provided, very low commercial potential.',
                'score_factors'   => array(
                    'intent'     => 10,
                    'fit'        => 8,
                    'quality'    => 6,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cool #2 ---
            array(
                'name'            => 'Priya Sharma',
                'email'           => 'p.sharma.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Priya Sharma',
                    'Email'   => 'p.sharma.demo@example.com',
                    'Service' => 'General Inquiry',
                    'Message' => 'Im a student working on a class project. Can you share some tips on marketing strategy?',
                ),
                'score'           => 22,
                'score_label'     => 'cool',
                'score_reasoning' => 'Student requesting free advice for a class project. No commercial intent or purchasing authority whatsoever.',
                'score_factors'   => array(
                    'intent'     => 8,
                    'fit'        => 6,
                    'quality'    => 5,
                    'engagement' => 3,
                ),
                'status'          => 'new',
            ),
            // --- Cold #1 ---
            array(
                'name'            => 'Test User',
                'email'           => 'test.user.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Test User',
                    'Email'   => 'test.user.demo@example.com',
                    'Message' => 'Just testing the form.',
                ),
                'score'           => 8,
                'score_label'     => 'cold',
                'score_reasoning' => 'Obvious test submission with no real inquiry. Name is generic "Test User" and message confirms form testing.',
                'score_factors'   => array(
                    'intent'     => 2,
                    'fit'        => 2,
                    'quality'    => 2,
                    'engagement' => 2,
                ),
                'status'          => 'new',
            ),
            // --- Cold #2 (spam) ---
            array(
                'name'            => 'SEO Guru',
                'email'           => 'seo.guru.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'SEO Guru',
                    'Email'   => 'seo.guru.demo@example.com',
                    'Message' => 'GET RANKED #1 ON GOOGLE GUARANTEED!!! Visit our site for cheap SEO packages $49/month!!! www.spammy-seo-example.com',
                ),
                'score'           => 3,
                'score_label'     => 'cold',
                'score_reasoning' => 'Spam submission promoting third-party SEO services. Contains promotional link and all-caps sales pitch. Zero lead value.',
                'score_factors'   => array(
                    'intent'     => 1,
                    'fit'        => 1,
                    'quality'    => 1,
                    'engagement' => 0,
                ),
                'status'          => 'new',
            ),
        ),

        // -----------------------------------------------------------------
        // E-COMMERCE
        // -----------------------------------------------------------------
        'ecommerce' => array(
            // --- Hot #1 ---
            array(
                'name'            => 'Linda Chen',
                'email'           => 'l.chen.demo@example.com',
                'company'         => 'Bloom & Vine Co.',
                'phone'           => '555-7201',
                'form_data'       => array(
                    'Name'           => 'Linda Chen',
                    'Email'          => 'l.chen.demo@example.com',
                    'Company'        => 'Bloom & Vine Co.',
                    'Phone'          => '555-7201',
                    'Order Size'     => '500+ units',
                    'Product'        => 'Custom gift baskets',
                    'Timeline'       => 'Need delivery by March 15',
                    'Message'        => 'We are placing our annual corporate gift order for 500 custom baskets. Budget is pre-approved at $25,000. Please send wholesale pricing and confirm you can meet our March 15 delivery date.',
                ),
                'score'           => 96,
                'score_label'     => 'hot',
                'score_reasoning' => 'Large wholesale order of 500+ units with a pre-approved $25K budget and a firm delivery deadline. Repeat corporate buyer showing very high purchase intent.',
                'score_factors'   => array(
                    'intent'     => 39,
                    'fit'        => 29,
                    'quality'    => 19,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Hot #2 ---
            array(
                'name'            => 'Marcus Johnson',
                'email'           => 'm.johnson.demo@example.com',
                'company'         => 'Urban Outfitters Local',
                'phone'           => '555-7202',
                'form_data'       => array(
                    'Name'           => 'Marcus Johnson',
                    'Email'          => 'm.johnson.demo@example.com',
                    'Company'        => 'Urban Outfitters Local',
                    'Phone'          => '555-7202',
                    'Order Size'     => '200 units',
                    'Product'        => 'Handmade candles - Signature Collection',
                    'Timeline'       => 'This week',
                    'Message'        => 'We want to stock your Signature Collection in our 3 retail locations. Looking to order 200 units immediately with monthly reorders. Can you handle this volume?',
                ),
                'score'           => 91,
                'score_label'     => 'hot',
                'score_reasoning' => 'Retail buyer requesting immediate bulk order with ongoing monthly reorder commitment. Multi-location distribution indicates strong long-term revenue potential.',
                'score_factors'   => array(
                    'intent'     => 37,
                    'fit'        => 27,
                    'quality'    => 18,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Warm #1 ---
            array(
                'name'            => 'Sophie Taylor',
                'email'           => 's.taylor.demo@example.com',
                'company'         => 'Taylor Event Planning',
                'phone'           => '555-7203',
                'form_data'       => array(
                    'Name'           => 'Sophie Taylor',
                    'Email'          => 's.taylor.demo@example.com',
                    'Company'        => 'Taylor Event Planning',
                    'Phone'          => '555-7203',
                    'Order Size'     => '50 units',
                    'Product'        => 'Party favors assortment',
                    'Timeline'       => '3-4 weeks',
                    'Message'        => 'Planning a corporate event and interested in your party favors as gifts. Budget is around $1,500. Can you send a catalog with pricing?',
                ),
                'score'           => 74,
                'score_label'     => 'warm',
                'score_reasoning' => 'Defined need with a stated budget and reasonable timeline. Mid-size order for a specific event shows genuine purchase interest.',
                'score_factors'   => array(
                    'intent'     => 29,
                    'fit'        => 22,
                    'quality'    => 15,
                    'engagement' => 8,
                ),
                'status'          => 'new',
            ),
            // --- Warm #2 ---
            array(
                'name'            => 'Brian Foster',
                'email'           => 'b.foster.demo@example.com',
                'company'         => '',
                'phone'           => '555-7204',
                'form_data'       => array(
                    'Name'           => 'Brian Foster',
                    'Email'          => 'b.foster.demo@example.com',
                    'Phone'          => '555-7204',
                    'Product'        => 'Kitchen gadgets bundle',
                    'Timeline'       => 'Next month',
                    'Message'        => 'I saw your kitchen gadgets set on Instagram and love it. Want to order one for myself and maybe 5 more as holiday gifts. What is the bulk discount?',
                ),
                'score'           => 67,
                'score_label'     => 'warm',
                'score_reasoning' => 'Consumer with clear purchase intent for multiple units. Referred from social media, asking about bulk pricing, but relatively small order size.',
                'score_factors'   => array(
                    'intent'     => 26,
                    'fit'        => 20,
                    'quality'    => 14,
                    'engagement' => 7,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #1 ---
            array(
                'name'            => 'Jenny Kim',
                'email'           => 'j.kim.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Jenny Kim',
                    'Email'   => 'j.kim.demo@example.com',
                    'Product' => 'Various',
                    'Message' => 'What is your return policy? And do you ship internationally?',
                ),
                'score'           => 52,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Policy inquiry without a specific product or order intent. International shipping question could indicate a future purchase, but no commitment signals.',
                'score_factors'   => array(
                    'intent'     => 20,
                    'fit'        => 15,
                    'quality'    => 11,
                    'engagement' => 6,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #2 ---
            array(
                'name'            => 'Derek Owens',
                'email'           => 'd.owens.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Derek Owens',
                    'Email'   => 'd.owens.demo@example.com',
                    'Product' => 'Wireless earbuds',
                    'Message' => 'How do your earbuds compare to AirPods? I have been comparing prices on a few sites.',
                ),
                'score'           => 44,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Price-comparison shopper comparing across competitors. Shows product interest but low commitment. Likely evaluating multiple options.',
                'score_factors'   => array(
                    'intent'     => 17,
                    'fit'        => 13,
                    'quality'    => 9,
                    'engagement' => 5,
                ),
                'status'          => 'new',
            ),
            // --- Cool #1 ---
            array(
                'name'            => 'Amy Green',
                'email'           => 'a.green.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Amy Green',
                    'Email'   => 'a.green.demo@example.com',
                    'Message' => 'Do you ever have free giveaways or sample products? I cannot afford anything right now but would love to try your stuff.',
                ),
                'score'           => 25,
                'score_label'     => 'cool',
                'score_reasoning' => 'Explicitly states inability to purchase. Looking for free products only. Very low conversion probability.',
                'score_factors'   => array(
                    'intent'     => 8,
                    'fit'        => 7,
                    'quality'    => 6,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cool #2 ---
            array(
                'name'            => 'Oscar Fernandez',
                'email'           => 'o.fernandez.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Oscar Fernandez',
                    'Email'   => 'o.fernandez.demo@example.com',
                    'Message' => 'I want to become an influencer and promote your products for free items. I have 300 followers on TikTok.',
                ),
                'score'           => 21,
                'score_label'     => 'cool',
                'score_reasoning' => 'Aspiring influencer with a negligible audience seeking free products. No purchasing intent and minimal marketing value at 300 followers.',
                'score_factors'   => array(
                    'intent'     => 6,
                    'fit'        => 6,
                    'quality'    => 5,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cold #1 ---
            array(
                'name'            => 'asdf',
                'email'           => 'asdf.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'asdf',
                    'Email'   => 'asdf.demo@example.com',
                    'Message' => 'asdfasdf',
                ),
                'score'           => 5,
                'score_label'     => 'cold',
                'score_reasoning' => 'Gibberish submission with no real content. Keyboard-mash name and message indicate a test or accidental form submission.',
                'score_factors'   => array(
                    'intent'     => 1,
                    'fit'        => 1,
                    'quality'    => 2,
                    'engagement' => 1,
                ),
                'status'          => 'new',
            ),
            // --- Cold #2 (spam) ---
            array(
                'name'            => 'Best Replica Watches',
                'email'           => 'replica.watches.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Best Replica Watches',
                    'Email'   => 'replica.watches.demo@example.com',
                    'Message' => 'BUY CHEAP ROLEX REPLICA WATCHES!! 90% OFF!! FREE SHIPPING!! Visit www.fake-watches-example.com NOW!!!',
                ),
                'score'           => 2,
                'score_label'     => 'cold',
                'score_reasoning' => 'Blatant spam advertising counterfeit goods. All-caps promotional message with external link. No legitimate business value.',
                'score_factors'   => array(
                    'intent'     => 0,
                    'fit'        => 1,
                    'quality'    => 1,
                    'engagement' => 0,
                ),
                'status'          => 'new',
            ),
        ),

        // -----------------------------------------------------------------
        // HEALTHCARE
        // -----------------------------------------------------------------
        'healthcare' => array(
            // --- Hot #1 ---
            array(
                'name'            => 'Patricia Morales',
                'email'           => 'p.morales.demo@example.com',
                'company'         => 'Sunrise Wellness Center',
                'phone'           => '555-6101',
                'form_data'       => array(
                    'Name'          => 'Patricia Morales',
                    'Email'         => 'p.morales.demo@example.com',
                    'Company'       => 'Sunrise Wellness Center',
                    'Phone'         => '555-6101',
                    'Service'       => 'Full dental implant procedure',
                    'Insurance'     => 'Blue Cross PPO',
                    'Preferred Date' => 'Next available',
                    'Message'       => 'I need full dental implants for my upper arch. My dentist referred me to your clinic specifically. I have Blue Cross PPO and am ready to schedule as soon as possible. I have already completed the X-rays.',
                ),
                'score'           => 92,
                'score_label'     => 'hot',
                'score_reasoning' => 'Referred patient ready for a high-value procedure with insurance verified and X-rays completed. Requests earliest available appointment, indicating strong immediate intent.',
                'score_factors'   => array(
                    'intent'     => 37,
                    'fit'        => 28,
                    'quality'    => 18,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Hot #2 ---
            array(
                'name'            => 'Robert Keane',
                'email'           => 'r.keane.demo@example.com',
                'company'         => 'Keane Family Medical',
                'phone'           => '555-6102',
                'form_data'       => array(
                    'Name'          => 'Robert Keane',
                    'Email'         => 'r.keane.demo@example.com',
                    'Company'       => 'Keane Family Medical',
                    'Phone'         => '555-6102',
                    'Service'       => 'Corporate wellness program',
                    'Employees'     => '200+',
                    'Budget'        => '$50,000/year',
                    'Message'       => 'Our company wants to implement a comprehensive wellness program for 200+ employees including annual physicals, mental health support, and fitness assessments. We have a $50K annual budget allocated. Looking to start in Q2.',
                ),
                'score'           => 87,
                'score_label'     => 'hot',
                'score_reasoning' => 'Corporate contract opportunity with a $50K annual budget for 200+ employees. Q2 start date and detailed requirements show serious commitment and planning.',
                'score_factors'   => array(
                    'intent'     => 35,
                    'fit'        => 26,
                    'quality'    => 17,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Warm #1 ---
            array(
                'name'            => 'Diana Ross-Williams',
                'email'           => 'd.rosswilliams.demo@example.com',
                'company'         => '',
                'phone'           => '555-6103',
                'form_data'       => array(
                    'Name'          => 'Diana Ross-Williams',
                    'Email'         => 'd.rosswilliams.demo@example.com',
                    'Phone'         => '555-6103',
                    'Service'       => 'Cosmetic dentistry consultation',
                    'Insurance'     => 'Aetna',
                    'Preferred Date' => 'Sometime next month',
                    'Message'       => 'I have been thinking about veneers for my front teeth. Would like a consultation to discuss options and costs. I am flexible on scheduling.',
                ),
                'score'           => 70,
                'score_label'     => 'warm',
                'score_reasoning' => 'Interested in elective cosmetic procedure with insurance. Flexible timeline suggests genuine interest but no urgency. Consultation stage.',
                'score_factors'   => array(
                    'intent'     => 27,
                    'fit'        => 22,
                    'quality'    => 14,
                    'engagement' => 7,
                ),
                'status'          => 'new',
            ),
            // --- Warm #2 ---
            array(
                'name'            => 'Ahmed Hassan',
                'email'           => 'a.hassan.demo@example.com',
                'company'         => '',
                'phone'           => '555-6104',
                'form_data'       => array(
                    'Name'          => 'Ahmed Hassan',
                    'Email'         => 'a.hassan.demo@example.com',
                    'Phone'         => '555-6104',
                    'Service'       => 'Physical therapy - knee recovery',
                    'Insurance'     => 'United Healthcare',
                    'Preferred Date' => 'Within 2 weeks',
                    'Message'       => 'I had ACL surgery 6 weeks ago and need to start physical therapy. My surgeon recommended 3 sessions per week for 8 weeks. Does your clinic accept United Healthcare?',
                ),
                'score'           => 68,
                'score_label'     => 'warm',
                'score_reasoning' => 'Post-surgical patient requiring ongoing therapy sessions. Medically necessary treatment with insurance, but needs insurance verification before committing.',
                'score_factors'   => array(
                    'intent'     => 26,
                    'fit'        => 21,
                    'quality'    => 14,
                    'engagement' => 7,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #1 ---
            array(
                'name'            => 'Sarah Mitchell',
                'email'           => 's.mitchell.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Sarah Mitchell',
                    'Email'   => 's.mitchell.demo@example.com',
                    'Service' => 'General checkup',
                    'Message' => 'Do you accept walk-ins? What are your hours? I do not have insurance.',
                ),
                'score'           => 46,
                'score_label'     => 'neutral',
                'score_reasoning' => 'General inquiry about availability and hours. No insurance reduces revenue potential. Walk-in question suggests low commitment to scheduling.',
                'score_factors'   => array(
                    'intent'     => 18,
                    'fit'        => 13,
                    'quality'    => 10,
                    'engagement' => 5,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #2 ---
            array(
                'name'            => 'Kevin Brooks',
                'email'           => 'k.brooks.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Kevin Brooks',
                    'Email'   => 'k.brooks.demo@example.com',
                    'Service' => 'Teeth whitening',
                    'Message' => 'How much does teeth whitening cost? I have been looking at a few places. Do you have any promotions running?',
                ),
                'score'           => 41,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Price-shopping for a commodity service across multiple providers. Promotion-seeking behavior indicates price sensitivity over provider loyalty.',
                'score_factors'   => array(
                    'intent'     => 16,
                    'fit'        => 12,
                    'quality'    => 8,
                    'engagement' => 5,
                ),
                'status'          => 'new',
            ),
            // --- Cool #1 ---
            array(
                'name'            => 'Lisa Tran',
                'email'           => 'l.tran.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Lisa Tran',
                    'Email'   => 'l.tran.demo@example.com',
                    'Message' => 'Can you give me medical advice? I have been having headaches for a week. What should I take?',
                ),
                'score'           => 30,
                'score_label'     => 'cool',
                'score_reasoning' => 'Seeking free medical advice via contact form rather than booking an appointment. Unlikely to convert to a paying patient visit.',
                'score_factors'   => array(
                    'intent'     => 10,
                    'fit'        => 9,
                    'quality'    => 7,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cool #2 ---
            array(
                'name'            => 'Carlos Mendez',
                'email'           => 'c.mendez.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Carlos Mendez',
                    'Email'   => 'c.mendez.demo@example.com',
                    'Message' => 'I am a pre-med student. Would it be possible to shadow a doctor at your clinic for a school assignment?',
                ),
                'score'           => 24,
                'score_label'     => 'cool',
                'score_reasoning' => 'Student requesting shadowing opportunity, not a patient inquiry. No revenue potential. Administrative burden only.',
                'score_factors'   => array(
                    'intent'     => 8,
                    'fit'        => 7,
                    'quality'    => 5,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cold #1 ---
            array(
                'name'            => 'Definitely Real Person',
                'email'           => 'def.real.demo@example.com',
                'company'         => '',
                'phone'           => '555-0000',
                'form_data'       => array(
                    'Name'    => 'Definitely Real Person',
                    'Email'   => 'def.real.demo@example.com',
                    'Phone'   => '555-0000',
                    'Message' => 'test test test 123',
                ),
                'score'           => 7,
                'score_label'     => 'cold',
                'score_reasoning' => 'Fake name and test message. Clearly not a real patient inquiry. Likely a bot or someone testing the contact form.',
                'score_factors'   => array(
                    'intent'     => 2,
                    'fit'        => 2,
                    'quality'    => 2,
                    'engagement' => 1,
                ),
                'status'          => 'new',
            ),
            // --- Cold #2 (spam) ---
            array(
                'name'            => 'Dr. Miracle Pills',
                'email'           => 'miracle.pills.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Dr. Miracle Pills',
                    'Email'   => 'miracle.pills.demo@example.com',
                    'Message' => 'LOSE 30 POUNDS IN 30 DAYS!!! Our revolutionary supplement is doctor-approved! Click here: www.fake-pills-example.com SPECIAL OFFER TODAY ONLY!!!',
                ),
                'score'           => 1,
                'score_label'     => 'cold',
                'score_reasoning' => 'Spam advertising fake health supplements. Promotional message with suspicious external link and all-caps urgency tactics. Zero patient value.',
                'score_factors'   => array(
                    'intent'     => 0,
                    'fit'        => 0,
                    'quality'    => 1,
                    'engagement' => 0,
                ),
                'status'          => 'new',
            ),
        ),

        // -----------------------------------------------------------------
        // REAL ESTATE
        // -----------------------------------------------------------------
        'realestate' => array(
            // --- Hot #1 ---
            array(
                'name'            => 'Jonathan Blake',
                'email'           => 'j.blake.demo@example.com',
                'company'         => 'Blake Ventures LLC',
                'phone'           => '555-5501',
                'form_data'       => array(
                    'Name'          => 'Jonathan Blake',
                    'Email'         => 'j.blake.demo@example.com',
                    'Company'       => 'Blake Ventures LLC',
                    'Phone'         => '555-5501',
                    'Property Type' => 'Commercial office space',
                    'Budget'        => '$800,000 - $1,200,000',
                    'Timeline'      => 'Pre-approved, ready to close in 30 days',
                    'Message'       => 'We are relocating our headquarters and need 5,000+ sq ft of Class A office space downtown. Financing is pre-approved through Chase. We want to close within 30 days. Can you show us available listings this week?',
                ),
                'score'           => 95,
                'score_label'     => 'hot',
                'score_reasoning' => 'Pre-approved commercial buyer with a seven-figure budget and 30-day close timeline. Specific space requirements and same-week showing request demonstrate immediate purchase readiness.',
                'score_factors'   => array(
                    'intent'     => 39,
                    'fit'        => 28,
                    'quality'    => 19,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Hot #2 ---
            array(
                'name'            => 'Samantha Reeves',
                'email'           => 's.reeves.demo@example.com',
                'company'         => '',
                'phone'           => '555-5502',
                'form_data'       => array(
                    'Name'          => 'Samantha Reeves',
                    'Email'         => 's.reeves.demo@example.com',
                    'Phone'         => '555-5502',
                    'Property Type' => 'Single-family home',
                    'Budget'        => '$450,000 - $550,000',
                    'Timeline'      => 'Moving by June — need to close in 60 days',
                    'Location'      => 'Westside school district',
                    'Message'       => 'We are relocating for my husband\'s job and need a 3-4 bedroom home in the Westside school district. We are pre-approved for $550K through Wells Fargo. We have already sold our current home. Flexible on showings any day this week.',
                ),
                'score'           => 89,
                'score_label'     => 'hot',
                'score_reasoning' => 'Pre-approved buyer who has already sold their current home, eliminating a major contingency. Specific location and size requirements with a firm relocation deadline.',
                'score_factors'   => array(
                    'intent'     => 36,
                    'fit'        => 27,
                    'quality'    => 17,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Warm #1 ---
            array(
                'name'            => 'Nathan Cruz',
                'email'           => 'n.cruz.demo@example.com',
                'company'         => '',
                'phone'           => '555-5503',
                'form_data'       => array(
                    'Name'          => 'Nathan Cruz',
                    'Email'         => 'n.cruz.demo@example.com',
                    'Phone'         => '555-5503',
                    'Property Type' => 'Investment property — duplex or triplex',
                    'Budget'        => '$300,000 - $400,000',
                    'Timeline'      => '3-6 months',
                    'Message'       => 'I am looking to buy my first rental property and want a duplex or triplex with good cash flow potential. Pre-qualification in progress with my bank. I would like to start touring properties when I get approved.',
                ),
                'score'           => 71,
                'score_label'     => 'warm',
                'score_reasoning' => 'Serious buyer with defined budget and property type, but pre-qualification is still in progress. 3-6 month timeline allows for nurturing.',
                'score_factors'   => array(
                    'intent'     => 28,
                    'fit'        => 22,
                    'quality'    => 14,
                    'engagement' => 7,
                ),
                'status'          => 'new',
            ),
            // --- Warm #2 ---
            array(
                'name'            => 'Emily Watson',
                'email'           => 'e.watson.demo@example.com',
                'company'         => '',
                'phone'           => '555-5504',
                'form_data'       => array(
                    'Name'          => 'Emily Watson',
                    'Email'         => 'e.watson.demo@example.com',
                    'Phone'         => '555-5504',
                    'Property Type' => 'Condo',
                    'Budget'        => '$200,000 - $280,000',
                    'Timeline'      => 'When our lease ends in August',
                    'Message'       => 'My partner and I are tired of renting and want to buy our first condo. We have been saving for a down payment and want to explore what is available in the $200-280K range. We are first-time buyers.',
                ),
                'score'           => 63,
                'score_label'     => 'warm',
                'score_reasoning' => 'First-time buyers with savings and a lease-end deadline providing natural urgency. Budget is defined, but financing details are unclear.',
                'score_factors'   => array(
                    'intent'     => 24,
                    'fit'        => 20,
                    'quality'    => 13,
                    'engagement' => 6,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #1 ---
            array(
                'name'            => 'Frank DeLuca',
                'email'           => 'f.deluca.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Frank DeLuca',
                    'Email'   => 'f.deluca.demo@example.com',
                    'Message' => 'What does it cost to sell a house in this market? We are just starting to think about selling in a year or two.',
                ),
                'score'           => 50,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Potential seller but with a 1-2 year horizon. Very early research stage with no commitment to listing. Worth a nurture drip.',
                'score_factors'   => array(
                    'intent'     => 19,
                    'fit'        => 15,
                    'quality'    => 10,
                    'engagement' => 6,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #2 ---
            array(
                'name'            => 'Megan Patel',
                'email'           => 'm.patel.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'          => 'Megan Patel',
                    'Email'         => 'm.patel.demo@example.com',
                    'Property Type' => 'Any',
                    'Message'       => 'Can you send me a list of all homes under $200K in the area? Just browsing to see what is out there.',
                ),
                'score'           => 43,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Browsing-stage inquiry with a broad, non-specific request. "Just browsing" language signals low urgency and undefined buying intent.',
                'score_factors'   => array(
                    'intent'     => 16,
                    'fit'        => 13,
                    'quality'    => 9,
                    'engagement' => 5,
                ),
                'status'          => 'new',
            ),
            // --- Cool #1 ---
            array(
                'name'            => 'Tyler Jennings',
                'email'           => 't.jennings.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Tyler Jennings',
                    'Email'   => 't.jennings.demo@example.com',
                    'Message' => 'I am writing a school report on real estate trends. Can you answer some interview questions about the housing market?',
                ),
                'score'           => 26,
                'score_label'     => 'cool',
                'score_reasoning' => 'Student seeking interview for a school assignment. No buying or selling intent. Zero transaction potential.',
                'score_factors'   => array(
                    'intent'     => 8,
                    'fit'        => 8,
                    'quality'    => 6,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cool #2 ---
            array(
                'name'            => 'Grace Kim',
                'email'           => 'g.kim.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Grace Kim',
                    'Email'   => 'g.kim.demo@example.com',
                    'Budget'  => 'Under $50,000',
                    'Message' => 'I have $50K saved. Is there anything at all I can buy? Maybe a piece of land or something?',
                ),
                'score'           => 20,
                'score_label'     => 'cool',
                'score_reasoning' => 'Extremely limited budget for the market with vague property goals. Very unlikely to transact on a traditional property at this price point.',
                'score_factors'   => array(
                    'intent'     => 7,
                    'fit'        => 5,
                    'quality'    => 5,
                    'engagement' => 3,
                ),
                'status'          => 'new',
            ),
            // --- Cold #1 ---
            array(
                'name'            => 'qwerty',
                'email'           => 'qwerty.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'qwerty',
                    'Email'   => 'qwerty.demo@example.com',
                    'Message' => 'qwerty qwerty',
                ),
                'score'           => 4,
                'score_label'     => 'cold',
                'score_reasoning' => 'Keyboard-mash submission with no meaningful content. Test or bot submission with zero real estate inquiry.',
                'score_factors'   => array(
                    'intent'     => 1,
                    'fit'        => 1,
                    'quality'    => 1,
                    'engagement' => 1,
                ),
                'status'          => 'new',
            ),
            // --- Cold #2 (spam) ---
            array(
                'name'            => 'Cash Home Buyers',
                'email'           => 'cash.buyers.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Cash Home Buyers',
                    'Email'   => 'cash.buyers.demo@example.com',
                    'Message' => 'WE BUY HOUSES FAST FOR CASH!!! ANY CONDITION!! CALL NOW 1-800-FAKE-NUM!! Visit www.cash-house-scam-example.com for instant offer!!!',
                ),
                'score'           => 2,
                'score_label'     => 'cold',
                'score_reasoning' => 'Spam from a competing house-flipping operation. Promotional message with fake phone number and suspicious link. Not a legitimate buyer or seller.',
                'score_factors'   => array(
                    'intent'     => 0,
                    'fit'        => 1,
                    'quality'    => 1,
                    'engagement' => 0,
                ),
                'status'          => 'new',
            ),
        ),

        // -----------------------------------------------------------------
        // SAAS / TECHNOLOGY
        // -----------------------------------------------------------------
        'saas' => array(
            // --- Hot #1 ---
            array(
                'name'            => 'Christine Albright',
                'email'           => 'c.albright.demo@example.com',
                'company'         => 'DataStream Analytics',
                'phone'           => '555-4401',
                'form_data'       => array(
                    'Name'        => 'Christine Albright',
                    'Email'       => 'c.albright.demo@example.com',
                    'Company'     => 'DataStream Analytics',
                    'Phone'       => '555-4401',
                    'Job Title'   => 'VP of Engineering',
                    'Team Size'   => '50-100 users',
                    'Budget'      => '$2,000 - $5,000/month',
                    'Timeline'    => 'Evaluating for Q2 rollout',
                    'Message'     => 'We are migrating off a legacy platform and need an enterprise solution for 75 engineers. Already evaluated Competitor A and Competitor B. Your product fills the gaps we identified. Can we get a demo this week and discuss volume pricing?',
                ),
                'score'           => 93,
                'score_label'     => 'hot',
                'score_reasoning' => 'VP-level decision-maker from a mid-size tech company with confirmed budget and defined team size. Already evaluated competitors, indicating late-stage buying process. Demo request signals imminent purchase decision.',
                'score_factors'   => array(
                    'intent'     => 38,
                    'fit'        => 28,
                    'quality'    => 18,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Hot #2 ---
            array(
                'name'            => 'Michael Torres',
                'email'           => 'm.torres.demo@example.com',
                'company'         => 'Pinnacle Financial Tech',
                'phone'           => '555-4402',
                'form_data'       => array(
                    'Name'        => 'Michael Torres',
                    'Email'       => 'm.torres.demo@example.com',
                    'Company'     => 'Pinnacle Financial Tech',
                    'Phone'       => '555-4402',
                    'Job Title'   => 'CTO',
                    'Team Size'   => '25-50 users',
                    'Budget'      => '$1,500/month',
                    'Timeline'    => 'This month',
                    'Message'     => 'Our current tool is sunsetting and we need to migrate 40 users by end of month. We need SSO, SOC 2 compliance, and API access. Pricing page shows your enterprise tier covers this. Can we get on a call today or tomorrow to finalize?',
                ),
                'score'           => 86,
                'score_label'     => 'hot',
                'score_reasoning' => 'CTO with an end-of-month forced migration deadline. Already reviewed pricing and identified specific enterprise requirements. Urgent call request shows deal-closing intent.',
                'score_factors'   => array(
                    'intent'     => 36,
                    'fit'        => 26,
                    'quality'    => 16,
                    'engagement' => 8,
                ),
                'status'          => 'contacted',
            ),
            // --- Warm #1 ---
            array(
                'name'            => 'Julia Andersen',
                'email'           => 'j.andersen.demo@example.com',
                'company'         => 'BrightPath HR',
                'phone'           => '555-4403',
                'form_data'       => array(
                    'Name'        => 'Julia Andersen',
                    'Email'       => 'j.andersen.demo@example.com',
                    'Company'     => 'BrightPath HR',
                    'Phone'       => '555-4403',
                    'Job Title'   => 'Head of Operations',
                    'Team Size'   => '10-25 users',
                    'Budget'      => 'Undecided',
                    'Timeline'    => 'Next quarter',
                    'Message'     => 'We are growing and our spreadsheet-based workflow is breaking down. Looking for a project management tool that integrates with Slack and Google Workspace. Can you send us a comparison sheet with your plans?',
                ),
                'score'           => 73,
                'score_label'     => 'warm',
                'score_reasoning' => 'Operations leader with a real pain point and integration requirements. Budget undecided and next-quarter timeline suggest active evaluation but no urgency.',
                'score_factors'   => array(
                    'intent'     => 28,
                    'fit'        => 23,
                    'quality'    => 14,
                    'engagement' => 8,
                ),
                'status'          => 'new',
            ),
            // --- Warm #2 ---
            array(
                'name'            => 'Ryan Choi',
                'email'           => 'r.choi.demo@example.com',
                'company'         => 'Spark Digital Agency',
                'phone'           => '',
                'form_data'       => array(
                    'Name'        => 'Ryan Choi',
                    'Email'       => 'r.choi.demo@example.com',
                    'Company'     => 'Spark Digital Agency',
                    'Job Title'   => 'Product Manager',
                    'Team Size'   => '5-10 users',
                    'Message'     => 'We tried your free tier last month and liked it. Considering upgrading to a paid plan for the team. Does the Pro plan include priority support and custom integrations?',
                ),
                'score'           => 66,
                'score_label'     => 'warm',
                'score_reasoning' => 'Existing free-tier user evaluating paid upgrade with specific feature questions. Already familiar with the product, reducing sales friction.',
                'score_factors'   => array(
                    'intent'     => 25,
                    'fit'        => 20,
                    'quality'    => 14,
                    'engagement' => 7,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #1 ---
            array(
                'name'            => 'Laura Bennett',
                'email'           => 'l.bennett.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Laura Bennett',
                    'Email'   => 'l.bennett.demo@example.com',
                    'Message' => 'Do you have a free plan? I am a freelancer and need something basic for tracking projects.',
                ),
                'score'           => 49,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Solo freelancer seeking a free plan. Low revenue potential unless they upgrade. Basic needs may fit free tier indefinitely.',
                'score_factors'   => array(
                    'intent'     => 19,
                    'fit'        => 14,
                    'quality'    => 10,
                    'engagement' => 6,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #2 ---
            array(
                'name'            => 'Daniel Okafor',
                'email'           => 'd.okafor.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Daniel Okafor',
                    'Email'   => 'd.okafor.demo@example.com',
                    'Message' => 'Is your API publicly documented? I want to see if I can build an integration before signing up for anything.',
                ),
                'score'           => 45,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Developer evaluating API capabilities before any commitment. Technical interest but no timeline, budget, or team context provided.',
                'score_factors'   => array(
                    'intent'     => 17,
                    'fit'        => 13,
                    'quality'    => 10,
                    'engagement' => 5,
                ),
                'status'          => 'new',
            ),
            // --- Cool #1 ---
            array(
                'name'            => 'Zach Murray',
                'email'           => 'z.murray.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Zach Murray',
                    'Email'   => 'z.murray.demo@example.com',
                    'Message' => 'I am a computer science student building a SaaS clone for my capstone project. Can I get a free developer account to study how your app works?',
                ),
                'score'           => 27,
                'score_label'     => 'cool',
                'score_reasoning' => 'Student requesting free access for academic purposes. No commercial intent and no path to becoming a paying customer in the near term.',
                'score_factors'   => array(
                    'intent'     => 8,
                    'fit'        => 8,
                    'quality'    => 7,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cool #2 ---
            array(
                'name'            => 'Hannah Yee',
                'email'           => 'h.yee.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Hannah Yee',
                    'Email'   => 'h.yee.demo@example.com',
                    'Message' => 'Your product looks cool but way too expensive. I will just use the free version of Notion instead. Just wanted to let you know your pricing is unreasonable.',
                ),
                'score'           => 23,
                'score_label'     => 'cool',
                'score_reasoning' => 'Negative feedback about pricing with an explicit statement they will use a competitor. Zero purchase intent and confrontational tone.',
                'score_factors'   => array(
                    'intent'     => 7,
                    'fit'        => 7,
                    'quality'    => 5,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cold #1 ---
            array(
                'name'            => 'aaa bbb',
                'email'           => 'aaa.bbb.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'aaa bbb',
                    'Email'   => 'aaa.bbb.demo@example.com',
                    'Message' => 'testing 1 2 3',
                ),
                'score'           => 6,
                'score_label'     => 'cold',
                'score_reasoning' => 'Obvious form test with a placeholder name and test message. No real product interest or business inquiry.',
                'score_factors'   => array(
                    'intent'     => 2,
                    'fit'        => 1,
                    'quality'    => 2,
                    'engagement' => 1,
                ),
                'status'          => 'new',
            ),
            // --- Cold #2 (spam) ---
            array(
                'name'            => 'Cheap Software Keys',
                'email'           => 'cheap.keys.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Cheap Software Keys',
                    'Email'   => 'cheap.keys.demo@example.com',
                    'Message' => 'GET WINDOWS 11 PRO FOR $12!!! OFFICE 365 LIFETIME $29!! VISIT www.pirated-keys-example.com FOR HUGE DISCOUNTS!!! 100% GENUINE GUARANTEED!',
                ),
                'score'           => 2,
                'score_label'     => 'cold',
                'score_reasoning' => 'Spam promoting pirated software licenses. All-caps promotional message with external scam link. Zero legitimate business interest.',
                'score_factors'   => array(
                    'intent'     => 0,
                    'fit'        => 1,
                    'quality'    => 1,
                    'engagement' => 0,
                ),
                'status'          => 'new',
            ),
        ),

        // -----------------------------------------------------------------
        // GENERAL BUSINESS (OTHER)
        // -----------------------------------------------------------------
        'other' => array(
            // --- Hot #1 ---
            array(
                'name'            => 'Victoria Sato',
                'email'           => 'v.sato.demo@example.com',
                'company'         => 'Sato & Partners Accounting',
                'phone'           => '555-3301',
                'form_data'       => array(
                    'Name'      => 'Victoria Sato',
                    'Email'     => 'v.sato.demo@example.com',
                    'Company'   => 'Sato & Partners Accounting',
                    'Phone'     => '555-3301',
                    'Budget'    => '$5,000 - $10,000',
                    'Timeline'  => 'Before tax season (2 weeks)',
                    'Service'   => 'Business process automation',
                    'Message'   => 'We need to automate our client onboarding and invoicing before tax season hits. Currently everything is manual and it is costing us 20+ hours a week. Budget is approved for up to $10K. When can we start?',
                ),
                'score'           => 90,
                'score_label'     => 'hot',
                'score_reasoning' => 'Approved budget with an urgent seasonal deadline. Clearly quantified pain point (20+ hours/week manual work) and a direct "when can we start" close signal.',
                'score_factors'   => array(
                    'intent'     => 37,
                    'fit'        => 27,
                    'quality'    => 17,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Hot #2 ---
            array(
                'name'            => 'William Grant',
                'email'           => 'w.grant.demo@example.com',
                'company'         => 'Grant Manufacturing Co.',
                'phone'           => '555-3302',
                'form_data'       => array(
                    'Name'      => 'William Grant',
                    'Email'     => 'w.grant.demo@example.com',
                    'Company'   => 'Grant Manufacturing Co.',
                    'Phone'     => '555-3302',
                    'Budget'    => '$20,000+',
                    'Timeline'  => 'Immediately',
                    'Service'   => 'Equipment maintenance contract',
                    'Message'   => 'Our current maintenance vendor just went out of business and we need an emergency replacement for our factory floor equipment. We have 50+ machines that need quarterly servicing. Previous contract was $24K/year. Need someone onsite by next Monday.',
                ),
                'score'           => 85,
                'score_label'     => 'hot',
                'score_reasoning' => 'Emergency vendor replacement with a $24K annual contract value. Monday deadline and existing budget benchmark make this an immediate high-value opportunity.',
                'score_factors'   => array(
                    'intent'     => 35,
                    'fit'        => 25,
                    'quality'    => 16,
                    'engagement' => 9,
                ),
                'status'          => 'contacted',
            ),
            // --- Warm #1 ---
            array(
                'name'            => 'Angela Ruiz',
                'email'           => 'a.ruiz.demo@example.com',
                'company'         => 'Ruiz Bakery',
                'phone'           => '555-3303',
                'form_data'       => array(
                    'Name'      => 'Angela Ruiz',
                    'Email'     => 'a.ruiz.demo@example.com',
                    'Company'   => 'Ruiz Bakery',
                    'Phone'     => '555-3303',
                    'Budget'    => '$2,000 - $3,000',
                    'Timeline'  => 'Next month',
                    'Service'   => 'Point-of-sale system setup',
                    'Message'   => 'We are opening a second bakery location and need a POS system installed. Looking for something with inventory tracking and employee scheduling. Comparing a few options right now.',
                ),
                'score'           => 75,
                'score_label'     => 'warm',
                'score_reasoning' => 'Expanding small business with a defined need, budget, and timeline. Comparing vendors signals active buying cycle but no urgency to close immediately.',
                'score_factors'   => array(
                    'intent'     => 29,
                    'fit'        => 23,
                    'quality'    => 15,
                    'engagement' => 8,
                ),
                'status'          => 'new',
            ),
            // --- Warm #2 ---
            array(
                'name'            => 'Derek Huang',
                'email'           => 'd.huang.demo@example.com',
                'company'         => 'Huang Auto Repair',
                'phone'           => '555-3304',
                'form_data'       => array(
                    'Name'      => 'Derek Huang',
                    'Email'     => 'd.huang.demo@example.com',
                    'Company'   => 'Huang Auto Repair',
                    'Phone'     => '555-3304',
                    'Budget'    => '$1,000 - $2,000',
                    'Timeline'  => '1-2 months',
                    'Service'   => 'Customer appointment booking system',
                    'Message'   => 'We are still using a paper appointment book and it is causing double bookings. Need an online scheduling system that sends reminders to customers. What do you recommend for a small auto shop?',
                ),
                'score'           => 62,
                'score_label'     => 'warm',
                'score_reasoning' => 'Real operational pain point with stated budget and reasonable timeline. Small business size limits deal value, but the problem is clearly defined.',
                'score_factors'   => array(
                    'intent'     => 24,
                    'fit'        => 19,
                    'quality'    => 12,
                    'engagement' => 7,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #1 ---
            array(
                'name'            => 'Natalie Wong',
                'email'           => 'n.wong.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Natalie Wong',
                    'Email'   => 'n.wong.demo@example.com',
                    'Service' => 'General inquiry',
                    'Message' => 'I have a small Etsy shop and I am wondering if your services would be useful for someone at my scale. What is your cheapest option?',
                ),
                'score'           => 47,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Micro-business owner asking about the cheapest option. Scale may be too small for paid services, but shows some interest in growing.',
                'score_factors'   => array(
                    'intent'     => 18,
                    'fit'        => 14,
                    'quality'    => 10,
                    'engagement' => 5,
                ),
                'status'          => 'new',
            ),
            // --- Neutral #2 ---
            array(
                'name'            => 'Steve Kowalski',
                'email'           => 's.kowalski.demo@example.com',
                'company'         => 'Kowalski Lawn Care',
                'phone'           => '555-3306',
                'form_data'       => array(
                    'Name'      => 'Steve Kowalski',
                    'Email'     => 's.kowalski.demo@example.com',
                    'Company'   => 'Kowalski Lawn Care',
                    'Phone'     => '555-3306',
                    'Message'   => 'Do you work with seasonal businesses? We are only busy April through October and I am not sure if it makes sense to pay for something year-round.',
                ),
                'score'           => 40,
                'score_label'     => 'neutral',
                'score_reasoning' => 'Seasonal business with concerns about year-round costs. Shows some interest but the cost-benefit doubt signals potential deal resistance.',
                'score_factors'   => array(
                    'intent'     => 15,
                    'fit'        => 12,
                    'quality'    => 8,
                    'engagement' => 5,
                ),
                'status'          => 'new',
            ),
            // --- Cool #1 ---
            array(
                'name'            => 'Brittany Flores',
                'email'           => 'b.flores.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Brittany Flores',
                    'Email'   => 'b.flores.demo@example.com',
                    'Message' => 'Are you hiring? I am looking for a remote customer service job. Please let me know about any openings.',
                ),
                'score'           => 29,
                'score_label'     => 'cool',
                'score_reasoning' => 'Job inquiry submitted through the sales contact form. Not a prospective customer. Should be redirected to careers page.',
                'score_factors'   => array(
                    'intent'     => 10,
                    'fit'        => 8,
                    'quality'    => 7,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cool #2 ---
            array(
                'name'            => 'Ethan Marshall',
                'email'           => 'e.marshall.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Ethan Marshall',
                    'Email'   => 'e.marshall.demo@example.com',
                    'Message' => 'I want to start a business but I have no money yet. Can you give me some free advice on how to get started?',
                ),
                'score'           => 22,
                'score_label'     => 'cool',
                'score_reasoning' => 'Pre-revenue aspiring entrepreneur seeking free advice. No budget, no existing business, and no timeline to become a paying customer.',
                'score_factors'   => array(
                    'intent'     => 7,
                    'fit'        => 6,
                    'quality'    => 5,
                    'engagement' => 4,
                ),
                'status'          => 'new',
            ),
            // --- Cold #1 ---
            array(
                'name'            => 'John Smith',
                'email'           => 'john.smith.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'John Smith',
                    'Email'   => 'john.smith.demo@example.com',
                    'Message' => 'test',
                ),
                'score'           => 9,
                'score_label'     => 'cold',
                'score_reasoning' => 'Generic placeholder name with a single-word test message. Classic form-testing submission with no business intent.',
                'score_factors'   => array(
                    'intent'     => 3,
                    'fit'        => 2,
                    'quality'    => 2,
                    'engagement' => 2,
                ),
                'status'          => 'new',
            ),
            // --- Cold #2 (spam) ---
            array(
                'name'            => 'Make Money Online',
                'email'           => 'make.money.demo@example.com',
                'company'         => '',
                'phone'           => '',
                'form_data'       => array(
                    'Name'    => 'Make Money Online',
                    'Email'   => 'make.money.demo@example.com',
                    'Message' => 'EARN $5000/DAY FROM HOME WITH THIS ONE SIMPLE TRICK!!! NO EXPERIENCE NEEDED!!! Click www.money-scam-example.com and start TODAY!! Limited spots available!!!',
                ),
                'score'           => 1,
                'score_label'     => 'cold',
                'score_reasoning' => 'Get-rich-quick spam submission. All-caps promotional message with scam link and impossible earnings claims. Zero business value.',
                'score_factors'   => array(
                    'intent'     => 0,
                    'fit'        => 0,
                    'quality'    => 1,
                    'engagement' => 0,
                ),
                'status'          => 'new',
            ),
        ),
    );

    return isset( $leads[ $industry ] ) ? $leads[ $industry ] : array();
}
