<?php
/**
 * Webhook Handler - sends lead data to external services with retry support
 *
 * @package FormRankLS\Core
 */

namespace FormRankLS\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Webhook_Handler {

    /**
     * Maximum retry attempts for failed webhooks
     */
    const MAX_RETRIES = 3;

    /**
     * Retry delay in seconds (exponential backoff base)
     */
    const RETRY_DELAY_BASE = 60; // 1 minute, 2 minutes, 4 minutes

    /**
     * Trigger webhooks for a scored lead
     *
     * @param int $lead_id The lead ID
     * @param array $score_result The scoring result
     */
    public function trigger(int $lead_id, array $score_result): void {
        $settings = get_option('formrank_settings', []);
        $webhooks = $settings['webhooks'] ?? [];

        if (empty($webhooks)) {
            return;
        }

        // Get full lead data
        $lead = $this->get_lead($lead_id);

        if (!$lead) {
            return;
        }

        foreach ($webhooks as $index => $webhook) {
            if (!$this->should_trigger_webhook($webhook, $score_result['label'])) {
                continue;
            }

            $result = $this->send_webhook_with_retry($webhook, $lead, $score_result);

            // If immediate send fails with retryable error, schedule async retry
            if (!$result['success'] && $this->is_retryable_error($result)) {
                $this->schedule_webhook_retry($webhook, $lead_id, $score_result, $index);
            }
        }
    }

    /**
     * Send webhook with immediate retry on transient failures
     *
     * @param array $webhook Webhook configuration
     * @param array $lead Lead data
     * @param array $score_result Score result
     * @return array Result with success flag
     */
    private function send_webhook_with_retry(array $webhook, array $lead, array $score_result): array {
        $result = $this->send_webhook($webhook, $lead, $score_result);

        if ($result['success']) {
            return $result;
        }

        // Check if error is retryable (network errors, 5xx, 429)
        if (!$this->is_retryable_error($result)) {
            return $result;
        }

        // One immediate retry for transient errors
        usleep(500000); // 500ms delay
        return $this->send_webhook($webhook, $lead, $score_result);
    }

    /**
     * Schedule webhook retry via WordPress cron
     *
     * @param array $webhook Webhook configuration
     * @param int $lead_id Lead ID
     * @param array $score_result Score result
     * @param int $webhook_index Webhook index
     */
    private function schedule_webhook_retry(array $webhook, int $lead_id, array $score_result, int $webhook_index): void {
        // Store retry data
        $retry_data = [
            'webhook_index' => $webhook_index,
            'lead_id' => $lead_id,
            'score_result' => $score_result,
            'attempt' => 1,
            'created_at' => current_time('mysql'),
        ];

        $retry_key = 'formrank_webhook_retry_' . md5($webhook['url'] . $lead_id);

        // Don't schedule if already pending
        if (get_transient($retry_key)) {
            return;
        }

        // Store retry data
        set_transient($retry_key, $retry_data, DAY_IN_SECONDS);

        // Schedule the retry
        if (!wp_next_scheduled('formrank_webhook_retry', [$retry_key])) {
            wp_schedule_single_event(
                time() + self::RETRY_DELAY_BASE,
                'formrank_webhook_retry',
                [$retry_key]
            );
        }

        error_log(sprintf(
            'FormRank: Scheduled webhook retry for lead %d to %s',
            $lead_id,
            $webhook['url']
        ));
    }

    /**
     * Process scheduled webhook retry (called by cron)
     *
     * @param string $retry_key Retry transient key
     */
    public function process_retry(string $retry_key): void {
        $retry_data = get_transient($retry_key);

        if (!$retry_data) {
            return;
        }

        $settings = get_option('formrank_settings', []);
        $webhook = $settings['webhooks'][$retry_data['webhook_index']] ?? null;

        if (!$webhook) {
            delete_transient($retry_key);
            return;
        }

        $lead = $this->get_lead($retry_data['lead_id']);

        if (!$lead) {
            delete_transient($retry_key);
            return;
        }

        $result = $this->send_webhook($webhook, $lead, $retry_data['score_result']);

        if ($result['success']) {
            delete_transient($retry_key);
            error_log(sprintf(
                'FormRank: Webhook retry succeeded for lead %d (attempt %d)',
                $retry_data['lead_id'],
                $retry_data['attempt']
            ));
            return;
        }

        // Check if we should retry again
        if ($retry_data['attempt'] < self::MAX_RETRIES && $this->is_retryable_error($result)) {
            $retry_data['attempt']++;
            set_transient($retry_key, $retry_data, DAY_IN_SECONDS);

            // Exponential backoff
            $delay = self::RETRY_DELAY_BASE * pow(2, $retry_data['attempt'] - 1);

            wp_schedule_single_event(
                time() + $delay,
                'formrank_webhook_retry',
                [$retry_key]
            );

            error_log(sprintf(
                'FormRank: Rescheduled webhook retry for lead %d (attempt %d, delay %ds)',
                $retry_data['lead_id'],
                $retry_data['attempt'],
                $delay
            ));
        } else {
            // Max retries reached or non-retryable error
            delete_transient($retry_key);
            error_log(sprintf(
                'FormRank: Webhook permanently failed for lead %d after %d attempts: %s',
                $retry_data['lead_id'],
                $retry_data['attempt'],
                $result['message'] ?? 'Unknown error'
            ));
        }
    }

    /**
     * Check if error is retryable
     *
     * @param array $result Send result
     * @return bool
     */
    private function is_retryable_error(array $result): bool {
        // Network errors are retryable
        if (!empty($result['is_wp_error'])) {
            return true;
        }

        // 5xx server errors and 429 rate limit are retryable
        $status = $result['status_code'] ?? 0;
        return $status >= 500 || $status === 429;
    }

    /**
     * Check if webhook should be triggered based on score label
     */
    private function should_trigger_webhook(array $webhook, string $label): bool {
        if (empty($webhook['enabled']) || empty($webhook['url'])) {
            return false;
        }

        $trigger = $webhook['trigger'] ?? 'all';

        if ($trigger === 'all') {
            return true;
        }

        // Support comma-separated triggers (e.g., "hot,warm")
        $triggers = array_map('trim', explode(',', $trigger));

        return in_array($label, $triggers);
    }

    /**
     * Send webhook payload
     *
     * @param array $webhook Webhook config
     * @param array $lead Lead data
     * @param array $score_result Score result
     * @return array Result with success, status_code, message, is_wp_error
     */
    private function send_webhook(array $webhook, array $lead, array $score_result): array {
        $payload = $this->build_payload($webhook, $lead, $score_result);

        $response = wp_remote_post($webhook['url'], [
            'timeout' => 15,
            'headers' => [
                'Content-Type' => 'application/json',
                'User-Agent' => 'FormRank-LS/' . FORMRANK_LS_VERSION
            ],
            'body' => wp_json_encode($payload)
        ]);

        // Handle WP_Error
        if (is_wp_error($response)) {
            $this->log_webhook_result($webhook['name'] ?? 'Unknown', $response);

            return [
                'success' => false,
                'is_wp_error' => true,
                'message' => $response->get_error_message(),
                'status_code' => 0
            ];
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $success = $status_code >= 200 && $status_code < 300;

        // Log result
        $this->log_webhook_result($webhook['name'] ?? 'Unknown', $response);

        return [
            'success' => $success,
            'is_wp_error' => false,
            'message' => wp_remote_retrieve_response_message($response),
            'status_code' => $status_code
        ];
    }

    /**
     * Build webhook payload
     */
    private function build_payload(array $webhook, array $lead, array $score_result): array {
        $form_data = json_decode($lead['form_data'], true) ?: [];

        return [
            'event' => 'lead_scored',
            'timestamp' => current_time('c'),
            'lead' => [
                'id' => $lead['id'],
                'email' => $lead['email'],
                'name' => $lead['name'],
                'company' => $lead['company'],
                'phone' => $lead['phone'],
                'form_data' => $form_data,
                'source_url' => $lead['source_url'],
                'created_at' => $lead['created_at']
            ],
            'score' => [
                'value' => $score_result['score'],
                'label' => $score_result['label'],
                'reasoning' => $score_result['reasoning'],
                'factors' => $score_result['factors'] ?? []
            ],
            'form' => [
                'plugin' => $lead['form_plugin'],
                'id' => $lead['form_id']
            ],
            'meta' => [
                'site_url' => get_site_url(),
                'site_name' => get_bloginfo('name'),
                'plugin_version' => FORMRANK_LS_VERSION
            ]
        ];
    }

    /**
     * Get lead data by ID
     */
    private function get_lead(int $lead_id): ?array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $lead_id
        ), ARRAY_A);
    }

    /**
     * Log webhook result for debugging
     */
    private function log_webhook_result(string $name, $response): void {
        $log_enabled = defined('WP_DEBUG') && WP_DEBUG;

        if (!$log_enabled) {
            return;
        }

        if (is_wp_error($response)) {
            error_log(sprintf(
                'FormRank Webhook [%s] failed: %s',
                $name,
                $response->get_error_message()
            ));
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code < 200 || $status_code >= 300) {
            error_log(sprintf(
                'FormRank Webhook [%s] returned status %d',
                $name,
                $status_code
            ));
        }
    }

    /**
     * Test webhook endpoint
     */
    public function test_webhook(string $url): array {
        $test_payload = [
            'event' => 'test',
            'timestamp' => current_time('c'),
            'message' => 'This is a test webhook from FormRank',
            'lead' => [
                'id' => 0,
                'email' => 'test@example.com',
                'name' => 'Test Lead',
                'company' => 'Test Company'
            ],
            'score' => [
                'value' => 75,
                'label' => 'warm',
                'reasoning' => 'This is a test score'
            ]
        ];

        $response = wp_remote_post($url, [
            'timeout' => 15,
            'headers' => [
                'Content-Type' => 'application/json',
                'User-Agent' => 'FormRank-LS/' . FORMRANK_LS_VERSION
            ],
            'body' => wp_json_encode($test_payload)
        ]);

        if (is_wp_error($response)) {
            return [
                'success' => false,
                'message' => $response->get_error_message()
            ];
        }

        $status_code = wp_remote_retrieve_response_code($response);

        return [
            'success' => $status_code >= 200 && $status_code < 300,
            'message' => "Received HTTP {$status_code}",
            'status_code' => $status_code
        ];
    }

    /**
     * Send Slack notification for hot leads
     */
    public function send_slack_notification(int $lead_id, array $score_result): void {
        $settings = get_option('formrank_settings', []);
        $slack_webhook = $settings['notifications']['slack_webhook'] ?? '';

        if (empty($slack_webhook)) {
            return;
        }

        $lead = $this->get_lead($lead_id);

        if (!$lead) {
            return;
        }

        $emoji = $this->get_score_emoji($score_result['label']);
        $color = $this->get_score_color($score_result['label']);

        $payload = [
            'attachments' => [
                [
                    'color' => $color,
                    'pretext' => "{$emoji} New Lead Alert from FormRank",
                    'title' => $lead['name'] ?: 'New Lead',
                    'title_link' => admin_url("admin.php?page=formrank-lead-scoring-lead&id={$lead_id}"),
                    'fields' => [
                        [
                            'title' => 'Score',
                            'value' => "{$score_result['score']} ({$score_result['label']})",
                            'short' => true
                        ],
                        [
                            'title' => 'Email',
                            'value' => $lead['email'] ?: 'N/A',
                            'short' => true
                        ],
                        [
                            'title' => 'Company',
                            'value' => $lead['company'] ?: 'N/A',
                            'short' => true
                        ],
                        [
                            'title' => 'Reasoning',
                            'value' => $score_result['reasoning'],
                            'short' => false
                        ]
                    ],
                    'footer' => 'FormRank',
                    'ts' => time()
                ]
            ]
        ];

        wp_remote_post($slack_webhook, [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => wp_json_encode($payload)
        ]);
    }

    private function get_score_emoji(string $label): string {
        $emojis = [
            'hot' => ':fire:',
            'warm' => ':sun_with_face:',
            'neutral' => ':neutral_face:',
            'cool' => ':snowflake:',
            'cold' => ':ice_cube:'
        ];

        return $emojis[$label] ?? ':grey_question:';
    }

    private function get_score_color(string $label): string {
        $colors = [
            'hot' => '#ef4444',
            'warm' => '#f59e0b',
            'neutral' => '#10b981',
            'cool' => '#3b82f6',
            'cold' => '#6b7280'
        ];

        return $colors[$label] ?? '#6b7280';
    }
}
