<?php
/**
 * AI Scoring Engine - processes leads through Claude API
 *
 * @package FormRankLS\Core
 */

namespace FormRankLS\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Scoring_Engine {

    private $api_client;
    private $cache_manager;

    /**
     * Free tier AI rescore limit per month
     */
    const FREE_TIER_AI_LIMIT = 15;

    public function __construct() {
        $this->api_client = new API_Client();
        $this->cache_manager = new Cache_Manager();
    }

    /**
     * Check if AI scoring is available (has quota remaining)
     *
     * @return array ['available' => bool, 'reason' => string, 'usage' => int, 'limit' => int]
     */
    public function check_ai_availability(): array {
        // Pro users have unlimited AI rescores
        if ($this->is_pro_user()) {
            // Reset usage counter on upgrade from free to paid
            $this->maybe_reset_usage_on_upgrade();

            return [
                'available' => true,
                'reason' => 'Pro plan - unlimited AI scoring',
                'usage' => $this->get_ai_usage_count(),
                'limit' => -1 // unlimited
            ];
        }

        // Track that user is currently on free plan
        update_option('formrank_previous_plan_status', 'free');

        // Free tier: check usage
        $usage = $this->get_ai_usage_count();
        $limit = self::FREE_TIER_AI_LIMIT;

        if ($usage >= $limit) {
            return [
                'available' => false,
                'reason' => sprintf(
                    'Free tier limit reached (%d/%d AI rescores this month). Upgrade to Pro for unlimited AI scoring.',
                    $usage,
                    $limit
                ),
                'usage' => $usage,
                'limit' => $limit
            ];
        }

        return [
            'available' => true,
            'reason' => sprintf('%d/%d AI rescores remaining this month', $limit - $usage, $limit),
            'usage' => $usage,
            'limit' => $limit
        ];
    }

    /**
     * Get AI usage count for current month
     */
    public function get_ai_usage_count(): int {
        $usage = get_option('formrank_ai_usage', []);
        $month_key = gmdate('Y-m');
        return $usage[$month_key] ?? 0;
    }

    /**
     * Increment AI usage counter
     */
    private function increment_ai_usage(): void {
        $usage = get_option('formrank_ai_usage', []);
        $month_key = gmdate('Y-m');
        $usage[$month_key] = ($usage[$month_key] ?? 0) + 1;

        // Keep only last 3 months
        $usage = array_slice($usage, -3, 3, true);

        update_option('formrank_ai_usage', $usage);
    }

    /**
     * Reset AI usage counter when user upgrades from free to paid plan.
     * Prevents stale free-tier counters from showing after upgrade.
     */
    private function maybe_reset_usage_on_upgrade(): void {
        $previous_plan = get_option('formrank_previous_plan_status', 'free');

        if ($previous_plan !== 'pro') {
            // Plan just changed to pro - reset the usage counter
            update_option('formrank_ai_usage', []);
            update_option('formrank_previous_plan_status', 'pro');
        }
    }

    /**
     * Check if current user has Pro license
     */
    private function is_pro_user(): bool {
        // Check via global helper first (handles Freemius namespace correctly)
        if (function_exists('formrank_ls_is_pro') && formrank_ls_is_pro()) {
            return true;
        }

        // Check Freemius license directly (fr_fs is in FormRankLS namespace)
        if (function_exists('\FormRankLS\fr_fs')) {
            $fs = \FormRankLS\fr_fs();

            // Check if user is on a paid plan
            if ($fs->is_paying() || $fs->is_trial()) {
                return true;
            }

            // Check specific plan names
            if ($fs->is_plan('pro') || $fs->is_plan('agency') || $fs->is_plan('business')) {
                return true;
            }
        }

        // Allow filtering for custom license implementations
        return apply_filters('formrank_ls_is_pro', false);
    }

    /**
     * Get current license status for display
     */
    public function get_license_status(): array {
        if (function_exists('\FormRankLS\fr_fs')) {
            $fs = \FormRankLS\fr_fs();

            if ($fs->is_paying()) {
                $plan = $fs->get_plan();
                return [
                    'status' => 'active',
                    'plan' => $plan ? $plan->name : 'pro',
                    'is_pro' => true,
                    'message' => __('Pro license active - Unlimited AI scoring', 'formrank-lead-scoring')
                ];
            }

            if ($fs->is_trial()) {
                return [
                    'status' => 'trial',
                    'plan' => 'trial',
                    'is_pro' => true,
                    'message' => __('Trial active - Unlimited AI scoring during trial', 'formrank-lead-scoring')
                ];
            }

            if ($fs->is_registered()) {
                return [
                    'status' => 'free',
                    'plan' => 'free',
                    'is_pro' => false,
                    'message' => sprintf(
                        /* translators: %d: number of AI rescores per month */
                        __('Free plan - %d AI rescores/month', 'formrank-lead-scoring'),
                        self::FREE_TIER_AI_LIMIT
                    )
                ];
            }
        }

        return [
            'status' => 'free',
            'plan' => 'free',
            'is_pro' => false,
            'message' => sprintf(
                /* translators: %d: number of AI rescores per month */
                __('Free plan - %d AI rescores/month', 'formrank-lead-scoring'),
                self::FREE_TIER_AI_LIMIT
            )
        ];
    }

    /**
     * Score a lead submission
     *
     * @param array $lead_data Form submission data
     * @param array $context Business context for scoring
     * @return array Score result with reasoning
     */
    public function score_lead(array $lead_data, array $context = []): array {
        // Check cache first
        $cache_key = $this->generate_cache_key($lead_data);
        $cached = $this->cache_manager->get($cache_key);

        if ($cached !== false) {
            return $cached;
        }

        // Build the prompt
        $prompt = $this->build_scoring_prompt($lead_data, $context);

        // Call Claude API
        try {
            $response = $this->api_client->complete($prompt);
            $result = $this->parse_scoring_response($response);

            // Cache the result
            $settings = get_option('formrank_settings', []);
            $cache_duration = $settings['advanced']['cache_duration'] ?? 3600;
            $this->cache_manager->set($cache_key, $result, $cache_duration);

            return $result;

        } catch (\Exception $e) {
            // Log error
            error_log('FormRank API Error: ' . $e->getMessage());

            // Fallback to Local Scoring Engine when API fails
            try {
                $local_engine = new Local_Scoring_Engine();
                $local_result = $local_engine->score_lead($lead_data, $context);

                // Mark that this was a fallback result
                $local_result['scoring_method'] = 'local_fallback';
                $local_result['fallback_reason'] = $e->getMessage();

                // Cache the local result with shorter duration
                $this->cache_manager->set($cache_key, $local_result, 1800); // 30 minutes

                error_log('FormRank: Fell back to local scoring due to API error');

                return $local_result;

            } catch (\Exception $local_error) {
                // If local scoring also fails, return default score
                error_log('FormRank Local Fallback Error: ' . $local_error->getMessage());

                return [
                    'score' => 50,
                    'label' => 'neutral',
                    'reasoning' => 'Unable to score lead at this time. Please try again later.',
                    'factors' => [],
                    'scoring_method' => 'error_fallback',
                    'error' => $e->getMessage()
                ];
            }
        }
    }

    /**
     * Build the scoring prompt for Claude
     */
    private function build_scoring_prompt(array $lead_data, array $context): string {
        $settings = get_option('formrank_settings', []);

        $business_context = $context['business_context'] ?? $settings['scoring']['business_context'] ?? 'General business';
        $ideal_customer = $context['ideal_customer'] ?? $settings['scoring']['ideal_customer'] ?? 'Any qualified buyer';

        $form_data_str = $this->format_form_data($lead_data);

        $prompt = 'You are an expert lead qualification specialist. Analyze this form submission and provide a lead score from 1-100.' . "\n\n"
            . '## Business Context' . "\n"
            . $business_context . "\n\n"
            . '## Ideal Customer Profile' . "\n"
            . $ideal_customer . "\n\n"
            . '## Form Submission Data' . "\n"
            . $form_data_str . "\n\n"
            . '## Scoring Instructions' . "\n"
            . 'Score this lead based on:' . "\n"
            . '1. INTENT (40%): Urgency signals, specific questions, timeline mentions, budget references' . "\n"
            . '2. FIT (30%): Company size, industry, role/title, geographic relevance' . "\n"
            . '3. QUALITY (20%): Email domain (business vs free), form completion, professionalism' . "\n"
            . '4. ENGAGEMENT (10%): Message detail, specific questions, clear next steps' . "\n\n"
            . '## Response Format' . "\n"
            . 'Respond ONLY with valid JSON in this exact format:' . "\n"
            . '{' . "\n"
            . '    "score": <number 1-100>,' . "\n"
            . '    "label": "<hot|warm|neutral|cool|cold>",' . "\n"
            . '    "reasoning": "<1-2 sentence summary>",' . "\n"
            . '    "factors": {' . "\n"
            . '        "intent": {"score": <0-40>, "notes": "<brief note>"},' . "\n"
            . '        "fit": {"score": <0-30>, "notes": "<brief note>"},' . "\n"
            . '        "quality": {"score": <0-20>, "notes": "<brief note>"},' . "\n"
            . '        "engagement": {"score": <0-10>, "notes": "<brief note>"}' . "\n"
            . '    }' . "\n"
            . '}' . "\n\n"
            . 'Labels:' . "\n"
            . '- hot (80-100): High intent, perfect fit - contact immediately' . "\n"
            . '- warm (60-79): Good potential - follow up within 24 hours' . "\n"
            . '- neutral (40-59): Possible fit - standard follow-up' . "\n"
            . '- cool (20-39): Low priority - nurture sequence' . "\n"
            . '- cold (1-19): Likely spam or poor fit - archive';

        return $prompt;
    }

    /**
     * Format form data for the prompt
     */
    private function format_form_data(array $lead_data): string {
        $output = [];

        foreach ($lead_data as $key => $value) {
            // Skip internal fields
            if (strpos($key, '_') === 0) {
                continue;
            }

            if (is_array($value)) {
                $value = implode(', ', array_filter($value));
            }

            if (empty($value)) {
                continue;
            }

            $clean_key = ucwords(str_replace(['_', '-'], ' ', $key));
            $output[] = "- {$clean_key}: {$value}";
        }

        return implode("\n", $output);
    }

    /**
     * Parse Claude's JSON response
     */
    private function parse_scoring_response(string $response): array {
        // Extract JSON from response (in case there's extra text)
        $json_match = preg_match('/\{[\s\S]*\}/', $response, $matches);

        if (!$json_match) {
            throw new \Exception('Invalid response format from AI');
        }

        $data = json_decode($matches[0], true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception('Failed to parse AI response: ' . esc_html(json_last_error_msg()));
        }

        // Validate required fields
        if (!isset($data['score']) || !isset($data['label'])) {
            throw new \Exception('Missing required fields in AI response');
        }

        // Ensure score is within bounds
        $data['score'] = max(1, min(100, (int) $data['score']));

        // Validate label
        $valid_labels = ['hot', 'warm', 'neutral', 'cool', 'cold'];
        if (!in_array($data['label'], $valid_labels)) {
            $data['label'] = $this->get_label_from_score($data['score']);
        }

        return $data;
    }

    /**
     * Get label from score if API returns invalid label
     */
    private function get_label_from_score(int $score): string {
        $settings = get_option('formrank_settings', []);

        $hot = $settings['scoring']['score_threshold_hot'] ?? 80;
        $warm = $settings['scoring']['score_threshold_warm'] ?? 60;
        $neutral = $settings['scoring']['score_threshold_neutral'] ?? 40;
        $cool = $settings['scoring']['score_threshold_cool'] ?? 20;

        if ($score >= $hot) {
            return 'hot';
        } elseif ($score >= $warm) {
            return 'warm';
        } elseif ($score >= $neutral) {
            return 'neutral';
        } elseif ($score >= $cool) {
            return 'cool';
        }

        return 'cold';
    }

    /**
     * Generate cache key for lead data
     */
    private function generate_cache_key(array $lead_data): string {
        // Include settings hash so cache invalidates on settings change
        $settings = get_option('formrank_settings', []);
        $settings_hash = md5(serialize($settings['scoring'] ?? []));

        return 'formrank_' . md5(serialize($lead_data) . $settings_hash);
    }

    /**
     * Manually rescore a lead using AI
     *
     * @param int $lead_id Lead ID to rescore
     * @param bool $force_ai Force AI scoring even if limit reached (for admin override)
     * @return array Score result with reasoning
     * @throws \Exception On errors or quota exceeded
     */
    public function rescore_lead(int $lead_id, bool $force_ai = false): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Check AI availability first (unless forced)
        if (!$force_ai) {
            $availability = $this->check_ai_availability();
            if (!$availability['available']) {
                throw new \Exception(esc_html($availability['reason']));
            }
        }

        $lead = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $lead_id
        ), ARRAY_A);

        if (!$lead) {
            throw new \Exception('Lead not found');
        }

        $form_data = json_decode($lead['form_data'], true);

        // Clear cache for this lead
        $cache_key = $this->generate_cache_key($form_data);
        $this->cache_manager->delete($cache_key);

        // Score using AI (with automatic fallback to local scoring on error)
        $result = $this->score_lead($form_data);

        // Check if we got a fallback result
        $was_fallback = isset($result['scoring_method']) &&
                        in_array($result['scoring_method'], ['local_fallback', 'error_fallback'], true);

        // Set scoring method based on actual result
        $scoring_method = $was_fallback ? $result['scoring_method'] : 'ai';

        // Update lead record
        $wpdb->update($table, [
            'score' => $result['score'],
            'score_label' => $result['label'],
            'score_reasoning' => $result['reasoning'],
            'score_factors' => wp_json_encode($result['factors'] ?? []),
            'scoring_method' => $scoring_method,
            'scored_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        ], ['id' => $lead_id]);

        // Only increment AI usage counter if we actually used AI (not fallback)
        if (!$force_ai && !$was_fallback) {
            $this->increment_ai_usage();
        }

        // Return result with the actual method used
        $result['scoring_method'] = $scoring_method;

        return $result;
    }

    /**
     * Get AI scoring stats for display
     */
    public function get_ai_stats(): array {
        $is_pro = $this->is_pro_user();

        // Ensure usage counter is reset on plan upgrade
        if ($is_pro) {
            $this->maybe_reset_usage_on_upgrade();
        }

        $usage = $this->get_ai_usage_count();
        $limit = $is_pro ? -1 : self::FREE_TIER_AI_LIMIT;

        return [
            'is_pro' => $is_pro,
            'usage' => $usage,
            'limit' => $limit,
            'remaining' => $is_pro ? -1 : max(0, $limit - $usage),
            'percentage' => $is_pro ? 0 : min(100, ($usage / $limit) * 100)
        ];
    }
}
