<?php
/**
 * Local Scoring Engine - Rule-based and learning-based scoring without external API
 *
 * This engine handles two scenarios:
 * 1. Cold Start: Uses rule-based BANT scoring when no historical data exists
 * 2. Learning Mode: Improves scoring based on user feedback and conversions
 *
 * @package FormRankLS\Core
 */

namespace FormRankLS\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Local_Scoring_Engine {

    /**
     * Minimum leads needed before switching from pure rules to learning mode
     */
    const MIN_LEADS_FOR_LEARNING = 50;

    /**
     * Minimum conversions needed to start using ML-like adjustments
     */
    const MIN_CONVERSIONS_FOR_ML = 10;

    /**
     * Score a lead using local algorithms (no external API)
     *
     * @param array $lead_data Form submission data
     * @param array $context Optional business context
     * @return array Score result with reasoning
     */
    public function score_lead( array $lead_data, array $context = [] ): array {
        try {
            // Validate input
            if ( empty( $lead_data ) ) {
                $this->log_scoring_event( 'warning', 'Empty lead data provided', [ 'context' => $context ] );
                return $this->get_default_score( 'Empty lead data' );
            }

            $stats = $this->get_learning_stats();

            // Determine scoring mode
            if ( $stats['total_leads'] < self::MIN_LEADS_FOR_LEARNING ) {
                // Cold start: Pure rule-based scoring
                return $this->score_with_rules( $lead_data, $context );
            }

            if ( $stats['total_conversions'] < self::MIN_CONVERSIONS_FOR_ML ) {
                // Have leads but no conversion data: Rules + basic heuristics
                return $this->score_with_enhanced_rules( $lead_data, $context, $stats );
            }

            // Learning mode: Rules + learned adjustments from conversion data
            return $this->score_with_learning( $lead_data, $context, $stats );

        } catch ( \Exception $e ) {
            $this->log_scoring_event( 'error', 'Scoring failed: ' . $e->getMessage(), [
                'lead_data_keys' => array_keys( $lead_data ),
                'trace' => $e->getTraceAsString()
            ] );

            return $this->get_default_score( 'Scoring error: ' . $e->getMessage() );
        }
    }

    /**
     * Get default score for error cases
     *
     * @param string $reason Reason for default score
     * @return array Default score result
     */
    private function get_default_score( string $reason ): array {
        return [
            'score'       => 50,
            'label'       => 'neutral',
            'reasoning'   => 'Default score applied. ' . $reason,
            'factors'     => [],
            'mode'        => 'error_fallback',
            'confidence'  => 0.3,
            'error'       => true
        ];
    }

    /**
     * Log scoring events for debugging
     *
     * @param string $level Log level (debug, info, warning, error)
     * @param string $message Log message
     * @param array $context Additional context
     */
    private function log_scoring_event( string $level, string $message, array $context = [] ): void {
        // Only log if WP_DEBUG is enabled
        if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG ) {
            return;
        }

        $log_levels = [ 'debug' => 0, 'info' => 1, 'warning' => 2, 'error' => 3 ];
        $min_level = defined( 'FORMRANK_LOG_LEVEL' ) ? FORMRANK_LOG_LEVEL : 'warning';

        // Skip if level is below minimum
        if ( isset( $log_levels[ $level ] ) && isset( $log_levels[ $min_level ] ) ) {
            if ( $log_levels[ $level ] < $log_levels[ $min_level ] ) {
                return;
            }
        }

        $context_str = ! empty( $context ) ? ' | Context: ' . wp_json_encode( $context ) : '';
        error_log( sprintf( 'FormRank [%s]: %s%s', strtoupper( $level ), $message, $context_str ) );
    }

    /**
     * Cold Start: Pure rule-based BANT scoring
     * Used when we have no historical data
     */
    private function score_with_rules( array $lead_data, array $context ): array {
        $fit_score = $this->calculate_fit_score( $lead_data );
        $engagement_score = $this->calculate_engagement_score( $lead_data );

        // Two-dimensional scoring (research shows this outperforms single-score)
        $combined = ( $fit_score * 0.4 ) + ( $engagement_score * 0.6 );

        $label = $this->get_label_from_score( $combined );
        $factors = $this->get_scoring_factors( $lead_data, $fit_score, $engagement_score );

        return [
            'score'       => round( $combined ),
            'label'       => $label,
            'reasoning'   => $this->generate_reasoning( $factors, $combined ),
            'factors'     => $factors,
            'mode'        => 'cold_start',
            'confidence'  => 0.6, // Lower confidence in cold start mode
        ];
    }

    /**
     * Enhanced Rules: Rule-based with basic pattern detection
     * Used when we have leads but insufficient conversion data
     */
    private function score_with_enhanced_rules( array $lead_data, array $context, array $stats ): array {
        $base_result = $this->score_with_rules( $lead_data, $context );

        // Apply learned patterns from high-engagement leads
        $patterns = $this->detect_high_engagement_patterns( $stats );
        $adjustment = $this->calculate_pattern_adjustment( $lead_data, $patterns );

        $adjusted_score = max( 1, min( 100, $base_result['score'] + $adjustment ) );

        $base_result['score'] = round( $adjusted_score );
        $base_result['label'] = $this->get_label_from_score( $adjusted_score );
        $base_result['mode'] = 'enhanced_rules';
        $base_result['confidence'] = 0.7;

        if ( abs( $adjustment ) > 5 ) {
            $base_result['reasoning'] .= ' (Adjusted based on similar leads.)';
        }

        return $base_result;
    }

    /**
     * Learning Mode: Rules + ML-like adjustments from conversion data
     * Used when we have sufficient conversion data to learn from
     */
    private function score_with_learning( array $lead_data, array $context, array $stats ): array {
        $base_result = $this->score_with_rules( $lead_data, $context );

        // Get learned feature weights from conversion data
        $learned_weights = $this->get_learned_weights();

        // Calculate learning adjustment
        $adjustment = $this->calculate_learning_adjustment( $lead_data, $learned_weights );

        $adjusted_score = max( 1, min( 100, $base_result['score'] + $adjustment ) );

        $base_result['score'] = round( $adjusted_score );
        $base_result['label'] = $this->get_label_from_score( $adjusted_score );
        $base_result['mode'] = 'learning';
        $base_result['confidence'] = min( 0.95, 0.7 + ( $stats['total_conversions'] / 500 ) ); // Confidence grows with data

        if ( abs( $adjustment ) > 5 ) {
            $base_result['reasoning'] .= ' (Score adjusted based on your conversion patterns.)';
        }

        return $base_result;
    }

    /**
     * Calculate FIT score (demographics/firmographics)
     * Based on BANT: Budget, Authority, Need, Timeline
     */
    private function calculate_fit_score( array $lead_data ): float {
        $score = 0;
        $fields = $lead_data['fields'] ?? $lead_data;

        // Normalize field names for matching
        $normalized = $this->normalize_field_names( $fields );

        // BUDGET scoring (25 points max)
        $budget_score = $this->score_budget( $normalized );
        $score += $budget_score;

        // AUTHORITY scoring (25 points max) - based on job title/role
        $authority_score = $this->score_authority( $normalized );
        $score += $authority_score;

        // NEED scoring (25 points max) - detected from message/inquiry
        $need_score = $this->score_need( $normalized );
        $score += $need_score;

        // TIMELINE scoring (25 points max)
        $timeline_score = $this->score_timeline( $normalized );
        $score += $timeline_score;

        return min( 100, $score );
    }

    /**
     * Calculate ENGAGEMENT score (behavioral signals)
     */
    private function calculate_engagement_score( array $lead_data ): float {
        $score = 30; // Base score for submitting a form
        $fields = $lead_data['fields'] ?? $lead_data;
        $normalized = $this->normalize_field_names( $fields );

        // Form completion quality
        $filled_fields = count( array_filter( $normalized, function( $v ) {
            return ! empty( $v ) && $v !== 'N/A';
        }));

        if ( $filled_fields >= 5 ) {
            $score += 15; // Detailed form submission
        } elseif ( $filled_fields >= 3 ) {
            $score += 10;
        }

        // Message length/detail (if message field exists)
        $message = $normalized['message'] ?? $normalized['comments'] ?? $normalized['inquiry'] ?? '';
        $word_count = str_word_count( $message );

        if ( $word_count >= 50 ) {
            $score += 20; // Detailed message shows high engagement
        } elseif ( $word_count >= 20 ) {
            $score += 10;
        } elseif ( $word_count >= 10 ) {
            $score += 5;
        }

        // Specific questions or requests (high intent signals)
        $high_intent_patterns = [
            '/\bpric(e|ing)\b/i'        => 15,
            '/\bquote\b/i'              => 15,
            '/\bdemo\b/i'               => 15,
            '/\bmeeting\b/i'            => 12,
            '/\bcall\b/i'               => 10,
            '/\basap|urgent|immediate/i' => 12,
            '/\bbudget\b/i'             => 10,
            '/\bwhen can\b/i'           => 10,
            '/\bhow (much|soon)\b/i'    => 10,
        ];

        foreach ( $high_intent_patterns as $pattern => $points ) {
            if ( preg_match( $pattern, $message ) ) {
                $score += $points;
            }
        }

        // Business email domain (vs free email)
        $email = $lead_data['email'] ?? '';
        if ( $this->is_business_email( $email ) ) {
            $score += 10;
        }

        return min( 100, $score );
    }

    /**
     * Score BUDGET component
     */
    private function score_budget( array $normalized ): int {
        // Look for budget field
        $budget = $normalized['budget'] ?? $normalized['budget_range'] ?? $normalized['project_budget'] ?? '';

        if ( empty( $budget ) ) {
            return 10; // No budget info = neutral
        }

        $budget_lower = strtolower( $budget );

        // Extract numeric value if present
        preg_match( '/[\$]?([\d,]+)/', $budget, $matches );
        $amount = isset( $matches[1] ) ? (int) str_replace( ',', '', $matches[1] ) : 0;

        // Score based on amount
        if ( $amount >= 50000 || strpos( $budget_lower, '50k' ) !== false || strpos( $budget_lower, '50,000' ) !== false ) {
            return 25;
        } elseif ( $amount >= 25000 || strpos( $budget_lower, '25k' ) !== false ) {
            return 22;
        } elseif ( $amount >= 10000 || strpos( $budget_lower, '10k' ) !== false ) {
            return 18;
        } elseif ( $amount >= 5000 || strpos( $budget_lower, '5k' ) !== false ) {
            return 15;
        } elseif ( $amount > 0 ) {
            return 10;
        }

        // Text-based budget indicators
        $budget_keywords = [
            'enterprise'    => 25,
            'unlimited'     => 22,
            'flexible'      => 18,
            'negotiable'    => 15,
            'limited'       => 8,
            'small'         => 5,
            'tight'         => 5,
            'no budget'     => 3,
            'free'          => 2,
        ];

        foreach ( $budget_keywords as $keyword => $score ) {
            if ( strpos( $budget_lower, $keyword ) !== false ) {
                return $score;
            }
        }

        return 10;
    }

    /**
     * Score AUTHORITY component (job title/role)
     */
    private function score_authority( array $normalized ): int {
        $title = $normalized['job_title'] ?? $normalized['title'] ?? $normalized['role'] ?? $normalized['position'] ?? '';

        if ( empty( $title ) ) {
            return 12; // No title = neutral
        }

        $title_lower = strtolower( $title );

        // C-Level executives
        if ( preg_match( '/\b(ceo|cto|cfo|coo|cmo|cio|chief|president|founder|owner)\b/i', $title_lower ) ) {
            return 25;
        }

        // VP level
        if ( preg_match( '/\b(vp|vice president|svp|evp)\b/i', $title_lower ) ) {
            return 22;
        }

        // Director level
        if ( preg_match( '/\b(director|head of)\b/i', $title_lower ) ) {
            return 18;
        }

        // Manager level
        if ( preg_match( '/\b(manager|lead|supervisor|team lead)\b/i', $title_lower ) ) {
            return 14;
        }

        // Senior individual contributor
        if ( preg_match( '/\b(senior|sr\.|principal)\b/i', $title_lower ) ) {
            return 12;
        }

        // Individual contributor
        if ( preg_match( '/\b(specialist|analyst|coordinator|associate)\b/i', $title_lower ) ) {
            return 10;
        }

        // Entry level / Students
        if ( preg_match( '/\b(intern|student|trainee|junior|jr\.)\b/i', $title_lower ) ) {
            return 5;
        }

        return 10;
    }

    /**
     * Score NEED component (detected from message/inquiry)
     */
    private function score_need( array $normalized ): int {
        $message = $normalized['message'] ?? $normalized['comments'] ?? $normalized['inquiry'] ?? $normalized['question'] ?? '';
        $services = $normalized['services'] ?? $normalized['interest'] ?? $normalized['looking_for'] ?? '';

        $combined_text = strtolower( $message . ' ' . $services );

        if ( empty( trim( $combined_text ) ) ) {
            return 10;
        }

        $score = 10; // Base score

        // Pain point indicators (high need)
        $pain_indicators = [
            '/\b(problem|issue|challenge|struggling|frustrated|need help)\b/i' => 8,
            '/\b(not working|broken|failing|underperform)\b/i'                 => 8,
            '/\b(upgrade|replace|switch|migrate)\b/i'                          => 6,
            '/\b(looking for|searching for|want to find)\b/i'                  => 5,
            '/\b(must have|require|essential|critical)\b/i'                    => 7,
        ];

        foreach ( $pain_indicators as $pattern => $points ) {
            if ( preg_match( $pattern, $combined_text ) ) {
                $score += $points;
            }
        }

        // Specific service interest
        if ( ! empty( $services ) ) {
            $score += 5;
        }

        return min( 25, $score );
    }

    /**
     * Score TIMELINE component
     */
    private function score_timeline( array $normalized ): int {
        $timeline = $normalized['timeline'] ?? $normalized['timeframe'] ?? $normalized['when'] ?? $normalized['start_date'] ?? '';
        $message = $normalized['message'] ?? $normalized['comments'] ?? '';

        $combined_text = strtolower( $timeline . ' ' . $message );

        if ( empty( trim( $combined_text ) ) ) {
            return 10;
        }

        // Immediate timeline
        if ( preg_match( '/\b(asap|immediate|urgent|right away|this week|tomorrow|today)\b/i', $combined_text ) ) {
            return 25;
        }

        // Near-term timeline
        if ( preg_match( '/\b(this month|next week|within 2 weeks|soon|1-2 weeks)\b/i', $combined_text ) ) {
            return 22;
        }

        // This quarter
        if ( preg_match( '/\b(this quarter|next month|1-3 months|within 30 days)\b/i', $combined_text ) ) {
            return 18;
        }

        // This year
        if ( preg_match( '/\b(this year|next quarter|3-6 months|Q[1-4])\b/i', $combined_text ) ) {
            return 12;
        }

        // Long-term / Just researching
        if ( preg_match( '/\b(next year|just (looking|researching|browsing)|no rush|eventually|someday)\b/i', $combined_text ) ) {
            return 5;
        }

        return 10;
    }

    /**
     * Normalize field names for consistent matching
     */
    private function normalize_field_names( array $fields ): array {
        $normalized = [];

        foreach ( $fields as $key => $value ) {
            // Convert to lowercase, replace common separators
            $normalized_key = strtolower( $key );
            $normalized_key = str_replace( [ '-', '_', ' ' ], '_', $normalized_key );
            $normalized_key = preg_replace( '/[^a-z0-9_]/', '', $normalized_key );

            $normalized[ $normalized_key ] = $value;

            // Also map common variations
            $mappings = [
                'email_address'  => 'email',
                'your_email'     => 'email',
                'company_name'   => 'company',
                'organization'   => 'company',
                'business_name'  => 'company',
                'phone_number'   => 'phone',
                'telephone'      => 'phone',
                'mobile'         => 'phone',
                'your_message'   => 'message',
                'comments'       => 'message',
                'inquiry'        => 'message',
                'how_can_we_help' => 'message',
                'job_role'       => 'job_title',
                'your_role'      => 'job_title',
                'position'       => 'job_title',
            ];

            if ( isset( $mappings[ $normalized_key ] ) ) {
                $normalized[ $mappings[ $normalized_key ] ] = $value;
            }
        }

        return $normalized;
    }

    /**
     * Check if email is from a business domain
     */
    private function is_business_email( string $email ): bool {
        if ( empty( $email ) || ! is_email( $email ) ) {
            return false;
        }

        $free_domains = [
            'gmail.com', 'yahoo.com', 'hotmail.com', 'outlook.com', 'aol.com',
            'icloud.com', 'mail.com', 'protonmail.com', 'zoho.com', 'yandex.com',
            'live.com', 'msn.com', 'me.com', 'inbox.com', 'gmx.com', 'fastmail.com',
        ];

        $domain = strtolower( substr( strrchr( $email, '@' ), 1 ) );

        return ! in_array( $domain, $free_domains, true );
    }

    /**
     * Get label from score
     */
    private function get_label_from_score( float $score ): string {
        $settings = get_option( 'formrank_settings', [] );

        $hot     = $settings['scoring']['score_threshold_hot'] ?? 80;
        $warm    = $settings['scoring']['score_threshold_warm'] ?? 60;
        $neutral = $settings['scoring']['score_threshold_neutral'] ?? 40;
        $cool    = $settings['scoring']['score_threshold_cool'] ?? 20;

        if ( $score >= $hot ) {
            return 'hot';
        } elseif ( $score >= $warm ) {
            return 'warm';
        } elseif ( $score >= $neutral ) {
            return 'neutral';
        } elseif ( $score >= $cool ) {
            return 'cool';
        }

        return 'cold';
    }

    /**
     * Get scoring factors breakdown
     */
    private function get_scoring_factors( array $lead_data, float $fit_score, float $engagement_score ): array {
        $fields = $lead_data['fields'] ?? $lead_data;
        $normalized = $this->normalize_field_names( $fields );

        return [
            'fit' => [
                'score' => round( $fit_score * 0.4 ),
                'max'   => 40,
                'notes' => $this->get_fit_notes( $normalized ),
            ],
            'engagement' => [
                'score' => round( $engagement_score * 0.6 ),
                'max'   => 60,
                'notes' => $this->get_engagement_notes( $normalized, $lead_data['email'] ?? '' ),
            ],
        ];
    }

    /**
     * Generate human-readable fit notes
     */
    private function get_fit_notes( array $normalized ): string {
        $notes = [];

        // Budget
        $budget = $normalized['budget'] ?? '';
        if ( ! empty( $budget ) ) {
            $notes[] = "Budget: {$budget}";
        }

        // Title
        $title = $normalized['job_title'] ?? $normalized['title'] ?? '';
        if ( ! empty( $title ) ) {
            $notes[] = "Role: {$title}";
        }

        // Company
        $company = $normalized['company'] ?? '';
        if ( ! empty( $company ) ) {
            $notes[] = "Company: {$company}";
        }

        // Timeline
        $timeline = $normalized['timeline'] ?? '';
        if ( ! empty( $timeline ) ) {
            $notes[] = "Timeline: {$timeline}";
        }

        return ! empty( $notes ) ? implode( '; ', $notes ) : 'Limited fit information provided';
    }

    /**
     * Generate human-readable engagement notes
     */
    private function get_engagement_notes( array $normalized, string $email ): string {
        $notes = [];

        // Message detail
        $message = $normalized['message'] ?? '';
        $word_count = str_word_count( $message );

        if ( $word_count >= 50 ) {
            $notes[] = 'Detailed inquiry';
        } elseif ( $word_count >= 20 ) {
            $notes[] = 'Good message detail';
        } elseif ( $word_count > 0 ) {
            $notes[] = 'Brief message';
        }

        // Email type
        if ( $this->is_business_email( $email ) ) {
            $notes[] = 'Business email';
        } else {
            $notes[] = 'Personal email';
        }

        // High intent signals
        if ( preg_match( '/\b(pric|quote|demo)\b/i', $message ) ) {
            $notes[] = 'Shows purchase intent';
        }

        if ( preg_match( '/\b(urgent|asap|immediate)\b/i', $message ) ) {
            $notes[] = 'Urgent need';
        }

        return ! empty( $notes ) ? implode( '; ', $notes ) : 'Standard engagement';
    }

    /**
     * Generate reasoning summary
     */
    private function generate_reasoning( array $factors, float $score ): string {
        $label = $this->get_label_from_score( $score );

        $label_descriptions = [
            'hot'     => 'High-intent lead with strong fit indicators.',
            'warm'    => 'Good potential lead showing interest.',
            'neutral' => 'Moderate fit, may need nurturing.',
            'cool'    => 'Low priority, limited engagement signals.',
            'cold'    => 'Poor fit or likely not a qualified prospect.',
        ];

        $base = $label_descriptions[ $label ] ?? 'Lead scored based on available information.';

        // Add specific notes
        $fit_notes = $factors['fit']['notes'] ?? '';
        $engagement_notes = $factors['engagement']['notes'] ?? '';

        if ( strpos( $engagement_notes, 'purchase intent' ) !== false ) {
            $base .= ' Shows clear purchase intent.';
        }

        if ( strpos( $engagement_notes, 'Urgent' ) !== false ) {
            $base .= ' Has urgent timeline.';
        }

        return $base;
    }

    /**
     * Get learning statistics
     */
    public function get_learning_stats(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Total leads
        $total_leads = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );

        // Converted leads (requires conversion tracking)
        $total_conversions = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$table} WHERE status = 'converted'"
        );

        // Average score of converted leads
        $avg_converted_score = (float) $wpdb->get_var(
            "SELECT AVG(score) FROM {$table} WHERE status = 'converted' AND score IS NOT NULL"
        );

        // Score distribution
        $score_distribution = $wpdb->get_results(
            "SELECT
                CASE
                    WHEN score >= 80 THEN 'hot'
                    WHEN score >= 60 THEN 'warm'
                    WHEN score >= 40 THEN 'neutral'
                    WHEN score >= 20 THEN 'cool'
                    ELSE 'cold'
                END as segment,
                COUNT(*) as count
            FROM {$table}
            WHERE score IS NOT NULL
            GROUP BY segment",
            ARRAY_A
        );

        return [
            'total_leads'        => $total_leads,
            'total_conversions'  => $total_conversions,
            'conversion_rate'    => $total_leads > 0 ? round( $total_conversions / $total_leads * 100, 2 ) : 0,
            'avg_converted_score' => round( $avg_converted_score, 1 ),
            'score_distribution' => $score_distribution,
            'scoring_mode'       => $this->get_current_scoring_mode( $total_leads, $total_conversions ),
        ];
    }

    /**
     * Get current scoring mode description
     */
    private function get_current_scoring_mode( int $total_leads, int $total_conversions ): string {
        if ( $total_leads < self::MIN_LEADS_FOR_LEARNING ) {
            return 'cold_start';
        }

        if ( $total_conversions < self::MIN_CONVERSIONS_FOR_ML ) {
            return 'enhanced_rules';
        }

        return 'learning';
    }

    /**
     * Detect patterns from high-engagement leads
     */
    private function detect_high_engagement_patterns( array $stats ): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Get form data from high-scoring leads
        $high_score_leads = $wpdb->get_results(
            "SELECT form_data FROM {$table}
            WHERE score >= 70
            ORDER BY created_at DESC
            LIMIT 50",
            ARRAY_A
        );

        if ( empty( $high_score_leads ) ) {
            return [];
        }

        $patterns = [
            'common_industries' => [],
            'common_titles'     => [],
            'common_budgets'    => [],
            'avg_message_length' => 0,
        ];

        $message_lengths = [];

        foreach ( $high_score_leads as $lead ) {
            $form_data = json_decode( $lead['form_data'], true ) ?: [];
            $normalized = $this->normalize_field_names( $form_data );

            // Track industries
            $industry = $normalized['industry'] ?? '';
            if ( ! empty( $industry ) ) {
                $patterns['common_industries'][ $industry ] = ( $patterns['common_industries'][ $industry ] ?? 0 ) + 1;
            }

            // Track titles
            $title = $normalized['job_title'] ?? $normalized['title'] ?? '';
            if ( ! empty( $title ) ) {
                $patterns['common_titles'][ strtolower( $title ) ] = ( $patterns['common_titles'][ strtolower( $title ) ] ?? 0 ) + 1;
            }

            // Track message lengths
            $message = $normalized['message'] ?? '';
            if ( ! empty( $message ) ) {
                $message_lengths[] = str_word_count( $message );
            }
        }

        // Calculate average message length
        if ( ! empty( $message_lengths ) ) {
            $patterns['avg_message_length'] = array_sum( $message_lengths ) / count( $message_lengths );
        }

        // Sort by frequency
        arsort( $patterns['common_industries'] );
        arsort( $patterns['common_titles'] );

        return $patterns;
    }

    /**
     * Calculate score adjustment based on detected patterns
     */
    private function calculate_pattern_adjustment( array $lead_data, array $patterns ): float {
        if ( empty( $patterns ) ) {
            return 0;
        }

        $adjustment = 0;
        $fields = $lead_data['fields'] ?? $lead_data;
        $normalized = $this->normalize_field_names( $fields );

        // Boost if industry matches common high-scoring industries
        $industry = $normalized['industry'] ?? '';
        if ( ! empty( $industry ) && ! empty( $patterns['common_industries'] ) ) {
            $top_industries = array_slice( array_keys( $patterns['common_industries'] ), 0, 5 );
            if ( in_array( $industry, $top_industries, true ) ) {
                $adjustment += 5;
            }
        }

        // Boost if title matches common high-scoring titles
        $title = strtolower( $normalized['job_title'] ?? $normalized['title'] ?? '' );
        if ( ! empty( $title ) && ! empty( $patterns['common_titles'] ) ) {
            foreach ( array_keys( array_slice( $patterns['common_titles'], 0, 10 ) ) as $common_title ) {
                if ( strpos( $title, $common_title ) !== false || strpos( $common_title, $title ) !== false ) {
                    $adjustment += 5;
                    break;
                }
            }
        }

        // Boost if message length is above average for high-scoring leads
        $message = $normalized['message'] ?? '';
        $message_length = str_word_count( $message );
        if ( $message_length > ( $patterns['avg_message_length'] ?? 0 ) * 1.2 ) {
            $adjustment += 5;
        }

        return $adjustment;
    }

    /**
     * Get learned feature weights from conversion data
     */
    private function get_learned_weights(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Get converted vs non-converted leads
        $converted = $wpdb->get_results(
            "SELECT form_data, score FROM {$table}
            WHERE status = 'converted'
            ORDER BY created_at DESC
            LIMIT 100",
            ARRAY_A
        );

        $not_converted = $wpdb->get_results(
            "SELECT form_data, score FROM {$table}
            WHERE status IN ('lost', 'disqualified')
            ORDER BY created_at DESC
            LIMIT 100",
            ARRAY_A
        );

        if ( empty( $converted ) ) {
            return [];
        }

        // Analyze what's different between converted and non-converted
        $weights = [
            'budget_weight'          => 1.0,
            'title_weight'           => 1.0,
            'business_email_weight'  => 1.0,
            'message_length_weight'  => 1.0,
            'high_intent_weight'     => 1.0,
        ];

        // Calculate conversion rates by feature presence
        $converted_features = $this->analyze_feature_presence( $converted );
        $not_converted_features = $this->analyze_feature_presence( $not_converted );

        // Adjust weights based on conversion rate differences
        foreach ( $weights as $feature => &$weight ) {
            $feature_key = str_replace( '_weight', '', $feature );
            $converted_rate = $converted_features[ $feature_key ] ?? 0.5;
            $not_converted_rate = $not_converted_features[ $feature_key ] ?? 0.5;

            if ( $converted_rate > $not_converted_rate ) {
                // This feature is more common in converted leads - boost it
                $weight = 1.0 + ( $converted_rate - $not_converted_rate );
            } else {
                // This feature is more common in non-converted leads - reduce it
                $weight = 1.0 - ( $not_converted_rate - $converted_rate ) * 0.5;
            }

            $weight = max( 0.5, min( 2.0, $weight ) ); // Clamp between 0.5 and 2.0
        }

        return $weights;
    }

    /**
     * Analyze feature presence in a set of leads
     */
    private function analyze_feature_presence( array $leads ): array {
        if ( empty( $leads ) ) {
            return [];
        }

        $features = [
            'budget'         => 0,
            'title'          => 0,
            'business_email' => 0,
            'message_length' => 0,
            'high_intent'    => 0,
        ];

        foreach ( $leads as $lead ) {
            $form_data = json_decode( $lead['form_data'], true ) ?: [];
            $normalized = $this->normalize_field_names( $form_data );

            if ( ! empty( $normalized['budget'] ) ) {
                $features['budget']++;
            }

            if ( ! empty( $normalized['job_title'] ?? $normalized['title'] ) ) {
                $features['title']++;
            }

            // We don't have email in form_data, so skip business_email

            $message = $normalized['message'] ?? '';
            if ( str_word_count( $message ) >= 30 ) {
                $features['message_length']++;
            }

            if ( preg_match( '/\b(pric|quote|demo|urgent|asap)\b/i', $message ) ) {
                $features['high_intent']++;
            }
        }

        // Convert to rates
        $total = count( $leads );
        foreach ( $features as $key => &$value ) {
            $value = $value / $total;
        }

        return $features;
    }

    /**
     * Calculate learning adjustment based on learned weights
     */
    private function calculate_learning_adjustment( array $lead_data, array $weights ): float {
        if ( empty( $weights ) ) {
            return 0;
        }

        $adjustment = 0;
        $fields = $lead_data['fields'] ?? $lead_data;
        $normalized = $this->normalize_field_names( $fields );

        // Apply budget weight
        if ( ! empty( $normalized['budget'] ) ) {
            $adjustment += ( $weights['budget_weight'] - 1.0 ) * 10;
        }

        // Apply title weight
        if ( ! empty( $normalized['job_title'] ?? $normalized['title'] ) ) {
            $adjustment += ( $weights['title_weight'] - 1.0 ) * 8;
        }

        // Apply business email weight
        $email = $lead_data['email'] ?? '';
        if ( $this->is_business_email( $email ) ) {
            $adjustment += ( $weights['business_email_weight'] - 1.0 ) * 5;
        }

        // Apply message length weight
        $message = $normalized['message'] ?? '';
        if ( str_word_count( $message ) >= 30 ) {
            $adjustment += ( $weights['message_length_weight'] - 1.0 ) * 8;
        }

        // Apply high intent weight
        if ( preg_match( '/\b(pric|quote|demo|urgent|asap)\b/i', $message ) ) {
            $adjustment += ( $weights['high_intent_weight'] - 1.0 ) * 10;
        }

        return $adjustment;
    }

    /**
     * Record user feedback on a lead (for learning)
     *
     * @param int    $lead_id The lead ID
     * @param string $feedback 'converted', 'lost', 'disqualified', 'correct_score', 'wrong_score'
     * @param array  $details Optional additional details
     */
    public function record_feedback( int $lead_id, string $feedback, array $details = [] ): bool {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $valid_statuses = [ 'converted', 'lost', 'disqualified', 'nurturing', 'contacted' ];

        if ( in_array( $feedback, $valid_statuses, true ) ) {
            $update = [
                'status'     => $feedback,
                'updated_at' => current_time( 'mysql' ),
            ];

            if ( ! empty( $details['notes'] ) ) {
                $update['notes'] = sanitize_textarea_field( $details['notes'] );
            }

            $result = $wpdb->update( $table, $update, [ 'id' => $lead_id ] );

            // Log for learning system
            $this->log_feedback_for_learning( $lead_id, $feedback, $details );

            return $result !== false;
        }

        return false;
    }

    /**
     * Log feedback for learning system analysis
     */
    private function log_feedback_for_learning( int $lead_id, string $feedback, array $details ): void {
        $log = get_option( 'formrank_feedback_log', [] );

        $log[] = [
            'lead_id'   => $lead_id,
            'feedback'  => $feedback,
            'details'   => $details,
            'timestamp' => current_time( 'mysql' ),
        ];

        // Keep last 1000 feedback entries
        $log = array_slice( $log, -1000 );

        update_option( 'formrank_feedback_log', $log );
    }

    /**
     * Get comprehensive conversion analytics for the UI
     *
     * @return array Conversion analytics data
     */
    public function get_conversion_analytics(): array {
        $stats = $this->get_learning_stats();

        return [
            'segments'           => $this->get_segment_conversion_stats(),
            'funnel'             => $this->calculate_funnel_stats(),
            'accuracy'           => $this->calculate_accuracy_metrics(),
            'recent_conversions' => $this->get_recent_conversions( 10 ),
            'learning_status'    => $this->get_learning_status_detail( $stats ),
        ];
    }

    /**
     * Get conversion stats by score segment
     *
     * @return array Segment statistics
     */
    private function get_segment_conversion_stats(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $segments = [
            'hot'     => [ 'min' => 80, 'max' => 100, 'color' => '#ef4444' ],
            'warm'    => [ 'min' => 60, 'max' => 79,  'color' => '#f59e0b' ],
            'neutral' => [ 'min' => 40, 'max' => 59,  'color' => '#8b5cf6' ],
            'cool'    => [ 'min' => 20, 'max' => 39,  'color' => '#6b7280' ],
            'cold'    => [ 'min' => 0,  'max' => 19,  'color' => '#3b82f6' ],
        ];

        $result = [];

        foreach ( $segments as $name => $config ) {
            $total = (int) $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table} WHERE score >= %d AND score <= %d",
                $config['min'],
                $config['max']
            ) );

            $converted = (int) $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table} WHERE score >= %d AND score <= %d AND status = 'converted'",
                $config['min'],
                $config['max']
            ) );

            $result[ $name ] = [
                'range'     => $config['min'] . '-' . $config['max'],
                'color'     => $config['color'],
                'total'     => $total,
                'converted' => $converted,
                'rate'      => $total > 0 ? round( ( $converted / $total ) * 100 ) : 0,
            ];
        }

        return $result;
    }

    /**
     * Calculate conversion funnel statistics
     *
     * @return array Funnel stats by stage
     */
    private function calculate_funnel_stats(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Total leads
        $total = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );

        // Contacted (status is contacted, qualified, or converted)
        $contacted = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$table} WHERE status IN ('contacted', 'qualified', 'converted')"
        );

        // Qualified (score >= 60 OR status is qualified/converted)
        $qualified = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$table} WHERE score >= 60 OR status IN ('qualified', 'converted')"
        );

        // Converted
        $converted = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$table} WHERE status = 'converted'"
        );

        return [
            'new'       => [
                'count'   => $total,
                'percent' => 100,
            ],
            'contacted' => [
                'count'   => $contacted,
                'percent' => $total > 0 ? round( ( $contacted / $total ) * 100 ) : 0,
            ],
            'qualified' => [
                'count'   => $qualified,
                'percent' => $total > 0 ? round( ( $qualified / $total ) * 100 ) : 0,
            ],
            'converted' => [
                'count'   => $converted,
                'percent' => $total > 0 ? round( ( $converted / $total ) * 100 ) : 0,
            ],
        ];
    }

    /**
     * Calculate scoring accuracy metrics
     *
     * @param int $threshold Score threshold for prediction (default 60)
     * @return array Accuracy metrics
     */
    private function calculate_accuracy_metrics( int $threshold = 60 ): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // True Positives: High score AND converted
        $tp = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE score >= %d AND status = 'converted'",
            $threshold
        ) );

        // False Positives: High score BUT not converted (and has final status)
        $fp = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE score >= %d AND status IN ('lost', 'disqualified', 'archived')",
            $threshold
        ) );

        // True Negatives: Low score AND not converted (and has final status)
        $tn = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE score < %d AND status IN ('lost', 'disqualified', 'archived')",
            $threshold
        ) );

        // False Negatives: Low score BUT converted
        $fn = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE score < %d AND status = 'converted'",
            $threshold
        ) );

        $total = $tp + $fp + $tn + $fn;
        $accuracy = $total > 0 ? round( ( ( $tp + $tn ) / $total ) * 100 ) : 0;
        $precision = ( $tp + $fp ) > 0 ? round( ( $tp / ( $tp + $fp ) ) * 100 ) : 0;
        $recall = ( $tp + $fn ) > 0 ? round( ( $tp / ( $tp + $fn ) ) * 100 ) : 0;

        return [
            'accuracy'        => $accuracy,
            'precision'       => $precision,
            'recall'          => $recall,
            'true_positives'  => $tp,
            'false_positives' => $fp,
            'true_negatives'  => $tn,
            'false_negatives' => $fn,
            'threshold'       => $threshold,
            'total_evaluated' => $total,
        ];
    }

    /**
     * Get recent conversions for timeline display
     *
     * @param int $limit Number of conversions to return
     * @return array Recent conversions
     */
    private function get_recent_conversions( int $limit = 10 ): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $results = $wpdb->get_results( $wpdb->prepare(
            "SELECT id, name, email, score, updated_at
             FROM {$table}
             WHERE status = 'converted'
             ORDER BY updated_at DESC
             LIMIT %d",
            $limit
        ), ARRAY_A );

        return array_map( function( $lead ) {
            return [
                'id'           => $lead['id'],
                'name'         => $lead['name'] ?: __( 'Unknown', 'formrank-lead-scoring' ),
                'email'        => $lead['email'],
                'score'        => (int) $lead['score'],
                'segment'      => $this->get_segment_for_score( (int) $lead['score'] ),
                'converted_at' => $lead['updated_at'],
                'time_ago'     => human_time_diff( strtotime( $lead['updated_at'] ) ) . ' ' . __( 'ago', 'formrank-lead-scoring' ),
            ];
        }, $results ?: [] );
    }

    /**
     * Get segment name for a given score
     *
     * @param int $score The lead score
     * @return string Segment name
     */
    public function get_segment_for_score( int $score ): string {
        if ( $score >= 80 ) {
            return 'hot';
        }
        if ( $score >= 60 ) {
            return 'warm';
        }
        if ( $score >= 40 ) {
            return 'neutral';
        }
        if ( $score >= 20 ) {
            return 'cool';
        }
        return 'cold';
    }

    /**
     * Get detailed learning status for UI display
     *
     * @param array $stats Learning stats from get_learning_stats()
     * @return array Learning status details
     */
    private function get_learning_status_detail( array $stats ): array {
        $total_leads = $stats['total_leads'];
        $total_conversions = $stats['total_conversions'];

        // Determine current mode and progress
        if ( $total_leads < self::MIN_LEADS_FOR_LEARNING ) {
            $mode = 'cold_start';
            $mode_label = __( 'Cold Start', 'formrank-lead-scoring' );
            $mode_icon = '🌱';
            $mode_color = '#3b82f6';
            $mode_description = __( 'Building baseline data with rule-based scoring', 'formrank-lead-scoring' );
            $progress_target = self::MIN_LEADS_FOR_LEARNING;
            $progress_current = $total_leads;
            $progress_type = 'leads';
            $next_mode = __( 'Enhanced Rules', 'formrank-lead-scoring' );
        } elseif ( $total_conversions < self::MIN_CONVERSIONS_FOR_ML ) {
            $mode = 'enhanced_rules';
            $mode_label = __( 'Enhanced Rules', 'formrank-lead-scoring' );
            $mode_icon = '📊';
            $mode_color = '#f59e0b';
            $mode_description = __( 'Rules active with pattern detection from high-scoring leads', 'formrank-lead-scoring' );
            $progress_target = self::MIN_CONVERSIONS_FOR_ML;
            $progress_current = $total_conversions;
            $progress_type = 'conversions';
            $next_mode = __( 'Learning Mode', 'formrank-lead-scoring' );
        } else {
            $mode = 'learning_mode';
            $mode_label = __( 'Learning Mode', 'formrank-lead-scoring' );
            $mode_icon = '🧠';
            $mode_color = '#10b981';
            $mode_description = __( 'Full ML-style weight adjustments active based on your conversion data', 'formrank-lead-scoring' );
            $progress_target = null;
            $progress_current = null;
            $progress_type = null;
            $next_mode = null;
        }

        $progress_percent = $progress_target ? min( 100, round( ( $progress_current / $progress_target ) * 100 ) ) : 100;
        $progress_remaining = $progress_target ? max( 0, $progress_target - $progress_current ) : 0;

        return [
            'mode'             => $mode,
            'mode_label'       => $mode_label,
            'mode_icon'        => $mode_icon,
            'mode_color'       => $mode_color,
            'mode_description' => $mode_description,
            'total_leads'      => $total_leads,
            'total_conversions' => $total_conversions,
            'conversion_rate'  => $stats['conversion_rate'],
            'avg_converted_score' => $stats['avg_converted_score'],
            'progress'         => [
                'target'    => $progress_target,
                'current'   => $progress_current,
                'type'      => $progress_type,
                'percent'   => $progress_percent,
                'remaining' => $progress_remaining,
                'next_mode' => $next_mode,
            ],
        ];
    }

    /**
     * Check if segment conversion rates show an anomaly
     * (lower segments converting better than higher ones)
     *
     * @param array $segments Segment stats
     * @return bool True if anomaly detected
     */
    public function has_segment_anomaly( array $segments ): bool {
        $rates = [];
        $order = [ 'hot', 'warm', 'neutral', 'cool', 'cold' ];

        foreach ( $order as $segment ) {
            if ( isset( $segments[ $segment ] ) && $segments[ $segment ]['total'] >= 5 ) {
                $rates[] = $segments[ $segment ]['rate'];
            }
        }

        // Check if rates decrease (with some tolerance)
        for ( $i = 0; $i < count( $rates ) - 1; $i++ ) {
            // If a lower segment has significantly higher rate (>10% difference)
            if ( isset( $rates[ $i + 1 ] ) && $rates[ $i + 1 ] > $rates[ $i ] + 10 ) {
                return true;
            }
        }

        return false;
    }
}
