<?php
/**
 * Enrichment Engine - validates prerequisites, calls the Worker /enrich endpoint,
 * applies quality thresholds, and stores enrichment data via upsert.
 *
 * @package FormRankLS\Core
 */

namespace FormRankLS\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Enrichment_Engine {

    /**
     * API client instance
     *
     * @var API_Client
     */
    private $api_client;

    public function __construct() {
        $this->api_client = new API_Client();
    }

    // =========================================================================
    // PUBLIC METHODS
    // =========================================================================

    /**
     * Enrich a lead — main entry point.
     *
     * Flow:
     *  1. Extend PHP execution time limit for long-running web search
     *  2. Validate enrichment is available (plan, cooldown, lead data)
     *  3. Fetch lead name + email from DB
     *  4. Call Worker /enrich endpoint
     *  5. Apply quality threshold rejection
     *  6. Set first-use notice flag (formrank_enrichment_notice_pending)
     *  7. Map promoted columns
     *  8. Upsert into wp_formrank_enrichments
     *  9. Set 30-second cooldown transient
     *  10. Return enrichment summary
     *
     * @param int $lead_id Lead ID to enrich.
     * @return array Enrichment summary with person, company, confidence, email_type, usage.
     * @throws \Exception On validation failure, API error, or threshold rejection.
     */
    public function enrich_lead(int $lead_id): array {
        global $wpdb;

        // Allow long-running web search (15-45 s typical; shared hosts cap at 30 s by default).
        @set_time_limit(120); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged

        // 1. Validate availability (plan, cooldown, lead data checks).
        $availability = $this->check_enrichment_availability($lead_id);
        if (!$availability['available']) {
            throw new \Exception(esc_html($availability['reason']));
        }

        // 2. Fetch lead from DB.
        $leads_table = $wpdb->prefix . 'formrank_leads';
        $lead        = $wpdb->get_row(
            $wpdb->prepare("SELECT id, name, email FROM {$leads_table} WHERE id = %d", $lead_id),
            ARRAY_A
        );

        if (!$lead) {
            throw new \Exception(esc_html__('Lead not found.', 'formrank-lead-scoring'));
        }

        // 3. Call Worker.
        $response = $this->api_client->enrich($lead['name'], $lead['email']);

        // 4. Extract person/company sub-objects.
        $person  = $response['data']['person']  ?? [];
        $company = $response['data']['company'] ?? [];

        // 5. Quality threshold rejection — stops early before any DB write.
        if ($this->is_below_threshold($response)) {
            throw new \Exception(
                esc_html__('Person not found — try with a different email.', 'formrank-lead-scoring')
            );
        }

        // 6. Set first-use notice flag (only on the very first enrichment for this site).
        if (!get_option('formrank_enrichment_notice_pending')) {
            update_option('formrank_enrichment_notice_pending', 1, false); // autoload=false
        }

        // 7. Map promoted columns (null-coalescent; %s in prepare() stores '' for null).
        $job_title        = $person['job_title']                            ?? '';
        $company_name     = $company['name'] ?? $person['company']          ?? '';
        $seniority_level  = $person['seniority_level']                      ?? '';
        $linkedin_url     = $person['linkedin_url']                         ?? '';
        $company_industry = $company['industry']                            ?? '';
        $company_size     = $company['size']                                ?? '';
        $confidence       = $response['data']['confidence']                 ?? 'low';
        $email_type       = $response['email_type']                         ?? 'personal';
        $enrichment_data  = wp_json_encode($response['data']);
        $now              = current_time('mysql');

        // 8. Upsert — UNIQUE KEY on lead_id prevents duplicates; re-enrichment overwrites.
        $enrichments_table = $wpdb->prefix . 'formrank_enrichments';
        $wpdb->query(
            $wpdb->prepare(
                "INSERT INTO {$enrichments_table}
                    (lead_id, job_title, company_name, seniority_level, linkedin_url,
                     company_industry, company_size, confidence, email_type,
                     enrichment_data, enriched_at, updated_at)
                 VALUES
                    (%d, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)
                 ON DUPLICATE KEY UPDATE
                    job_title        = VALUES(job_title),
                    company_name     = VALUES(company_name),
                    seniority_level  = VALUES(seniority_level),
                    linkedin_url     = VALUES(linkedin_url),
                    company_industry = VALUES(company_industry),
                    company_size     = VALUES(company_size),
                    confidence       = VALUES(confidence),
                    email_type       = VALUES(email_type),
                    enrichment_data  = VALUES(enrichment_data),
                    enriched_at      = VALUES(enriched_at),
                    updated_at       = VALUES(updated_at)",
                $lead_id,
                $job_title,
                $company_name,
                $seniority_level,
                $linkedin_url,
                $company_industry,
                $company_size,
                $confidence,
                $email_type,
                $enrichment_data,
                $now,
                $now
            )
        );

        // 9. 30-second cooldown — prevents rapid re-enrichment of same lead.
        set_transient('formrank_enrich_cooldown_' . $lead_id, 1, 30);

        // 10. Return enrichment summary.
        return [
            'enrichment' => [
                'person'     => $person,
                'company'    => $company,
                'confidence' => $confidence,
                'email_type' => $email_type,
            ],
            'usage'       => $response['usage']      ?? 0,
            'limit'       => $response['limit']       ?? 0,
            'remaining'   => $response['remaining']   ?? 0,
            'enriched_at' => $now,
        ];
    }

    /**
     * Check whether enrichment is available for a lead.
     *
     * Checks in order: plan → cooldown → lead data completeness.
     *
     * @param int $lead_id Lead ID to check.
     * @return array ['available' => bool, 'reason' => string]
     */
    public function check_enrichment_availability(int $lead_id): array {
        global $wpdb;

        // a. Plan check — enrichment is a PRO-only feature.
        if (!$this->is_pro_user()) {
            return [
                'available' => false,
                'reason'    => __('Lead enrichment requires a PRO or Agency plan. Upgrade to access this feature.', 'formrank-lead-scoring'),
            ];
        }

        // b. Cooldown check — prevent rapid re-enrichment.
        if (get_transient('formrank_enrich_cooldown_' . $lead_id)) {
            return [
                'available' => false,
                'reason'    => __('Please wait 30 seconds before re-enriching this lead.', 'formrank-lead-scoring'),
            ];
        }

        // c. Lead data check — must have both name and email.
        $leads_table = $wpdb->prefix . 'formrank_leads';
        $lead        = $wpdb->get_row(
            $wpdb->prepare("SELECT name, email FROM {$leads_table} WHERE id = %d", $lead_id),
            ARRAY_A
        );

        if (!$lead || empty($lead['name']) || empty($lead['email'])) {
            return [
                'available' => false,
                'reason'    => __('Lead must have both name and email to be enriched.', 'formrank-lead-scoring'),
            ];
        }

        return [
            'available' => true,
            'reason'    => '',
        ];
    }

    /**
     * Get total enrichment count (plugin-side, for dashboard display).
     *
     * @return int Number of enriched leads in the enrichments table.
     */
    public function get_enrichment_usage_count(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_enrichments';
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
    }

    /**
     * Get the enrichment row for a given lead (used by Lead_Detail template in Phase 3).
     *
     * @param int $lead_id Lead ID.
     * @return array|null Enrichment row as associative array, or null if not enriched.
     */
    public function get_enrichment_by_lead(int $lead_id): ?array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_enrichments';

        $row = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$table} WHERE lead_id = %d", $lead_id),
            ARRAY_A
        );

        return $row ?: null;
    }

    // =========================================================================
    // PRIVATE METHODS
    // =========================================================================

    /**
     * Determine if a Worker response is below the quality threshold.
     *
     * Rejects when confidence is 'low' AND all three key person-identity fields
     * are empty (job_title, company_name, linkedin_url).
     *
     * @param array $response Full decoded Worker response body.
     * @return bool True if result is below threshold (should be rejected).
     */
    private function is_below_threshold(array $response): bool {
        $person  = $response['data']['person']  ?? [];
        $company = $response['data']['company'] ?? [];

        $confidence   = $response['data']['confidence'] ?? 'low';
        $job_title    = $person['job_title']             ?? '';
        $company_name = $company['name'] ?? $person['company'] ?? '';
        $linkedin_url = $person['linkedin_url']          ?? '';

        if ($confidence !== 'low') {
            return false;
        }

        // Reject only when all three key fields are empty.
        return empty($job_title) && empty($company_name) && empty($linkedin_url);
    }

    /**
     * Check if current user has a PRO or Agency license.
     *
     * Mirrors Scoring_Engine::is_pro_user() exactly.
     *
     * @return bool
     */
    private function is_pro_user(): bool {
        // Check via global helper first (handles Freemius namespace correctly).
        if (function_exists('formrank_ls_is_pro') && formrank_ls_is_pro()) {
            return true;
        }

        // Check Freemius license directly (fr_fs is in FormRankLS namespace).
        if (function_exists('\FormRankLS\fr_fs')) {
            $fs = \FormRankLS\fr_fs();

            // Check if user is on a paid plan.
            if ($fs->is_paying() || $fs->is_trial()) {
                return true;
            }

            // Check specific plan names.
            if ($fs->is_plan('pro') || $fs->is_plan('agency') || $fs->is_plan('business')) {
                return true;
            }
        }

        // Allow filtering for custom license implementations.
        return apply_filters('formrank_ls_is_pro', false);
    }
}
