<?php
/**
 * Historical Data Importer - Import existing leads from form plugins
 *
 * Supports:
 * - WPForms
 * - Gravity Forms
 * - Contact Form 7 (via Flamingo)
 * - Fluent Forms
 * - Formidable Forms
 * - Ninja Forms
 *
 * @package FormRankLS\Core
 */

namespace FormRankLS\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Data_Importer {

    /**
     * Supported form plugins with their detection methods
     */
    private $supported_plugins = [
        'wpforms' => [
            'name'        => 'WPForms',
            'detect'      => 'wpforms',
            'class'       => 'wpforms',
            'has_entries' => true,
        ],
        'gravity' => [
            'name'        => 'Gravity Forms',
            'detect'      => 'GFAPI',
            'class'       => 'GFAPI',
            'has_entries' => true,
        ],
        'cf7' => [
            'name'        => 'Contact Form 7',
            'detect'      => 'wpcf7',
            'class'       => 'WPCF7_ContactForm',
            'has_entries' => false, // CF7 doesn't store entries natively
            'flamingo'    => true,  // But Flamingo addon does
        ],
        'fluent' => [
            'name'        => 'Fluent Forms',
            'detect'      => 'fluentform',
            'class'       => 'FluentForm\\App\\Modules\\Form\\Form',
            'has_entries' => true,
        ],
        'formidable' => [
            'name'        => 'Formidable Forms',
            'detect'      => 'FrmForm',
            'class'       => 'FrmEntry',
            'has_entries' => true,
        ],
        'ninja' => [
            'name'        => 'Ninja Forms',
            'detect'      => 'Ninja_Forms',
            'class'       => 'Ninja_Forms',
            'has_entries' => true,
        ],
    ];

    /**
     * Local scoring engine instance
     */
    private $scoring_engine;

    public function __construct() {
        $this->scoring_engine = new Local_Scoring_Engine();
    }

    /**
     * Detect which form plugins are installed and have data
     *
     * @return array List of detected plugins with entry counts
     */
    public function detect_form_plugins(): array {
        $detected = [];

        foreach ( $this->supported_plugins as $key => $plugin ) {
            $is_active = $this->is_plugin_active( $plugin );

            if ( ! $is_active ) {
                continue;
            }

            $entry_count = $this->get_entry_count( $key );
            $forms = $this->get_available_forms( $key );

            $detected[ $key ] = [
                'name'        => $plugin['name'],
                'slug'        => $key,
                'active'      => true,
                'entry_count' => $entry_count,
                'form_count'  => count( $forms ),
                'forms'       => $forms,
                'can_import'  => $entry_count > 0,
            ];
        }

        return $detected;
    }

    /**
     * Check if a form plugin is active
     */
    private function is_plugin_active( array $plugin ): bool {
        // Check by function
        if ( function_exists( $plugin['detect'] ) ) {
            return true;
        }

        // Check by class
        if ( class_exists( $plugin['class'] ) ) {
            return true;
        }

        // Special case for CF7 with Flamingo
        if ( isset( $plugin['flamingo'] ) && class_exists( 'Flamingo_Inbound_Message' ) ) {
            return true;
        }

        return false;
    }

    /**
     * Get total entry count for a plugin
     */
    private function get_entry_count( string $plugin_key ): int {
        switch ( $plugin_key ) {
            case 'wpforms':
                return $this->get_wpforms_entry_count();

            case 'gravity':
                return $this->get_gravity_entry_count();

            case 'cf7':
                return $this->get_cf7_entry_count();

            case 'fluent':
                return $this->get_fluent_entry_count();

            case 'formidable':
                return $this->get_formidable_entry_count();

            case 'ninja':
                return $this->get_ninja_entry_count();

            default:
                return 0;
        }
    }

    /**
     * Get available forms for a plugin
     */
    private function get_available_forms( string $plugin_key ): array {
        switch ( $plugin_key ) {
            case 'wpforms':
                return $this->get_wpforms_forms();

            case 'gravity':
                return $this->get_gravity_forms();

            case 'cf7':
                return $this->get_cf7_forms();

            case 'fluent':
                return $this->get_fluent_forms();

            case 'formidable':
                return $this->get_formidable_forms();

            case 'ninja':
                return $this->get_ninja_forms();

            default:
                return [];
        }
    }

    // ==================== WPForms ====================

    private function get_wpforms_entry_count(): int {
        if ( ! function_exists( 'wpforms' ) ) {
            return 0;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'wpforms_entries';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
            return 0;
        }

        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );
    }

    private function get_wpforms_forms(): array {
        if ( ! function_exists( 'wpforms' ) ) {
            return [];
        }

        $forms = wpforms()->form->get( '', [
            'orderby' => 'title',
            'order'   => 'ASC',
        ] );

        if ( empty( $forms ) ) {
            return [];
        }

        $form_list = [];
        foreach ( $forms as $form ) {
            $entry_count = $this->get_wpforms_form_entry_count( $form->ID );
            $form_list[] = [
                'id'          => $form->ID,
                'title'       => $form->post_title,
                'entry_count' => $entry_count,
            ];
        }

        return $form_list;
    }

    private function get_wpforms_form_entry_count( int $form_id ): int {
        global $wpdb;
        $table = $wpdb->prefix . 'wpforms_entries';

        return (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE form_id = %d",
            $form_id
        ) );
    }

    /**
     * Import WPForms entries
     */
    public function import_wpforms_entries( array $form_ids = [], int $limit = 0, int $offset = 0 ): array {
        if ( ! function_exists( 'wpforms' ) ) {
            return [ 'success' => false, 'error' => 'WPForms not active' ];
        }

        global $wpdb;
        $entries_table = $wpdb->prefix . 'wpforms_entries';
        $formrank_table = $wpdb->prefix . 'formrank_leads';

        // Build query
        $query = "SELECT * FROM {$entries_table}";
        $params = [];

        if ( ! empty( $form_ids ) ) {
            $placeholders = implode( ',', array_fill( 0, count( $form_ids ), '%d' ) );
            $query .= " WHERE form_id IN ({$placeholders})";
            $params = $form_ids;
        }

        $query .= " ORDER BY date DESC";

        if ( $limit > 0 ) {
            $query .= $wpdb->prepare( " LIMIT %d OFFSET %d", $limit, $offset );
        }

        if ( ! empty( $params ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
            $query = $wpdb->prepare( $query, $params );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $entries = $wpdb->get_results( $query, ARRAY_A );

        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ( $entries as $entry ) {
            // Check if already imported
            $exists = $wpdb->get_var( $wpdb->prepare(
                "SELECT id FROM {$formrank_table}
                WHERE form_plugin = 'wpforms' AND form_id = %d AND entry_id = %d",
                $entry['form_id'],
                $entry['entry_id']
            ) );

            if ( $exists ) {
                $skipped++;
                continue;
            }

            // Parse fields
            $fields = json_decode( $entry['fields'], true ) ?: [];
            $lead_data = $this->extract_wpforms_lead_data( $fields );

            // Insert lead
            $result = $this->insert_imported_lead( $lead_data, [
                'form_plugin' => 'wpforms',
                'form_id'     => $entry['form_id'],
                'entry_id'    => $entry['entry_id'],
                'source_url'  => $entry['referer'] ?? '',
                'ip_address'  => $entry['ip_address'] ?? '',
                'user_agent'  => $entry['user_agent'] ?? '',
                'created_at'  => $entry['date'],
            ] );

            if ( $result ) {
                $imported++;
            } else {
                $errors[] = "Failed to import entry {$entry['entry_id']}";
            }
        }

        return [
            'success'  => true,
            'imported' => $imported,
            'skipped'  => $skipped,
            'errors'   => $errors,
            'total'    => count( $entries ),
        ];
    }

    private function extract_wpforms_lead_data( array $fields ): array {
        $lead_data = [];
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        foreach ( $fields as $field ) {
            $field_name = sanitize_text_field( $field['name'] ?? "field_{$field['id']}" );
            $field_value = $this->get_wpforms_field_value( $field );

            if ( ! empty( $field_value ) ) {
                $lead_data[ $field_name ] = $field_value;
            }

            $field_type = $field['type'] ?? '';
            $field_name_lower = strtolower( $field_name );

            // Email detection
            if ( $field_type === 'email' || strpos( $field_name_lower, 'email' ) !== false ) {
                $email = is_array( $field_value ) ? ( $field_value['primary'] ?? '' ) : $field_value;
            }

            // Name detection - check field type first, then label patterns
            if ( $field_type === 'name' || in_array( $field_name_lower, [ 'name', 'full name', 'your name', 'full_name', 'fullname' ], true ) ) {
                if ( is_array( $field['value_raw'] ?? null ) ) {
                    $name_parts = $field['value_raw'];
                    $name = trim(
                        ( $name_parts['first'] ?? '' ) . ' ' .
                        ( $name_parts['middle'] ?? '' ) . ' ' .
                        ( $name_parts['last'] ?? '' )
                    );
                } else {
                    $name = $field_value;
                }
            } elseif ( strpos( $field_name_lower, 'name' ) !== false && strpos( $field_name_lower, 'company' ) === false && strpos( $field_name_lower, 'business' ) === false ) {
                // Fuzzy name match (but exclude company/business name)
                if ( in_array( $field_name_lower, [ 'first name', 'first_name', 'firstname' ], true ) ) {
                    $first_name = $field_value;
                } elseif ( in_array( $field_name_lower, [ 'last name', 'last_name', 'lastname' ], true ) ) {
                    $last_name = $field_value;
                } elseif ( empty( $name ) ) {
                    $name = $field_value;
                }
            }

            // Company detection
            if ( strpos( $field_name_lower, 'company' ) !== false || strpos( $field_name_lower, 'business' ) !== false ) {
                $company = $field_value;
            }

            // Phone detection
            if ( $field_type === 'phone' || strpos( $field_name_lower, 'phone' ) !== false ) {
                $phone = $field_value;
            }
        }

        // Combine first + last name if no full name was found
        if ( empty( $name ) && ( ! empty( $first_name ) || ! empty( $last_name ) ) ) {
            $name = trim( ( $first_name ?? '' ) . ' ' . ( $last_name ?? '' ) );
        }

        return [
            'fields'  => $lead_data,
            'email'   => $email,
            'name'    => $name,
            'company' => $company,
            'phone'   => $phone,
        ];
    }

    private function get_wpforms_field_value( array $field ): string {
        $value = $field['value'] ?? '';

        if ( is_array( $value ) ) {
            return implode( ', ', array_filter( $value ) );
        }

        return sanitize_textarea_field( $value );
    }

    // ==================== Gravity Forms ====================

    private function get_gravity_entry_count(): int {
        if ( ! class_exists( 'GFAPI' ) ) {
            return 0;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'gf_entry';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
            return 0;
        }

        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table} WHERE status = 'active'" );
    }

    private function get_gravity_forms(): array {
        if ( ! class_exists( 'GFAPI' ) ) {
            return [];
        }

        $forms = \GFAPI::get_forms();

        if ( empty( $forms ) ) {
            return [];
        }

        $form_list = [];
        foreach ( $forms as $form ) {
            $entry_count = \GFAPI::count_entries( $form['id'] );
            $form_list[] = [
                'id'          => $form['id'],
                'title'       => $form['title'],
                'entry_count' => $entry_count,
            ];
        }

        return $form_list;
    }

    /**
     * Import Gravity Forms entries
     */
    public function import_gravity_entries( array $form_ids = [], int $limit = 0, int $offset = 0 ): array {
        if ( ! class_exists( 'GFAPI' ) ) {
            return [ 'success' => false, 'error' => 'Gravity Forms not active' ];
        }

        global $wpdb;
        $formrank_table = $wpdb->prefix . 'formrank_leads';

        $imported = 0;
        $skipped = 0;
        $errors = [];
        $total = 0;

        // If no form IDs specified, get all forms
        if ( empty( $form_ids ) ) {
            $forms = \GFAPI::get_forms();
            $form_ids = array_column( $forms, 'id' );
        }

        foreach ( $form_ids as $form_id ) {
            $form = \GFAPI::get_form( $form_id );
            if ( ! $form ) {
                continue;
            }

            $search_criteria = [ 'status' => 'active' ];
            $paging = $limit > 0 ? [ 'offset' => $offset, 'page_size' => $limit ] : null;

            $entries = \GFAPI::get_entries( $form_id, $search_criteria, null, $paging );

            foreach ( $entries as $entry ) {
                $total++;

                // Check if already imported
                $exists = $wpdb->get_var( $wpdb->prepare(
                    "SELECT id FROM {$formrank_table}
                    WHERE form_plugin = 'gravity' AND form_id = %d AND entry_id = %d",
                    $form_id,
                    $entry['id']
                ) );

                if ( $exists ) {
                    $skipped++;
                    continue;
                }

                // Extract lead data
                $lead_data = $this->extract_gravity_lead_data( $entry, $form );

                // Insert lead
                $result = $this->insert_imported_lead( $lead_data, [
                    'form_plugin' => 'gravity',
                    'form_id'     => $form_id,
                    'entry_id'    => $entry['id'],
                    'source_url'  => $entry['source_url'] ?? '',
                    'ip_address'  => $entry['ip'] ?? '',
                    'user_agent'  => $entry['user_agent'] ?? '',
                    'created_at'  => $entry['date_created'],
                ] );

                if ( $result ) {
                    $imported++;
                } else {
                    $errors[] = "Failed to import entry {$entry['id']} from form {$form_id}";
                }
            }
        }

        return [
            'success'  => true,
            'imported' => $imported,
            'skipped'  => $skipped,
            'errors'   => $errors,
            'total'    => $total,
        ];
    }

    private function extract_gravity_lead_data( array $entry, array $form ): array {
        $lead_data = [];
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        foreach ( $form['fields'] as $field ) {
            $field_id = $field->id;
            $field_label = sanitize_text_field( $field->label );
            $field_type = $field->type;

            $value = $this->get_gravity_field_value( $entry, $field );

            if ( ! empty( $value ) ) {
                $lead_data[ $field_label ] = $value;
            }

            $field_label_lower = strtolower( $field_label );

            if ( $field_type === 'email' || strpos( $field_label_lower, 'email' ) !== false ) {
                $email = $value;
            }

            // Name detection - check field type first, then label patterns
            if ( $field_type === 'name' ) {
                $name = $value;
            } elseif ( in_array( $field_label_lower, [ 'name', 'full name', 'your name', 'full_name', 'fullname' ], true ) ) {
                $name = $value;
            } elseif ( in_array( $field_label_lower, [ 'first name', 'first_name', 'firstname' ], true ) ) {
                $first_name = $value;
            } elseif ( in_array( $field_label_lower, [ 'last name', 'last_name', 'lastname' ], true ) ) {
                $last_name = $value;
            } elseif ( strpos( $field_label_lower, 'name' ) !== false && strpos( $field_label_lower, 'company' ) === false && strpos( $field_label_lower, 'business' ) === false && empty( $name ) ) {
                $name = $value;
            }

            if ( strpos( $field_label_lower, 'company' ) !== false || strpos( $field_label_lower, 'business' ) !== false ) {
                $company = $value;
            }

            if ( $field_type === 'phone' || strpos( $field_label_lower, 'phone' ) !== false ) {
                $phone = $value;
            }
        }

        // Combine first + last name if no full name was found
        if ( empty( $name ) && ( ! empty( $first_name ) || ! empty( $last_name ) ) ) {
            $name = trim( ( $first_name ?? '' ) . ' ' . ( $last_name ?? '' ) );
        }

        return [
            'fields'  => $lead_data,
            'email'   => $email,
            'name'    => $name,
            'company' => $company,
            'phone'   => $phone,
        ];
    }

    private function get_gravity_field_value( array $entry, $field ): string {
        $field_id = $field->id;
        $field_type = $field->type;

        // Handle complex field types
        if ( in_array( $field_type, [ 'name', 'address' ], true ) ) {
            $inputs = $field->inputs ?? [];
            $parts = [];

            foreach ( $inputs as $input ) {
                $value = $entry[ $input['id'] ] ?? '';
                if ( ! empty( $value ) ) {
                    $parts[] = sanitize_text_field( $value );
                }
            }

            return implode( ' ', $parts );
        }

        // Handle checkbox/multiselect
        if ( in_array( $field_type, [ 'checkbox', 'multiselect' ], true ) ) {
            $inputs = $field->inputs ?? [];
            $choices = [];

            foreach ( $inputs as $input ) {
                $value = $entry[ $input['id'] ] ?? '';
                if ( ! empty( $value ) ) {
                    $choices[] = sanitize_text_field( $value );
                }
            }

            return implode( ', ', $choices );
        }

        return sanitize_textarea_field( $entry[ $field_id ] ?? '' );
    }

    // ==================== Contact Form 7 (via Flamingo) ====================

    private function get_cf7_entry_count(): int {
        if ( ! class_exists( 'Flamingo_Inbound_Message' ) ) {
            return 0;
        }

        $args = [
            'post_type'      => 'flamingo_inbound',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ];

        $query = new \WP_Query( $args );
        return $query->found_posts;
    }

    private function get_cf7_forms(): array {
        if ( ! class_exists( 'WPCF7_ContactForm' ) ) {
            return [];
        }

        $forms = \WPCF7_ContactForm::find();

        if ( empty( $forms ) ) {
            return [];
        }

        $form_list = [];
        foreach ( $forms as $form ) {
            $entry_count = $this->get_cf7_form_entry_count( $form->id() );
            $form_list[] = [
                'id'          => $form->id(),
                'title'       => $form->title(),
                'entry_count' => $entry_count,
            ];
        }

        return $form_list;
    }

    private function get_cf7_form_entry_count( int $form_id ): int {
        if ( ! class_exists( 'Flamingo_Inbound_Message' ) ) {
            return 0;
        }

        global $wpdb;

        // Flamingo stores CF7 form ID in post meta
        return (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'flamingo_inbound'
            AND pm.meta_key = '_field_post_id'
            AND pm.meta_value = %d",
            $form_id
        ) );
    }

    /**
     * Import Contact Form 7 entries (via Flamingo)
     */
    public function import_cf7_entries( array $form_ids = [], int $limit = 0, int $offset = 0 ): array {
        if ( ! class_exists( 'Flamingo_Inbound_Message' ) ) {
            return [ 'success' => false, 'error' => 'Flamingo plugin not active (required for CF7 entry storage)' ];
        }

        global $wpdb;
        $formrank_table = $wpdb->prefix . 'formrank_leads';

        $args = [
            'post_type'      => 'flamingo_inbound',
            'posts_per_page' => $limit > 0 ? $limit : -1,
            'offset'         => $offset,
            'orderby'        => 'date',
            'order'          => 'DESC',
        ];

        if ( ! empty( $form_ids ) ) {
            $args['meta_query'] = [
                [
                    'key'     => '_field_post_id',
                    'value'   => $form_ids,
                    'compare' => 'IN',
                ],
            ];
        }

        $query = new \WP_Query( $args );
        $messages = $query->posts;

        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ( $messages as $message ) {
            $form_id = get_post_meta( $message->ID, '_field_post_id', true );

            // Check if already imported
            $exists = $wpdb->get_var( $wpdb->prepare(
                "SELECT id FROM {$formrank_table}
                WHERE form_plugin = 'cf7' AND form_id = %d AND entry_id = %d",
                $form_id,
                $message->ID
            ) );

            if ( $exists ) {
                $skipped++;
                continue;
            }

            // Get message fields
            $meta = get_post_meta( $message->ID );
            $fields = [];

            foreach ( $meta as $key => $value ) {
                if ( strpos( $key, '_field_' ) === 0 && $key !== '_field_post_id' ) {
                    $field_name = str_replace( '_field_', '', $key );
                    $fields[ $field_name ] = is_array( $value ) ? $value[0] : $value;
                }
            }

            $lead_data = $this->extract_cf7_lead_data( $fields );

            $result = $this->insert_imported_lead( $lead_data, [
                'form_plugin' => 'cf7',
                'form_id'     => $form_id,
                'entry_id'    => $message->ID,
                'source_url'  => get_post_meta( $message->ID, '_meta_referer', true ) ?: '',
                'ip_address'  => get_post_meta( $message->ID, '_meta_remote_ip', true ) ?: '',
                'user_agent'  => get_post_meta( $message->ID, '_meta_user_agent', true ) ?: '',
                'created_at'  => $message->post_date,
            ] );

            if ( $result ) {
                $imported++;
            } else {
                $errors[] = "Failed to import CF7 message {$message->ID}";
            }
        }

        return [
            'success'  => true,
            'imported' => $imported,
            'skipped'  => $skipped,
            'errors'   => $errors,
            'total'    => count( $messages ),
        ];
    }

    private function extract_cf7_lead_data( array $fields ): array {
        $lead_data = $fields;
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        foreach ( $fields as $key => $value ) {
            $key_lower = strtolower( $key );

            if ( strpos( $key_lower, 'email' ) !== false || filter_var( $value, FILTER_VALIDATE_EMAIL ) ) {
                $email = $value;
            }

            // Name detection - exact matches first, then fuzzy
            if ( in_array( $key_lower, [ 'name', 'your-name', 'your_name', 'fullname', 'full-name', 'full_name', 'your name', 'full name' ], true ) ) {
                $name = $value;
            } elseif ( in_array( $key_lower, [ 'first-name', 'first_name', 'firstname', 'first name' ], true ) ) {
                $first_name = $value;
            } elseif ( in_array( $key_lower, [ 'last-name', 'last_name', 'lastname', 'last name' ], true ) ) {
                $last_name = $value;
            } elseif ( strpos( $key_lower, 'name' ) !== false && strpos( $key_lower, 'company' ) === false && strpos( $key_lower, 'business' ) === false && empty( $name ) ) {
                $name = $value;
            }

            if ( strpos( $key_lower, 'company' ) !== false || strpos( $key_lower, 'business' ) !== false ) {
                $company = $value;
            }

            if ( strpos( $key_lower, 'phone' ) !== false || strpos( $key_lower, 'tel' ) !== false ) {
                $phone = $value;
            }
        }

        // Combine first + last name if no full name was found
        if ( empty( $name ) && ( ! empty( $first_name ) || ! empty( $last_name ) ) ) {
            $name = trim( ( $first_name ?? '' ) . ' ' . ( $last_name ?? '' ) );
        }

        return [
            'fields'  => $lead_data,
            'email'   => $email,
            'name'    => $name,
            'company' => $company,
            'phone'   => $phone,
        ];
    }

    // ==================== Fluent Forms ====================

    private function get_fluent_entry_count(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'fluentform_submissions';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
            return 0;
        }

        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table} WHERE status = 'read' OR status = 'unread'" );
    }

    private function get_fluent_forms(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'fluentform_forms';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
            return [];
        }

        $forms = $wpdb->get_results( "SELECT id, title FROM {$table} WHERE status = 'published'", ARRAY_A );

        $form_list = [];
        foreach ( $forms as $form ) {
            $entry_count = $this->get_fluent_form_entry_count( $form['id'] );
            $form_list[] = [
                'id'          => $form['id'],
                'title'       => $form['title'],
                'entry_count' => $entry_count,
            ];
        }

        return $form_list;
    }

    private function get_fluent_form_entry_count( int $form_id ): int {
        global $wpdb;
        $table = $wpdb->prefix . 'fluentform_submissions';

        return (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE form_id = %d AND (status = 'read' OR status = 'unread')",
            $form_id
        ) );
    }

    /**
     * Import Fluent Forms entries
     */
    public function import_fluent_entries( array $form_ids = [], int $limit = 0, int $offset = 0 ): array {
        global $wpdb;
        $submissions_table = $wpdb->prefix . 'fluentform_submissions';
        $formrank_table = $wpdb->prefix . 'formrank_leads';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$submissions_table}'" ) !== $submissions_table ) {
            return [ 'success' => false, 'error' => 'Fluent Forms not installed' ];
        }

        $query = "SELECT * FROM {$submissions_table} WHERE (status = 'read' OR status = 'unread')";
        $params = [];

        if ( ! empty( $form_ids ) ) {
            $placeholders = implode( ',', array_fill( 0, count( $form_ids ), '%d' ) );
            $query .= " AND form_id IN ({$placeholders})";
            $params = $form_ids;
        }

        $query .= " ORDER BY created_at DESC";

        if ( $limit > 0 ) {
            $query .= $wpdb->prepare( " LIMIT %d OFFSET %d", $limit, $offset );
        }

        if ( ! empty( $params ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
            $query = $wpdb->prepare( $query, $params );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $entries = $wpdb->get_results( $query, ARRAY_A );

        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ( $entries as $entry ) {
            $exists = $wpdb->get_var( $wpdb->prepare(
                "SELECT id FROM {$formrank_table}
                WHERE form_plugin = 'fluent' AND form_id = %d AND entry_id = %d",
                $entry['form_id'],
                $entry['id']
            ) );

            if ( $exists ) {
                $skipped++;
                continue;
            }

            $fields = json_decode( $entry['response'], true ) ?: [];
            $lead_data = $this->extract_generic_lead_data( $fields );

            $result = $this->insert_imported_lead( $lead_data, [
                'form_plugin' => 'fluent',
                'form_id'     => $entry['form_id'],
                'entry_id'    => $entry['id'],
                'source_url'  => $entry['source_url'] ?? '',
                'ip_address'  => $entry['ip'] ?? '',
                'user_agent'  => $entry['user_agent'] ?? '',
                'created_at'  => $entry['created_at'],
            ] );

            if ( $result ) {
                $imported++;
            } else {
                $errors[] = "Failed to import Fluent entry {$entry['id']}";
            }
        }

        return [
            'success'  => true,
            'imported' => $imported,
            'skipped'  => $skipped,
            'errors'   => $errors,
            'total'    => count( $entries ),
        ];
    }

    // ==================== Formidable Forms ====================

    private function get_formidable_entry_count(): int {
        if ( ! class_exists( 'FrmEntry' ) ) {
            return 0;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'frm_items';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
            return 0;
        }

        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table} WHERE is_draft = 0" );
    }

    private function get_formidable_forms(): array {
        if ( ! class_exists( 'FrmForm' ) ) {
            return [];
        }

        global $wpdb;
        $table = $wpdb->prefix . 'frm_forms';

        $forms = $wpdb->get_results( "SELECT id, name FROM {$table} WHERE status = 'published'", ARRAY_A );

        $form_list = [];
        foreach ( $forms as $form ) {
            $entry_count = $this->get_formidable_form_entry_count( $form['id'] );
            $form_list[] = [
                'id'          => $form['id'],
                'title'       => $form['name'],
                'entry_count' => $entry_count,
            ];
        }

        return $form_list;
    }

    private function get_formidable_form_entry_count( int $form_id ): int {
        global $wpdb;
        $table = $wpdb->prefix . 'frm_items';

        return (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE form_id = %d AND is_draft = 0",
            $form_id
        ) );
    }

    /**
     * Import Formidable Forms entries
     */
    public function import_formidable_entries( array $form_ids = [], int $limit = 0, int $offset = 0 ): array {
        if ( ! class_exists( 'FrmEntry' ) || ! class_exists( 'FrmEntryMeta' ) ) {
            return [ 'success' => false, 'error' => 'Formidable Forms not active' ];
        }

        global $wpdb;
        $items_table = $wpdb->prefix . 'frm_items';
        $meta_table = $wpdb->prefix . 'frm_item_metas';
        $fields_table = $wpdb->prefix . 'frm_fields';
        $formrank_table = $wpdb->prefix . 'formrank_leads';

        $query = "SELECT * FROM {$items_table} WHERE is_draft = 0";
        $params = [];

        if ( ! empty( $form_ids ) ) {
            $placeholders = implode( ',', array_fill( 0, count( $form_ids ), '%d' ) );
            $query .= " AND form_id IN ({$placeholders})";
            $params = $form_ids;
        }

        $query .= " ORDER BY created_at DESC";

        if ( $limit > 0 ) {
            $query .= $wpdb->prepare( " LIMIT %d OFFSET %d", $limit, $offset );
        }

        if ( ! empty( $params ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
            $query = $wpdb->prepare( $query, $params );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $entries = $wpdb->get_results( $query, ARRAY_A );

        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ( $entries as $entry ) {
            $exists = $wpdb->get_var( $wpdb->prepare(
                "SELECT id FROM {$formrank_table}
                WHERE form_plugin = 'formidable' AND form_id = %d AND entry_id = %d",
                $entry['form_id'],
                $entry['id']
            ) );

            if ( $exists ) {
                $skipped++;
                continue;
            }

            // Get entry meta/fields
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
            $metas = $wpdb->get_results( $wpdb->prepare(
                "SELECT m.meta_value, f.name, f.type
                FROM {$meta_table} m
                INNER JOIN {$fields_table} f ON m.field_id = f.id
                WHERE m.item_id = %d",
                $entry['id']
            ), ARRAY_A );

            $fields = [];
            foreach ( $metas as $meta ) {
                $fields[ $meta['name'] ] = $meta['meta_value'];
            }

            $lead_data = $this->extract_generic_lead_data( $fields );

            $result = $this->insert_imported_lead( $lead_data, [
                'form_plugin' => 'formidable',
                'form_id'     => $entry['form_id'],
                'entry_id'    => $entry['id'],
                'source_url'  => '',
                'ip_address'  => $entry['ip'] ?? '',
                'user_agent'  => '',
                'created_at'  => $entry['created_at'],
            ] );

            if ( $result ) {
                $imported++;
            } else {
                $errors[] = "Failed to import Formidable entry {$entry['id']}";
            }
        }

        return [
            'success'  => true,
            'imported' => $imported,
            'skipped'  => $skipped,
            'errors'   => $errors,
            'total'    => count( $entries ),
        ];
    }

    // ==================== Ninja Forms ====================

    private function get_ninja_entry_count(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'nf3_submissions';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
            return 0;
        }

        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );
    }

    private function get_ninja_forms(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'nf3_forms';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$table}'" ) !== $table ) {
            return [];
        }

        $forms = $wpdb->get_results( "SELECT id, title FROM {$table}", ARRAY_A );

        $form_list = [];
        foreach ( $forms as $form ) {
            $entry_count = $this->get_ninja_form_entry_count( $form['id'] );
            $form_list[] = [
                'id'          => $form['id'],
                'title'       => $form['title'],
                'entry_count' => $entry_count,
            ];
        }

        return $form_list;
    }

    private function get_ninja_form_entry_count( int $form_id ): int {
        global $wpdb;
        $table = $wpdb->prefix . 'nf3_submissions';

        return (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE form_id = %d",
            $form_id
        ) );
    }

    /**
     * Import Ninja Forms entries
     */
    public function import_ninja_entries( array $form_ids = [], int $limit = 0, int $offset = 0 ): array {
        global $wpdb;
        $submissions_table = $wpdb->prefix . 'nf3_submissions';
        $values_table = $wpdb->prefix . 'nf3_submission_values';
        $fields_table = $wpdb->prefix . 'nf3_fields';
        $formrank_table = $wpdb->prefix . 'formrank_leads';

        if ( $wpdb->get_var( "SHOW TABLES LIKE '{$submissions_table}'" ) !== $submissions_table ) {
            return [ 'success' => false, 'error' => 'Ninja Forms not installed' ];
        }

        $query = "SELECT * FROM {$submissions_table}";
        $params = [];

        if ( ! empty( $form_ids ) ) {
            $placeholders = implode( ',', array_fill( 0, count( $form_ids ), '%d' ) );
            $query .= " WHERE form_id IN ({$placeholders})";
            $params = $form_ids;
        }

        $query .= " ORDER BY created_at DESC";

        if ( $limit > 0 ) {
            $query .= $wpdb->prepare( " LIMIT %d OFFSET %d", $limit, $offset );
        }

        if ( ! empty( $params ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
            $query = $wpdb->prepare( $query, $params );
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $entries = $wpdb->get_results( $query, ARRAY_A );

        $imported = 0;
        $skipped = 0;
        $errors = [];

        foreach ( $entries as $entry ) {
            $exists = $wpdb->get_var( $wpdb->prepare(
                "SELECT id FROM {$formrank_table}
                WHERE form_plugin = 'ninja' AND form_id = %d AND entry_id = %d",
                $entry['form_id'],
                $entry['id']
            ) );

            if ( $exists ) {
                $skipped++;
                continue;
            }

            // Get submission values
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
            $values = $wpdb->get_results( $wpdb->prepare(
                "SELECT v.value, f.label
                FROM {$values_table} v
                INNER JOIN {$fields_table} f ON v.field_id = f.id
                WHERE v.submission_id = %d",
                $entry['id']
            ), ARRAY_A );

            $fields = [];
            foreach ( $values as $val ) {
                $fields[ $val['label'] ] = $val['value'];
            }

            $lead_data = $this->extract_generic_lead_data( $fields );

            $result = $this->insert_imported_lead( $lead_data, [
                'form_plugin' => 'ninja',
                'form_id'     => $entry['form_id'],
                'entry_id'    => $entry['id'],
                'source_url'  => '',
                'ip_address'  => '',
                'user_agent'  => '',
                'created_at'  => $entry['created_at'],
            ] );

            if ( $result ) {
                $imported++;
            } else {
                $errors[] = "Failed to import Ninja entry {$entry['id']}";
            }
        }

        return [
            'success'  => true,
            'imported' => $imported,
            'skipped'  => $skipped,
            'errors'   => $errors,
            'total'    => count( $entries ),
        ];
    }

    // ==================== Generic Helpers ====================

    /**
     * Extract lead data from generic field array
     */
    private function extract_generic_lead_data( array $fields ): array {
        $lead_data = $fields;
        $email = null;
        $name = null;
        $first_name = null;
        $last_name = null;
        $company = null;
        $phone = null;

        foreach ( $fields as $key => $value ) {
            // Handle Fluent Forms names array
            if ( $key === 'names' && is_array( $value ) ) {
                $name = trim(
                    ( $value['first_name'] ?? '' ) . ' ' .
                    ( $value['middle_name'] ?? '' ) . ' ' .
                    ( $value['last_name'] ?? '' )
                );
                $lead_data[ $key ] = $name;
                continue;
            }

            if ( is_array( $value ) ) {
                $value = implode( ', ', array_filter( $value ) );
            }

            $key_lower = strtolower( str_replace( [ '-', '_' ], ' ', $key ) );

            // Email detection
            if ( strpos( $key_lower, 'email' ) !== false || filter_var( $value, FILTER_VALIDATE_EMAIL ) ) {
                $email = $value;
            }

            // Name detection - exact matches first, then fuzzy (excluding company/business name)
            if ( preg_match( '/\b(full name|your name|fullname)\b/', $key_lower ) ) {
                $name = $value;
            } elseif ( preg_match( '/\b(first name|firstname)\b/', $key_lower ) ) {
                $first_name = $value;
            } elseif ( preg_match( '/\b(last name|lastname|surname)\b/', $key_lower ) ) {
                $last_name = $value;
            } elseif ( preg_match( '/\bname\b/', $key_lower ) && ! preg_match( '/\b(company|business|organization)\b/', $key_lower ) && empty( $name ) ) {
                $name = $value;
            }

            // Company detection
            if ( preg_match( '/\b(company|business|organization)\b/', $key_lower ) ) {
                $company = $value;
            }

            // Phone detection
            if ( preg_match( '/\b(phone|tel|mobile|cell)\b/', $key_lower ) ) {
                $phone = $value;
            }
        }

        // Combine first + last name if no full name was found
        if ( empty( $name ) && ( ! empty( $first_name ) || ! empty( $last_name ) ) ) {
            $name = trim( ( $first_name ?? '' ) . ' ' . ( $last_name ?? '' ) );
        }

        return [
            'fields'  => $lead_data,
            'email'   => $email,
            'name'    => $name,
            'company' => $company,
            'phone'   => $phone,
        ];
    }

    /**
     * Insert an imported lead into FormRank table
     */
    private function insert_imported_lead( array $lead_data, array $meta ): bool {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Prepare insert data
        $insert_data = [
            'form_plugin' => sanitize_text_field( $meta['form_plugin'] ),
            'form_id'     => absint( $meta['form_id'] ),
            'entry_id'    => isset( $meta['entry_id'] ) ? absint( $meta['entry_id'] ) : null,
            'email'       => sanitize_email( $lead_data['email'] ?? '' ),
            'name'        => sanitize_text_field( $lead_data['name'] ?? '' ),
            'company'     => sanitize_text_field( $lead_data['company'] ?? '' ),
            'phone'       => sanitize_text_field( $lead_data['phone'] ?? '' ),
            'form_data'   => wp_json_encode( $lead_data['fields'] ?? [] ),
            'source_url'  => esc_url_raw( $meta['source_url'] ?? '' ),
            'ip_address'  => sanitize_text_field( $meta['ip_address'] ?? '' ),
            'user_agent'  => sanitize_text_field( substr( $meta['user_agent'] ?? '', 0, 512 ) ),
            'status'      => 'imported',
            'created_at'  => $meta['created_at'] ?? current_time( 'mysql' ),
            'updated_at'  => current_time( 'mysql' ),
        ];

        $result = $wpdb->insert( $table, $insert_data );

        if ( ! $result ) {
            return false;
        }

        $lead_id = $wpdb->insert_id;

        // Score the lead
        $this->score_imported_lead( $lead_id, $lead_data );

        return true;
    }

    /**
     * Score an imported lead using local scoring engine
     */
    private function score_imported_lead( int $lead_id, array $lead_data ): void {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $score_result = $this->scoring_engine->score_lead( $lead_data );

        $wpdb->update( $table, [
            'score'           => $score_result['score'],
            'score_label'     => $score_result['label'],
            'score_reasoning' => $score_result['reasoning'],
            'score_factors'   => wp_json_encode( $score_result['factors'] ?? [] ),
            'scored_at'       => current_time( 'mysql' ),
            'updated_at'      => current_time( 'mysql' ),
        ], [ 'id' => $lead_id ] );
    }

    /**
     * Import all entries from a specific plugin
     *
     * @param string $plugin_key Plugin identifier (wpforms, gravity, etc.)
     * @param array  $form_ids   Optional specific form IDs to import
     * @param int    $limit      Max entries to import (0 = all)
     * @param int    $offset     Starting offset
     * @return array Import results
     */
    public function import_from_plugin( string $plugin_key, array $form_ids = [], int $limit = 0, int $offset = 0 ): array {
        switch ( $plugin_key ) {
            case 'wpforms':
                return $this->import_wpforms_entries( $form_ids, $limit, $offset );

            case 'gravity':
                return $this->import_gravity_entries( $form_ids, $limit, $offset );

            case 'cf7':
                return $this->import_cf7_entries( $form_ids, $limit, $offset );

            case 'fluent':
                return $this->import_fluent_entries( $form_ids, $limit, $offset );

            case 'formidable':
                return $this->import_formidable_entries( $form_ids, $limit, $offset );

            case 'ninja':
                return $this->import_ninja_entries( $form_ids, $limit, $offset );

            default:
                return [ 'success' => false, 'error' => 'Unknown plugin: ' . $plugin_key ];
        }
    }

    /**
     * Get import summary - shows what can be imported
     */
    public function get_import_summary(): array {
        $detected = $this->detect_form_plugins();

        $summary = [
            'total_available_entries' => 0,
            'plugins'                 => $detected,
            'already_imported'        => $this->get_imported_count(),
        ];

        foreach ( $detected as $plugin ) {
            $summary['total_available_entries'] += $plugin['entry_count'];
        }

        return $summary;
    }

    /**
     * Get count of already imported leads
     */
    private function get_imported_count(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table} WHERE status = 'imported'" );
    }
}
