<?php
/**
 * Cache Manager - handles caching of API responses
 *
 * @package FormRankLS\Core
 */

namespace FormRankLS\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Cache_Manager {

    private $use_object_cache;

    public function __construct() {
        // Check if persistent object cache is available
        $this->use_object_cache = wp_using_ext_object_cache();
    }

    /**
     * Get cached value
     *
     * @param string $key Cache key
     * @return mixed|false Cached value or false if not found
     */
    public function get(string $key) {
        if ($this->use_object_cache) {
            return wp_cache_get($key, 'formrank');
        }

        return get_transient($key);
    }

    /**
     * Set cache value
     *
     * @param string $key Cache key
     * @param mixed $value Value to cache
     * @param int $expiration Expiration time in seconds
     * @return bool Success
     */
    public function set(string $key, $value, int $expiration = 3600): bool {
        if ($this->use_object_cache) {
            return wp_cache_set($key, $value, 'formrank', $expiration);
        }

        return set_transient($key, $value, $expiration);
    }

    /**
     * Delete cached value
     *
     * @param string $key Cache key
     * @return bool Success
     */
    public function delete(string $key): bool {
        if ($this->use_object_cache) {
            return wp_cache_delete($key, 'formrank');
        }

        return delete_transient($key);
    }

    /**
     * Flush all plugin cache
     */
    public function flush(): bool {
        if ($this->use_object_cache) {
            return wp_cache_flush();
        }

        // For transients, we need to delete them from the database
        global $wpdb;

        $wpdb->query(
            "DELETE FROM {$wpdb->options}
             WHERE option_name LIKE '_transient_formrank_%'
             OR option_name LIKE '_transient_timeout_formrank_%'"
        );

        return true;
    }

    /**
     * Get cache statistics
     */
    public function get_stats(): array {
        global $wpdb;

        $stats = [
            'using_object_cache' => $this->use_object_cache,
            'cached_scores' => 0
        ];

        if (!$this->use_object_cache) {
            $stats['cached_scores'] = (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->options}
                 WHERE option_name LIKE '_transient_formrank_%'"
            );
        }

        return $stats;
    }

    /**
     * Cleanup expired cache entries (called via cron)
     */
    public static function cleanup() {
        // WordPress handles transient cleanup automatically
        // This is for any additional cleanup if needed
        global $wpdb;

        // Delete expired transients
        $wpdb->query(
            "DELETE a, b FROM {$wpdb->options} a, {$wpdb->options} b
             WHERE a.option_name LIKE '_transient_%'
             AND a.option_name NOT LIKE '_transient_timeout_%'
             AND b.option_name = CONCAT('_transient_timeout_', SUBSTRING(a.option_name, 12))
             AND b.option_value < UNIX_TIMESTAMP()"
        );
    }
}

// Register cleanup cron handler
add_action('formrank_cleanup_old_cache', [Cache_Manager::class, 'cleanup']);
